/*
 * kaParse.c -- KQML API parsing routines
 *
 * Copyright (c)  1993, 1994 Enterprise Integration Technologies Corporation,
 * Lockheed Missiles and Space Company, Inc., and Stanford University.
 *
 * Permission to use, copy, modify, distribute, and sell this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name of
 * Enterprise Integration Technologies Corporation and Lockheed Missiles
 * and Space Company may not be used in any advertising or publicity
 * relating to the software without the specific, prior written permission
 * of Enterprise Integration Technologies Corporation and Lockheed Missiles
 * and Space Company.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL ENTERPRISE INTEGRATION TECHNOLOGIES CORPORATION OR
 * LOCKHEED MISSILES AND SPACE COMPANY BE LIABLE FOR ANY SPECIAL,
 * INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY
 * DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY
 * THEORY OF LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 */

/* from code written by Greg Olsen, Stanford University */

#include <kapi.h>
#include <kapi_int.h>

#ifndef MAX_WORD
#define MAX_WORD 80
#endif

#define true 1
#define false 0

#define WHITESPACE " \t\n\r"

/*----------------------*/
/* Forward declarations */
/*----------------------*/
static int GetValue _ANSI_ARGS_((char**, char*));

/*
 *----------------------------------------------------------------------
 *
 * KGetField --
 *
 *	Extracts the field of the given key.  Returns the performative
 *      if the key is NULL.
 *
 * Results:
 *	K_OK = success
 *	K_BAD_KQML = string is not proper kqml
 *      K_NO_FIELD = field not found
 *
 * Side effects:
 *    Malloced space is returned in resultField.
 *
 *
 *----------------------------------------------------------------------
 */

int
#if defined(__STDC__) || defined(__cplusplus)
KGetField(char *inputstr, char *keyStr, char **resultField, long bufSize)
#else
KGetField(inputstr, keyStr, resultField, bufSize) 
     char *inputstr;
     char *keyStr;
     char **resultField;
     long bufSize;
#endif
{
  int  wordlen, found=false;
  char tmpString[K_MAX_MESSAGE_SIZE];
  char *substr;
  char *endstr;
  char tmpVal[K_MAX_MESSAGE_SIZE];
  long size;

  if (inputstr == NULL) goto error;

  strcpy(tmpString,inputstr);
  substr = &tmpString[0];

  /* Check for basics, string should begin with a '(' and
     have a ')' and at least one character in between */
  if (*substr != '('  ||
      (endstr=strrchr(substr,')')) == NULL ||
      ((int) endstr - (int) substr) < 2) goto error;

  /* Replace the last ')' with a null terminator and move to 
     the first character after the opening '(' */
  *endstr = '\0';
  substr++;

  /* Scan past whitespace to first token */
  substr += strspn( substr, WHITESPACE );  

  /* Return or skip past performative */
  wordlen = strcspn(substr,WHITESPACE);
  if (keyStr == NULL) {
      /* If keyStr == NULL, return the performative */
      size = wordlen+1;
      if (*resultField == NULL || size > bufSize) {
	  *resultField = (char *) malloc(size);
      }
      strncpy(*resultField,substr,wordlen);
      (*resultField)[wordlen] = NULL;
      return(K_OK);
  }
  substr += wordlen;
  
  /* Advance to next token */
  substr += strspn( substr, WHITESPACE );

  /* Search for correct keyword - value pair */
  while (*substr != '\0') {

    /* Get Keyword */
    if (*substr != ':') goto error;
    wordlen = strcspn(substr,WHITESPACE);
    if ( !strncasecmp(substr, keyStr, wordlen) ) {
	found = true;
    }
    substr += wordlen;

    /* Advance to next token */
    substr += strspn( substr, WHITESPACE );

    /* Get Value */
    if(GetValue(&substr,tmpVal)) goto error;

    if (found) {
	size = strlen(tmpVal)+1;
	if (*resultField == NULL || size > bufSize) {
	    *resultField = (char *) malloc(size);
	}
	strcpy(*resultField, tmpVal);
	return(K_OK);
    }

    /* Advance to next token */
    substr += strspn( substr, WHITESPACE );

  } /* end while */
  return(K_NO_FIELD);

 error:
  return(K_BAD_KQML);
}
/*
 *----------------------------------------------------------------------
 *
 * KSetField --
 *
 *	Sets the field of the given key.  Sets the performative
 *      if keyStr is NULL.  Creates a new string if inputStr is
 *      NULL.
 *
 * Results:
 *	K_OK = success
 *	K_BAD_KQML = incoming message is bad kqml
 *
 * Side effects:
 *    Malloced space is returned in resultField.
 *
 *
 *----------------------------------------------------------------------
 */

int
#if defined(__STDC__) || defined(__cplusplus)
KSetField(char *inputstr, char *keyStr, char *valStr, 
	  char **resultField, long bufSize)
#else
KSetField(inputstr, keyStr, valStr, resultField, bufSize) 
     char *inputstr;
     char *keyStr;
     char *valStr;
     char **resultField;
     long bufSize;
#endif
{
  int  wordlen, found=false;
  char tmpString[K_MAX_MESSAGE_SIZE];
  char *substr, *newStr = NULL;
  char *endstr;
  char tmpVal[K_MAX_MESSAGE_SIZE];
  long size;

  if (inputstr == NULL) { /* New kqml string */
      size = strlen(valStr) + 4;
      if (keyStr == NULL) {
	  if (*resultField == NULL || size > bufSize) {
	      *resultField = (char *) malloc(size);
	  }
	  newStr = *resultField;
	  *newStr = '(';
	  strcpy(newStr+1, valStr);
	  size = strlen(newStr);
	  newStr[size] = ')';
	  newStr[size+1] = NULL;
      } else {
	  size += strlen(keyStr) + 2 + strlen("no-op");
	  if (*resultField == NULL || size > bufSize) {
	      *resultField = (char *) malloc(size);
	  }
	  newStr = *resultField;
	  *newStr = '(';
	  strcpy(newStr+1, "no-op ");
	  strcat(newStr, keyStr);
	  strcat(newStr, " ");
	  strcat(newStr, valStr);
	  size = strlen(newStr);
	  newStr[size] = ')';
	  newStr[size+1] = NULL;
      }
      *resultField = newStr;
      return K_OK;
  }

  strcpy(tmpString,inputstr);
  substr = &tmpString[0];
  size = strlen(inputstr) + 3 + strlen(valStr) +
      ((keyStr == NULL) ? 0 : strlen(keyStr));
  if (*resultField == NULL || size > bufSize) {
      *resultField = (char *) malloc(size);
  }
  newStr = *resultField;

  strcpy(newStr, "(");

  /* Check for basics, string should begin with a '(' and
     have a ')' and at least one character in between */
  if (*substr != '('  ||
      (endstr=strrchr(substr,')')) == NULL ||
      ((int) endstr - (int) substr) < 2) {
      free(newStr);
      goto error;
  }

  /* Replace the last ')' with a null terminator and move to 
     the first character after the opening '(' */
  *endstr = '\0';
  substr++;

  /* Setting the performative */
  wordlen = strcspn(substr,WHITESPACE);
  if (keyStr == NULL) {
      /* If keyStr == NULL, set the performative */
      strcat(newStr, valStr);
      strcat(newStr, substr + wordlen);
      strcat(newStr, ")");
      *resultField = newStr;
      return(K_OK);
  }
  strncat(newStr, substr, wordlen);
  substr += wordlen;
  
  /* Advance to next token */
  substr += strspn( substr, WHITESPACE );

  /* Search for correct keyword - value pair */
  while (*substr != '\0') {

    /* Get Keyword */
    if (*substr != ':') { 
	free(newStr);
	goto error;
    }
    wordlen = strcspn(substr,WHITESPACE);
    if ( !strncasecmp(substr, keyStr, wordlen) ) {
	found = true;
    }
    strcat(newStr, " ");
    strncat(newStr, substr, wordlen);
    substr += wordlen;

    /* Advance to next token */
    substr += strspn( substr, WHITESPACE );

    /* Get Value */
    if(GetValue(&substr,tmpVal)) {
	free(newStr);
	goto error;
    }

    /* If found copy add the new value and the rest of the inputStr */
    if (found) {
	strcat(newStr, " ");
	strcat(newStr, valStr);
	strcat(newStr, substr);
	strcat(newStr, ")");
	*resultField = newStr;
	return(K_OK);
    } else {
	strcat(newStr, " ");
	strcat(newStr, tmpVal);
    }

    /* Advance to next token */
    substr += strspn( substr, WHITESPACE );

  } /* end while */

  /* If the key is not found, add the key & value at the end */
  if (!found) {
      strcat(newStr, " ");
      strcat(newStr, keyStr);
      strcat(newStr, " ");
      strcat(newStr, valStr);
      strcat(newStr, ")");
  }
  *resultField = newStr;
  return(K_OK);

 error:
  return(K_BAD_KQML);
}
	  
/*
 *----------------------------------------------------------------------
 *
 * KParse --
 *
 *	Parses a KQML string
 *
 * Results:
 *	K_OK = success
 *	K_BAD_KQML = string is not proper kqml
 *
 * Side effects:
 *
 *
 *----------------------------------------------------------------------
 */

int
#if defined(__STDC__) || defined(__cplusplus)
KParse(char *inputstr, char **perform, 
       KeydPair *table, KeyList **lftover)
#else
KParse(inputstr, perform, table, lftover)
     char *inputstr;
     char **perform;
     KeydPair *table;
     KeyList **lftover;
#endif
{
  int wordlen,i;
  char tmpString[K_MAX_MESSAGE_SIZE];
  char *substr;
  char *endstr;
  char tmpPerform[MAX_WORD];
  char tmpKey[MAX_WORD];
  char tmpVal[K_MAX_MESSAGE_SIZE];
  KeydPair *pair;
  KeyList *tmpKList;

  *lftover = NULL;

  if (inputstr == NULL) goto error;

  strcpy(tmpString,inputstr);
  substr = &tmpString[0];

  /* Check for basics, string should begin with a '(' and
     have a ')' and at least one character in between */
  if (*substr != '('  ||
      (endstr=strrchr(substr,')')) == NULL ||
      ((int) endstr - (int) substr) < 2) goto error;

  /* Replace the last ')' with a null terminator and move to 
     the first character after the opening '(' */
  *endstr = '\0';
  substr++;

  /* Scan past whitespace to first token */
  substr += strspn( substr, WHITESPACE );  

  /* Get Performative */
  wordlen = strcspn(substr,WHITESPACE);
  strncpy(tmpPerform,substr,wordlen);
  tmpPerform[wordlen] = '\0';

  (*perform) = strdup(tmpPerform);
  substr += wordlen;

  /* Advance to next token */
  substr += strspn( substr, WHITESPACE );

  /* Get keyword - value pairs */
  while (*substr != '\0') {

    /* Get Keyword */
    if (*substr != ':') goto error;
    wordlen = strcspn(substr,WHITESPACE);
    strncpy(tmpKey,substr,wordlen);
    tmpKey[wordlen] = '\0';
    substr += wordlen;

    /* Advance to next token */
    substr += strspn( substr, WHITESPACE );

    /* Get Value */
    if(GetValue(&substr,tmpVal)) goto error;

    /* Check keyword table.  Insert if located.
       Otherwise add pair to KeyList. */
    i=0;
    pair = &table[i];
    while (pair->keyword!=NULL) {
      if (strcasecmp(pair->keyword,tmpKey)==0) {
	*(pair->value_address) = strdup(tmpVal);
	break;
      }
      pair = &table[++i];
    }

    if (pair->keyword == NULL) {
      tmpKList = (KeyList *) malloc(sizeof(KeyList));
      tmpKList->keyword = strdup(tmpKey);
      tmpKList->value = strdup(tmpVal);
      tmpKList->prev = *lftover;
      *lftover = tmpKList;
    }

    /* Advance to next token */
    substr += strspn( substr, WHITESPACE );

  } /* end while */
  return(K_OK);

 error:
  return(K_BAD_KQML);
}
	  
/*
 *----------------------------------------------------------------------
 *
 * KConstruct --
 *
 *	Construct a KQML string
 *
 * Results:
 *	K_OK = success
 *
 * Side effects:
 *
 *
 *----------------------------------------------------------------------
 */

int
#if defined(__STDC__) || defined(__cplusplus)
KConstruct(char *perform, KeydPair *table, KeyList *others, 
       char **resultStr, long bufSize)
#else
KConstruct(perform, table, others, resultStr, bufSize)
     char *perform;
     KeydPair *table;
     KeyList *others;
     char **resultStr;
     long bufSize;
#endif
{
    int len, i;
    KeyList *cur;
    KeydPair *pair;
    char *ptr;
    long size;

    /* Start with the size of the performative & () & the NULL */
    size = strlen(perform) + 3;

    /* Add the sizes of all keys & fields in the table */
    i=0;
    pair = &table[i];
    while (pair->keyword != NULL) {
	if ( *(pair->value_address) != NULL ) {
	    size += strlen(pair->keyword) + strlen(*(pair->value_address)) + 2;
	}
	pair = &table[++i];
    }

    /* Add the sizes of the keys & fields in the linked list */
    cur = others;
    while ( cur != NULL ) {
	size += strlen(cur->keyword) + strlen(cur->value) + 2;
	cur = cur->prev;
    }

    if (*resultStr == NULL || size > bufSize) {
	*resultStr = (char *) malloc(size);
    }

    ptr = *resultStr;
    *ptr++ = '(';

    /* write the performative */
    len = strlen(perform);
    strncpy(ptr, perform, len);
    ptr += len;

    /* Write items from the table */
    i=0;
    pair = &table[i];
    while (pair->keyword != NULL) {
	if ( *(pair->value_address) != NULL ) {
	    *ptr++ = ' ';
	    len = strlen(pair->keyword);
	    strncpy(ptr, pair->keyword, len);
	    ptr += len;

	    *ptr++ = ' ';
	    len = strlen(*(pair->value_address));
	    strncpy(ptr, *(pair->value_address), len);
	    ptr += len;
	}
	pair = &table[++i];
    }

    /* Write items from the list of other keys */
    cur = others;
    while ( cur != NULL ) {
	*ptr++ = ' ';
	len = strlen(cur->keyword);
	strncpy(ptr, cur->keyword, len);
	ptr += len;

	*ptr++ = ' ';
	len = strlen(cur->value);
	strncpy(ptr, cur->value, len);
	ptr += len;

	cur = cur->prev;
    }

    /* Write the last paren */
    strcpy(ptr, ")");

    return K_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * GetValue --
 *
 *	Get the value associated with a keyword
 *
 * Results:
 *
 * Side effects:
 *
 *
 *----------------------------------------------------------------------
 */
#define DQUOTE '"'    /* to make life easier for formatters */

static int 
#if defined(__STDC__) || defined(__cplusplus)
GetValue(char **substr, char *val)
#else
GetValue(substr, val)
     char **substr, *val;
#endif
{
  char *local_substr = *substr;
  char tmpVal[K_MAX_MESSAGE_SIZE];
  int len, parens, i, cnt, maxlen;

  /* Quoted Expression Case ******************/
  if (*local_substr=='\'' || *local_substr=='`') {
    val[0] = *local_substr;
    val[1] = '\0';
    local_substr++;
    if (GetValue(&local_substr,tmpVal)) return(K_BAD_KQML);
    strcat(val,tmpVal);
  }

  /* Sexp Case *******************************/
  else if (*local_substr=='(') {
    parens = 1;
    val[0] = *local_substr;
    local_substr++;
    i = 1;
    while (parens>0) {
      if (*local_substr=='\0') return(1);

      if (*local_substr==DQUOTE) {
	val[i] = '\0';
	if (GetValue(&local_substr,tmpVal)) return(1);
	strcat(val,tmpVal);
	i += strlen(tmpVal);
      }

      else if (strncmp(local_substr,"{#!",3)==0) {
	val[i] = '\0';
	if (GetValue(&local_substr,tmpVal)) return(1);
	strcat(val,"{#!");
	strcat(val,tmpVal);
	strcat(val,"!#}");
	i += strlen(tmpVal) + 6;
      }
      
      else {
	val[i] = *local_substr;
	if (val[i]=='(')
	  parens++;
	else if (val[i]==')')
	  parens--;
	i++;
	local_substr++;
      }
    }
    val[i] = '\0';
  }

  /* String Case *****************************/
  else if (*local_substr==DQUOTE) {
    val[0] = *local_substr;
    i = 1;
    local_substr++;
    while (*local_substr!=DQUOTE) {

      if(*local_substr=='\0') return(1);

      val[i] = *local_substr;
      if (val[i]=='\\') {
	i++; local_substr++;
	val[i] = *local_substr;
      }
      i++; local_substr++;
    }
    val[i]=DQUOTE;
    local_substr++;
    val[i+1]='\0';
  }

  /* Enclosed Raw ****************************/
  else if (strncmp(local_substr,"{#!",3)==0) {
    char *tmpstr;
    local_substr += 3;
    if  ((tmpstr=strstr(local_substr,"!#}")) != NULL) {
      len = (int) tmpstr - (int) local_substr;
      strncpy(val,local_substr,len);
      local_substr += len + 3;
      val[len]='\0';
    }
    else
      return(1);
  }

  /* Symbol Case *****************************/
  else {
    /* This section was modified to accomodate non-conforming KQML that
       has embedded spaces in value fields */
    len = (int) strcspn(local_substr,WHITESPACE);
    cnt = len+1;
    maxlen =  strlen(local_substr);
    while(cnt < maxlen) {
      if (local_substr[cnt] ==':')
	break;
      else if ((local_substr[cnt] !=' ')  &&
	       (local_substr[cnt] !='\t') &&
	       (local_substr[cnt] !='\n') &&
	       (local_substr[cnt] !='\r')) {
	cnt += (int) strcspn(&local_substr[cnt],WHITESPACE);
	len = cnt;
      }
      cnt++;
    }
    strncpy(val,local_substr,len);
    local_substr += len;
    val[len]='\0';
  }
  *substr = local_substr;
  return(0);
}


/*
 *----------------------------------------------------------------------
 *
 * deleteKeyList --
 *
 *	Delete a key list
 *
 * Results:
 *
 * Side effects:
 *
 *
 *----------------------------------------------------------------------
 */

void
#if defined(__STDC__) || defined(__cplusplus)
deleteKeyList(KeyList *kl)
#else
deleteKeyList(kl)
     KeyList *kl;
#endif
{
  KeyList *tmp_kl = kl, *old_kl;

  while (tmp_kl!=NULL) {
    if (tmp_kl->keyword!=NULL) free(tmp_kl->keyword);
    if (tmp_kl->value!=NULL) free(tmp_kl->value);
    old_kl = tmp_kl;
    tmp_kl = tmp_kl->prev;
    free(old_kl);
  }
}



