/*
 * kaMBUS.c -- KQML API MBUS transport specific routines
 *
 * Copyright (c)  1993, 1994 Enterprise Integration Technologies Corporation
 * and Lockheed Missiles and Space Company, Inc.
 *
 * Permission to use, copy, modify, distribute, and sell this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name of
 * Enterprise Integration Technologies Corporation and Lockheed Missiles
 * and Space Company may not be used in any advertising or publicity
 * relating to the software without the specific, prior written permission
 * of Enterprise Integration Technologies Corporation and Lockheed Missiles
 * and Space Company.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL ENTERPRISE INTEGRATION TECHNOLOGIES CORPORATION OR
 * LOCKHEED MISSILES AND SPACE COMPANY BE LIABLE FOR ANY SPECIAL,
 * INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY
 * DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY
 * THEORY OF LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 */

#ifdef DO_MBUS_TRANS

#include <kapi_int.h>

#include <kasexp.h>
#include <api.h>
#include <kaMBUS.h>

int MBLogLevel = 0;

/*
 *----------------------------------------------------------------------
 *
 * mbusParseURL --
 *
 *	Parses URLs beginning with 'mbus://'
 *
 * Results:
 *	ParsedURL if successful
 *      NULL for failure
 *
 * Side effects:
 *      none
 *
 *----------------------------------------------------------------------
 */

ParsedURL *
#if defined(__STDC__) || defined(__cplusplus)
mbusParseURL(char *urlstring, int ttype)
#else
mbusParseURL(urlstring, ttype)
     char *urlstring;
     int ttype;
#endif
{
  ParsedURL *purl;
  ParsedURL *tmpurl = NULL;
  MBUSInfo *info;
  MBUSInfo *tmpinfo = NULL;

  HashSearch *searchPtr = NULL;
  HashEntry *entry = NULL;

  char  tmpbuffer[MAXURLLENGTH];
  char *hostNport = NULL, *t = NULL;

  tmpbuffer[0] = '\0';

  if (urlstring == NULL || strncasecmp(urlstring,"mbus://", 7))
    return(NULL);

  strcpy(tmpbuffer, &urlstring[7]); /* Skip "mbus://" part */
  
  if ((hostNport = strtok(tmpbuffer, "/")) == NULL ||
      (t = strtok(NULL, "")) == NULL) {
    DPRINT((stderr, "Bad mbus URL: %s\n", urlstring));
    return NULL;
  }

  purl = (ParsedURL *) calloc(1,sizeof(ParsedURL));
  info = (MBUSInfo *) calloc(1,sizeof(MBUSInfo));

  strcpy(info->agentname, t);
  strcpy(info->host, strtok(hostNport, ":"));

  if ((t=strtok(NULL,"")) && (info->port = atoi(t))) {
    /* specified port is fine */
  }
  else {
    info->port = MBUSPORT;
  }

  info->mbconn = 0;

  purl->transtype = ttype;
  purl->state = CLOSED_CON;
  purl->fd = -1;
  purl->hashed = 0;
  purl->alternate = NULL;
  purl->transinfo = (void *) info;

  strcpy(purl->url, urlstring);
  strcpy(purl->name, info->agentname);

  searchPtr = (HashSearch *) calloc(1, sizeof(HashSearch));
  entry = (HashEntry *) FirstHashEntry( &OpenConns, searchPtr );

  while ( entry != NULL ) {
      tmpurl = (ParsedURL *) GetHashValue(entry);
      if ( tmpurl->state == SHARED_CON ) {
	  tmpinfo = (MBUSInfo *) tmpurl->transinfo;
	  if ((strcasecmp(info->host, tmpinfo->host) == 0) &&
	      (info->port == tmpinfo->port)) {

	      info->mbconn = tmpinfo->mbconn;
	      purl->fd     = tmpurl->fd;
	      purl->state  = SHARED_CON;

	      break;
	  } 
      }
      entry = NextHashEntry( searchPtr );
  }

  free(searchPtr);

  return(purl);
}


/*
 *----------------------------------------------------------------------
 *
 * mbusListen --
 *
 *	Listen on mbus url host & port
 *
 * Results:
 *	1 if successful
 *      0 for failure
 *
 * Side effects:
 *      Possible establishment of connection
 *
 *----------------------------------------------------------------------
 */

int
#if defined(__STDC__) || defined(__cplusplus)
mbusListen(ParsedURL *purl)
#else
mbusListen(purl)
     ParsedURL *purl;
#endif
{
  MBUSInfo *info = (MBUSInfo *) purl->transinfo;
  char buf[K_MAX_MESSAGE_SIZE];
  char key[MAXNAMESIZE];

  buf[0] = key[0] = '\0';

  /* connect to the appropriate mbus and register interest in
     your messages */

  sprintf(buf, "(accept \".*\" %s)", info->agentname);

  if ((info->mbconn = MBConnect(info->host, info->port)) &&
      (MBtransmit_Cstring(info->mbconn, buf) == strlen(buf))) {

    purl->state = SHARED_CON;
    purl->fd = MBconnection_fd(info->mbconn);

  } else {
    purl->state = CONNECTION_ERROR;
    return(0);
  }
    
  return(1);
}


/*
 *----------------------------------------------------------------------
 *
 * mbusConnect --
 *
 *	Accept connection on mbus url host & port
 *
 * Results:
 *	parsedURL of new connection
 *      NULL for failure
 *
 * Side effects:
 *
 *----------------------------------------------------------------------
 */

ParsedURL *
#if defined(__STDC__) || defined(__cplusplus)
mbusConnect(ParsedURL *purl, int ttype)
#else
mbusConnect(purl, ttype)
     ParsedURL *purl;
     int ttype;
#endif
{
    ParsedURL* newpurl;
    MBUSInfo*  newinfo;
    MBUSInfo*  info;

    if (strncmp( LOCALNAME, purl->name, strlen( LOCALNAME )) != 0 ) {
	return(purl);
    }

    newpurl = (ParsedURL*) calloc(1, sizeof(ParsedURL));
    newinfo = (MBUSInfo *) calloc(1,sizeof(MBUSInfo));
    info    = (MBUSInfo *) purl->transinfo;

    newpurl->state = SHARED_CON;
    newpurl->transtype = purl->transtype;
    newpurl->fd = purl->fd;
    newpurl->alternate = NULL;
    newpurl->hashed = 0;
    
    newpurl->transinfo = newinfo;
    strcpy( newinfo->host, info->host );
    newinfo->mbconn = info->mbconn;
    newinfo->port   = info->port;

    return(newpurl);
}

/*
 *----------------------------------------------------------------------
 *
 * mbusDisconnect --
 *
 *      Terminate connection on mbus url host & port
 *
 * Results:
 *	returns 1
 *
 * Side effects:
 *      Removal of connection.
 *
 *----------------------------------------------------------------------
 */

int
#if defined(__STDC__) || defined(__cplusplus)
mbusDisconnect(ParsedURL *purl)
#else
mbusDisconnect(purl)
     ParsedURL *purl;
#endif
{
  MBUSInfo *info = (MBUSInfo *) purl->transinfo;

  MBtransmit_Cstring(info->mbconn, "(close)");
  purl->state = CLOSED_CON;
  close(purl->fd);
  purl->fd = -1;
  return(1);
}

/*
 *----------------------------------------------------------------------
 *
 * mbusGet --
 *
 *      Get message on mbus url.
 *
 * Results:
 *	returns message if successful.
 *      NULL for failure.
 *
 * Side effects:
 *
 *----------------------------------------------------------------------
 */

char *
#if defined(__STDC__) || defined(__cplusplus)
mbusGet(ParsedURL *purl, char *str)
#else
mbusGet(purl, str)
     ParsedURL *purl;
     char *str;
#endif
{
  t_sexp MBNextMessage(), MBcar(), MBnthcdr();
  char *MBCstring();
  t_sexp nm, val;
  MBUSInfo *info = (MBUSInfo *) purl->transinfo;

  /* The sleep call was put in a previous version with
     no documentation as to why.

     This appears to have been in there because mbus
     sometimes requires a wait between the time the
     fd is selected and the time a read on it will 
     return valid data.

     sleep(1);
     */

  val = MBNextMessage(info->mbconn);
  if (val == NULL) return NULL;

  nm = MBcar(MBnthcdr(val, 2));
  if (nm == NULL) return NULL;

  KSexpToString(nm, str);
  MBfree(nm);
  return(str);
}


/*
 *----------------------------------------------------------------------
 *
 * mbusSend --
 *
 *      Send message on mbus url
 *
 * Results:
 *	returns 1 if successful
 *      -1 for failure
 *
 * Side effects:
 *      Possible new connection.
 *
 *----------------------------------------------------------------------
 */

int
#if defined(__STDC__) || defined(__cplusplus)
mbusSend(ParsedURL *purl, char *buf)
#else
mbusSend(purl, buf)
     ParsedURL *purl;
     char *buf;
#endif
{
  MBUSInfo *info = (MBUSInfo *) purl->transinfo;
  char tmpbuf[MAXNAMESIZE];
  char buf2[K_MAX_MESSAGE_SIZE];
  int  paren=0,quote=0,bs=0,inquote=0,iqparen=0,nextnormal=0;
  char *foo;
  char *sender   = NULL;
  char *receiver = NULL;

  if ( KGetField( buf, ":sender", &sender, 0 ) ) {
      DPRINT(( stderr, "mbusSend: No sender field - message not sent.\n" ));
      return( -1 );
  }

  if ( KGetField( buf, ":receiver", &receiver, 0 ) ) {
      DPRINT(( stderr, "mbusSend: No receiver field - message not sent.\n" ));
      return( -1 );
  }

  /* This function contains legacy code that apparently fixes a problem
     with mbus's parsing behavior */

  /* first, expand quote characters into (quote ...) to accommodate
     mbus's sexpression format */

#define CANDG  {*foo++ = *buf;continue;}
#define NORMAL(c) ((c!=' ')&&(c!='(')&&(c!='"'))&&(c!=')')

  sprintf(foo=buf2, "(\"%s\" \"%s\" ", sender, receiver );
    
  while (*foo) foo++;
  do {
    if (*buf=='\\') {bs^=1;CANDG}
    if (bs) CANDG
      if (*buf=='"') {quote^=1;CANDG}
    if (quote) CANDG
      if(*buf=='(') {paren++;CANDG}
      else if(*buf==')') paren--;
    if (!inquote)
      {
	if(*buf=='\'')
	  {
	    sprintf(foo,"(quote ");
	    iqparen=paren;
	    foo+=7;
	    inquote=1;
	    if(NORMAL(*(buf+1)))
	      nextnormal=1;
	  }
	else
	  *foo++ = *buf;
      }
    else
      {
	if(!nextnormal) {
	  if((!quote)&&(iqparen==paren))
	    {
	      inquote=0;
	      *foo++=')';
	    }
	}
	else
          {
	    if(!NORMAL(*buf))
	      {
	        inquote=0;
	        *foo++=')';
	        nextnormal=0;
	      }
	  }
	*foo++ = *buf;
      }
  } while(*++buf);

  if(inquote) *foo++=')';
  *foo++ = ')'; /* the close-paren of the triple */
  *foo='\0';

  if ((purl->state == CLOSED_CON) || (purl->state == CONNECTION_ERROR)) {
    /* connect to the appropriate mbus and register interest in
       your messages */

    sprintf(tmpbuf, "(accept \".*\" %s)", sender);
    if ( sender != NULL ) {
	free( sender );
    }

    if ((info->mbconn = MBConnect(info->host, info->port)) &&
	(MBtransmit_Cstring(info->mbconn, tmpbuf) == strlen(tmpbuf))) {

      purl->state = SHARED_CON;
      purl->fd    = MBconnection_fd(info->mbconn);

    } else {
      purl->state = CONNECTION_ERROR;
      return( -1 );
    }
  }

  DPRINT((stderr, "Sending on MBUS: %s\n", buf2));

  return(MBtransmit_Cstring(info->mbconn, buf2));
}


#endif

    
