#ifndef KAHASH_H
#define KAHASH_H

/* kaHash.h 
 *
 * Hash Table code extracted from the Tcl Release by John K. Ousterhout
 *
 */

/*
 * Copyright (c) 1987-1993 The Regents of the University of California.
 * All rights reserved.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 */

/*
 * Definitions that allow this header file to be used either with or
 * without ANSI C features like function prototypes.
 */

#include <kapi_int.h>

#ifdef __cplusplus
#define EXTERN extern "C"
#else
#define EXTERN extern
#endif

/*
 * Macro to use instead of "void" for arguments that must have
 * type "void *" in ANSI C;  maps them to type "char *" in
 * non-ANSI systems.
 */

#ifdef __STDC__
#define VOID void
#else
#define VOID char
#endif

/*
 * Miscellaneous declarations (to allow Tcl to be used stand-alone,
 * without the rest of Sprite).
 */

#ifndef NULL
#define NULL 0
#endif

#  define ckalloc(x) malloc(x)
#  define ckfree(x)  free(x)
#  define ckrealloc(x,y) realloc(x,y)

#ifdef __STDC__
typedef void *ClientData;
#else
typedef int *ClientData;
#endif /* __STDC__ */

#ifdef __cplusplus
struct HashTable;
#endif

/*
 * Structure definition for an entry in a hash table.  No-one outside
 * Tcl should access any of these fields directly;  use the macros
 * defined below.
 */

typedef struct HashEntry {
    struct HashEntry *nextPtr;	/* Pointer to next entry in this
					 * hash bucket, or NULL for end of
					 * chain. */
    struct HashTable *tablePtr;	/* Pointer to table containing entry. */
    struct HashEntry **bucketPtr;	/* Pointer to bucket that points to
					 * first entry in this entry's chain:
					 * used for deleting the entry. */
    ClientData clientData;		/* Application stores something here
					 * with SetHashValue. */
    union {				/* Key has one of these forms: */
	char *oneWordValue;		/* One-word value for key. */
	int words[1];			/* Multiple integer words for key.
					 * The actual size will be as large
					 * as necessary for this table's
					 * keys. */
	char string[4];			/* String for key.  The actual size
					 * will be as large as needed to hold
					 * the key. */
    } key;				/* MUST BE LAST FIELD IN RECORD!! */
} HashEntry;

/*
 * Structure definition for a hash table.  Must be in tcl.h so clients
 * can allocate space for these structures, but clients should never
 * access any fields in this structure.
 */

#define SMALL_HASH_TABLE 4
typedef struct HashTable {
    HashEntry **buckets;		/* Pointer to bucket array.  Each
					 * element points to first entry in
					 * bucket's hash chain, or NULL. */
    HashEntry *staticBuckets[SMALL_HASH_TABLE];
					/* Bucket array used for small tables
					 * (to avoid mallocs and frees). */
    int numBuckets;			/* Total number of buckets allocated
					 * at **bucketPtr. */
    int numEntries;			/* Total number of entries present
					 * in table. */
    int rebuildSize;			/* Enlarge table when numEntries gets
					 * to be this large. */
    int downShift;			/* Shift count used in hashing
					 * function.  Designed to use high-
					 * order bits of randomized keys. */
    int mask;				/* Mask value used in hashing
					 * function. */
    int keyType;			/* Type of keys used in this table. 
					 * It's either STRING_KEYS,
					 * ONE_WORD_KEYS, or an integer
					 * giving the number of ints in a
					 */
    HashEntry *(*findProc) _ANSI_ARGS_((struct HashTable *tablePtr,
	    char *key));
    HashEntry *(*createProc) _ANSI_ARGS_((struct HashTable *tablePtr,
	    char *key, int *newPtr));
} HashTable;

/*
 * Structure definition for information used to keep track of searches
 * through hash tables:
 */

typedef struct HashSearch {
    HashTable *tablePtr;	     /* Table being searched. */
    int nextIndex;		     /* Index of next bucket to be
				      * enumerated after present one. */
    HashEntry *nextEntryPtr;	     /* Next entry to be enumerated in the
				      * the current bucket. */
} HashSearch;

/*
 * Acceptable key types for hash tables:
 */

#define STRING_KEYS	0
#define ONE_WORD_KEYS	1

/*
 * Macros for clients to use to access fields of hash entries:
 */

#define GetHashValue(h) ((h)->clientData)
#define SetHashValue(h, value) ((h)->clientData = (ClientData) (value))
#define GetHashKey(tablePtr, h) \
    ((char *) (((tablePtr)->keyType == ONE_WORD_KEYS) ? (h)->key.oneWordValue \
						: (h)->key.string))

/*
 * Macros to use for clients to use to invoke find and create procedures
 * for hash tables:
 */

#define FindHashEntry(tablePtr, key) \
	(*((tablePtr)->findProc))(tablePtr, key)
#define CreateHashEntry(tablePtr, key, newPtr) \
	(*((tablePtr)->createProc))(tablePtr, key, newPtr)

extern void InitHashTable _ANSI_ARGS_((register HashTable *tablePtr, 
				       int keyType));
extern void DeleteHashEntry _ANSI_ARGS_((HashEntry *entryPtr));
extern void DeleteHashTable _ANSI_ARGS_((register HashTable *tablePtr));
extern HashEntry *NextHashEntry _ANSI_ARGS_((register HashSearch *searchPtr));
extern HashEntry *FirstHashEntry _ANSI_ARGS_((HashTable *tablePtr, 
					      HashSearch *searchPtr));
extern char *HashStats _ANSI_ARGS_((HashTable *tablePtr));

#endif
