/*
 * kaHTTP.c -- KQML API HTTP transport specific routines
 *
 * Copyright (c)  1993, 1994 Enterprise Integration Technologies Corporation
 * and Lockheed Missiles and Space Company, Inc.
 *
 * Permission to use, copy, modify, distribute, and sell this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name of
 * Enterprise Integration Technologies Corporation and Lockheed Missiles
 * and Space Company may not be used in any advertising or publicity
 * relating to the software without the specific, prior written permission
 * of Enterprise Integration Technologies Corporation and Lockheed Missiles
 * and Space Company.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL ENTERPRISE INTEGRATION TECHNOLOGIES CORPORATION OR
 * LOCKHEED MISSILES AND SPACE COMPANY BE LIABLE FOR ANY SPECIAL,
 * INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY
 * DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY
 * THEORY OF LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 */

#ifdef DO_HTTP_TRANS

#include <kapi_int.h>
#include <kaHTTP.h>

/*
 *----------------------------------------------------------------------
 *
 * httpParseURL --
 *
 *	Parses URLs beginning with 'http://'
 *
 * Results:
 *	ParsedURL if successful
 *      NULL for failure
 *
 * Side effects:
 *      none
 *
 *----------------------------------------------------------------------
 */

ParsedURL *
#if defined(__STDC__) || defined(__cplusplus)
httpParseURL(char *urlstring, int ttype)
#else
httpParseURL(urlstring, ttype)
     char *urlstring;
     int ttype;
#endif
{
  ParsedURL *purl;
  HTTPInfo *info;
  char buffer[MAXURLLENGTH], *t = NULL;

  buffer[0] = '\0';

  if (urlstring == NULL || strncasecmp(urlstring,"http://", 7))
    return(NULL);

  purl = (ParsedURL *) calloc(1,sizeof(ParsedURL));
  info = (HTTPInfo *) calloc(1,sizeof(HTTPInfo));
  strcpy(buffer, &urlstring[7]); /* Skip "http://" part */

  strcpy(info->host, strtok(buffer, ":"));

  if ((t=strtok(NULL,"")) && 
      (info->port = atoi(t))) {
    /* specified port is fine */
  }
  else {
    info->port = HTTPPORT;
  }

  info->server = 0; /* default to client */
  purl->fd = -1;
  purl->transtype = ttype;
  purl->state = CLOSED_CON;
  purl->transinfo = (void *) info;
  purl->hashed = 0;

  strcpy(purl->url, urlstring);
  strcpy(purl->name, "none");

  purl->alternate = NULL;

  return(purl);
}


/*
 *----------------------------------------------------------------------
 *
 * httpListen --
 *
 *	Listen on http url host & port
 *
 * Results:
 *	1 if successful
 *      0 for failure
 *
 * Side effects:
 *      Possible establishment of connection
 *
 *----------------------------------------------------------------------
 */

int
#if defined(__STDC__) || defined(__cplusplus)
httpListen(ParsedURL *purl)
#else
httpListen(purl)
     ParsedURL *purl;
#endif
{
  HTTPInfo *info = (HTTPInfo *) purl->transinfo;

  if ((purl->fd = getTCPSocketAddr(info->port)) != -1) {
    if (TCPlisten(purl->fd) != -1) {
      purl->state = LISTENING_CON;
    } else {
      DPRINT((stderr, "httpListen: Can't listen to socket.\n"));
      DERROR( "httpListen" );
      purl->state = CONNECTION_ERROR;
      return(0);
    }
  } else {

    DPRINT((stderr, "httpListen: Can't find socket.\n"));
    
    purl->state = CONNECTION_ERROR;
    return(0);
  }
    
  return(1);
}


/*
 *----------------------------------------------------------------------
 *
 * httpConnect --
 *
 *	Accept connection on http url host & port
 *
 * Results:
 *	parsedURL of new connection
 *      NULL for failure
 *
 * Side effects:
 *      Possible establishment of connection.
 *
 *----------------------------------------------------------------------
 */

ParsedURL *
#if defined(__STDC__) || defined(__cplusplus)
httpConnect(ParsedURL *purl, int ttype)
#else
httpConnect(purl, ttype)
     ParsedURL *purl;
     int ttype;
#endif
{
  int fd;
  ParsedURL *newpURL;
  HTTPInfo *info = NULL;
  char connectionName[MAXNAMESIZE];

  connectionName[0] = '\0';

  fd = TCPaccept(purl->fd);
  if (fd < 0)
    return NULL;

  newpURL = (ParsedURL *) calloc(1,sizeof(ParsedURL));
  info = (HTTPInfo *) calloc(1,sizeof(HTTPInfo));

  newpURL->fd = fd;
  strcpy(info->host, "unknown");
  info->port   = 0;
  info->server = 1;

  newpURL->state = OPEN_CON;
  newpURL->transinfo = info;
  newpURL->transtype = ttype;
  newpURL->hashed = 0;

  return(newpURL);
}


/*
 *----------------------------------------------------------------------
 *
 * httpDisConnect --
 *
 *      Terminate connection on http url host & port
 *
 * Results:
 *	Returns 1
 *
 * Side effects:
 *      Removal of connection.
 *
 *----------------------------------------------------------------------
 */

int
#if defined(__STDC__) || defined(__cplusplus)
httpDisconnect(ParsedURL *purl)
#else
httpDisconnect(purl)
     ParsedURL *purl;
#endif
{  
  struct linger l;
  HTTPInfo *info = NULL;

  if (purl->state == OPEN_CON) {
    l.l_onoff = 0;
    l.l_linger = 0.0;
    setsockopt(purl->fd, SOL_SOCKET, SO_LINGER, (char *) &l, sizeof(l));
    info = (HTTPInfo *) purl->transinfo;

    DPRINT((stderr, "Closing HTTP file descriptor on port %d\n",
	    info->port));

    close(purl->fd);
    purl->fd = -1;
    purl->state = CLOSED_CON;
  }

  return(1);
}


/*
 *----------------------------------------------------------------------
 *
 * httpGet --
 *
 *      Get message on http url.
 *
 * Results:
 *	returns message if successful.
 *      NULL for failure.
 *
 * Side effects:
 *      Possible removal of connection, updating of OpenConns hashtable.
 *
 *----------------------------------------------------------------------
 */

char *
#if defined(__STDC__) || defined(__cplusplus)
httpGet(ParsedURL *purl, char *str)
#else
httpGet(purl, str)
     ParsedURL *purl;
     char *str;
#endif
{
  char buffer[K_MAX_MESSAGE_SIZE];
  char *kqmlstr = NULL;
  HashEntry *entry = NULL;
  HTTPInfo *info = (HTTPInfo *) purl->transinfo;

  buffer[0] = '\0';

  HTTPrecv(buffer, sizeof(buffer), purl->fd);

  /* If acting as a client, need to close connection */
  if (!info->server) {
    purl->state = CLOSED_CON;
    close(purl->fd);
    purl->fd = -1;
    if (( purl->hashed ) &&
	((entry = FindHashEntry(&OpenConns, purl->name)) != NULL)) {
	DeleteHashEntry(entry);
    }
  }

  if (buffer[0] == '\0') 
    return NULL;

  if ((kqmlstr = mime2kqml(buffer)) == NULL)
    return NULL;

  strcpy(str, kqmlstr);
  free(kqmlstr);

  return(str);
}


/*
 *----------------------------------------------------------------------
 *
 * httpSend --
 *
 *      Send message on http url
 *
 * Results:
 *	returns 1 if successful
 *      -1 for failure
 *
 * Side effects:
 *      Possible new connection, updating of OpenConns hashtable.
 *
 *----------------------------------------------------------------------
 */

int
#if defined(__STDC__) || defined(__cplusplus)
httpSend(ParsedURL *purl, char *buf)
#else
httpSend(purl, buf)
     ParsedURL *purl;
     char *buf;
#endif
{
  int status;
  char buffer[K_MAX_MESSAGE_SIZE];
  char *tmpbuf = NULL;
  HTTPInfo *info = (HTTPInfo *) purl->transinfo;
  HashEntry *entry = NULL;

  /* turn the KQML message in buf into a proper HTTP request, e.g.
     by turning the performative name into a method name, a
     :object parameter value into the method's argument, the
     :language parameter value into a Content-type header (KB/KIF
     by default), the :content parm value into the MIME body, and
     other parameter name/value pairs into header lines.  Take
     the server's response and feed it into KGetString's buffers.
     */

  buffer[0] = '\0';

  tmpbuf = kqml2mime(buf);
  if (tmpbuf == NULL) 
    return(-1);

  strcpy(buffer, tmpbuf);
  free(tmpbuf);

  /* Since HTTP queries are transaction-based, need to reconnect to
     the HTTP server everytime a new send is required. */

  if (purl->state != OPEN_CON) {
    purl->fd = TCPconnect(info->host, info->port);

    if (purl->fd == -1)
      return -1;

    purl->state = OPEN_CON;
  }

  status = TCPsend(buffer, purl->fd);

  /* If acting as a server, close connection after send */
  if (info->server) {
    purl->state = CLOSED_CON;
    close(purl->fd);
    purl->fd = -1;
    entry = FindHashEntry(&OpenConns, purl->name);
    DeleteHashEntry(entry);
  }

  if (status == -1) {
    DPRINT((stderr, "httpSend: Could not send message to server.\n"));
    return -1;
  }

  return(1);
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/*       HTTP-based ANS (Agent Name Server) Code                   */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

int
#if defined(__STDC__) || defined(__cplusplus)
httpQueryANS(char* aname, ParsedURL *anspurl, char *returnbuf)
#else
httpQueryANS(aname, anspurl, returnbuf)
     char *aname;        /* the name of the agent to lookup */
     ParsedURL *anspurl; /* the location information for the ANS */
     char *returnbuf;    /* string containing newline delimited urls */
#endif
{
  char buffer[K_MAX_MESSAGE_SIZE];
  HTTPInfo *info = (HTTPInfo *) anspurl->transinfo;

  buffer[0] = '\0';

  /* Make a HTTP connection to ANS. 
     For now, each agent will have its own file whose contents are composed
     of a set of URLs as described in kapi.h.
     
     The files are stored under $HTTPD_HOME/htdocs/agent/ where HTTPD_HOME
     corresponds to the location where the NCSA httpd server was installed.
     */
  if (anspurl->fd > 0) {
    close(anspurl->fd);
  }
 
  if ((anspurl->fd = TCPconnect(info->host, info->port)) == -1) {
      anspurl->state = CLOSED_CON;
      DPRINT((stderr, 
	      "httpQueryANS: ANS server via HTTP is not available.\n"));
      return K_NO_ANS_RESPONSE;
  }
  
  anspurl->state = OPEN_CON;
  
  /* Authentication might be needed */
  
  sprintf(buffer, "GET /agent/%s\n", aname);
  TCPsend(buffer, anspurl->fd);
  
  TCPrecv(buffer, sizeof(buffer), anspurl->fd);
  
  if(buffer[0] == '\0')
    return(K_ERROR);

  strcpy(returnbuf, buffer);
  close(anspurl->fd);
  anspurl->fd = -1;
  anspurl->state = CLOSED_CON;
  
  /* Return status. */
  return(K_OK);
}

/*      
 *----------------------------------------------------------------------
 * Make a HTTP connection to ANS. 
 * For now, each agent will have its own file whose contents are composed
 * of a set of URLs as described in kapi.h.
 *
 * The files are stored under $HTTPD_HOME/htdocs/agent/ where HTTPD_HOME
 * corresponds to the location where the NCSA httpd server was installed.
 *----------------------------------------------------------------------
 */
int
#if defined(__STDC__) || defined(__cplusplus)
httpRegisterANS(ParsedURL *anspurl, char* aname, char *aurl)
#else
httpRegisterANS(anspurl, aname, aurl)
     ParsedURL *anspurl;  /* the name server parsed url */
     char *aname;        /* the name of the agent to lookup */
     char *aurl;         /* the url to add */
#endif
{
  char buffer[K_MAX_MESSAGE_SIZE];
  HTTPInfo *info = (HTTPInfo *) anspurl->transinfo;
  
  if (anspurl->fd > 0) {
    close(anspurl->fd);
  }
 
  if ((anspurl->fd = TCPconnect(info->host, info->port)) == -1) {
    anspurl->state = CLOSED_CON;
    DPRINT((stderr, "ANSLookup: ANS server via HTTP is not available.\n"));
    return K_NO_ANS_RESPONSE;
  }
  
  /* Authentication might be needed */
  sprintf(buffer, "GET /cgi-bin/kapi_register?%s=%s\n", aname, aurl);
  TCPsend(buffer, anspurl->fd);

  bzero(buffer, sizeof(buffer));

  HTTPrecv(buffer, sizeof(buffer), anspurl->fd);
  
  close(anspurl->fd);
  anspurl->fd = -1;
  anspurl->state = CLOSED_CON;  
  
  return(K_OK);
}

/*      */
int
#if defined(__STDC__) || defined(__cplusplus)
httpUnRegisterANS(ParsedURL *anspurl, char* aname, char *aurl)
#else
httpUnRegisterANS(anspurl, aname, aurl)
     ParsedURL *anspurl;  /* the name server parsed url */
     char *aname;         /* the name of the agent to lookup */
     char *aurl;          /* the url to delete */
#endif
{
  char buffer[K_MAX_MESSAGE_SIZE];
  HTTPInfo *info = (HTTPInfo *) anspurl->transinfo;

  if (anspurl->fd > 0) {
    close(anspurl->fd);
  }
 
  if ((anspurl->fd = TCPconnect(info->host, info->port)) == -1) {
    anspurl->state = CLOSED_CON;
    DPRINT((stderr, 
	    "httpUnRegisterANS: ANS server via HTTP is not available.\n"));
    return K_NO_ANS_RESPONSE;
  }
  
  /* Authentication might be needed */
  
  sprintf(buffer, "GET /cgi-bin/kapi_unregister?%s=%s\n", aname, aurl);
  TCPsend(buffer, anspurl->fd);
  
  bzero(buffer, sizeof(buffer));

  HTTPrecv(buffer, sizeof(buffer), anspurl->fd);
  
  close(anspurl->fd);
  anspurl->fd = -1;
  anspurl->state = CLOSED_CON;

  return(K_OK);
}
  

#endif
  
