/*
 * kaGet.c -- KQML API get routines
 *
 * Copyright (c)  1993, 1994 Enterprise Integration Technologies Corporation
 * and Lockheed Missiles and Space Company, Inc.
 *
 * Permission to use, copy, modify, distribute, and sell this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name of
 * Enterprise Integration Technologies Corporation and Lockheed Missiles
 * and Space Company may not be used in any advertising or publicity
 * relating to the software without the specific, prior written permission
 * of Enterprise Integration Technologies Corporation and Lockheed Missiles
 * and Space Company.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL ENTERPRISE INTEGRATION TECHNOLOGIES CORPORATION OR
 * LOCKHEED MISSILES AND SPACE COMPANY BE LIABLE FOR ANY SPECIAL,
 * INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY
 * DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY
 * THEORY OF LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <sys/time.h>
#include <kapi.h>
#include <kapi_int.h>

/*----------------------*/
/* Forward declarations */
/*----------------------*/
static ParsedURL* isActiveURL _ANSI_ARGS_((ParsedURL*, fd_set*, int*));

/*
 *----------------------------------------------------------------------
 *
 * getActiveURL --
 *
 *	Finds a url with data waiting to be read.
 *
 * Results:
 *	ParsedURL if successful
 *      NULL for failure
 *
 * Side effects:
 *
 *
 *----------------------------------------------------------------------
 */

ParsedURL *
#if defined(__STDC__) || defined(__cplusplus)
getActiveURL(int wait)
#else
getActiveURL(wait)
     int wait;
#endif
{
  int numActiveFds = 0;
  char key[MAXNAMESIZE];
  ParsedURL *purl = NULL;
  ParsedURL* activePurl = NULL;
  HashSearch *searchPtr = (HashSearch *) calloc(1, sizeof(HashSearch));
  HashEntry *entry = (HashEntry *) FirstHashEntry(&OpenConns, searchPtr);

  static struct timeval timeout;
  static int numWaitingfds = 0;
  static fd_set ActiveFDs;

  bzero(key, sizeof(key));

  if (numWaitingfds <= 0) {

    FD_ZERO(&ActiveFDs);

    while(entry != NULL) {
      purl = (ParsedURL *) GetHashValue(entry);

      while( purl != NULL ) {
	  if (( purl->fd > 0 ) && ( purl->state != CLOSED_CON ) ) {
	      FD_SET(purl->fd, &ActiveFDs);
	      numActiveFds++;
	  }
	  purl = purl->alternate;
      }

      entry = NextHashEntry(searchPtr);
    }

    /* Call select to find out number of waiting file descriptors. */
    if (numActiveFds > 0) {
      if (wait) { /* Blocking */
	numWaitingfds = select(FD_SETSIZE,&ActiveFDs,NULL,NULL,NULL);
      }
      else { /* Non-blocking */
	timeout.tv_sec = 0.0;
	timeout.tv_usec = 0.0;
	numWaitingfds = select(FD_SETSIZE,&ActiveFDs,NULL, NULL, &timeout);
      };
    }
    else {
      DPRINT((stderr, "getActiveURL: No incoming transports active.\n"));
      free(searchPtr);
      return NULL;
    };
    
    if (numWaitingfds == 0) {
      free(searchPtr);
      return NULL;
    }
    
    if (numWaitingfds < 0) {
      /* At this error condition we should possibly try to find the
	 problem file descriptor and fix things */
      DPRINT((stderr, "getActiveURL: select has issued an error.\n"));
      free(searchPtr);
      return NULL;
    };
  }


  /* Find waiting descriptor and look up or generate ParsedURL */
  entry = (HashEntry *) FirstHashEntry(&OpenConns, searchPtr);

  while((entry != NULL) && (activePurl == NULL) && (numWaitingfds > 0)) {
    purl = (ParsedURL *) GetHashValue(entry);

    while((purl != NULL) && (activePurl == NULL) && (numWaitingfds > 0)) {
	activePurl = isActiveURL( purl, &ActiveFDs, &numWaitingfds );
	purl = purl->alternate;
    }

    entry = NextHashEntry(searchPtr);
  } /* while loop */
  
  free(searchPtr);
  return(activePurl);
}


/*
 *----------------------------------------------------------------------
 *
 * isActiveURL --
 *
 *	Locates a ParsedURL if it's in the set of readable fd's.
 *
 * Results:
 *	ParsedURL if successful.
 *      NULL otherwise.
 *
 * Side effects:
 *      Transport specific connection to socket.
 *
 *----------------------------------------------------------------------
 */
static ParsedURL*
#if defined(__STDC__) || defined(__cplusplus)
isActiveURL( ParsedURL* purl, fd_set *activefds, int *numwaitingfds )
#else
isActiveURL(purl, activefds, numwaitingfds )
     ParsedURL* purl;
     fd_set*    activefds;
     int*       numwaitingfds;
#endif
{
    ParsedURL *(*ConnectFunc) _ANSI_ARGS_((ParsedURL *, int));

    if (FD_ISSET(purl->fd, activefds) != 0) {
	FD_CLR(purl->fd, activefds);
	(*numwaitingfds)--;

	/* Check to see if fd is in listening state */
	if ( purl->state == LISTENING_CON ) {
	    ConnectFunc = Transports[purl->transtype]->Connect;
	    if ((purl = (*ConnectFunc)(purl, purl->transtype)) != NULL) 
	      return( purl );
	    /* Reaching this point is an error condition.  We should possibly
	       try to repair the listening socket.  For now we just ignore 
	       it and continue to look for messages on other URLs */
	    DPRINT((stderr, "Accept Connection failure\n"));

	} else if ( purl->state == SHARED_CON ) {
	    ConnectFunc = Transports[purl->transtype]->Connect;
	    purl = (*ConnectFunc)(purl, purl->transtype); 
	    return( purl );
	    
	} else if ( purl->state == CONNECTIONLESS ) {
	    /* This is an error condition.  CONNECTIONLESS purl->fd's
	       should never be select'ed, since they should be -1. */
	    DPRINT((stderr, "Select'ed connectionless socket.\n"));

	} else {
	    return( purl );  /* OPEN_CON */
	}
    }

    return((ParsedURL*)NULL);
}


/*
 *----------------------------------------------------------------------
 *
 * getFromTransport --
 *
 *	Retrieves data from a given url
 *
 * Results:
 *	message string if successful
 *      NULL for failure
 *
 * Side effects:
 *
 *
 *----------------------------------------------------------------------
 */

char *
#if defined(__STDC__) || defined(__cplusplus)
getFromTransport(ParsedURL *purl, char *str)
#else
getFromTransport(purl, str)
     ParsedURL *purl;
     char *str;            /* a buffer for the returned string */
#endif
{
  char *(*GetFunc) _ANSI_ARGS_((ParsedURL *, char *));

  GetFunc = Transports[purl->transtype]->Get;
  return((*GetFunc)(purl, str));
}


/*
 *----------------------------------------------------------------------
 *
 * getParsedMessage --
 *
 *	Retrieves a parsed KQML message from a given url
 *
 * Results:
 *	parsed message if successful
 *      NULL for failure
 *
 * Side effects:
 *
 *
 *----------------------------------------------------------------------
 */

ParsedKQMLMess *
#if defined(__STDC__) || defined(__cplusplus)
getParsedMessage(ParsedURL *purl)
#else
getParsedMessage(purl)
     ParsedURL *purl;    
#endif
{
  char *sender  = NULL;
  ParsedKQMLMess *pmessage = 
    (ParsedKQMLMess *) calloc(1, sizeof(ParsedKQMLMess));
  char buffer[K_MAX_MESSAGE_SIZE];

  /* The following approach is used to keep non-ANSI compilers happy.
     gcc allows direct initialization of the KTable as in
   KeydPair KTable[] = {
    {":sender", &sender},
    {":receiver", &(pmessage->receiver)},
    {(char *) NULL, (char **) NULL}
  }; */
  KeydPair KTable[3];
  char *key1 = ":sender";
  char *key2 = ":receiver";
  KTable[0].keyword = key1;
  KTable[0].value_address = &sender;
  KTable[1].keyword = key2;
  KTable[1].value_address = &(pmessage->receiver);
  KTable[2].keyword = (char *) NULL;
  KTable[2].value_address = (char **) NULL;

  bzero(buffer, sizeof(buffer));

  if (purl && getFromTransport(purl, buffer)) {
    if ((KParse(buffer, 
		&(pmessage->performative), 
		KTable, 
		&(pmessage->other_pairs)) != K_OK) || 
	*(pmessage->performative) == NULL) {

      /* Parse failure. Trying to receive invalid KQML message */
      DPRINT((stderr,
	      "getParsedMessage: Received unparsable KQML message. Receive Aborted\n"));
      return(NULL);
    }

    if (sender == NULL) {
      sender = strdup( "unknown" );
    }

    pmessage->sender = sender;

    return(pmessage);
  }

  return NULL;
}


/*
 *----------------------------------------------------------------------
 *
 * getString --
 *
 *	Retrieves a KQML string from waiting messages
 *
 * Results:
 *	K_OK = success
 *	K_NO_MSG = no message is available
 *	K_BAD_KQML = incoming message is bad kqml
 *      K_NO_ACTIVE_URL = not active urls to get from
 *
 * Side effects:
 *
 *
 *----------------------------------------------------------------------
 */
int
#if defined(__STDC__) || defined(__cplusplus)
getString(char* str, int wait)
#else
getString(str, wait)
	char  *str;    /* a buffer for the string received */
        int   wait;    /* wait flag */
#endif
{
    int result = K_OK;
    ParsedURL *purl = NULL;
    ParsedKQMLMess *pmessage = NULL;
    char* sender = NULL;

    if ((purl = getActiveURL(wait)) == NULL) {
	return K_NO_ACTIVE_URL;
    }

    if ((pmessage = getParsedMessage(purl)) == NULL) {
	if ( !purl->hashed ) DeletePurl( purl );
	return K_NO_MSG;
    }

    if (ParsedMessToString(pmessage, str) == NULL) {
	if ( !purl->hashed ) DeletePurl( purl );
	DeletePMessage(pmessage);
	return K_BAD_KQML;

    } 

    KGetField( str, ":sender", &sender, 0 );

    if ( sender == NULL ) {
	if ( !purl->hashed ) DeletePurl( purl );	
	DeletePMessage(pmessage);
	return K_NO_SENDER;
    } 

    if ( insertConnectionTable( sender, purl ) == NULL ) {
	if ( !purl->hashed ) DeletePurl( purl );
	/* Couldn't hash but message is still valid - should
	 * a parsed message be returned?  Maybe the agent
	 * can figure out the name from the ANS, so it's a 
	 * probably worthwhile trying ...
	 */

    } else {
	if ( duplicatedName( purl->name, sender ) ) {
	    /* Then we've modified the name because it was a duplicate.
	     * So change the sender field and re-parse the message
	     * with the modified sender field.
	     */
	    free( pmessage->sender );
	    pmessage->sender = strdup( purl->name );
	    ParsedMessToString(pmessage, str);
	}
    }

    if ( sender != NULL ) free( sender );

#if (DEBUG>0)
    printf( "------------- Connections Cache -----------------\n" );
    printf( "-------------------------------------------------\n" );
    printCache( &OpenConns );
    printf( "-------------------------------------------------\n" );
#endif

    return(result);
}


/*
 *----------------------------------------------------------------------
 *
 * KGetString --
 *
 *	Retrieves a KQML string from waiting messages
 *
 * Results:
 *	K_OK = success
 *	K_NO_MSG = no message is available
 *	K_BAD_KQML = incoming message is bad kqml
 *      K_NO_ACTIVE_URL = not active urls to get from
 *
 * Side effects:
 *
 *----------------------------------------------------------------------
 */

int
#if defined(__STDC__) || defined(__cplusplus)
KGetString(char* str)
#else
KGetString(str)
     char *str;    /* a buffer for the string received */
#endif
{
    return( getString( str, 0 ) );
}


/*
 *----------------------------------------------------------------------
 *
 * KGetString_or_Wait --
 *
 *	Blocking version of KGetString
 *
 * Results:
 *	K_OK = success
 *	K_BAD_KQML = incoming message is bad kqml
 *      K_NO_ACTIVE_URL = not active urls to get from
 *	K_ERROR = unknown error occured
 *
 * Side effects:
 *
 *
 *----------------------------------------------------------------------
 */

int
#if defined(__STDC__) || defined(__cplusplus)
KGetString_or_Wait(char* str)
#else
KGetString_or_Wait(str)
	char  *str;    /* a buffer for the string received */
#endif
{
    return( getString( str, 1 ) );
}


/*
 *----------------------------------------------------------------------
 *
 * KPeek --
 *
 *	Check to see if any file descriptor is ready to be read.
 *
 * Results:
 *	K_OK     for TRUE.
 *      K_NO_MSG for FALSE.
 *
 * Side effects:
 *
 *
 *----------------------------------------------------------------------
 */

int
#if defined(__STDC__) || defined(__cplusplus)
KPeek()
#else
KPeek()
#endif
{
  int numActiveFds = 0;
  ParsedURL *purl = NULL;
  HashSearch *searchPtr = (HashSearch *) calloc(1, sizeof(HashSearch));
  HashEntry *entry = (HashEntry *) FirstHashEntry(&OpenConns, searchPtr);

  static struct timeval timeout;
  static int numWaitingfds = 0;
  static fd_set ActiveFDs;

  FD_ZERO(&ActiveFDs);

  while(entry != NULL) {
      purl = (ParsedURL *) GetHashValue(entry);
      FD_SET(purl->fd, &ActiveFDs);
      entry = NextHashEntry(searchPtr);
      numActiveFds++;
  }
  
  /* Call select to find out number of waiting file descriptors. */
  if (numActiveFds > 0) {
      timeout.tv_sec = 0.0;
      timeout.tv_usec = 0.0;
      numWaitingfds = select(FD_SETSIZE,&ActiveFDs,NULL, NULL, &timeout);

  } else {
      DPRINT((stderr, "getActiveURL: No incoming transports active.\n"));
      free(searchPtr);
      return K_NO_MSG;
  };
  
  if (numWaitingfds == 0) {
      free(searchPtr);
      return K_NO_MSG;
  }
    
  if (numWaitingfds < 0) {
      /* At this error condition we should possibly try to find the
	 problem file descriptor and fix things */
      DPRINT((stderr, "getActiveURL: select has issued an error.\n"));
      DERROR( "getActiveURL" );
      free(searchPtr);
      return K_NO_MSG;
  }

  return K_OK;
}


