/*
 * kaConnect.c -- KQML API initialization and connection routines
 *
 * Copyright (c)  1993, 1994 Enterprise Integration Technologies Corporation
 * and Lockheed Missiles and Space Company, Inc.
 *
 * Permission to use, copy, modify, distribute, and sell this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name of
 * Enterprise Integration Technologies Corporation and Lockheed Missiles
 * and Space Company may not be used in any advertising or publicity
 * relating to the software without the specific, prior written permission
 * of Enterprise Integration Technologies Corporation and Lockheed Missiles
 * and Space Company.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL ENTERPRISE INTEGRATION TECHNOLOGIES CORPORATION OR
 * LOCKHEED MISSILES AND SPACE COMPANY BE LIABLE FOR ANY SPECIAL,
 * INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY
 * DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY
 * THEORY OF LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 */

/* kapi.h MUST be included last for ifndef MB_API_H in kapi.h to work*/
#include <kapi.h>
#include <kapi_int.h>

/* Local variables */
static int Listeners = 0;  /* Gensym index */


/*
 *----------------------------------------------------------------------
 *
 * KInit --
 *
 *	KInit initializes global variables and hash tables.
 *
 * Results:
 *	K_OK = success
 *
 * Side effects:
 *
 *
 *----------------------------------------------------------------------
 */

int
#if defined(__STDC__) || defined(__cplusplus)
KInit(void)
#else
KInit()
#endif
{
  InitHashTable(&OpenConns, STRING_KEYS);

  /* ANS disabled at this time */
  gANSpurl = NULL;

  return K_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * KSetUp --
 *
 *	KSetUp initializes global variables and hash tables.  If ANS URL
 *      is given and Lookup_and_Listen flag is set, this function queries
 *      ANS and attempts to listen on returned URLs.  Otherwise KListen
 *      must be called to set up listening.
 *
 * Results:
 *	K_OK = success
 *      K_NO_ANS_RESPONSE = name server did not respond
 *      K_BAD_URL = couldn't parse url to listen on
 *
 * Side effects:
 *
 *
 *----------------------------------------------------------------------
 */

int
#if defined(__STDC__) || defined(__cplusplus)
KSetUp(char* myname, char* ansurl, int Lookup_and_Listen)
#else
KSetUp(myname, ansurl, Lookup_and_Listen)
	char *myname;         /* the transport-name of this agent */
	char *ansurl;         /* a pointer to an agent name server */
	int Lookup_and_Listen; /* flag to determine whether to use the name
				 server to start this agent listening */
#endif
{
  ParsedURL *purl = NULL;
  ParsedURL *nextpurl = NULL;

  InitHashTable(&OpenConns, STRING_KEYS);

  if (ansurl == NULL) {
      gANSpurl = NULL;
      return K_OK; /* ANS disabled in this session */
  }
  else if ((gANSpurl = ParseURL(ansurl)) != NULL) {

    if (Lookup_and_Listen) {
    
      if ((purl = ANSLookup(myname, gANSpurl)) == NULL) {
	DPRINT((stderr, "KSetUp: Could not connect to ANS\n"));
	return(K_NO_ANS_RESPONSE);
      }

      while (purl != NULL) {
	nextpurl = purl->alternate;
	purl->alternate = NULL;

	if (ListenPurl(purl) > 0) {
	    char name[MAXNAMESIZE];
	    sprintf(name, "%s%d", LOCALNAME, Listeners++);
	    strcpy(purl->name, name);
	    if ( insertConnectionTable( name, purl ) == NULL ) {
		DPRINT((stderr, 
			"KSetUp: Couldn't hash URL: %s.\n", purl->url));
		return K_ERROR;
	    }
	} else {
	    DeletePurl(purl);
        }
	purl = nextpurl;
      }
    }
    return K_OK; /* Successful in parsing URL */
  } else {

    DPRINT((stderr, "KSetUp: Could not parse URL.\n"));
    return K_BAD_URL; /* Could not parse URL */
  }
}


/*
 *----------------------------------------------------------------------
 *
 * KSetDefaultANS --
 *
 *	KSetDefaultANS changes the default ans url for name lookup.
 *
 * Results:
 *	K_OK = success
 *      K_BAD_URL = couldn't parse url
 *
 * Side effects:
 *
 *
 *----------------------------------------------------------------------
 */

int
#if defined(__STDC__) || defined(__cplusplus)
KSetDefaultANS(char* ansurl)
#else
KSetDefaultANS(ansurl)
	char *ansurl;         /* a pointer to an agent name server */
#endif
{
  ParsedURL *purl = NULL;

  if (ansurl == NULL) {
      if (gANSpurl != NULL) {
	  DeletePurl(purl);
      }
      gANSpurl = NULL;
      return K_OK; /* ANS disabled in this session */
  }

  purl = ParseURL(ansurl);

  if (purl == NULL) {
    DPRINT((stderr, "KSetDefaultANS: Could not parse URL\n"));
    return K_BAD_URL; /* Could not parse URL */

  } else {
      if (gANSpurl != NULL) {
	  DeletePurl(purl);
      }
      gANSpurl = purl;
  }

  return K_OK; /* Successful in parsing URL */
}


/*
 *----------------------------------------------------------------------
 *
 * KListen --
 *
 *	Listen on a given URL string
 *
 * Results:
 *	K_OK = success
 *      K_BAD_URL = can't parse url
 *      K_ERROR = error setting up listen
 *
 * Side effects:
 *
 *
 *----------------------------------------------------------------------
 */

int
#if defined(__STDC__) || defined(__cplusplus)
KListen(char *url)
#else
KListen(url)
     char *url;     /* will be set up as an incoming connection */
#endif
{
    ParsedURL *purl = NULL;

    if ((purl = ParseURL(url)) != NULL) {
	if (ListenPurl(purl) > 0) {
	    char name[MAXNAMESIZE];
	    sprintf(name, "%s%d", LOCALNAME, Listeners++);
	    strcpy(purl->name, name);
	    if ( insertConnectionTable( name, purl ) == NULL ) {
		return K_ERROR;
	    }
	    return K_OK;
	} else {
	    DeletePurl(purl);
	    return K_ERROR;
	}
    } else {
	return K_BAD_URL;
    }
}


/*
 *----------------------------------------------------------------------
 *
 * ListenPurl --
 *
 *	Listen on a given parsed URL
 *
 * Results:
 *	1 if successful
 *      -1 for failure
 *
 * Side effects:
 *
 *
 *----------------------------------------------------------------------
 */

int
#if defined(__STDC__) || defined(__cplusplus)
ListenPurl(ParsedURL *purl)
#else
ListenPurl(purl)
     ParsedURL *purl;      
#endif
{
  int (*ListenFunc) _ANSI_ARGS_((ParsedURL *));

  ListenFunc = Transports[purl->transtype]->Listen;
  return((*ListenFunc)(purl));
}


/*
 *----------------------------------------------------------------------
 *
 * KDisconnect --
 *
 *	Disconnect all open connections
 *
 * Results:
 *	K_OK = success
 *      K_ERROR = all other errors
 *
 * Side effects:
 *      Hash tables deleted. All open connections are closed.
 *
 *----------------------------------------------------------------------
 */

int
#if defined(__STDC__) || defined(__cplusplus)
KDisconnect()
#else
KDisconnect()
#endif
{
  HashSearch *searchPtr = (HashSearch *) calloc(1, sizeof(HashSearch));
  HashEntry *entry = (HashEntry *) FirstHashEntry(&OpenConns, searchPtr);
  ParsedURL *purl = NULL;
  int (*DisconnectFunc) _ANSI_ARGS_((ParsedURL *));

  while (entry != NULL) {
    purl = (void *) GetHashValue(entry);
    DisconnectFunc = Transports[purl->transtype]->Disconnect;
    (*DisconnectFunc)(purl);
    DeletePurl(purl);
    entry = NextHashEntry(searchPtr);
  }

  ckfree(searchPtr);
  DeleteHashTable(&OpenConns);

  return(K_OK);
}


/*
 *----------------------------------------------------------------------
 *
 * KCacheURL --
 *
 *	Cache the URL for a given agent name
 *
 * Results:
 *	K_OK = success
 *      K_BAD_URL = can't parse url
 *
 * Side effects:
 *
 *      
 *----------------------------------------------------------------------
 */

int
#if defined(__STDC__) || defined(__cplusplus)
KCacheURL(char *aname, char *url)
#else
KCacheURL(aname, url)
     char *aname;               /* the name of the agent */
     char *url;                 /* the url */
#endif
{
  HashEntry *entry;
  ParsedURL *purl;
  int flg;

  if ((purl = ParseURL(url)) == NULL)
    return(K_BAD_URL);

  /* Check to see if agent_name is cached locally */
  entry = CreateHashEntry(&OpenConns, aname, &flg);
  strcpy(purl->name, aname);

  if (!flg) {
      purl->alternate = (ParsedURL *) GetHashValue(entry);
  }

  SetHashValue(entry, purl);
  purl->hashed = 1;

  return(K_OK);
}


/*
 *----------------------------------------------------------------------
 *
 * KUnCacheURL --
 *
 *	UnCache the URLs for a given agent name and url.
 *       - if both non-NULL, only delete the url for agent name 
 *       - if only url == NULL, delete entries for agent name
 *       - if only aname == NULL, delete the url in all entries.
 *            (even though different agents shouldn't share urls)
 *       - if both NULL, delete all entries
 *
 * Results:
 *	K_OK if successful.
 *      K_ERROR if url not found.
 *
 * Side effects:
 *
 *
 *----------------------------------------------------------------------
 */

int
#if defined(__STDC__) || defined(__cplusplus)
KUnCacheURL(char *aname, char *url)
#else
KUnCacheURL(aname, url)
     char *aname;               /* name of the agent */
     char *url;                 /* URL to uncache    */
#endif
{
    if ((aname != NULL) && (url != NULL)) { 
	return( deletePurlCacheEntryURL( aname, url, &OpenConns ) );

    } else if ((aname != NULL) && (url == NULL)) { 
	return( deletePurlCacheEntry( aname, &OpenConns ) );

    } else if ((aname == NULL) && (url != NULL)) { 
	return( deletePurlCacheEntriesURL( url, &OpenConns ) );

    } else if ((aname == NULL) && (url == NULL)) { 
	return( flushPurlCache( &OpenConns ) ); 
    }

    /* Should never get here. */
    return( K_ERROR );
}


/*
 *----------------------------------------------------------------------
 *
 * KGetOpenFd
 *
 *      If first is set,  KGetOpenFd( 1 ), get the first fd from the
 *      OpenConns hash table.
 *
 *      Otherwise, get the next fd from the hash table. Position in the
 *      hash table is maintained internally via the hash table interface.
 *
 * Results:
 *      fd if successful
 *      K_ERROR for failure (or no more hash entries)
 *
 * Side effects:
 *
 *
 *----------------------------------------------------------------------
 */
int
#if defined(__STDC__) || defined(__cplusplus)
KGetOpenFd( int first )
#else
KGetOpenFd( first )
int first;
#endif
{
  ParsedURL *purl = NULL;
  static HashEntry *entry = NULL;
  static HashSearch searchInfo;
 
  if ( first ) {
      bzero( (char *)&searchInfo, sizeof( HashSearch ) );
      entry = (HashEntry *) FirstHashEntry(&OpenConns, &searchInfo);
  } else {
      entry = NextHashEntry(&searchInfo);
  }      
 
  if (entry == NULL) return( K_ERROR );
  
  purl = (ParsedURL*) GetHashValue(entry);
  
  while (( purl->fd == -1 ) || 
	 ( purl->state == CLOSED_CON ) ||
	 ( purl->state == CONNECTIONLESS ) ||
	 ( purl->state == CONNECTION_ERROR ) ) {
      entry = NextHashEntry(&searchInfo);

      if ( entry == NULL ) return( K_ERROR );

      purl = (ParsedURL*) GetHashValue(entry);
  }
 
  return( purl->fd );
}

