/*
 * kaANS.c -- KQML API interface to the Agent Naming Service
 *
 * Copyright (c)  1993, 1994 Enterprise Integration Technologies Corporation
 * and Lockheed Missiles and Space Company, Inc.
 *
 * Permission to use, copy, modify, distribute, and sell this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name of
 * Enterprise Integration Technologies Corporation and Lockheed Missiles
 * and Space Company may not be used in any advertising or publicity
 * relating to the software without the specific, prior written permission
 * of Enterprise Integration Technologies Corporation and Lockheed Missiles
 * and Space Company.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL ENTERPRISE INTEGRATION TECHNOLOGIES CORPORATION OR
 * LOCKHEED MISSILES AND SPACE COMPANY BE LIABLE FOR ANY SPECIAL,
 * INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY
 * DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY
 * THEORY OF LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 */

/* kapi.h MUST be included last for ifndef MB_API_H in kapi.h to work */
#include <kapi.h>
#include <kapi_int.h>


/*
 *----------------------------------------------------------------------
 *
 * AgentLookup --
 *
 *	Given an agent name and a an ANS parsed URL returns a list of
 *      url strings.
 *
 * Results:
 *	K_OK = success
 *      K_BAD_URL = can't parse url
 *      K_NO_ANS_RESPONSE = no response from the name server
 *      K_ERROR = to use AgentLookup you must compile with HTTP support,
 *                or another error occured
 *
 * Side effects:
 *      none
 *
 *----------------------------------------------------------------------
 */

int
#if defined(__STDC__) || defined(__cplusplus)
AgentLookup(char* aname, ParsedURL *anspurl, char *returnbuf)
#else
AgentLookup(aname, anspurl, returnbuf)
     char *aname;        /* the name of the agent to lookup */
     ParsedURL *anspurl; /* the location information for the ANS */
     char *returnbuf;    /* string containing newline delimited urls */
#endif
{
  /* Go to ANS server specified by URL in anspurl and request list of
     transports this agent called aname can handle.
     ANS server will provide an ordered list of URLs. */

  if (anspurl->transtype != TransportID("http")) {
    DPRINT((stderr,"Only HTTP ANS's are currently supported\n"));
    return(K_BAD_URL);
  }

#ifndef DO_HTTP_TRANS
  fprintf(stderr,"Need HTTP support to use ANS\n");
  return(K_ERROR);
#else
  /* Temporary Hook for dedicated HTTP ANS.  Eventually any agent should be
     able to act as ANS */
  return(httpQueryANS(aname, anspurl, returnbuf));
#endif
}


/*
 *----------------------------------------------------------------------
 *
 * KRegister --
 *
 *	Register a URL for agent with ANS given by url string
 *
 * Results:
 *	K_OK = success
 *      K_BAD_URL = can't parse url
 *      K_NO_ANS_RESPONSE = no response from the name server
 *      K_ERROR = to use KRegister you must compile with HTTP support
 *
 * Side effects:
 *      none
 *
 *----------------------------------------------------------------------
 */

int
#if defined(__STDC__) || defined(__cplusplus)
KRegister(char *ansurl, char *aname, char *aurl)
#else
KRegister(ansurl, aname, aurl)
     char *ansurl;              /* Agent Name Server url */
     char *aname;               /* the name of the agent */
     char *aurl;                /* transport protocol URL */
#endif
{
  int status = K_OK;
  ParsedURL *anspurl = ParseURL(ansurl);

  /* Go to ANS server specified by URL in anspurl and request list of
     transports this agent called aname can handle.
     ANS server will provide an ordered list of URLs. */

  if (anspurl == NULL || anspurl->transtype != TransportID("http")) {
    DPRINT((stderr,"Only HTTP ANS's are currently supported\n"));
    return(K_BAD_URL);
  }

#ifndef DO_HTTP_TRANS
  fprintf(stderr,"Need HTTP support to use ANS\n");
  return(K_ERROR);
#else
  status = httpRegisterANS(anspurl, aname, aurl);

  DeletePurl(anspurl);
  return(status);
#endif
}


/*
 *----------------------------------------------------------------------
 *
 * KUnRegister --
 *
 *	UnRegister a URL for agent with ANS given by url string
 *
 * Results:
 *	K_OK = success
 *      K_BAD_URL = can't parse url
 *      K_NO_ANS_RESPONSE = no response from the name server
 *      K_ERROR = to use KRegister you must compile with HTTP support
 *
 * Side effects:
 *      none
 *
 *----------------------------------------------------------------------
 */

int
#if defined(__STDC__) || defined(__cplusplus)
KUnRegister(char *ansurl, char *aname, char *aurl)
#else
KUnRegister(ansurl, aname, aurl)
     char *ansurl;              /* Agent Name Server url */
     char *aname;               /* the name of the agent */
     char *aurl;                /* transport protocol URL */
#endif
{
  int status = K_OK;
  ParsedURL *anspurl = ParseURL(ansurl);

  /* Go to ANS server specified by URL in anspurl and request list of
     transports this agent called aname can handle.
     ANS server will provide an ordered list of URLs. */

  if (anspurl == NULL || anspurl->transtype != TransportID("http")) {
    DPRINT((stderr,"Only HTTP ANS's are currently supported\n"));
    return(K_BAD_URL);
  }

#ifndef DO_HTTP_TRANS
  fprintf(stderr,"Need HTTP support to use ANS\n");
  return(K_ERROR);
#else
  status = httpUnRegisterANS(anspurl, aname, aurl);
  return(status);
#endif
}


/*
 *----------------------------------------------------------------------
 *
 * ANSLookup --
 *
 *	Get a parsed url for a given agent name and ANS parsed url.
 *
 * Results:
 *	ParsedURL if successful
 *      NULL for failure
 *
 * Side effects:
 *      none
 *
 *----------------------------------------------------------------------
 */

ParsedURL *
#if defined(__STDC__) || defined(__cplusplus)
ANSLookup(char *aname, ParsedURL *anspurl)
#else
ANSLookup(aname, anspurl)
     char *aname;               /* the name of the agent */
     ParsedURL *anspurl;        /* Agent Name Server parsed url */
#endif
{
  char URLstrings[MAXURLLENGTH*MAXANSURLS];
  ParsedURL *purl, *purl1, *purl2;
  int  start = 0, count = 0;
  char buffer[MAXURLLENGTH];

  if (AgentLookup(aname, anspurl, URLstrings) != K_OK) {
      return NULL;
  }
  count = strcspn(URLstrings, "\n");
  strncpy(buffer, URLstrings, count);
  buffer[count] = '\0';
  start += count;

  if ((purl = ParseURL(buffer)) == NULL) 
    return(NULL);

  /* Construct Linked list of alternate urls */
  purl1 = purl;
  strcpy( purl1->name, aname );

  while (*(URLstrings+start) == '\n') {
    start++;
    if ((count = strcspn(URLstrings+start,"\n")) == 0) break;
    strncpy(buffer, URLstrings+start, count);
    buffer[count] = '\0';
    purl2 = ParseURL(buffer);
    if ( purl2 != NULL ) {
	strcpy( purl2->name, aname );
	purl1->alternate = purl2;
	purl1 = purl2;
    }
    start += count;
  }

  return(purl);
}


/*
 *----------------------------------------------------------------------
 *
 * KLookup --
 *
 *	Get a list of url strings for a given agent name and ANS url string.
 *
 * Results:
 *	K_OK = success
 *      K_BAD_URL = couldn't parse url to look up, or ans url
 *      K_NO_ANS_RESPONSE = no response from the name server
 *      K_ERROR = to use KLookup you must compile with HTTP support,
 *                or another error occured
 *
 * Side effects:
 *      none
 *
 *----------------------------------------------------------------------
 */

int
#if defined(__STDC__) || defined(__cplusplus)
KLookup(char *ansurl, char *aname, char *returnbuf)
#else
KLookup(ansurl, aname, returnbuf)
     char *ansurl;        /* Agent Name Server parsed url */
     char *aname;               /* the name of the agent */
     char *returnbuf;           /* the return parsed url */
#endif
{
  int result;
  ParsedURL *anspurl;

  if ((anspurl = ParseURL(ansurl)) == NULL)
    return(K_BAD_URL);

  result = AgentLookup(aname, anspurl, returnbuf);

  DeletePurl(anspurl);

  return(result);
}


