;;  CogTool Copyright Notice and Distribution Terms
;;  CogTool 1.0, Copyright (c) 2005-2006 Carnegie Mellon University
;;  This software is distributed under the terms of the FSF Lesser
;;  Gnu Public License (see LGPL.txt).
;; 
;;  CogTool is free software; you can redistribute it and/or modify
;;  it under the terms of the GNU Lesser General Public License as published by
;;  the Free Software Foundation; either version 2.1 of the License, or
;;  (at your option) any later version.
;; 
;;  CogTool is distributed in the hope that it will be useful,
;;  but WITHOUT ANY WARRANTY; without even the implied warranty of
;;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;  GNU Lesser General Public License for more details.
;; 
;;  You should have received a copy of the GNU Lesser General Public License
;;  along with CogTool; if not, write to the Free Software
;;  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
;;
;;  
;;  CogTool makes use of several third-party components, with the 
;;  following notices:
;;
;;  Eclipse SWT 3.1
;;  Eclipse GEF Draw2D 3.1
;;
;;  Unless otherwise indicated, all Content made available by the Eclipse 
;;  Foundation is provided to you under the terms and conditions of the Eclipse 
;;  Public License Version 1.0 ("EPL"). A copy of the EPL is provided with this 
;;  Content and is also available at http://www.eclipse.org/legal/epl-v10.html.
;;
;;  clisp 2.35
;;
;;  Copyright (c) 1992-2004 Bruno Haible and Michael Stoll
;;  This software is distributed under the terms of the FSF Gnu Public License.
;;  See COPYRIGHT file in clisp installation folder for more information.
;;  
;;  ACT-R 5.1
;;
;;  Copyright (c) 1998-2005 Dan Bothell, Mike Byrne, Christian Lebiere & 
;;                          John R Anderson. 
;;  This software is distributed under the terms of the FSF Lesser
;;  Gnu Public License (see LGPL.txt).
;;
;;  Apache Batik 1.6
;;  Apache Jakarta Commons-Collections 3.1
;;                        -Concurrent 1.3.2
;;                        -Lang 2.1
;;                        -Logging 1.6.2
;;
;;  This product contains software developed by the Apache Software Foundation
;;  (http://www.apache.org/)
;;
;;  Hibernate 3.0.5
;;
;;  This software is distributed under the terms of the FSF Lesser
;;  Gnu Public License (see LGPL.txt).

;; When loaded, print what the current features are.
(print *features*)

;; Debug flag -- set to 1 for basic tracing, 2 for verbose, and 0 for none
(defvar *cogtool-debug* 0)

(defmacro cogtool-debug (level message &rest args)
  ;; Note that this is a macro, so neither message nor args are eval'd
  ;; unless we are at or above the given debug level.
  `(when (>= *cogtool-debug* ,level)
     (format t "~&~5TCogTool: ~@?~%" ,message ,@args)))

;; For now at least we keep the module name in a variable, which is set from
;; the Java side. The visualization code needs to know the name, and we keep
;; changing it, so it seems best to keep it in a shared constant on the Java
;; side so we don't have to worry about keep two disparate bits of code in synch.
(defvar *cogtool-module-name* nil)

;; Declare a global variable, in which we will store the current design
(defvar *cogtool-design* nil)

;; Set up a disembodied plist to pass side-band information from CogTool.
(defvar *cogtool-plist* nil)

;; Use a counter to keep track of whether or not the start of the current
;; script step has been reported yet.
(defvar *last-script-step-count* 0)

(defmacro get-cogtool-plist (key)
  `(getf *cogtool-plist* ,key))

;; When updating the plist we defer changes to :destination-frame, as it needs to be kept around in
;; some circumstances.
(defun set-cogtool-plist (&rest keys 
			  &key (destination-frame (getf *cogtool-plist* :destination-frame) df-supplied)
			  &allow-other-keys)
  (cogtool-debug 2 "At time: ~A" (mp-time))
  (let ((script-step (getf keys :script-step))
	(buffers (getf keys :buffers))
	(script-step-count (getf keys :script-step-count)))
    (when (and script-step (> script-step-count *last-script-step-count*))
      (apply #'emit-trace nil 'START-SCRIPT-STEP script-step (and buffers (list buffers))))
    (setq *last-script-step-count* (max *last-script-step-count* script-step-count)))
  (unless df-supplied
    (setq keys (list* :destination-frame destination-frame keys)))
  (setq *cogtool-plist* keys))

(defun clear-cogtool-plist-element (key)
  (when (get-cogtool-plist key)
    (let ((result (copy-seq *cogtool-plist*)))
      (setf (getf result key) nil)
      (setq *cogtool-plist* result))))

(defun emit-trace (pseudo-module &rest args)
  (let ((*print-pretty* nil)
	(details (format nil #.(formatter "~{~S~^ ~}") args)))
    (format t +format-event-event-string+
	    (mp-time)
	    nil nil
	    nil nil nil
	    (max-module-name-length) (or pseudo-module *cogtool-module-keyword*)
	    details details
	    nil nil nil nil)))

;; The monkey business below with pending-delays is to handle the rather obscure
;; case where we do a system-wait before an already pending system-wait has
;; finished. We queue up the new and start it as soon as the old one finishes,
;; without refreshing the display until all system-waits are done.

(defun schedule-delay-finished (seconds)
  (schedule-event-relative
     seconds
     #'cogtool-delay-finished
     :priority :max
     :details (format nil "Restoring display at end of system wait (~D)" seconds)))

(defun set-cogtool-delay (seconds)
  (if (visible-p *cogtool-design*)
      (progn
	(schedule-delay-finished seconds)
	(setf (visible-p *cogtool-design*) nil)
	(proc-display))
      (setf (pending-delays *cogtool-design*)
	    (nconc (pending-delays *cogtool-design*) (list seconds)))))

(defun cogtool-delay-finished ()
  (if (pending-delays *cogtool-design*)
      (schedule-delay-finished (pop (pending-delays *cogtool-design*)))
      (progn
	(setf (visible-p *cogtool-design*) t)
	(proc-display))))

;;; movement style for a single graffiti gesture

(defstyle graffiti-gesture () key)

(defmethod compute-exec-time ((mtr-mod motor-module) (self graffiti-gesture))
  (+ (init-time mtr-mod) .580))

(defmethod queue-output-events ((mtr-mod motor-module) (self graffiti-gesture))
  (queue-command
     :where :device
     :command #'output-graffiti
     :time (exec-time self)
     :params (key self)))
      
(defgeneric output-graffiti (devin key)
  (:documentation  "Request that the device register a graffiti output for the  given key."))

(defmethod output-graffiti ((devin device-interface) key)
    (if (eq key 'mouse)
      () ;;; Check to see if the key is actually gesturable.
      (device-handle-graffiti (device devin) key)))
   
(defgeneric device-handle-graffiti (device key)
  (:documentation  "Handle the graffiti entry of the given key."))

(defmethod device-handle-graffiti (device key)
  (declare (ignore key))
  (error "No method defined for device-handle-graffiti for object ~S." device))

(defmethod feat-differences ((move1 graffiti-gesture) (move2 graffiti-gesture))
  0)
  
(defmethod pm-module-request :around ((motor motor-module) buffer-name chunk-spec)
   (if (eql (chunk-spec-chunk-type chunk-spec) 'graffiti-gesture)
     (let ((key (and (slot-in-chunk-spec-p chunk-spec 'key) 
		     (verify-single-explicit-value 
		       (chunk-spec-slot-spec chunk-spec 'key) 
		       :motor 'graffiti-gesture 'key))))
       (when key
         (schedule-event-relative 
            0 
            'graffiti-gesture
            :destination :motor
            :params (list :key key)
            :module :motor
            :output 'low)))
     (call-next-method)))

;;;;;;;;;; Define cogtool model classes

(defclass cogtool-device (device-interface) 
  ((frames :accessor frames :initform (make-hash-table :test #'equal)
           :documentation "hashtable of (string => frame)")
   ;; init position really happens cogtool-run-model
   (cursor-x :accessor cursor-x :initform 0)
   (cursor-y :accessor cursor-y :initform 0)
   (curframe :accessor curframe :initform nil)
   (initial-frame-name :accessor initial-frame-name :initarg :initial-frame-name)
   (visible-p :accessor visible-p :initform t)
   (pending-delays :accessor pending-delays :initform nil)))

(defclass cogtool-frame () 
  ((name :accessor name :initarg :name :initform ""
         :documentation "name of the frame")
   (widgets :accessor widgets :initarg :widgets :initform nil
            :documentation "list of widgets")))

(defclass cogtool-widget () 
  ((name :accessor name :initarg :name :initform ""
         :documentation "name of the widget")
   (title :accessor title :initarg :title :initform ""
         :documentation "title of the widget")
   (x :accessor x :initarg :x :initform 0) ; x,y are the upper left corner, not center!
   (y :accessor y :initarg :y :initform 0)
   (width :accessor width :initarg :width :initform 0)
   (height :accessor height :initarg :height :initform 0)
   (wtype :accessor wtype :initarg :wtype :initform "cogtool-button")))

;;;;;;;;;; Actual ACT-R device-interface methods

(defmethod build-features-for ((design cogtool-device) (vis-mod vision-module))
  (if (or (not (visible-p design)) (null (curframe design)))
      (progn
	(cogtool-debug 1 "Clearing visual icon at time: ~D" (mp-time))
        (clear vis-mod)
        nil)
      ;; tell the vis-mod about all widgets
      (let ((features nil))
	(cogtool-debug 1 "Building visual icon at time: ~D" (mp-time))
        (dolist (widget (widgets (curframe design)) features)
          ;; NOTE: this reverses the widget-list
	  (cogtool-debug 2 "Making feature for widget: ~A" (name widget))
          (setf features (cons (make-feature widget design vis-mod) features))))))

(loop for wtype in '(cogtool-menu cogtool-submenu cogtool-menu-item cogtool-text
		     cogtool-text-box cogtool-pull-down-list cogtool-pull-down-item
		     cogtool-list-box-item cogtool-radio-button cogtool-checkbox
		     cogtool-non-interactive cogtool-graffiti)
    do (setf (get wtype 'surrogate-widget-type) 'cogtool-button))

;; Helper method for build-features-for -- turns a widget into an icon-feature
(defun make-feature (widget design vis-mod)
  ;; Note that the x,y position for an icon-feature needs to be its
  ;; center rather than its upper left corner.
  (make-instance 'icon-feature
      :value (name widget)
      :dmo-id (gentemp (name widget))
      :x (+ (x widget) (floor (width widget) 2))
      :y (+ (y widget) (floor (height widget) 2))
      :kind (let ((typ (wtype widget))) (or (get typ 'surrogate-widget-type) typ))
      :screen-obj design
      :height (height widget) 
      :width (width widget)))

;; Sets the cursor to a new location.
(defmethod device-move-cursor-to ((design cogtool-device) (xyloc vector))
  (cogtool-debug 1 "Move cursor to x: ~D y: ~D" (px xyloc) (py xyloc))
  ;; save position info
  (setf (cursor-x design) (px xyloc))
  (setf (cursor-y design) (py xyloc))
  ;; Follow a transition if needed.
  (transition (get-cogtool-plist :destination-frame) design))

;; Gets the current location of the cursor.
(defmethod get-mouse-coordinates ((design cogtool-device))
  (cogtool-debug 2 "Getting mouse coords x: ~D y: ~D" 
            (cursor-x design) (cursor-y design))
  (vector (cursor-x design) (cursor-y design)))

(defmethod device-handle-graffiti ((design cogtool-device) key)
  (cogtool-debug 1 "Graffiti: ~A" key)
  (transition (get-cogtool-plist :destination-frame) design))

(defmethod device-handle-keypress ((design cogtool-device) key)
  (cogtool-debug 1 "Keypress: ~A" key)
  (transition (get-cogtool-plist :destination-frame) design))

(defmethod device-handle-click ((design cogtool-device))
  (cogtool-debug 1 "Clicked")
  (transition (get-cogtool-plist :destination-frame) design))

;; Actually perform the transition to the specified frame
(defun transition (frame-name design)
  ;; do the transition
  (cogtool-debug 1 "transition called on ~S" frame-name)
  ;; only do the transition if not nil.
  (when (and frame-name (not (equal frame-name (name (curframe design)))))
    (let ((frame (gethash frame-name (frames design))))
      (emit-trace nil 'TRANSITION-TO (name frame))
      (setf (curframe design) frame))
    (proc-display)
    (clear-cogtool-plist-element :destination-frame)))
    
;; If we are using EMMA, don't let it put in any noise, just use
;; the mean of any normally noisey values.
#+emma
(defun add-gaussian-noise (x stddev)
  "Override the version in EMMA to add no noise."
  x)

#+emma
(defmethod initiate-eye-move :before ((eye-mod emma-vis-mod) (recog-time number) (r-theta vector))
  (emit-trace :vision 'PREPARE-EYE-MOVEMENT))

;;;
;;;  Run Code
;;;

;; This is the default set of values passed to sgp below; by setting 
;; *overridden-global-parameters* to a list of alternating keywords and values
;; before klm-p is called, one or more of those default global parameter values
;; can be overridden.
(defparameter *default-global-parameters* `(:v t
					    :trace-detail high
					    :esc t
					    :lf 0
					    ,@ #+emma '(:visual-encoding-factor 0.006 visual-encoding-exponent 0.4)
					    ))

(defvar *overridden-global-parameters* '())

(defun cogtool-run-model (start-with-mouse timeout debug &rest load-files)
  (let ((*cogtool-debug* (max *cogtool-debug* debug)))
    (when load-files
      (cogtool-debug 2 "Clearing ACT-R state")
      (clear-all)
      (mapcar #'(lambda (file)
		  (cogtool-debug 1 "Loading ~A" file)
		  (load file))
	      load-files))
    (let* ((d *cogtool-design*)
	   (n (initial-frame-name d))
	   (f (gethash n (frames d))))
      (cogtool-debug 2 "Intalling device ~S" d)
      (install-device d)
      (emit-trace nil 'SET-INITIAL-FRAME (name f))
      (setf (curframe d) f))
    (when start-with-mouse
      (cogtool-debug 1 "Moving hand to mouse")
      (start-hand-at-mouse))
    (cogtool-debug 1 "Setting cursor position to 0, 0")
    (set-cursor-position 0 0)
    (proc-display)
    (let ((*cogtool-plist*) (*last-script-step-count* 0))
      (cogtool-debug 2 "Running model")
      (values (run timeout)))))

(defmacro define-cogtool-model (&rest forms)
  (let ((params (loop :with result := (copy-seq *default-global-parameters*)
		  :for (key val) :on *overridden-global-parameters* :by #'cddr
		  :do (setf (getf result key) val)
		  :finally (return result))))
  `(define-model cogtool-model
     (cogtool-debug 1 "Setting params to ~A" ',params)
     (sgp ,@params)
     (sgp)
     (chunk-type (cogtool-button (:include visual-object)) ())
     (chunk-type (drag (:include motor-command)) hand)
     (chunk-type (graffiti-gesture (:include motor-command)) key)
     (chunk-type klm state)
     (add-dm
        (goal isa klm state 1)
	(graffiti-gesture isa motor-command)
	(cogtool-button isa cogtool-button))
     ,@forms
     (goal-focus goal))))

