#!/usr/bin/perl
## record all remote writes that happen at different 
## places, and collect a mapping of 
##   classname, function ->  [classname, field, [r|w] ] list
##
use strict;
use Travertine;
use Getopt::Std;
use IO::File;
require "MDUtils.pl";
use vars qw($opt_l $opt_f);

use constant READ => 0;
use constant WRITE => 1;
use constant FUNCALL_PRE => 2;
use constant FUNCALL_POST => 3;
use constant FUNC_WRITE => 4;

getopts ("l:f:");
our $LOG = $opt_l;
tdie "Log file must be defined (-l)" if !defined $LOG;
our $FIELD_FILE = defined $opt_f ? $opt_f : $ENV{'HOME'} . "/quake3/src/scripts/.extracted-ds";

our $LOG_DIR = `dirname $LOG`;
chomp $LOG_DIR;

our @CLASS_STRINGS = `cat $LOG_DIR/strings.out`;
chomp @CLASS_STRINGS;
########################################################################

our (@ENT_REV_MAP, @CLI_REV_MAP);
BuildReverseMaps ();
ProcessLog ($LOG);

########################################################################
sub BuildReverseMaps () {
    my (%ent_fields, %cli_fields);

    populateFieldHashes ($FIELD_FILE, \%ent_fields, \%cli_fields);
    @ENT_REV_MAP = getReverseMap (\%ent_fields);
    @CLI_REV_MAP = getReverseMap (\%cli_fields);
}

sub mapField ($) {
    my $field = shift;
    my $ent_fields = scalar @ENT_REV_MAP;
    
    if ($field >= $ent_fields) { 
	$field -= $ent_fields;
	if ($field >= scalar @CLI_REV_MAP) {
	    $field = "invalid_client_field";
	}
	else {
	    $field = $CLI_REV_MAP [$field];
	}
    }
    else {
	$field = $ENT_REV_MAP [$field];
    }

    return $field;
}

sub ProcessLog ($) { 
    my $logfile = shift;
    my @stack = ();
    my %nothink = ('read' => {}, 'write' => {});
    my %remotes = ();
    my %curfuncs = ();
    
    print STDERR "Determining length of log...\n";
    my $lines = `wc -l $logfile`;
    chomp $lines;
    print STDERR "Starting processing...\n";
    
    open F, $logfile or tdie "Could not open log file $logfile : $!";
    my $i = 0;
    my $perc = 0;
    while (<F>) {
	chomp;
	my ($type, $field, $ent_num, $class, $func);
	
	if (/(\d+)\t(\d+)\t(\d+)\t(\d+)\t(\w+)/) { 
	    $type = $1;
	    $field = $2;
	    $ent_num = $3;
	    $class = $4;
	    $func = $5;
	}
	elsif (/(\d+)\t(\d+)\t(\d+)\t(\d+)/) { 
	    $type = $1;
	    $field = $2;
	    $ent_num = $3;
	    $class = $4;
	    $func = "";
	}

	# give some progress indicator...;
	$i++;
	if ($perc != int($i * 100 / $lines)) {
	    $perc = int ($i * 100.0 / $lines);	    
	    print STDERR "$perc percent done...\r";
	}
	
	if ($type == FUNC_WRITE) {
	    my $key = "e$ent_num.f$field";
	    # print STDERR "recording func_write: field=" . mapField ($field) . " func=$func\n";
	    $curfuncs {$key} = $func;
	}
	elsif ($type == FUNCALL_PRE) { 
	    my $efunc;
	    my $key = "e$ent_num.f$field";
	    if (exists $curfuncs {$key}) {
		$efunc = $curfuncs {$key};
	    }
	    else {
		$efunc = "undef";
	    }
	    my $var = [ $ent_num, $field, $class, $efunc ];
	    push @stack, $var;
	}
	elsif ($type == FUNCALL_POST) {
	    pop @stack;
	}
	else { # read or write;
	    if (scalar @stack != 0) {     # don't care about stuff which is written to outside of entity functions.
		my $think_ent_num = $stack [-1]->[0];
		my $think_class   = $stack [-1]->[2];
		my $think_func    = $stack [-1]->[3];
		
		if ($think_ent_num != $ent_num) {
		    my $key = "c-$think_class-f-$think_func";
		    
		    if (!exists $remotes {$key}) { 
			$remotes {$key} = {};
			$remotes {$key}->{'reads'} = {};
			$remotes {$key}->{'writes'} = {};
		    }

		    # print STDERR "key: $key type=$type <" . $remotes {$key}->{$type} . ">\n";
		    my $okey = "C-$class-F-$field";
		    $remotes {$key}->{$type}++;
		    if ($type == READ) {
			$remotes {$key}->{'reads'}->{$okey} = 1;
		    } else {
			$remotes {$key}->{'writes'}->{$okey} = 1;
		    }
		}
	    }
	}
    }
    close F;
    print STDERR "Done!\n";

    my $fh = new IO::File "> $LOG_DIR/remoteWrites.stats";
    tdie "Could not open file $LOG_DIR/remoteWrites.stats: $!" if (!defined $fh);
    
    foreach my $key (keys %remotes) {
	$key =~ /c-(\d+)-f-(\w+)/;
	my $fclass = mapClass ($1);
	my $tfunc = $2;
	
	print $fh "=" x 80, "\n";
	print $fh "$fclass" . "::" . "$tfunc\n";
	foreach my $okey (keys %{$remotes {$key}->{'reads'}}) {
	    $okey =~ /C-(\d+)-F-(\d+)/;
	    my $c = mapClass ($1);
	    my $f = mapField ($2);
	    print $fh "r\t$c\t$f\n";
	}
	print $fh "-" x 80, "\n";
	foreach my $okey (keys %{$remotes {$key}->{'writes'}}) {
	    $okey =~ /C-(\d+)-F-(\d+)/;
	    my $c = mapClass ($1);
	    my $f = mapField ($2);
	    print $fh "w\t$c\t$f\n";
	}

    }
    close $fh;
}

sub mapClass ($) {
    my $c = shift;
    if ($c == -1 or $c >= scalar @CLASS_STRINGS) {
	return "null";
    }
    else {
	return $CLASS_STRINGS [$c];
    }
}
