#!/usr/bin/perl
use strict;
use Travertine;
use Getopt::Std;
require "MDUtils.pl";
use vars qw($opt_l $opt_f);

use constant READ => 0;
use constant WRITE => 1;
use constant FUNCALL_PRE => 2;
use constant FUNCALL_POST => 3;
use constant FUNC_WRITE => 4;

getopts ("l:f:");
our $LOG = $opt_l;
tdie "Log file must be defined (-l)" if !defined $LOG;
our $FIELD_FILE = defined $opt_f ? $opt_f : $ENV{'HOME'} . "/quake3/src/scripts/.extracted-ds";

our $LOG_DIR = `dirname $LOG`;
chomp $LOG_DIR;

our @CLASS_STRINGS = `cat $LOG_DIR/strings.out`;
chomp @CLASS_STRINGS;
########################################################################

our (@ENT_REV_MAP, @CLI_REV_MAP);
BuildReverseMaps ();
ProcessLog ($LOG);

########################################################################
sub BuildReverseMaps () {
    my (%ent_fields, %cli_fields);

    populateFieldHashes ($FIELD_FILE, \%ent_fields, \%cli_fields);
    @ENT_REV_MAP = getReverseMap (\%ent_fields);
    @CLI_REV_MAP = getReverseMap (\%cli_fields);
}

sub mapField ($) {
    my $field = shift;
    my $ent_fields = scalar @ENT_REV_MAP;
    
    if ($field >= $ent_fields) { 
	$field -= $ent_fields;
	if ($field >= scalar @CLI_REV_MAP) {
	    $field = "invalid_client_field";
	}
	else {
	    $field = $CLI_REV_MAP [$field];
	}
    }
    else {
	$field = $ENT_REV_MAP [$field];
    }

    return $field;
}

sub ProcessLog ($) { 
    my $logfile = shift;
    my @stack = ();
    my %nothink = ('read' => {}, 'write' => {});
    my %remotes = ();
    
    print STDERR "Determining length of log...\n";
    my $lines = `wc -l $logfile`;
    chomp $lines;
    print STDERR "Starting processing...\n";
    
    open F, $logfile or tdie "Could not open log file $logfile : $!";
    my $i = 0;
    my $perc = 0;
    while (<F>) {
	chomp;
	m/(\d+)\t(\d+)\t(\d+)\t(\d+)/;
	my $type = $1;
	my $field = $2;
	my $ent_num = $3;
	my $class = $4;

	# give some progress indicator...;
	$i++;
	if ($perc != int($i * 100 / $lines)) {
	    $perc = int ($i * 100.0 / $lines);	    
	    print STDERR "$perc percent done...\r";
	}
	
	if ($type == FUNCALL_PRE) { 
	    my $var = [ $ent_num, $field, $class ];
	    push @stack, $var;
	}
	elsif ($type == FUNCALL_POST) {
	    pop @stack;
	}
	else { # read or write;
	    if (scalar @stack == 0) {
		# we just want to know which entities (classnames) are read/written to 
		# outside the think functions we have instrumented;
		
		if (!exists $nothink{$class}) { 
		    $nothink {$class} = {};
		    $nothink {$class}->{'rf'} = {};
		    $nothink {$class}->{'wf'} = {};
		}
		
		if ($type == READ) { 
		    $nothink {$class}->{'rf'}->{$field} = 1;
		}
		else { 
		    $nothink {$class}->{'wf'}->{$field} = 1;
		}
	    }
	    else {
		my $think_ent_num = $stack [-1]->[0];
		my $think_class   = $stack [-1]->[2];
		
		if ($think_ent_num != $ent_num) {
		    my $key = "f-$think_class-t-$class";
		    if (!exists $remotes {$key}) { 
			$remotes {$key} = {};
			$remotes {$key}->{'rf'} = {};
			$remotes {$key}->{'wf'} = {};
		    }

		    # print STDERR "key: $key type=$type <" . $remotes {$key}->{$type} . ">\n";
		    $remotes {$key}->{$type}++;
		    if ($type == READ) {
			$remotes {$key}->{'rf'}->{$field} = 1;
		    } else {
			$remotes {$key}->{'wf'}->{$field} = 1;
		    }
		}
	    }
	}
    }
    close F;
    print STDERR "Done!\n";

    print "===INFO: Fields written outside instrumented think functions\n";
    foreach my $class (keys %nothink) { 
	print sprintf ("%s\treads=", mapClass ($class));
	print join (",", map { $_ = mapField ($_) } (keys %{$nothink {$class}->{'rf'}}));
	print "\twrites=";
	print join (",", map { $_ = mapField ($_) } (keys %{$nothink {$class}->{'wf'}}));
	print "\n";
    }
    
    print "===INFO: Remote reads or writes\n";
    foreach my $key (keys %remotes) {
	$key =~ /f-(\d+)-t-(\d+)/;
	my $fclass = mapClass ($1);
	my $tclass = mapClass ($2);
	
	print sprintf ("%s\t%s\treads=", $fclass, $tclass);
	print join (",", map { $_ = mapField ($_) } (keys %{$remotes {$key}->{'rf'}}));
	print "\twrites=";
	print join (",", map { $_ = mapField ($_) } (keys %{$remotes {$key}->{'wf'}}));
	print "\n";
    }
}

sub mapClass ($) {
    my $c = shift;
    if ($c == -1 or $c >= scalar @CLASS_STRINGS) {
	return "null";
    }
    else {
	return $CLASS_STRINGS [$c];
    }
}
