#!/usr/bin/perl
use strict;
use Travertine;
use Getopt::Std;
use IO::File;
use POSIX qw/ceil floor/;
use vars qw($opt_g $opt_p);

getopts ("gp:");

our $GENERATE_ONLY = defined $opt_g;           # dont extract information; use the scratch-files produced 
our $PREFIX = defined $opt_p ? $opt_p : $ENV{'HOME'} . "/quake3/src/quake3";

my $opt_o = "/dev/stdout";
our $OFH = new IO::File ("> $opt_o");
if (!$OFH) { 
    die "Could not open file $opt_o for writing: $!";
}

our $PROG = "$PREFIX/extractors/cilly -DGAME_HARD_LINKED";

print $OFH <<EOT;
/* 
 * This file has been automatically generated. You run the 
 * risk of loosing your changes if you edit it manually. A 
 * better place to incorporate your edits would be 
 * `scripts/DeriveMetadataInfo.pl'. 
 *
 * This file contains the following information extracted from 
 * the game module.
 *
 * * Addresses and declarations of functions _defined_ in the module
 *   Note that it does not include declarations referenced by the 
 *   module (the latter are not needed for our purposes.)
 *
 * * Addresses of global variables. For arrays, we "expand" the array
 *   (the static length is known) and list addresses for each element
 *   in the array. 
 *
 * * Collection of static strings referenced by the module. 
 *
 * * Detailed information about the gentity_t and gclient_t structures.
 *   This includes the types and sizes of sub-fields. 
 *
 */

#include <game/be_aas.h>
#include <game/botlib.h>
#include <game/bg_local.h>
#include <game/q_shared.h>
#include <game/be_ai_goal.h>
#include <game/be_ai_char.h>
#include <game/be_ai_chat.h>
#include <game/be_ai_weap.h>
#include <game/be_ai_move.h>
#include <game/be_ai_gen.h>
#include <game/be_ea.h>
#include <game/ai_main.h>
#include <game/ai_cmd.h>
#include <game/ai_vcmd.h>
#include <game/ai_chat.h>
#include <game/ai_team.h>
#include <game/ai_dmnet.h>
#include <game/ai_dmq3.h>

// HACKS galore... 
//
// These structures are defined in C files (not .h files) and 
// so when we declare the functions which use them in 
// ExtractedMetadata.cxx, compilation does not happen! Basically, 
// these structs and the functions have static linkage, but the 
// function pointers can still be used as think functions

// XXX perhaps these definitions can be obtained automagically?

typedef struct ipFilter_s
{
    unsigned        mask;
    unsigned        compare;
} ipFilter_t;

//ctf task preferences for a client
typedef struct bot_ctftaskpreference_s
{
        char            name[36];
        int                     preference;
} bot_ctftaskpreference_t;

typedef struct voiceCommand_s
{
    char *cmd;
    void (*func)(bot_state_t *bs, int client, int mode);
} voiceCommand_t;

typedef struct {
    int             clientNum;
    int             spawnTime;
} botSpawnQueue_t;

typedef struct {
    char    *name;
    void    (*spawn)(gentity_t *ent);
} spawn_t;

typedef struct {
    gentity_t       *ent;
    vec3_t  origin;
    vec3_t  angles;
    float   deltayaw;
} pushed_t;

typedef struct {
    vmCvar_t        *vmCvar;
    char            *cvarName;
    char            *defaultString;
    int                     cvarFlags;
    int                     modificationCount;  // for tracking changes
    qboolean        trackChange;        // track this variable, and announce if change
    qboolean teamShader;        // track and if changed, update shader state
} cvarTable_t;

typedef struct teamgame_s {
    float                   last_flag_capture;
    int                             last_capture_team;
    flagStatus_t    redStatus;      // CTF
    flagStatus_t    blueStatus;     // CTF
    flagStatus_t    flagStatus;     // One Flag CTF
    int                             redTakenTime;
    int                             blueTakenTime;
    int                             redObeliskAttackedTime;
    int                             blueObeliskAttackedTime;
} teamgame_t;                           

typedef struct {
    char oldShader[MAX_QPATH];
    char newShader[MAX_QPATH];
    float timeOffset;
} shaderRemap_t;

EOT

our $SCR_PREFIX = ".extracted";
if (!$GENERATE_ONLY) {
    # make a copy of g_local.h,g_client.c
    my @saves = ("g_local.h", "g_client.c", "g_public.h");
    psystem ("mkdir -p precious/");
    foreach (@saves) { psystem ("cp $PREFIX/game/$_ precious/"); }
    
    psystem ("perl -ni -e 'print if !/Quake3Entity/' $PREFIX/game/g_local.h");
    psystem ("perl -ni -e 'print if !/Quake3Entity/' $PREFIX/game/g_public.h");
    psystem ("perl -ni -e 'print if !/colyseus/' $PREFIX/game/g_client.c");

    my @filearray = glob "$PREFIX/game/*.c";
    my $files = "";
    foreach (@filearray) {
	next if /rankings/;
	next if /bg_lib/;
	$files .= " $_";
    }

    tinfo "Extracting function addresses...";
    psystem ("$PROG --dologfunctions $files > $SCR_PREFIX-functions");
    tinfo "Extracting global variable addresses...";
    psystem ("$PROG --dologglobals $files > $SCR_PREFIX-globals");
    tinfo "Extracting static strings...";
    psystem ("$PROG --dologstrings $files > $SCR_PREFIX-strings");
    tinfo "Extracting information about gentity_t and gclient_t...";
    psystem ("$PROG --doextractdatastruct $PREFIX/game/g_main.c > $SCR_PREFIX-ds");

    foreach (@saves) { psystem ("cp precious/$_ $PREFIX/game/"); }
    psystem ("rm -rf precious/");
}

extractFunctionPointers ();
extractGlobals ();
extractStrings ();
extractEntityClientInfos ();

close $OFH;

exit 0;

###################################################################################

sub extractFunctionPointers {
    my @decls = (); 
    my @names = ();
    my %seen  = ();

    open F, "$SCR_PREFIX-functions" or die "Could not open $SCR_PREFIX-functions: $!";
    while (<F>) {
	chomp;
	/name=(\w+) decl=(.*)$/;
	my $name = $1;
	my $decl = $2;
	if (!exists $seen {$name}) {
	    $seen {$name} = 1;
	    next if $name =~ /dllentry/i;
	    next if $name =~ /^\s*$/;
	    next if $name =~ /MatrixMultiply/;

	    # more hackyness; CIL forgets about 'const' and so 
	    # C++ thinks there are multiple overloaded functions 
	    # with the same name 

	    push @names, $name;
	    my $inHeaders = `grep -w $name $PREFIX/game/*.h | head -1`;
	    chomp $inHeaders;
	    if ($inHeaders eq '') {
		push @decls, $decl;
	    }
	}
    }
    close F;

    print $OFH "/* Function prototypes */\n\n";    
    foreach (@decls) { 
	print $OFH "extern ", $_, "\n";
    }
    print $OFH "\nvoid *g_FunctionPointers [] = \n{\n";
    foreach (@names) { 
	print $OFH "\t(void *) $_,\n";
    }
    print $OFH "};\n\n";
}

sub extractGlobals {
    open F, "$SCR_PREFIX-globals" or die "Could not open $SCR_PREFIX-globals: $!";
    my @lines = <F>;
    chomp @lines;
    close F;

    print $OFH "/* Pointers to globals */\n\n";

    foreach my $line (@lines) {
	next if ($line !~ /type=(\w+)\s+base=(\w+)\s+name=(\w+)/);	
	my $type = $1;
	my $basetype = $2;
	my $name = $3;
	
	next if ($basetype =~ /int/ or $basetype =~ /float/ or $basetype =~ /char/ or $basetype =~ /vec/);
	if ($type ne 'array') {
	    print $OFH "extern $basetype $name;\n" 
	}
	else {
	    print $OFH "extern $basetype $name\[\];\n";
	}	
    }
    print $OFH "\nvoid *g_GlobalsPointers [] = \n{\n";

    foreach my $line (@lines) {
	next if ($line !~ /type=(\w+)\s+base=(\w+)\s+name=(\w+)/);
	my $type = $1;
	my $basetype = $2;
	my $name = $3;
	
	next if ($name eq 'g_entities' or $name eq 'g_clients' or $name eq 'memoryPool');
	next if ($basetype =~ /int/ or $basetype =~ /float/ or $basetype =~ /char/ or $basetype =~ /vec/ or $basetype =~ /qboolean/);

	if ($type ne 'array') {
	    print $OFH "\t(void *) &$name,\n" 
	}
	else {
	    next if ($name eq 'pushed' or $name eq 'ipFilters');

	    $line =~ /len=(\d+)/;
	    my $len = $1;

	    for (my $i = 0; $i < $len; $i++) { 
		print $OFH "\t(void *) (& $name\[$i\]),\n";
	    }
	    print $OFH "\n";
	}	
    }
    print $OFH "};\n";
}

sub extractStrings {
    my @strings = ();

    open F, "$SCR_PREFIX-strings" or die "Could not open $SCR_PREFIX-strings: $!";
    while (<F>) {
	chomp;
	push @strings, $_;
    }
    close F;
    print $OFH "/* Static strings */\n\n";
    print $OFH "char *g_StaticStrings [] = {\n";
    foreach (@strings) { 
	print $OFH "\t$_,\n";
    }
    print $OFH "};\n\n";
}

sub log2 ($) {
    return ceil((log ($_[0]) / log 2.0));
}

sub dumpFieldInfo ($) {
    my $line = shift;

     # check regexps; 
    $line =~ /type=:([^:]+):\s+name=:([^:]+):/;
    my $type = $1;
    my $name = $2;

    my @ignore = ('isReplica', # this is a Colyseus field;
	    's.number', 's.clientNum', 'ps.clientNum', 's.otherEntityNum', 's.otherEntityNum2', # refer to local indices;
	    'ps.ping', # local ping measurement; 
	    'airOutTime', # dunno :P
	    'r.absmax', 'r.absmin' # SV_LinkEntity and UnlinkEntity change that anyway;
	    # 'r.linkcount',  # DONT IGNORE THIS. For some unknown reason, rocketlauncher stops working if you don't replicate this field. 
	                      # grepping through the code shows no sign of linkcount except in one place. i am dumbfounded.
	    );

    my %ignore = map { $_ => 1 } @ignore; 
    return if exists $ignore {$name};

    if ($type eq 'enum') { 
	$line =~ /nitems=(\d+)/;	
	my $maxbits = log2 ($1);
	if ($maxbits <= 8) { $type = "char"; }
	elsif ($maxbits <= 16) { $type = "short"; }
	else { $type = "int"; }
	    
	# XXX: all the above optimization wont work; g++ seems to allocate
	# space for an entire integer (4 bytes) even for a small enum :(  
	
	$type = "int";
	print $OFH "\tnew TypedField<$type> (FOFS($name)),\n";
    }
    elsif ($type eq 'int' or $type eq 'float') { 
	print $OFH "\tnew TypedField<$type> (FOFS($name)),\n";
    }
    elsif ($type =~ /char/) {  # get rid of the signed/unsigned, if any
	print $OFH "\tnew TypedField<char> (FOFS($name)),\n";
    }
    elsif ($type eq 'pointer') {
	$line =~ /base=:([^:]+):/;
	my $basetype = $1;
	if ($basetype =~ /_s\s*$/) { 
	    $basetype =~ s/struct (\w+)_s\s*/$1_t/;
	}
	elsif ($basetype =~ /function/) {
	    $basetype = "void";
	}
	elsif ($basetype =~ /char/) {  # get rid of the signed/unsigned if any;
	    $basetype = "char";
	}

	print $OFH "\tnew TypedField<$basetype *> (FOFS($name)),\n";
    }
    elsif ($type eq 'array') { 
	$line =~ /base=:([^:]+):/;
	my $basetype = $1;
	$line =~ /len=(\d+)/;
	my $length = $1;

	if ($basetype =~ /char/) { 
	    $basetype = "char";
	}

	print $OFH "\tnew TypedArray<$basetype> (FOFS($name), $length),\n";
    }
}

sub extractEntityClientInfos () {
    open F, "$SCR_PREFIX-ds" or die "Could not open $SCR_PREFIX-ds: $!";
    
    print $OFH <<EOT;
#undef  FOFS
#define FOFS(x)     #x,(int)&(((gentity_t *)0)->x)

const Field *g_GEntityFields [] = {
    
EOT
    while (<F>) {
	next if /TYPEINFO\s+typename/;          # get rid of the starting tag;
	last if /<\/TYPEINFO>/;
	
	dumpFieldInfo ($_);
    }
    print $OFH <<EOT;
    
};

#undef  FOFS
#define FOFS(x)     #x,(int)&(((gclient_t *)0)->x)

const Field *g_GClientFields [] = {
    
EOT
    while (<F>) {
	next if /TYPEINFO\s+typename/;          # get rid of the starting tag;
	last if /<\/TYPEINFO>/;
	
	dumpFieldInfo ($_);
    }

    print $OFH <<EOT;

};

EOT
    close F;
}
