/*
===========================================================================
Copyright (C) 1999-2005 Id Software, Inc.

This file is part of Quake III Arena source code.

Quake III Arena source code is free software; you can redistribute it
and/or modify it under the terms of the GNU General Public License as
published by the Free Software Foundation; either version 2 of the License,
or (at your option) any later version.

Quake III Arena source code is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Foobar; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
===========================================================================
*/
// sv_game.c -- interface to the game dll

#include "server.h"

#include "../game/botlib.h"
#include "../game/be_aas.h" // ASHWIN
#include "../game/be_ai_chat.h" // ASHWIN
#include "../game/be_ai_goal.h"
#include "../game/be_ai_move.h"
#include "../game/be_ai_weap.h"

botlib_export_t	*botlib_export;

void SV_GameError( const char *string ) {
	Com_Error( ERR_DROP, "%s", string );
}

void SV_GamePrint( const char *string ) {
	Com_Printf( "%s", string );
}

// these functions must be used instead of pointer arithmetic, because
// the game allocates gentities with private information after the server shared part
int	SV_NumForGentity( sharedEntity_t *ent ) {
	int		num;

	num = ( (byte *)ent - (byte *)sv.gentities ) / sv.gentitySize;

	return num;
}

sharedEntity_t *SV_GentityNum( int num ) {
	sharedEntity_t *ent;

	ent = (sharedEntity_t *)((byte *)sv.gentities + sv.gentitySize*(num));

	return ent;
}

playerState_t *SV_GameClientNum( int num ) {
	playerState_t	*ps;

	ps = (playerState_t *)((byte *)sv.gameClients + sv.gameClientSize*(num));

	return ps;
}

svEntity_t	*SV_SvEntityForGentity( sharedEntity_t *gEnt ) {
	if ( !gEnt || gEnt->s.number < 0 || gEnt->s.number >= MAX_GENTITIES ) {
		Com_Error( ERR_DROP, "SV_SvEntityForGentity: bad gEnt" );
	}
	return &sv.svEntities[ gEnt->s.number ];
}

sharedEntity_t *SV_GEntityForSvEntity( svEntity_t *svEnt ) {
	int		num;

	num = svEnt - sv.svEntities;
	return SV_GentityNum( num );
}

/*
===============
SV_GameSendServerCommand

Sends a command string to a client
===============
*/
void SV_GameSendServerCommand( int clientNum, const char *text ) {
	if ( clientNum == -1 ) {
		SV_SendServerCommand( NULL, "%s", text );
	} else {
		if ( clientNum < 0 || clientNum >= sv_maxclients->integer ) {
			return;
		}
		SV_SendServerCommand( svs.clients + clientNum, "%s", text );	
	}
}


/*
===============
SV_GameDropClient

Disconnects the client with a message
===============
*/
void SV_GameDropClient( int clientNum, const char *reason ) {
	if ( clientNum < 0 || clientNum >= sv_maxclients->integer ) {
		return;
	}
	SV_DropClient( svs.clients + clientNum, reason );	
}


/*
=================
SV_SetBrushModel

sets mins and maxs for inline bmodels
=================
*/
void SV_SetBrushModel( sharedEntity_t *ent, const char *name ) {
	clipHandle_t	h;
	vec3_t			mins, maxs;

	if (!name) {
		Com_Error( ERR_DROP, "SV_SetBrushModel: NULL" );
	}

	if (name[0] != '*') {
		Com_Error( ERR_DROP, "SV_SetBrushModel: %s isn't a brush model", name );
	}


	ent->s.modelindex = atoi( name + 1 );

	h = CM_InlineModel( ent->s.modelindex );
	CM_ModelBounds( h, mins, maxs );
	VectorCopy (mins, ent->r.mins);
	VectorCopy (maxs, ent->r.maxs);
	ent->r.bmodel = qtrue;

	ent->r.contents = -1;		// we don't know exactly what is in the brushes

	SV_LinkEntity( ent );		// FIXME: remove
}



/*
=================
SV_inPVS

Also checks portalareas so that doors block sight
=================
*/
qboolean SV_inPVS (vec3_t p1, vec3_t p2)
{
	int		leafnum;
	int		cluster;
	int		area1, area2;
	byte	*mask;

	leafnum = CM_PointLeafnum (p1);
	cluster = CM_LeafCluster (leafnum);
	area1 = CM_LeafArea (leafnum);
	mask = CM_ClusterPVS (cluster);

	leafnum = CM_PointLeafnum (p2);
	cluster = CM_LeafCluster (leafnum);
	area2 = CM_LeafArea (leafnum);
	if ( mask && (!(mask[cluster>>3] & (1<<(cluster&7)) ) ) )
		return qfalse;
	if (!CM_AreasConnected (area1, area2))
		return qfalse;		// a door blocks sight
	return qtrue;
}


/*
=================
SV_inPVSIgnorePortals

Does NOT check portalareas
=================
*/
qboolean SV_inPVSIgnorePortals( vec3_t p1, vec3_t p2)
{
	int		leafnum;
	int		cluster;
	int		area1, area2;
	byte	*mask;

	leafnum = CM_PointLeafnum (p1);
	cluster = CM_LeafCluster (leafnum);
	area1 = CM_LeafArea (leafnum);
	mask = CM_ClusterPVS (cluster);

	leafnum = CM_PointLeafnum (p2);
	cluster = CM_LeafCluster (leafnum);
	area2 = CM_LeafArea (leafnum);

	if ( mask && (!(mask[cluster>>3] & (1<<(cluster&7)) ) ) )
		return qfalse;

	return qtrue;
}


/*
========================
SV_AdjustAreaPortalState
========================
*/
void SV_AdjustAreaPortalState( sharedEntity_t *ent, qboolean open ) {
	svEntity_t	*svEnt;

	svEnt = SV_SvEntityForGentity( ent );
	if ( svEnt->areanum2 == -1 ) {
		return;
	}
	CM_AdjustAreaPortalState( svEnt->areanum, svEnt->areanum2, open );
}


/*
==================
SV_GameAreaEntities
==================
*/
qboolean	SV_EntityContact( vec3_t mins, vec3_t maxs, sharedEntity_t *gEnt, int capsule ) {
	float	*origin, *angles;
	clipHandle_t	ch;
	trace_t			trace;

	// check for exact collision
	origin = gEnt->r.currentOrigin;
	angles = gEnt->r.currentAngles;

	ch = SV_ClipHandleForEntity( gEnt );
	CM_TransformedBoxTrace ( &trace, vec3_origin, vec3_origin, mins, maxs,
		ch, -1, origin, angles, capsule );

	return trace.startsolid;
}


/*
===============
SV_GetServerinfo

===============
*/
void SV_GetServerinfo( char *buffer, int bufferSize ) {
	if ( bufferSize < 1 ) {
		Com_Error( ERR_DROP, "SV_GetServerinfo: bufferSize == %i", bufferSize );
	}
	Q_strncpyz( buffer, Cvar_InfoString( CVAR_SERVERINFO ), bufferSize );
}

/*
===============
SV_LocateGameData

===============
*/
void SV_LocateGameData( sharedEntity_t *gEnts, int numGEntities, int sizeofGEntity_t,
					   playerState_t *clients, int sizeofGameClient ) {
	sv.gentities = gEnts;
	sv.gentitySize = sizeofGEntity_t;
	sv.num_entities = numGEntities;

	sv.gameClients = clients;
	sv.gameClientSize = sizeofGameClient;
}


/*
===============
SV_GetUsercmd

===============
*/
void SV_GetUsercmd( int clientNum, usercmd_t *cmd ) {
	if ( clientNum < 0 || clientNum >= sv_maxclients->integer ) {
		Com_Error( ERR_DROP, "SV_GetUsercmd: bad clientNum:%i", clientNum );
	}
	*cmd = svs.clients[clientNum].lastUsercmd;
}

//==============================================

static int	FloatAsInt( float f ) {
	union
	{
	    int i;
	    float f;
	} temp;
	
	temp.f = f;
	return temp.i;
}

/*
====================
SV_GameSystemCalls

The module is making a system call
====================
*/
//rcg010207 - see my comments in VM_DllSyscall(), in qcommon/vm.c ...
#if ((defined __linux__) && (defined __powerpc__))
#define VMA(x) ((void *) args[x])
#else
#define	VMA(x) VM_ArgPtr(args[x])
#endif

#define	VMF(x)	((float *)args)[x]

extern "C" { 
    int SV_GameSystemCalls (int *args);
};

int SV_GameSystemCalls( int *args ) {
    switch( args[0] ) {
    case G_PRINT:
	Com_Printf ("%s",  VMA(1) );
	return 0;
    case G_ERROR:
	Com_Error (ERR_DROP, "%s",  VMA(1) );
	return 0;
    case G_MILLISECONDS:
	return Sys_Milliseconds();
    case G_CVAR_REGISTER:
	Cvar_Register (( vmCvar_t *)  VMA(1), ( const char *)  VMA(2), ( const char *)  VMA(3), (int)  args[4] );
	return 0;
    case G_CVAR_UPDATE:
	Cvar_Update (( vmCvar_t *)  VMA(1) );
	return 0;
    case G_CVAR_SET:
	Cvar_Set (( const char *)  (const char *)VMA(1), ( const char *)  (const char *)VMA(2) );
	return 0;
    case G_CVAR_VARIABLE_INTEGER_VALUE:
	return Cvar_VariableIntegerValue (( const char *)  (const char *)VMA(1) );
    case G_CVAR_VARIABLE_STRING_BUFFER:
	Cvar_VariableStringBuffer (( const char *)  VMA(1), ( char *)  VMA(2), ( int)  args[3] );
	return 0;
    case G_ARGC:
	return Cmd_Argc();
    case G_ARGV:
	Cmd_ArgvBuffer (( int )  args[1], ( char *)  VMA(2), ( int )  args[3] );
	return 0;
    case G_SEND_CONSOLE_COMMAND:
	Cbuf_ExecuteText ((int )  args[1], ( const char *)  VMA(2) );
	return 0;

    case G_FS_FOPEN_FILE:
	return FS_FOpenFileByMode (( const char *)  VMA(1), ( fileHandle_t *)  VMA(2), ( fsMode_t )  args[3] );
    case G_FS_READ:
	FS_Read2 (( void *)  VMA(1), ( int )  args[2], ( fileHandle_t )  args[3] );
	return 0;
    case G_FS_WRITE:
	FS_Write (( const void *)  VMA(1), ( int )  args[2], ( fileHandle_t )  args[3] );
	return 0;
    case G_FS_FCLOSE_FILE:
	FS_FCloseFile (( fileHandle_t )  args[1] );
	return 0;
    case G_FS_GETFILELIST:
	return FS_GetFileList ((  const char *)  VMA(1), ( const char *)  VMA(2), ( char *)  VMA(3), ( int)  args[4] );
    case G_FS_SEEK:
	return FS_Seek (( fileHandle_t )  args[1], ( long )  args[2], ( int )  args[3] );
    case G_LOCATE_GAME_DATA:
	SV_LocateGameData((sharedEntity_t *) VMA(1), args[2], args[3], (playerState_t *) VMA(4), args[5] );
	return 0;
    case G_DROP_CLIENT:
	SV_GameDropClient (( int )  args[1], ( const char *)  VMA(2) );
	return 0;
    case G_SEND_SERVER_COMMAND:
	SV_GameSendServerCommand (( int )  args[1], ( const char *)  VMA(2) );
	return 0;
    case G_LINKENTITY:
	SV_LinkEntity (( sharedEntity_t *)  VMA(1) );
	return 0;
    case G_UNLINKENTITY:
	SV_UnlinkEntity (( sharedEntity_t *)  VMA(1) );
	return 0;
    case G_ENTITIES_IN_BOX:
	return SV_AreaEntities ((vec_t *)  VMA(1), (vec_t *)  VMA(2), ( int *)  VMA(3), ( int )  args[4] );
    case G_ENTITY_CONTACT:
	return SV_EntityContact ((vec_t *)  VMA(1), (vec_t *)  VMA(2), ( sharedEntity_t *)  VMA(3), ( int )  /*int capsule*/ qfalse );
    case G_ENTITY_CONTACTCAPSULE:
	return SV_EntityContact ((vec_t *)  VMA(1), (vec_t *)  VMA(2), ( sharedEntity_t *)  VMA(3), ( int )  /*int capsule*/ qtrue );
    case G_TRACE:
	SV_Trace (( trace_t *)  VMA(1), (vec_t *)  VMA(2), (vec_t *)  VMA(3), (vec_t *)  VMA(4), (vec_t *)  VMA(5), ( int )  args[6], ( int )  args[7], ( int )  /*int capsule*/ qfalse );
	return 0;
    case G_TRACECAPSULE:
	SV_Trace (( trace_t *)  VMA(1), (vec_t *)  VMA(2), (vec_t *)  VMA(3), (vec_t *)  VMA(4), (vec_t *)  VMA(5), ( int )  args[6], ( int )  args[7], ( int )  /*int capsule*/ qtrue );
	return 0;
    case G_POINT_CONTENTS:
	return SV_PointContents ((vec_t *)  VMA(1), ( int )  args[2] );
    case G_SET_BRUSH_MODEL:
	SV_SetBrushModel (( sharedEntity_t *)  VMA(1), ( const char *)  VMA(2) );
	return 0;
    case G_IN_PVS:
	return SV_inPVS ((vec_t *)  VMA(1), (vec_t *)  VMA(2) );
    case G_IN_PVS_IGNORE_PORTALS:
	return SV_inPVSIgnorePortals ((vec_t *)  VMA(1), (vec_t *)  VMA(2) );
    case G_SET_CONFIGSTRING:
	SV_SetConfigstring ((int )  args[1], ( const char *)  VMA(2) );
	return 0;
    case G_GET_CONFIGSTRING:
	SV_GetConfigstring (( int )  args[1], ( char *)  VMA(2), ( int )  args[3] );
	return 0;
    case G_SET_USERINFO:
	SV_SetUserinfo (( int )  args[1], ( const char * )  VMA(2) );
	return 0;
    case G_GET_USERINFO:
	SV_GetUserinfo (( int )  args[1], ( char *)  VMA(2), ( int  )  args[3] );
	return 0;
    case G_GET_SERVERINFO:
	SV_GetServerinfo (( char *)  VMA(1), ( int )  args[2] );
	return 0;
    case G_ADJUST_AREA_PORTAL_STATE:
	SV_AdjustAreaPortalState (( sharedEntity_t *)  VMA(1), ( qboolean )  args[2] );
	return 0;
    case G_AREAS_CONNECTED:
	return CM_AreasConnected (( int )  args[1], ( int )  args[2] );

    case G_BOT_ALLOCATE_CLIENT:
	return SV_BotAllocateClient();
    case G_BOT_FREE_CLIENT:
	SV_BotFreeClient (( int )  args[1] );
	return 0;

    case G_GET_USERCMD:
	SV_GetUsercmd (( int )  args[1], ( usercmd_t *)  VMA(2) );
	return 0;
    case G_GET_ENTITY_TOKEN:
	{
	    const char	*s;

	    s = COM_Parse (&sv.entityParsePoint );
	    Q_strncpyz (( char *)  VMA(1), ( const char *)  s, ( int )  args[2] );
	    if ( !sv.entityParsePoint && !s[0] ) {
		return qfalse;
	    } else {
		return qtrue;
	    }
	}

    case G_DEBUG_POLYGON_CREATE:
	return BotImport_DebugPolygonCreate ((int )  args[1], ( int )  args[2], ( vec3_t *)  VMA(3) );
    case G_DEBUG_POLYGON_DELETE:
	BotImport_DebugPolygonDelete ((int )  args[1] );
	return 0;
    case G_REAL_TIME:
	return Com_RealTime ((qtime_t *)  VMA(1) );
    case G_SNAPVECTOR:
	Sys_SnapVector((float *) VMA(1) );
	return 0;

	//====================================

    case BOTLIB_SETUP:
	return SV_BotLibSetup();
    case BOTLIB_SHUTDOWN:
	return SV_BotLibShutdown();
    case BOTLIB_LIBVAR_SET:
	return botlib_export->BotLibVarSet ((char *)  VMA(1), ( char *)  VMA(2) );
    case BOTLIB_LIBVAR_GET:
	return botlib_export->BotLibVarGet ((char *)  VMA(1), ( char *)  VMA(2), ( int )  args[3] );

    case BOTLIB_PC_ADD_GLOBAL_DEFINE:
	return botlib_export->PC_AddGlobalDefine ((char *)  VMA(1) );
    case BOTLIB_PC_LOAD_SOURCE:
	return botlib_export->PC_LoadSourceHandle ((const char *)  VMA(1) );
    case BOTLIB_PC_FREE_SOURCE:
	return botlib_export->PC_FreeSourceHandle ((int )  args[1] );
    case BOTLIB_PC_READ_TOKEN:
	return botlib_export->PC_ReadTokenHandle ((int )  args[1], ( pc_token_t *)  VMA(2) );
    case BOTLIB_PC_SOURCE_FILE_AND_LINE:
	return botlib_export->PC_SourceFileAndLine ((int )  args[1], ( char *)  VMA(2), ( int *)  VMA(3) );

    case BOTLIB_START_FRAME:
	return botlib_export->BotLibStartFrame ((float )  VMF(1) );
    case BOTLIB_LOAD_MAP:
	return botlib_export->BotLibLoadMap ((const char *)  VMA(1) );
    case BOTLIB_UPDATENTITY:
	return botlib_export->BotLibUpdateEntity ((int )  args[1], ( bot_entitystate_t *)  VMA(2) );
    case BOTLIB_TEST:
	return botlib_export->Test ((int )  args[1], ( char *)  VMA(2), (vec_t *)  VMA(3), (vec_t *)  VMA(4) );

    case BOTLIB_GET_SNAPSHOT_ENTITY:
	return SV_BotGetSnapshotEntity (( int )  args[1], ( int )  args[2] );
    case BOTLIB_GET_CONSOLE_MESSAGE:
	return SV_BotGetConsoleMessage (( int )  args[1], ( char *)  VMA(2), ( int )  args[3] );
    case BOTLIB_USER_COMMAND:
	SV_ClientThink ((client_t *)  &svs.clients[args[1]], ( usercmd_t *)  VMA(2) );
	return 0;

    case BOTLIB_AAS_BBOX_AREAS:
	return botlib_export->aas.AAS_BBoxAreas ((vec_t *)  VMA(1), (vec_t *)  VMA(2), ( int *)  VMA(3), ( int )  args[4] );
    case BOTLIB_AAS_AREA_INFO:
	return botlib_export->aas.AAS_AreaInfo (( int )  args[1], ( aas_areainfo_t *)  VMA(2) );
    case BOTLIB_AAS_ALTERNATIVE_ROUTE_GOAL:
	return botlib_export->aas.AAS_AlternativeRouteGoals((vec_t *) VMA(1), args[2], (vec_t *) VMA(3), args[4], args[5], 
		(aas_altroutegoal_t *) VMA(6), args[7], args[8] );
    case BOTLIB_AAS_ENTITY_INFO:
	botlib_export->aas.AAS_EntityInfo ((int )  args[1], ( aas_entityinfo_t *)  VMA(2) );
	return 0;

    case BOTLIB_AAS_INITIALIZED:
	return botlib_export->aas.AAS_Initialized();
    case BOTLIB_AAS_PRESENCE_TYPE_BOUNDING_BOX:
	botlib_export->aas.AAS_PresenceTypeBoundingBox ((int )  args[1], (vec_t *)  VMA(2), (vec_t *)  VMA(3) );
	return 0;
    case BOTLIB_AAS_TIME:
	return FloatAsInt( botlib_export->aas.AAS_Time() );

    case BOTLIB_AAS_POINT_AREA_NUM:
	return botlib_export->aas.AAS_PointAreaNum ((vec_t *)  VMA(1) );
    case BOTLIB_AAS_POINT_REACHABILITY_AREA_INDEX:
	return botlib_export->aas.AAS_PointReachabilityAreaIndex (( vec_t *)  VMA(1) );
    case BOTLIB_AAS_TRACE_AREAS:
	return botlib_export->aas.AAS_TraceAreas ((vec_t *)  VMA(1), (vec_t *)  VMA(2), ( int *)  VMA(3), ( vec3_t *)  VMA(4), ( int )  args[5] );

    case BOTLIB_AAS_POINT_CONTENTS:
	return botlib_export->aas.AAS_PointContents ((vec_t *)  VMA(1) );
    case BOTLIB_AAS_NEXT_BSP_ENTITY:
	return botlib_export->aas.AAS_NextBSPEntity ((int )  args[1] );
    case BOTLIB_AAS_VALUE_FOR_BSP_EPAIR_KEY:
	return botlib_export->aas.AAS_ValueForBSPEpairKey ((int )  args[1], ( char *)  VMA(2), ( char *)  VMA(3), ( int )  args[4] );
    case BOTLIB_AAS_VECTOR_FOR_BSP_EPAIR_KEY:
	return botlib_export->aas.AAS_VectorForBSPEpairKey ((int )  args[1], ( char *)  VMA(2), (vec_t *)  VMA(3) );
    case BOTLIB_AAS_FLOAT_FOR_BSP_EPAIR_KEY:
	return botlib_export->aas.AAS_FloatForBSPEpairKey ((int )  args[1], ( char *)  VMA(2), ( float *)  VMA(3) );
    case BOTLIB_AAS_INT_FOR_BSP_EPAIR_KEY:
	return botlib_export->aas.AAS_IntForBSPEpairKey ((int )  args[1], ( char *)  VMA(2), ( int *)  VMA(3) );

    case BOTLIB_AAS_AREA_REACHABILITY:
	return botlib_export->aas.AAS_AreaReachability ((int )  args[1] );

    case BOTLIB_AAS_AREA_TRAVEL_TIME_TO_GOAL_AREA:
	return botlib_export->aas.AAS_AreaTravelTimeToGoalArea ((int )  args[1], (vec_t *)  VMA(2), ( int )  args[3], ( int )  args[4] );
    case BOTLIB_AAS_ENABLE_ROUTING_AREA:
	return botlib_export->aas.AAS_EnableRoutingArea ((int )  args[1], ( int )  args[2] );
    case BOTLIB_AAS_PREDICT_ROUTE:
	return botlib_export->aas.AAS_PredictRoute( (struct aas_predictroute_s *) VMA(1), args[2], (vec_t *) VMA(3), args[4], args[5], args[6], args[7], args[8], args[9], args[10], args[11] );

    case BOTLIB_AAS_SWIMMING:
	return botlib_export->aas.AAS_Swimming ((vec_t *)  VMA(1) );
    case BOTLIB_AAS_PREDICT_CLIENT_MOVEMENT:
	return botlib_export->aas.AAS_PredictClientMovement((struct aas_clientmove_s *) VMA(1), args[2], (vec_t *) VMA(3), args[4], args[5],
		(vec_t *) VMA(6), (vec_t *) VMA(7), args[8], args[9], VMF(10), args[11], args[12], args[13] );

    case BOTLIB_EA_SAY:
	botlib_export->ea.EA_Say ((int )  args[1], ( char *)  VMA(2) );
	return 0;
    case BOTLIB_EA_SAY_TEAM:
	botlib_export->ea.EA_SayTeam ((int )  args[1], ( char *)  VMA(2) );
	return 0;
    case BOTLIB_EA_COMMAND:
	botlib_export->ea.EA_Command ((int )  args[1], ( char *)  VMA(2) );
	return 0;

    case BOTLIB_EA_ACTION:
	botlib_export->ea.EA_Action ((int )  args[1], ( int )  args[2] );
	break;
    case BOTLIB_EA_GESTURE:
	botlib_export->ea.EA_Gesture ((int )  args[1] );
	return 0;
    case BOTLIB_EA_TALK:
	botlib_export->ea.EA_Talk ((int )  args[1] );
	return 0;
    case BOTLIB_EA_ATTACK:
	botlib_export->ea.EA_Attack ((int )  args[1] );
	return 0;
    case BOTLIB_EA_USE:
	botlib_export->ea.EA_Use ((int )  args[1] );
	return 0;
    case BOTLIB_EA_RESPAWN:
	botlib_export->ea.EA_Respawn ((int )  args[1] );
	return 0;
    case BOTLIB_EA_CROUCH:
	botlib_export->ea.EA_Crouch ((int )  args[1] );
	return 0;
    case BOTLIB_EA_MOVE_UP:
	botlib_export->ea.EA_MoveUp ((int )  args[1] );
	return 0;
    case BOTLIB_EA_MOVE_DOWN:
	botlib_export->ea.EA_MoveDown ((int )  args[1] );
	return 0;
    case BOTLIB_EA_MOVE_FORWARD:
	botlib_export->ea.EA_MoveForward ((int )  args[1] );
	return 0;
    case BOTLIB_EA_MOVE_BACK:
	botlib_export->ea.EA_MoveBack ((int )  args[1] );
	return 0;
    case BOTLIB_EA_MOVE_LEFT:
	botlib_export->ea.EA_MoveLeft ((int )  args[1] );
	return 0;
    case BOTLIB_EA_MOVE_RIGHT:
	botlib_export->ea.EA_MoveRight ((int )  args[1] );
	return 0;

    case BOTLIB_EA_SELECT_WEAPON:
	botlib_export->ea.EA_SelectWeapon ((int )  args[1], ( int )  args[2] );
	return 0;
    case BOTLIB_EA_JUMP:
	botlib_export->ea.EA_Jump ((int )  args[1] );
	return 0;
    case BOTLIB_EA_DELAYED_JUMP:
	botlib_export->ea.EA_DelayedJump ((int )  args[1] );
	return 0;
    case BOTLIB_EA_MOVE:
	botlib_export->ea.EA_Move ((int )  args[1], (vec_t *)  VMA(2), ( float )  VMF(3) );
	return 0;
    case BOTLIB_EA_VIEW:
	botlib_export->ea.EA_View ((int )  args[1], (vec_t *)  VMA(2) );
	return 0;

    case BOTLIB_EA_END_REGULAR:
	botlib_export->ea.EA_EndRegular ((int )  args[1], ( float )  VMF(2) );
	return 0;
    case BOTLIB_EA_GET_INPUT:
	botlib_export->ea.EA_GetInput ((int )  args[1], ( float )  VMF(2), ( bot_input_t *)  VMA(3) );
	return 0;
    case BOTLIB_EA_RESET_INPUT:
	botlib_export->ea.EA_ResetInput ((int )  args[1] );
	return 0;

    case BOTLIB_AI_LOAD_CHARACTER:
	return botlib_export->ai.BotLoadCharacter ((char *)  VMA(1), ( float )  VMF(2) );
    case BOTLIB_AI_FREE_CHARACTER:
	botlib_export->ai.BotFreeCharacter ((int )  args[1] );
	return 0;
    case BOTLIB_AI_CHARACTERISTIC_FLOAT:
	return FloatAsInt( botlib_export->ai.Characteristic_Float( args[1], args[2] ) );
    case BOTLIB_AI_CHARACTERISTIC_BFLOAT:
	return FloatAsInt( botlib_export->ai.Characteristic_BFloat( args[1], args[2], VMF(3), VMF(4) ) );
    case BOTLIB_AI_CHARACTERISTIC_INTEGER:
	return botlib_export->ai.Characteristic_Integer ((int )  args[1], ( int )  args[2] );
    case BOTLIB_AI_CHARACTERISTIC_BINTEGER:
	return botlib_export->ai.Characteristic_BInteger ((int )  args[1], ( int )  args[2], ( int )  args[3], ( int )  args[4] );
    case BOTLIB_AI_CHARACTERISTIC_STRING:
	botlib_export->ai.Characteristic_String ((int )  args[1], ( int )  args[2], ( char *)  VMA(3), ( int )  args[4] );
	return 0;

    case BOTLIB_AI_ALLOC_CHAT_STATE:
	return botlib_export->ai.BotAllocChatState();
    case BOTLIB_AI_FREE_CHAT_STATE:
	botlib_export->ai.BotFreeChatState ((int )  args[1] );
	return 0;
    case BOTLIB_AI_QUEUE_CONSOLE_MESSAGE:
	botlib_export->ai.BotQueueConsoleMessage ((int )  args[1], ( int )  args[2], ( char *)  VMA(3) );
	return 0;
    case BOTLIB_AI_REMOVE_CONSOLE_MESSAGE:
	botlib_export->ai.BotRemoveConsoleMessage ((int )  args[1], ( int )  args[2] );
	return 0;
    case BOTLIB_AI_NEXT_CONSOLE_MESSAGE:
	return botlib_export->ai.BotNextConsoleMessage ((int )  args[1], ( bot_consolemessage_t *)  VMA(2) );
    case BOTLIB_AI_NUM_CONSOLE_MESSAGE:
	return botlib_export->ai.BotNumConsoleMessages ((int )  args[1] );
    case BOTLIB_AI_INITIAL_CHAT:
	botlib_export->ai.BotInitialChat ((int )  args[1], ( char *)  VMA(2), ( int )  args[3], ( char *)  VMA(4), ( char *)  VMA(5), ( char *)  VMA(6), ( char *)  VMA(7), ( char *)  VMA(8), ( char *)  VMA(9), ( char *)  VMA(10), ( char *)  VMA(11) );
	return 0;
    case BOTLIB_AI_NUM_INITIAL_CHATS:
	return botlib_export->ai.BotNumInitialChats ((int )  args[1], ( char *)  VMA(2) );
    case BOTLIB_AI_REPLY_CHAT:
	return botlib_export->ai.BotReplyChat ((int )  args[1], ( char *)  VMA(2), ( int )  args[3], ( int )  args[4], ( char *)  VMA(5), ( char *)  VMA(6), ( char *)  VMA(7), ( char *)  VMA(8), ( char *)  VMA(9), ( char *)  VMA(10), ( char *)  VMA(11), ( char *)  VMA(12) );
    case BOTLIB_AI_CHAT_LENGTH:
	return botlib_export->ai.BotChatLength ((int )  args[1] );
    case BOTLIB_AI_ENTER_CHAT:
	botlib_export->ai.BotEnterChat ((int )  args[1], ( int )  args[2], ( int )  args[3] );
	return 0;
    case BOTLIB_AI_GET_CHAT_MESSAGE:
	botlib_export->ai.BotGetChatMessage ((int )  args[1], ( char *)  VMA(2), ( int )  args[3] );
	return 0;
    case BOTLIB_AI_STRING_CONTAINS:
	return botlib_export->ai.StringContains ((char *)  VMA(1), ( char *)  VMA(2), ( int )  args[3] );
    case BOTLIB_AI_FIND_MATCH:
	return botlib_export->ai.BotFindMatch ((char *)  VMA(1), ( bot_match_t *)  VMA(2), ( unsigned long int )  args[3] );
    case BOTLIB_AI_MATCH_VARIABLE:
	botlib_export->ai.BotMatchVariable ((bot_match_t *)  VMA(1), ( int )  args[2], ( char *)  VMA(3), ( int )  args[4] );
	return 0;
    case BOTLIB_AI_UNIFY_WHITE_SPACES:
	botlib_export->ai.UnifyWhiteSpaces ((char *)  VMA(1) );
	return 0;
    case BOTLIB_AI_REPLACE_SYNONYMS:
	botlib_export->ai.BotReplaceSynonyms ((char *)  VMA(1), ( unsigned long int )  args[2] );
	return 0;
    case BOTLIB_AI_LOAD_CHAT_FILE:
	return botlib_export->ai.BotLoadChatFile ((int )  args[1], ( char *)  VMA(2), ( char *)  VMA(3) );
    case BOTLIB_AI_SET_CHAT_GENDER:
	botlib_export->ai.BotSetChatGender ((int )  args[1], ( int )  args[2] );
	return 0;
    case BOTLIB_AI_SET_CHAT_NAME:
	botlib_export->ai.BotSetChatName ((int )  args[1], ( char *)  VMA(2), ( int )  args[3] );
	return 0;

    case BOTLIB_AI_RESET_GOAL_STATE:
	botlib_export->ai.BotResetGoalState ((int )  args[1] );
	return 0;
    case BOTLIB_AI_RESET_AVOID_GOALS:
	botlib_export->ai.BotResetAvoidGoals ((int )  args[1] );
	return 0;
    case BOTLIB_AI_REMOVE_FROM_AVOID_GOALS:
	botlib_export->ai.BotRemoveFromAvoidGoals ((int )  args[1], ( int )  args[2] );
	return 0;
    case BOTLIB_AI_PUSH_GOAL:
	botlib_export->ai.BotPushGoal ((int )  args[1], ( bot_goal_t *)  VMA(2) );
	return 0;
    case BOTLIB_AI_POP_GOAL:
	botlib_export->ai.BotPopGoal ((int )  args[1] );
	return 0;
    case BOTLIB_AI_EMPTY_GOAL_STACK:
	botlib_export->ai.BotEmptyGoalStack ((int )  args[1] );
	return 0;
    case BOTLIB_AI_DUMP_AVOID_GOALS:
	botlib_export->ai.BotDumpAvoidGoals ((int )  args[1] );
	return 0;
    case BOTLIB_AI_DUMP_GOAL_STACK:
	botlib_export->ai.BotDumpGoalStack ((int )  args[1] );
	return 0;
    case BOTLIB_AI_GOAL_NAME:
	botlib_export->ai.BotGoalName ((int )  args[1], ( char *)  VMA(2), ( int )  args[3] );
	return 0;
    case BOTLIB_AI_GET_TOP_GOAL:
	return botlib_export->ai.BotGetTopGoal ((int )  args[1], ( bot_goal_t *)  VMA(2) );
    case BOTLIB_AI_GET_SECOND_GOAL:
	return botlib_export->ai.BotGetSecondGoal ((int )  args[1], ( bot_goal_t *)  VMA(2) );
    case BOTLIB_AI_CHOOSE_LTG_ITEM:
	return botlib_export->ai.BotChooseLTGItem ((int )  args[1], (vec_t *)  VMA(2), ( int *)  VMA(3), ( int )  args[4] );
    case BOTLIB_AI_CHOOSE_NBG_ITEM:
	return botlib_export->ai.BotChooseNBGItem( args[1], (vec_t *) VMA(2), (int *) VMA(3), args[4], (bot_goal_t *) VMA(5), VMF(6) );
    case BOTLIB_AI_TOUCHING_GOAL:
	return botlib_export->ai.BotTouchingGoal ((vec_t *)  VMA(1), ( bot_goal_t *)  VMA(2) );
    case BOTLIB_AI_ITEM_GOAL_IN_VIS_BUT_NOT_VISIBLE:
	return botlib_export->ai.BotItemGoalInVisButNotVisible ((int )  args[1], (vec_t *)  VMA(2), (vec_t *)  VMA(3), ( bot_goal_t *)  VMA(4) );
    case BOTLIB_AI_GET_LEVEL_ITEM_GOAL:
	return botlib_export->ai.BotGetLevelItemGoal ((int )  args[1], ( char *)  VMA(2), ( bot_goal_t *)  VMA(3) );
    case BOTLIB_AI_GET_NEXT_CAMP_SPOT_GOAL:
	return botlib_export->ai.BotGetNextCampSpotGoal ((int )  args[1], ( bot_goal_t *)  VMA(2) );
    case BOTLIB_AI_GET_MAP_LOCATION_GOAL:
	return botlib_export->ai.BotGetMapLocationGoal ((char *)  VMA(1), ( bot_goal_t *)  VMA(2) );
    case BOTLIB_AI_AVOID_GOAL_TIME:
	return FloatAsInt( botlib_export->ai.BotAvoidGoalTime( args[1], args[2] ) );
    case BOTLIB_AI_SET_AVOID_GOAL_TIME:
	botlib_export->ai.BotSetAvoidGoalTime ((int )  args[1], ( int )  args[2], ( float )  VMF(3));
	return 0;
    case BOTLIB_AI_INIT_LEVEL_ITEMS:
	botlib_export->ai.BotInitLevelItems();
	return 0;
    case BOTLIB_AI_UPDATE_ENTITY_ITEMS:
	botlib_export->ai.BotUpdateEntityItems();
	return 0;
    case BOTLIB_AI_LOAD_ITEM_WEIGHTS:
	return botlib_export->ai.BotLoadItemWeights ((int )  args[1], ( char *)  VMA(2) );
    case BOTLIB_AI_FREE_ITEM_WEIGHTS:
	botlib_export->ai.BotFreeItemWeights ((int )  args[1] );
	return 0;
    case BOTLIB_AI_INTERBREED_GOAL_FUZZY_LOGIC:
	botlib_export->ai.BotInterbreedGoalFuzzyLogic ((int )  args[1], ( int )  args[2], ( int )  args[3] );
	return 0;
    case BOTLIB_AI_SAVE_GOAL_FUZZY_LOGIC:
	botlib_export->ai.BotSaveGoalFuzzyLogic ((int )  args[1], ( char *)  VMA(2) );
	return 0;
    case BOTLIB_AI_MUTATE_GOAL_FUZZY_LOGIC:
	botlib_export->ai.BotMutateGoalFuzzyLogic ((int )  args[1], ( float )  VMF(2) );
	return 0;
    case BOTLIB_AI_ALLOC_GOAL_STATE:
	return botlib_export->ai.BotAllocGoalState ((int )  args[1] );
    case BOTLIB_AI_FREE_GOAL_STATE:
	botlib_export->ai.BotFreeGoalState ((int )  args[1] );
	return 0;

    case BOTLIB_AI_RESET_MOVE_STATE:
	botlib_export->ai.BotResetMoveState ((int )  args[1] );
	return 0;
    case BOTLIB_AI_ADD_AVOID_SPOT:
	botlib_export->ai.BotAddAvoidSpot ((int )  args[1], (vec_t *)  VMA(2), ( float )  VMF(3), ( int )  args[4] );
	return 0;
    case BOTLIB_AI_MOVE_TO_GOAL:
	botlib_export->ai.BotMoveToGoal ((bot_moveresult_t *)  VMA(1), ( int )  args[2], ( bot_goal_t *)  VMA(3), ( int )  args[4] );
	return 0;
    case BOTLIB_AI_MOVE_IN_DIRECTION:
	botlib_export->ai.BotMoveInDirection ((int )  args[1], (vec_t *)  VMA(2), ( float )  VMF(3), ( int )  args[4] );
    case BOTLIB_AI_RESET_AVOID_REACH:
	botlib_export->ai.BotResetAvoidReach ((int )  args[1] );
	return 0;
    case BOTLIB_AI_RESET_LAST_AVOID_REACH:
	botlib_export->ai.BotResetLastAvoidReach ((int )  args[1] );
	return 0;
    case BOTLIB_AI_REACHABILITY_AREA:
	return botlib_export->ai.BotReachabilityArea ((vec_t *)  VMA(1), ( int )  args[2] );
    case BOTLIB_AI_MOVEMENT_VIEW_TARGET:
	return botlib_export->ai.BotMovementViewTarget ((int )  args[1], ( bot_goal_t *)  VMA(2), ( int )  args[3], ( float )  VMF(4), (vec_t *)  VMA(5) );
    case BOTLIB_AI_PREDICT_VISIBLE_POSITION:
	return botlib_export->ai.BotPredictVisiblePosition ((vec_t *)  VMA(1), ( int )  args[2], ( bot_goal_t *)  VMA(3), ( int )  args[4], (vec_t *)  VMA(5) );
    case BOTLIB_AI_ALLOC_MOVE_STATE:
	return botlib_export->ai.BotAllocMoveState();
    case BOTLIB_AI_FREE_MOVE_STATE:
	botlib_export->ai.BotFreeMoveState ((int )  args[1] );
	return 0;
    case BOTLIB_AI_INIT_MOVE_STATE:
	botlib_export->ai.BotInitMoveState ((int )  args[1], ( bot_initmove_t *)  VMA(2) );
	return 0;

    case BOTLIB_AI_CHOOSE_BEST_FIGHT_WEAPON:
	return botlib_export->ai.BotChooseBestFightWeapon ((int )  args[1], ( int *)  VMA(2) );
    case BOTLIB_AI_GET_WEAPON_INFO:
	botlib_export->ai.BotGetWeaponInfo ((int )  args[1], ( int )  args[2], ( weaponinfo_t *)  VMA(3) );
	return 0;
    case BOTLIB_AI_LOAD_WEAPON_WEIGHTS:
	return botlib_export->ai.BotLoadWeaponWeights ((int )  args[1], ( char *)  VMA(2) );
    case BOTLIB_AI_ALLOC_WEAPON_STATE:
	return botlib_export->ai.BotAllocWeaponState();
    case BOTLIB_AI_FREE_WEAPON_STATE:
	botlib_export->ai.BotFreeWeaponState ((int )  args[1] );
	return 0;
    case BOTLIB_AI_RESET_WEAPON_STATE:
	botlib_export->ai.BotResetWeaponState ((int )  args[1] );
	return 0;

    case BOTLIB_AI_GENETIC_PARENTS_AND_CHILD_SELECTION:
	return botlib_export->ai.GeneticParentsAndChildSelection ((int ) args[1], ( float *)  VMA(2), ( int *)  VMA(3), ( int *)  VMA(4), ( int *)  VMA(5));

    case TRAP_MEMSET:
	Com_Memset ((void* )  VMA(1), ( int )  args[2], ( size_t )  args[3] );
	return 0;

    case TRAP_MEMCPY:
	Com_Memcpy ((void* )  VMA(1), ( void* )  VMA(2), ( size_t )  args[3] );
	return 0;

    case TRAP_STRNCPY:
	return (int)strncpy((char *) VMA(1), (char *) VMA(2), args[3] );

    case TRAP_SIN:
	return FloatAsInt( sin( VMF(1) ) );

    case TRAP_COS:
	return FloatAsInt( cos( VMF(1) ) );

    case TRAP_ATAN2:
	return FloatAsInt( atan2( VMF(1), VMF(2) ) );

    case TRAP_SQRT:
	return FloatAsInt( sqrt( VMF(1) ) );

    case TRAP_MATRIXMULTIPLY:
	MatrixMultiply ((float (*)[3]) VMA(1), (float (*)[3]) VMA(2), (float (*)[3]) VMA(3) );
	return 0;

    case TRAP_ANGLEVECTORS:
	AngleVectors ((vec_t *)  VMA(1), (vec_t *)  VMA(2), (vec_t *)  VMA(3), (vec_t *)  VMA(4) );
	return 0;

    case TRAP_PERPENDICULARVECTOR:
	PerpendicularVector ((vec_t *)  VMA(1), (vec_t *)  VMA(2) );
	return 0;

    case TRAP_FLOOR:
	return FloatAsInt( floor( VMF(1) ) );

    case TRAP_CEIL:
	return FloatAsInt( ceil( VMF(1) ) );


    default:
	Com_Error ((int )  ERR_DROP, ( char *)  "Bad game system trap: %i",  args[0] );
    }
    return -1;
}
/*
===============
SV_ShutdownGameProgs

Called every time a map changes
===============
*/
void SV_ShutdownGameProgs( void ) {
	if ( !gvm ) {
		return;
	}
	VM_Call( gvm, GAME_SHUTDOWN, qfalse );
	VM_Free( gvm );
	gvm = NULL;
}

/*
==================
SV_InitGameVM

Called for both a full init and a restart
==================
*/
static void SV_InitGameVM( qboolean restart ) {
	int		i;

	// start the entity parsing at the beginning
	sv.entityParsePoint = CM_EntityString();

	// clear all gentity pointers that might still be set from
	// a previous level
	// https://zerowing.idsoftware.com/bugzilla/show_bug.cgi?id=522
	//   now done before GAME_INIT call
	for ( i = 0 ; i < sv_maxclients->integer ; i++ ) {
		svs.clients[i].gentity = NULL;
	}
	
	// use the current msec count for a random seed
	// init for this gamestate
	VM_Call( gvm, GAME_INIT, svs.time, Com_Milliseconds(), restart );
}



/*
===================
SV_RestartGameProgs

Called on a map_restart, but not on a normal map change
===================
*/
void SV_RestartGameProgs( void ) {
	if ( !gvm ) {
		return;
	}
	VM_Call( gvm, GAME_SHUTDOWN, qtrue );

	// do a restart instead of a free
	gvm = VM_Restart( gvm );
	if ( !gvm ) { // bk001212 - as done below
		Com_Error( ERR_FATAL, "VM_Restart on game failed" );
	}

	SV_InitGameVM( qtrue );
}


/*
===============
SV_InitGameProgs

Called on a normal map change, not on a map_restart
===============
*/
#ifdef GAME_HARD_LINKED
extern int qagame_vmMain
 (int command, int arg0, int arg1, int arg2, int arg3, int arg4, 
  int arg5, int arg6, int arg7, int arg8, int arg9, int arg10, int arg11);

extern void qagame_dllEntry
( int (QDECL *syscallptr)( int arg,... ) );
#endif


void SV_InitGameProgs( void ) {
	cvar_t	*var;
	//FIXME these are temp while I make bots run in vm
	extern int	bot_enable;

	var = Cvar_Get( "bot_enable", "1", CVAR_LATCH );
	if ( var ) {
		bot_enable = var->integer;
	}
	else {
		bot_enable = 0;
	}

	// load the dll or bytecode
#ifdef GAME_HARD_LINKED       // make sure this is defined when compiling
	gvm = VM_Create ("qagame", SV_GameSystemCalls, VMI_NATIVE_LINKED, (vmEntryFunc_t) qagame_vmMain, (vmSyscallSet_t) qagame_dllEntry);
#else
	gvm = VM_Create ("qagame", SV_GameSystemCalls, Cvar_VariableValue( "vm_game" ), NULL, NULL );
#endif
	if ( !gvm ) {
		Com_Error( ERR_FATAL, "VM_Create on game failed" );
	}

	SV_InitGameVM( qfalse );
}


/*
====================
SV_GameCommand

See if the current console command is claimed by the game
====================
*/
qboolean SV_GameCommand( void ) {
	if ( sv.state != SS_GAME ) {
		return qfalse;
	}

	return (qboolean) VM_Call( gvm, GAME_CONSOLE_COMMAND );
}

