////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

// Questions we want to answer using this instrumentation:
//  1. How many remote writes and remote reads happen? 
//      * whenever a write or read occurs, we need to know which 
//        entity's think function caused it, as well as which entity
//        this action occured on. 
//        
//      * 
//
//  2. ...
//  
// the log will look like
//
// event_type	field_index	entity_number	entity_classname

#include "../game/g_local.h"
#include <om/common.h>
#include <util/ExpLog.h>
#include <hash_map.h>
#include "metadata2.cxx"
#include "record.h"

/* Stuff copied from ../../Quake3Metadata.cpp */

#define GENTITY_FOFS(x) (int)&(((gentity_t *)0)->x)

static int s_DynamicStringOffsets [] = {
    GENTITY_FOFS(classname),
    GENTITY_FOFS(model),
    GENTITY_FOFS(model2),
    GENTITY_FOFS(target),
    GENTITY_FOFS(targetname),
    GENTITY_FOFS(message),
    GENTITY_FOFS(team),
    GENTITY_FOFS(targetShaderName),
    GENTITY_FOFS(targetShaderNewName),
    -1
};

struct hash_char_ptr {
    size_t operator () (char *s) const { return hash<char *>()(s); }
};
struct eq_char_ptr { 
    bool operator () (char *s1, char *s2) const { return !strcmp (s1, s2); }
};

template <typename T, class hashFunc, class eqFunc>
class PointerTable {
    typedef hash_map <T *, int, hashFunc, eqFunc> ptrHash;

    ptrHash        table;
    vector <T *>   array;

public:
    PointerTable () {} 
    ~PointerTable () {}

    void add (T* ptr) { 
	if (table.find (ptr) != table.end ()) {
	    return;
	}
	int n = array.size ();		
	array.push_back (ptr);
	table.insert (typename ptrHash::value_type (ptr, n));
    }

    int ptrToIndex (T* ptr) { 
	typename ptrHash::iterator it = table.find (ptr);
	if (it == table.end ())
	    return -1;
	else 
	    return it->second;
    }

    T* indexToPtr (int index) { 
	if (index >= (int) array.size ()) {
	    WARN << "index (" << index << ") out of bounds " << endl;
	    return NULL;    
	}

	return array [index];
    }

    void dump (FILE *fp) {
	int len = (int) array.size ();
	for (int i = 0; i < len; i++) { 
	    char *str = (char *) array [i];
	    if (strchr (str, '\n') || strchr (str, '\r')) 
		fprintf (fp, "Dont_care_string\n");
	    else 
		fprintf (fp, "%s\n", str);
	}
    }
};

typedef PointerTable<char, hash_char_ptr, eq_char_ptr> StringTable;
static StringTable s_StringsTable;
static bool s_Initialized = false;

#define ASIZE(arr)  (sizeof(arr)/sizeof(void *))

void InitializeStringsTable () {
    // add strings
    for (int i = 0, n = ASIZE (s_StaticStrings); i < n; i++) 
	s_StringsTable.add (s_StaticStrings [i]);
    
    gentity_t *ent = &g_entities [0];
    for (int i = 0; i < level.num_entities; i++, ent++) {
	if (!ent->inuse)
	    continue;
	
	for (int *off = s_DynamicStringOffsets; *off >= 0; off++) {
	    char *ptr = * (char **) ((byte *) ent + *off);
	    if (ptr)
		s_StringsTable.add (ptr);
	}
    }

    FILE *fp = fopen ("strings.out", "w");
    if (!fp) 
	Debug::die ("Could not open strings.out for writing!");
    
    s_StringsTable.dump (fp);
    fclose (fp);

    s_Initialized = true;
}

struct RWEntry : public ExpLogEntry 
{
    typedef enum { READ, WRITE, FUNCALL_PRE, FUNCALL_POST, FUNC_WRITE } EventType;
	
    EventType   type;         // type of the event
    int         ent_number;   
    char        ent_name [64];
    char        func_name [64];
    TYPE_FIELD_TYPE  field;        // which field was read/write/funcall'ed
    
    RWEntry () {};
    RWEntry (EventType type, gentity_t *ent, TYPE_FIELD_TYPE field) : 
	type (type), ent_number (ent->s.number), field (field) 
    {
	SetEntName (ent);
    }
    RWEntry (gentity_t *ent, TYPE_FIELD_TYPE field, const char *func) :
	type (FUNC_WRITE), ent_number (ent->s.number), field (field)
    {
	SetEntName (ent);
	strncpy (func_name, func, sizeof (func_name));
    }
		
    virtual ~RWEntry () {} 

    void SetEntName (gentity_t *ent) {
	memset (ent_name, 0, sizeof (ent_name));
	strcpy (ent_name, "null");
	if (ent->classname) 
	    strncpy (ent_name, ent->classname, sizeof (ent_name) - 1);
    }
    
    uint32 Dump (FILE *fp) { 
	int cn_index = s_StringsTable.ptrToIndex (ent_name);
	uint32 ret;
	
	ret = fprintf (fp, "%d\t%d\t%d\t%d", 
		(int) type, 
		field,
		ent_number, 
		cn_index);

	if (type == FUNC_WRITE) 
	    ret += fprintf (fp, "\t%s", func_name);
#if 0
	if (cn_index == -1)
	    ret += fprintf (fp, "\t%s\n", ent_name);
	else
#endif
	    ret += fprintf (fp, "\n");

	return ret;
    }
};

DECLARE_EXPLOG(RWLog, RWEntry);
IMPLEMENT_EXPLOG(RWLog, RWEntry);

extern gclient_t g_clients [];

template<class T>
int getIndex (void *ptr, T *begin) {
    int diff = ((byte *) ptr) - ((byte *) begin);
    return diff / (int) sizeof (T);
}

inline int getPointerIndex (void *ptr, int isClientPtr) { 
#if 0
    // gets initialized from InitializeStringsTable() which is called at an 
    // opportune moment from the top loop
    if (!s_Initialized) 
	return -1;
#endif

    if (!ptr)  // Happens due to FOFS
	return -1;

    // Entity pointer resolution
    int index;

    if (isClientPtr != 0) {
	index = getIndex<gclient_t> (ptr, &g_clients [0]);
	if (index < 0 || index >= MAX_CLIENTS)
	    return -1;

	// ASSERTDO (index >= 0 && index < MAX_CLIENTS, 
	//	WARN << merc_va ("ptr %p (field %s, isClient %d) not resolved", ptr, field, isClientPtr) << endl
	//	);
    }
    else {
	index = getIndex<gentity_t> (ptr, &g_entities [0]);
	if (index < 0 || index >= MAX_GENTITIES)
	    return -1;
	
	// ASSERTDO (index >= 0 && index < MAX_GENTITIES,
	//	WARN << merc_va ("ptr %p (field %s, isClient %d) not resolved", ptr, field, isClientPtr) << endl
	//       );
    }

    // dont log entities which are really not present in the game yet
    // but are checked (typically for the `inuse' field) inside a big
    // loop. 
    //
    if (index >= level.num_entities)
	return -1; 

    return index;
}

inline void Record (RWEntry::EventType etype, void *ptr, TYPE_FIELD_TYPE field, int isClientPtr)
{
    int index = getPointerIndex (ptr, isClientPtr);
    if (index < 0)
	return;
    gentity_t *ent = &g_entities [index];
    // ASSERT (ent->inuse);  // this aint true always. during initialization, random crap happens.

    RWEntry e (etype, ent, field);
    LOG (RWLog, e);
}

void RecordFuncWrite (void *ptr, TYPE_FIELD_TYPE field, const char *funcName) 
{
    int index = getPointerIndex (ptr, 0);   // these are never client fields
    if (index < 0)
	return;
    gentity_t *ent = &g_entities [index];
    RWEntry e (ent, field, funcName);
    LOG (RWLog, e);
}

// Why this duplication? Perhaps, at a later point in time, you may want
// to have different functionality for read/write/funcall recording
// 
void RecordWrite (void *ptr, TYPE_FIELD_TYPE field, int isClientPtr) 
{
    Record (RWEntry::WRITE, ptr, field, isClientPtr);
}
void RecordRead (void *ptr, TYPE_FIELD_TYPE field, int isClientPtr) 
{
    Record (RWEntry::READ, ptr, field, isClientPtr);
}
void RecordFuncallPre (void *ptr, TYPE_FIELD_TYPE field, int isClientPtr) 
{
    Record (RWEntry::FUNCALL_PRE, ptr, field, isClientPtr);
}
void RecordFuncallPost (void *ptr, TYPE_FIELD_TYPE field, int isClientPtr) 
{
    Record (RWEntry::FUNCALL_POST, ptr, field, isClientPtr);
}

void InitRecordingLogs () {
    if (!g_RWLog)
	INIT_EXPLOG(RWLog, g_LocalSID);
}

// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
