(*
 *
 * Copyright (c) 2001-2002, 
 *  George C. Necula    <necula@cs.berkeley.edu>
 *  Scott McPeak        <smcpeak@cs.berkeley.edu>
 *  Wes Weimer          <weimer@cs.berkeley.edu>
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 * 1. Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *
 * 3. The names of the contributors may not be used to endorse or promote
 * products derived from this software without specific prior written
 * permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
 * IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *)

open Pretty
open Cil
open Str
open List
open String
module E = Errormsg
module H = Hashtbl

class logStringVisitor = object
  inherit nopCilVisitor
  (* Create a prototype for the logging function, but don't put it in the 
   * file *)
  val printfFun =   
    let fdec = emptyFunction "syslog" in
    fdec.svar.vtype <- TFun(intType, 
                            Some [ ("prio", intType, []);
                                   ("format", charConstPtrType, []) ], 
                            true, []);
    fdec

(* the objective is to extract all strings; but try to avoid "stupid" 
   strings occuring within dprintf and other kinds of statements 
*)

  method vexpr (e: exp) : exp visitAction = 
    match e with 
	Const(c) -> begin
	  match c with 
	      CStr(string) -> 
		ignore (Pretty.printf "%a\n" d_exp e);
		SkipChildren
	    | _ -> DoChildren
	end
      | _ -> DoChildren

	
  method vinst (i: instr) : instr list visitAction = 
    match i with
	Call(_,name,args,_) -> 
	  (*
	  ignore (Pretty.printf "func: %a\n" d_exp name);
	  let f elem = 
	    ignore (Pretty.printf "arg: %a\n" d_exp elem)
	  in 
	    List.iter f args;
	  *)
	  let fname = Pretty.sprint 80 (Pretty.dprintf "%a" d_exp name) in 
	  let pat   = regexp "printf|^va|gi.error" in
	    (* if fname =~ /printf/ SkipChildren *)
	  let matched = 
	    try search_forward pat fname 0 with
		Not_found -> -1 in
	    if matched = -1 then begin
	      (* ignore (Pretty.printf "didn't match fname = %s\n" fname); *)
	      DoChildren;
	    end
	    else begin
	      (* ignore  (Pretty.printf "MATCHED fname = %s\n" fname); *) 
	      SkipChildren;
	    end
      | _ -> DoChildren


 (* for extracting global array declarations and some other global structures 
   from the quake source code *)

(*
  method vglob (g: global) : global list visitAction = 
    match g with 
	GVar(var, init, loc) -> begin
	  match var.vtype with
	      TArray(base_type, arr_exp, _) -> begin
		let len = lenOfArray(arr_exp) in
		  begin
		    ignore (Pretty.printf "type %a " d_type base_type); 
		    ignore (Pretty.printf "arrayvar %s len %d\n" var.vname len);
		    DoChildren
		  end
	      end
	    | TNamed(type_info, _) ->
		ignore (printf "type %s var %s\n" type_info.tname var.vname);
		DoChildren
	    | TComp(comp_info, attribs) ->
		let name = compFullName comp_info in 
		  ignore (Pretty.printf "type %s var %s\n" name var.vname);
		  DoChildren
	    | _ -> DoChildren
	end
      | _ -> DoChildren
*)

end

let feature : featureDescr = 
  { fd_name = "logstrings";
    fd_enabled = ref false;
    fd_description = "extracting of strings";
    fd_extraopt = [];
    fd_doit = 
    (function (f: file) -> 
      let lwVisitor = new logStringVisitor in
      visitCilFileSameGlobals lwVisitor f);
    fd_post_check = true;
  } 

