(*
 *
 * Copyright (c) 2001-2002, 
 *  George C. Necula    <necula@cs.berkeley.edu>
 *  Scott McPeak        <smcpeak@cs.berkeley.edu>
 *  Wes Weimer          <weimer@cs.berkeley.edu>
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 * 1. Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *
 * 3. The names of the contributors may not be used to endorse or promote
 * products derived from this software without specific prior written
 * permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
 * IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *)

open Pretty
open Cil
open Str
open List
open String
module E = Errormsg
module H = Hashtbl

let sprintf x = Pretty.sprint 80 x

let getPointerType t = 
  match t with 
    | TFun(_,_,_,_) -> "function"
    | _ -> sprintf (Pretty.dprintf "%a" d_type t)
    
let rec dumpType t prefix  = 
  match t with 
      TVoid(_) -> sprintf (Pretty.dprintf "type=:void: name=:%s:" prefix)
    | TInt(kind,_) -> sprintf (Pretty.dprintf "type=:%a: name=:%s:" d_ikind kind prefix)
    | TFloat(kind,_) -> sprintf (Pretty.dprintf "type=:%a: name=:%s:" d_fkind kind prefix)
    | TEnum(einfo,_) -> sprintf (Pretty.dprintf "type=:enum: name=:%s: nitems=%d" prefix (List.length einfo.eitems))
    | TPtr(pt,_) -> 
	sprintf (Pretty.dprintf "type=:pointer: name=:%s: base=:%s:" prefix (getPointerType pt))
    | TArray(bt,exp,_) -> 
        sprintf (Pretty.dprintf "type=:array: name=:%s: base=:%s: len=%d" prefix (getPointerType bt) (lenOfArray exp))
    | TComp(cinfo,_) -> 
	   let f x = 
	     let s = if prefix = "" then x.fname else prefix ^ "." ^ x.fname in 
		dumpType (Cil.unrollTypeDeep x.ftype) s 
	    in 	    
	        String.concat "\n" (List.map f cinfo.cfields) 
    | _ -> sprintf (Pretty.dprintf "unmatched type=%a" d_type t)

class extDSVisitor = object
  inherit nopCilVisitor
  (* Create a prototype for the logging function, but don't put it in the 
   * file *)
  val printfFun =   
    let fdec = emptyFunction "syslog" in
    fdec.svar.vtype <- TFun(intType, 
                            Some [ ("prio", intType, []);
                                   ("format", charConstPtrType, []) ], 
                            true, []);
    fdec

  method vglob (g: global) : global list visitAction = 
  match g with
   GVar(var, init, loc) -> begin
    match var.vname with 
     "g_clients" |
     "botstates" | 
     "g_entities" -> begin 
          match var.vtype with
              TArray(base_type, arr_exp, _) -> begin
	        (* ignore (Pretty.printf "unrolled=%a\n" d_type
		(Cil.unrollTypeDeep base_type)); *)
		ignore (Pretty.printf "<TYPEINFO typename='%s'>\n%s\n</TYPEINFO>\n" var.vname (dumpType (Cil.unrollTypeDeep base_type) ""));
		DoChildren;
              end
            | _ -> DoChildren
        end
     
      | _ -> DoChildren
    end
    | _ -> DoChildren

end

let feature : featureDescr = 
  { fd_name = "extractdatastruct";
    fd_enabled = ref false;
    fd_description = "extraction of the full definition of a datatype";
    fd_extraopt = [];
    fd_doit = 
    (function (f: file) -> 
      let lwVisitor = new extDSVisitor in
      visitCilFileSameGlobals lwVisitor f);
    fd_post_check = true;
  } 

