////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

using namespace std;

#include <unistd.h>
#include <q_exports.h>
#include <server/server.h>
#include <Quake3Metadata.h>
#include <Quake3DynamicEntity.h>
#include <algorithm>

extern void  AimAtTarget ( gentity_t * self);

gentity_t *allocEntity () { 
    gentity_t *ent = new gentity_t ();
    memset (ent, 0, sizeof (gentity_t));
    return ent;
}

int RunFrame ();
void TestSerialization ();
void TestPointerLookup ();

/**
 * Test whether the serialization-deserialization infrastructure 
 * works correctly or not.
 */
int main(int argc, char *argv[])
{

    g_VerbosityLevel = 2;
    /*
    // Get quake to initialize itself normally
    Sys_Main(argc, argv);
    
    // We must load from the DLL, not the vm
    Cvar_Set( "vm_game", "0" );
    
    // Load the quake map and startup the server
    // (constructing the world may require this step)
    SV_SpawnServer( "q3dm1", qtrue );

    // Run a few frames
    for (int i = 0; i < 5; i++) { 
	cout << " Running frame " << i << endl;
	RunFrame ();
    }

    */
    InitializeQuake3Metadata ();

    // check if the pointers are being looked up correctly
    TestSerialization ();

    return 0;
}

bool CheckDirty (gentity_t *ent, gentity_t *nent, DeltaMask &mask) {
    list<string> dirtylist;
    mask.Clear ();

    pair<bool, bool> dirty =
	g_DeltaEncoder->RecordChanges (ent, nent, NULL, NULL, mask, &dirtylist);

    cout << "dirty fields -> ";
    copy (dirtylist.begin(), dirtylist.end(), ostream_iterator<string> (cout, " "));
    cout << endl;

    int offset = 4 + Quake3Entity::NumEntityPointers ();    
    int ncl = g_DeltaEncoder->GetNumClusters ();
    
    // check if the right bits got encoded in the clusters or not
    for (list<string>::iterator it = dirtylist.begin (); it != dirtylist.end (); ++it) {
	const char *field = (*it).c_str ();
	int cli = g_DeltaEncoder->GetClusterForField (field);
	ASSERT (cli != -1);
	ASSERT (cli < ncl);

	DB (10) << "cluster #" << cli << " for field " << field << endl;
	// make sure the cluster found was correct
	const Cluster *c = g_DeltaEncoder->GetCluster (cli);

	bool found = false;
	for (int i = 0, n = c->GetNumFields (); i < n; i++) 
	    if (!strcmp (c->GetField (i), field))
		found = true;
	
	ASSERT (found);

	// make sure the correct bit is set in the mask
	ASSERTDO (mask.IsSet (offset + cli), { WARN << " field=" << field << " offset=" << offset << " cli=" << cli << " mask=" << mask << endl; });
    }
    return dirty.first;
}

void TestSerialization () {
    gentity_t *ent = allocEntity ();
    gentity_t *nent = allocEntity ();

    // make sure we get the pointer from the charPointerTable 
    nent->classname = Field::GetCharPointer ((char *) "Lightning Gun");
    nent->think = AimAtTarget;
    nent->target_ent = ent;

    DeltaMask mask;
    cout << "diff between ent and nent" << endl << "\t";
    CheckDirty (ent, nent, mask);

    ////////////////////////////////////////////////////
    Packet *pkt = new Packet ();
    g_DeltaEncoder->PackUpdate (nent, NULL, pkt, mask);
    
    gentity_t *other = allocEntity ();
    pkt->ResetBufPosition ();
    g_DeltaEncoder->UnpackUpdate (other, NULL, pkt, mask);
    
    cout << "diff between ent and nent, again" << endl << "\t";
    CheckDirty (ent, nent, mask);
    cout << "diff between ent and other" << endl << "\t";
    CheckDirty (ent, other, mask);

    cout << "diff between nent and other" << endl << "\t";
    CheckDirty (nent, other, mask);
}

int RunFrame () {
    // *** from Com_Frame

    // write config file if anything changed
    Com_WriteConfiguration(); 
    Com_EventLoop();
    Cbuf_Execute ();

    // *** from SV_Frame

    // the menu kills the server with this cvar
    if ( sv_killserver->integer ) {
	SV_Shutdown ("Server was killed.\n");
	Cvar_Set( "sv_killserver", "0" );
	return 1;
    }

    // if it isn't time for the next frame, do nothing
    if ( sv_fps->integer < 1 ) {
	Cvar_Set( "sv_fps", "10" );
    }

    // XXX -- DO SOMETHING ABOUT THIS!!!  if time is about to hit the
    // 32nd bit, kick all clients and clear sv.time, rather than
    // checking for negative time wraparound everywhere.
    // 2giga-milliseconds = 23 days, so it won't be too often
    if ( svs.time > 0x70000000 ) {
	SV_Shutdown( "Restarting server due to time wrapping" );
	Cbuf_AddText( "vstr nextmap\n" );
	return 1;
    }
    // this can happen considerably earlier when lots of clients play
    // and the map doesn't change
    if ( svs.nextSnapshotEntities >= 0x7FFFFFFE - svs.numSnapshotEntities ) {
	SV_Shutdown( "Restarting server due to numSnapshotEntities wrapping" );
	Cbuf_AddText( "vstr nextmap\n" );
	return 1;
    }

    // update infostrings if anything has been changed
    if ( cvar_modifiedFlags & CVAR_SERVERINFO ) {
	SV_SetConfigstring( CS_SERVERINFO, Cvar_InfoString( CVAR_SERVERINFO ) );
	cvar_modifiedFlags &= ~CVAR_SERVERINFO;
    }
    if ( cvar_modifiedFlags & CVAR_SYSTEMINFO ) {
	SV_SetConfigstring( CS_SYSTEMINFO, Cvar_InfoString_Big( CVAR_SYSTEMINFO ) );
	cvar_modifiedFlags &= ~CVAR_SYSTEMINFO;
    }

    // update ping based on the all received frames
    SV_CalcPings();

    SV_BotFrame( svs.time );

    // we assume we only run 1 sim chunk at a time
    svs.time += 50;

    // let everything in the world think and move
    // XXX: only do this for the primary objects!
    VM_Call( gvm, GAME_RUN_FRAME, svs.time );


    // *** from server/sv_main.c:SV_Frame

    // check timeouts
    SV_CheckTimeouts();

    // send messages back to the clients
    SV_SendClientMessages();

    // send a heartbeat to the master if needed
    SV_MasterHeartbeat();

    usleep(50 * 1000);

    return 0;
}

// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
