////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

using namespace std;

#include <iostream>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <gameapi/common.h>
#include <util/Benchmark.h>

#include <q_exports.h>
#include <BBox.h>
#include <util/Options.h>
#include <Options.h>
#include <om/Manager.h>

extern clipMap_t cm;
static char g_Mapname[255];
static bool g_DumpDims;

static OptionType g_Options[] = {
    { '/', "bbox-min", OPT_STR, "the min corner of world (default: read from map)", 
	g_QuakePreferences.bbox_min, "", NULL },
    { '/', "bbox-max", OPT_STR, "the max corner of world (default: read from map)", 
	g_QuakePreferences.bbox_max, "", NULL },
    { '/', "bbox-unit", OPT_FLT, "the unit granularity between each bbox point", 
	&g_QuakePreferences.bbox_unit, "64", NULL },
    { '/', "bbox-xystep", OPT_FLT, "the number of degrees to space each ray in xy", 
	&g_QuakePreferences.bbox_xystep, "1", NULL },
    { '/', "bbox-zstep", OPT_FLT, "the number of degrees to space each ray in z", 
	&g_QuakePreferences.bbox_zstep, "5", NULL },
    { '/', "dump-dims", OPT_NOARG | OPT_BOOL, "only dump map dimensions", 
	&g_DumpDims, "0", (void *) "1" },

    { '/', "bbox-file", OPT_STR, "file to output/input precomputed bboxes", 
	g_QuakePreferences.bbox_file, "", NULL },
    { '/', "map", OPT_STR, "map to generate bboxes for", 
	g_Mapname, "", NULL },
    { '/', "bvlevel", OPT_INT, "verbosity level for bbox stuff", 
	&g_VerbosityLevel, "-1", NULL },
    
    { '#', "open maps", OPT_SEP, "", NULL, "", NULL },
    { '/', "enable-open-map", OPT_NOARG | OPT_BOOL, "enable checks for an open map", 
	&g_QuakePreferences.open_map, "0", (void *) "1" },
    { '/', "max-bbox-width", OPT_FLT, "maximum width of a bbox", 
	&g_QuakePreferences.max_bbox_width, "2700", NULL },
    { '/', "max-bbox-height", OPT_FLT, "maximum height of a bbox", 
	&g_QuakePreferences.max_bbox_height, "2400", NULL },
    { '/', "open-threshold", OPT_FLT, "fraction of rays which must be 'open' for an open bbox /* unused */", 
	&g_QuakePreferences.open_bbox_detection_threshold, "0.3", NULL },

    { 0, 0, 0, 0, 0, 0, 0 }
};

//
// Compute the bounding boxes for points between 
// 'min' and 'max' and write them to a file
//
void DG_ExportBoundingBoxes(vec3_t min, vec3_t max) {

    if (!strcmp(g_QuakePreferences.bbox_file, "")) {
	WARN << "no bbox output file specified!" << endl;
	Debug::die("");
    }

    // open the bbox file to write to
    FILE *fp = fopen(g_QuakePreferences.bbox_file, "w");

    if (!fp) {
	WARN << "unable to open " 
	    << g_QuakePreferences.bbox_file << " for writing: "
	    << strerror(errno) << endl;
	Debug::die("");
    }

    // parse the x,y,z min and max for the space we are interested in
    // computing bboxes for...
    float step = g_QuakePreferences.bbox_unit;
    char *str;

    // calculate the number of buckets to calculate (int's cheaper than floats)
    int xbuckets = (int)( (max[0]-min[0])/step + 1 );
    int ybuckets = (int)( (max[1]-min[1])/step + 1 );
    int zbuckets = (int)( (max[2]-min[2])/step + 1 );
    int total    = xbuckets*ybuckets*zbuckets;
    float total_vol = BBoxVolume(min, max);

#ifdef COMPACT_FORMAT
    fwrite((void *)MAGIC_STRING, strlen(MAGIC_STRING), 1, fp); // magic
    fwrite((void *)min, sizeof(vec3_t), 1, fp);    // starting point
    fwrite((void *)&xbuckets, sizeof(int), 1, fp); // number of x points
    fwrite((void *)&ybuckets, sizeof(int), 1, fp); // number of y points
    fwrite((void *)&zbuckets, sizeof(int), 1, fp); // number of z points
    fwrite((void *)&step, sizeof(float), 1, fp);   // size of step between pts
#endif

    INFO << "bbox buckets = (" << xbuckets << "," 
	<< ybuckets << "," << zbuckets << ") = "
	<< total << " total, vol = " << total_vol << endl;

    // for each point in the space we are interested in, calculate and
    // write the bbox for that point.
    vec3_t orig, bmin, bmax;
    int x_index, y_index, z_index;
    int todo = total;

    BBoxer bboxer (min, max, g_QuakePreferences.bbox_xystep, g_QuakePreferences.bbox_zstep);


    int nlarge = 0, npoint = 0;
    for (x_index = 0; x_index < xbuckets; x_index++) {
	orig[0] = min[0] + step*x_index;

	for (y_index = 0; y_index < ybuckets; y_index++) {
	    orig[1] = min[1] + step*y_index;

	    for (z_index = 0; z_index < zbuckets; z_index++) {
		orig[2] = min[2] + step*z_index;
		START(BBOX_TIMER);

		BBoxType t = bboxer.ComputeVisibilityBox (orig, bmin, bmax);
		if (t == BBOX_MAXWIDTH) 
		    nlarge += 1;
		if (t == BBOX_POINTSIZE)
		    npoint += 1;

#ifndef COMPACT_FORMAT
		float vol = BBoxVolume(bmin, bmax);
		fprintf(fp, "%8.3f,%8.3f,%8.3f\t%8.3f,%8.3f,%8.3f\t%8.3f,%8.3f,%8.3f\t%7.5f\n",
			orig[0], orig[1], orig[2],
			bmin[0], bmin[1], bmin[2],
			bmax[0], bmax[1], bmax[2], vol/total_vol);
#else
		fwrite((void *)bmin, sizeof(vec3_t), 1, fp); // bbox min
		fwrite((void *)bmax, sizeof(vec3_t), 1, fp); // bbox max
#endif

		todo--;

		TimeVal now;
		gettimeofday(&now, NULL);
		STOP(BBOX_TIMER);

		// XXX This isn't very accurate since we aren't sampling points
		// from the map space randomly...
		//	PERIODIC(30000, now, 

		{
		    PERIODIC (10000, now, {
			    float time;
			    AVERAGE_ASSIGN(time, BBOX_TIMER);
			    fprintf(stderr, "Time Left: %7.1f min     Todo: %10d #large=%.2f%%, #point=%.2f%%\n", 
				((float)todo/(MSEC_IN_SEC*60))*time, 
				todo,
				(float) nlarge * 100.0 / (float) (total - todo),
				(float) npoint * 100.0 / (float) (total - todo)
				);

			    // Restart the timer average after the cache has been
			    // primed with a few runs to get an accurate ETA
			    static bool reset = 0;
			    if (!reset && todo < total - 100) {
			    RESTART(BBOX_TIMER);
			    reset = 1;
			    }
			    });
		}
	    }
	}
    }

    fclose(fp);
}

int main(int argc, char *argv[])
{
    argc = ProcessOptions(g_Options, argc, argv, false);

    if (! (strcmp (g_Mapname, "") && 
	    (strcmp (g_QuakePreferences.bbox_file, "") || g_DumpDims)) )  {
	cerr << "usage: bboxgen [options] [--dump-dims] --map <mapname> --bbox-file <output_file>" << endl;
	exit(1);
    }

    // Get quake to initialize itself normally
    Sys_Main(argc, argv);
    // ttyconsole doesn't work with nodelay disabled
    Cvar_Set( "ttycon", "0" );
    SV_SpawnServer( g_Mapname, qtrue );
    
    strcpy(g_ProgramName, argv[0]);
    DBG_INIT(&g_LocalSID);
    InitCPUMHz();
    Benchmark::init();

    vec3_t min, max;
    if (!strcmp(g_QuakePreferences.bbox_min, "") ||
	    !strcmp(g_QuakePreferences.bbox_max, "")) {

	ASSERT(cm.numSubModels > 0);

	for (int i=0; i<3; i++) {
	    min[i] = cm.cmodels[0].mins[i];
	    max[i] = cm.cmodels[0].maxs[i];
	}

    }
    else {
	char *str = strtok(g_QuakePreferences.bbox_min, ",");
	float step = g_QuakePreferences.bbox_unit;

	ASSERT(str);
	min[0] = atof(str) + step/2;
	str = strtok(NULL, ",");
	ASSERT(str);
	min[1] = atof(str) + step/2;
	str = strtok(NULL, ",");
	ASSERT(str);
	min[2] = atof(str) + step/2;

	str = strtok(g_QuakePreferences.bbox_max, ",");
	ASSERT(str);
	max[0] = atof(str) - step/2;
	str = strtok(NULL, ",");
	ASSERT(str);
	max[1] = atof(str) - step/2;
	str = strtok(NULL, ",");
	ASSERT(str);
	max[2] = atof(str) - step/2;
    }

    INFO << "Loaded map..." << endl;
    cout << "worldmin=" << min 
	<< " worldmax=" << max << endl;

    if (!g_DumpDims) {
	DG_ExportBoundingBoxes(min, max);
	INFO << " ---- done with bbox computation! ----" << endl;
    }

    SV_Shutdown ("finished.\n");
    Sys_ConsoleInputShutdown();

    return 0;
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
