////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

#ifndef __QUAKE3_WORLD_H__
#define __QUAKE3_WORLD_H__

#include <mercury/common.h>
#include <gameapi/GameWorld.h>
#include <gameapi/GameObject.h>
#include "BBox.h"

#ifdef USE_SHARED_MEMORY
#include <sys/mman.h>
#include <errno.h>

#define GET_OFFSET(x, y, z)     ((( (x) * g_ybuckets ) + (y)) * g_zbuckets + (z))
#define GET_BBOX(x, y, z)       (* (bbox_t *) ((byte *) g_bboxes + GET_OFFSET(x, y, z) * sizeof(bbox_t)))
#define GET_FLAGS(box)          Shmbbox_GetFlags(&(box))

#else // USE_SHARED_MEMORY

#define GET_BBOX(x, y, z)       (g_bboxes[x][y][z])
#define GET_FLAGS(box)          ((box).flags)

#endif // USE_SHARED_MEMORY

#define BBOX_VALID_BIT 0x1
#define BBOX_MARK_BIT  0x2

#define BBOX_SET_VALID(bbox, val) \
   ( (val) ? GET_FLAGS(bbox) |= BBOX_VALID_BIT : \
     GET_FLAGS(bbox) &= ~BBOX_VALID_BIT )
#define BBOX_IS_VALID(bbox)       \
   ( GET_FLAGS(bbox) & BBOX_VALID_BIT )
#define BBOX_SET_MARK(bbox, val)  \
   ( (val) ? GET_FLAGS(bbox) |= BBOX_MARK_BIT : \
     GET_FLAGS(bbox) &= ~BBOX_MARK_BIT )
#define BBOX_IS_MARK(bbox)        \
   ( GET_FLAGS(bbox) & BBOX_MARK_BIT )

///////////////////////////////////////////////////////////////////////////////

class Quake3DynamicEntity;

/**
 * Quake3World encapsulates the bbox information needed for a game.
 */
class Quake3World : public GameWorld {
 private:
    BBox extent;

    ///////////////////////////////////////////////////////////////////////////

    // legacy bbox variables from dg_bbox.h
    vec3_t g_worldmin, g_worldmax;
    vec3_t g_min;
    int    g_xbuckets, g_ybuckets, g_zbuckets;
    float  g_step, g_total_vol;
    bbox_t ***g_bboxes;

    BBoxer *bboxer;

    ///////////////////////////////////////////////////////////////////////////

#ifdef USE_SHARED_MEMORY
    /// separate space for the bbox flags 
    byte   ***g_bbox_flags;
    /// shared memory is attached here
    byte     *g_shmaddress;

    void Shmbbox_GetIndices(bbox_t *addr, int *x, int *y, int *z);
    byte& Shmbbox_GetFlags(bbox_t *addr);
#endif

    ///////////////////////////////////////////////////////////////////////////

    void IndexToPoint(vec3_t out, ivec3_t index);
    void ClearMarks(ivec3_t trunc, int width);
    bool ExpandingRingSearch(vec3_t min, vec3_t max, 
				vec3_t torig, int width, 
				bool all = false, int zlimit = 9999);

    /** Load precomputed bounding boxes */
    void LoadBoundingBoxFile(const char *);
    void LoadBBoxFromMmapedFile(const char *);
    void ReadBBoxesFromMmapedFile();
    FILE * LoadBBoxFromNormalFile(const char *);
    void ReadBBoxesFromNormalFile(FILE *fp);
    void LoadDefaultWorld();

    ///////////////////////////////////////////////////////////////////////////

    bool HasPrecompBBoxes();

    /** 
     * Get the bounding box from the closest precomputed point
     *
     * @param orig the origin to compute the bbox for
     * @param min  output min point of the bbox
     * @param max  output max point of the bbox
     * @param diameter instead of finding the closest bbox for this point,
     *        combine all the bboxes within this diameter of that point.
     * @param zlimit limit the z-diameter to this much.
     * @param maxvol the maximum volume of the bbox allowed (only useful if
     *        diameter is specified). If specified, if the merged bbox for
     *        a particular diameter is too big, we will reduce the diameter
     *        by 2 units and try again until we find one small enough. In
     *        the worst case, if we don't find one, then we will just fall
     *        back to using the closest bbox. The maxvol is represented as
     *        a fraction of the total world volume.
     *
     * @return the vol of the bbox as a fraction of total world volume
     */
    float GetPrecompBoundingBox(vec3_t orig, vec3_t min, vec3_t max,
				   float diameter = 0, float zlimit = 9999,
				   float maxvol = 1);
    
    /**
     * Get the bbox for the entire world if we loaded a precomputed bbox map
     */
    void GetPrecompWorldBBox(vec3_t min, vec3_t max);
    
    /** 
     * Get the bounding box using the "preferred" method from the above 3
     *
     * @return the vol of the bbox as a fraction of total world volume
     *         (only if we are using precomputed bboxes.. otherwise unknown)
     */
    float GetBoundingBox(vec3_t orig, vec3_t min, vec3_t max);

 public:
    Quake3World(const char *bboxfile = NULL);
    virtual ~Quake3World ();

    virtual const BBox& GetExtent();
    
    virtual void Link(GameObject *obj);
    virtual void Unlink(GameObject *obj);
    virtual void Update(GameObject *obj);

    /** no prediction */
    BBox GetViewableBBox(Quake3DynamicEntity *ent);
    /** 
     * with prediction 
     * XXX: at some point change this interface to take into account known
     * min and max and trajectories.
     */
    BBox GetPredictedBBox(Quake3DynamicEntity *ent, 
			  float diameter, float zlimit);

    /** without prediction */
    virtual uint32 GetAreaOfInterest(list<BBox> *toFill, 
				     DynamicGameObject *obj);
    virtual const bool PublishDim(uint32 dim);

    virtual BBox Clip(const BBox& box);
    virtual Vec3 Clip(const Vec3& clip);
};

#endif
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
