////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

#include "World.h"
#include "Quake3DynamicEntity.h"
#include "Options.h"
#include <util/Benchmark.h>
#include <sys/types.h>
#include <limits.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>

///////////////////////////////////////////////////////////////////////////////

#ifdef USE_SHARED_MEMORY

///////////////////////////////////////////////////////
// Get the bbox indices from the address 
// 
void Quake3World::Shmbbox_GetIndices(bbox_t *addr, int *x, int *y, int *z)
{
    int offset = addr - (bbox_t *) g_bboxes;
	
    *z = offset % g_zbuckets;
    offset -= *z; offset /= g_zbuckets;
	
    *y = offset % g_ybuckets;
    offset -= *y; offset /= g_ybuckets;

    *x = offset;
}

byte& Quake3World::Shmbbox_GetFlags(bbox_t *addr)
{
    int x, y, z;
    Shmbbox_GetIndices(addr, &x, &y, &z);
    return g_bbox_flags[x][y][z];
}

void Quake3World::LoadBBoxFromMmapedFile(const char *filename)
{
    int fd = open(filename, O_RDONLY);
    if (fd < 0) {
	perror("open");
	WARN << "can't open bbox file: " << filename << endl;
	exit(1);
    }
	
    fprintf(stderr, "opened file successfully...\n");
    struct stat buf;
    if (fstat(fd, &buf) < 0) { 
	perror("fstat"); 
	Debug::die("fstat failed!"); 
    }
    
    fprintf(stderr, "stat happenned successfully...file length = %d\n", (int)buf.st_size);
    g_shmaddress = (byte *) mmap(NULL, buf.st_size, PROT_READ, MAP_SHARED, fd, 0);
    if (g_shmaddress == NULL) {
	fprintf(stderr, "stat error...\n");
	perror("mmap");
	WARN << "error mmaping bbox file: " << filename << endl;
	exit(1);
    }
    fprintf(stderr, "mmap happened successfully...\n");
    if (strncmp((char *) g_shmaddress, MAGIC_STRING, strlen(MAGIC_STRING))) {
	fprintf(stderr, "bbox file doesn't begin with magic string: " 
		MAGIC_STRING "\n");
	exit(1);
    }	
    g_shmaddress += strlen(MAGIC_STRING);
    
    memcpy(g_min, g_shmaddress, sizeof(vec3_t));  g_shmaddress += sizeof(vec3_t);
    memcpy(&g_xbuckets, g_shmaddress, sizeof(int)); g_shmaddress += sizeof(int);
    memcpy(&g_ybuckets, g_shmaddress, sizeof(int)); g_shmaddress += sizeof(int);
    memcpy(&g_zbuckets, g_shmaddress, sizeof(int)); g_shmaddress += sizeof(int);
    memcpy(&g_step, g_shmaddress, sizeof(float)); g_shmaddress += sizeof(float);
}

void Quake3World::ReadBBoxesFromMmapedFile()
{
    g_bboxes = (bbox_t ***) g_shmaddress;

    g_bbox_flags = new byte**[g_xbuckets];
    for (int x_index = 0; x_index < g_xbuckets; x_index++) {
	g_bbox_flags[x_index] = new byte*[g_ybuckets];
	for (int y_index = 0; y_index < g_ybuckets; y_index++) {
	    g_bbox_flags[x_index][y_index] = new byte[g_zbuckets];
	    
	    for (int z_index = 0; z_index < g_zbuckets; z_index++) {
		g_bbox_flags[x_index][y_index][z_index] = 0;
	    }
	}
    }
}

#endif

///////////////////////////////////////////////////////////////////////////////

FILE *  Quake3World::LoadBBoxFromNormalFile(const char *filename)
{
    FILE *fp = fopen(filename, "r");
    if (!fp) {
	WARN << "can't open bbox file: " << filename << endl;
	exit(1);
    }

    char *junk = (char *)calloc(strlen(MAGIC_STRING)+1, 1);
    fread((void *)junk, strlen(MAGIC_STRING), 1, fp); // magic

    if (strcmp(junk, MAGIC_STRING)) {
	fprintf(stderr, "bbox file doesn't begin with magic string: " 
		MAGIC_STRING "\n");
	exit(1);
    }
    free(junk);

    fread((void *)g_min, sizeof(vec3_t), 1, fp);    // starting point
    fread((void *)&g_xbuckets, sizeof(int), 1, fp); // number of x points
    fread((void *)&g_ybuckets, sizeof(int), 1, fp); // number of y points
    fread((void *)&g_zbuckets, sizeof(int), 1, fp); // number of z points
    fread((void *)&g_step, sizeof(float), 1, fp);   // size of step between pts

    return fp;
}

void  Quake3World::ReadBBoxesFromNormalFile(FILE *fp)
{
    g_bboxes = new bbox_t**[g_xbuckets];
	
    for (int x_index = 0; x_index < g_xbuckets; x_index++) {
	g_bboxes[x_index] = new bbox_t*[g_ybuckets];
	for (int y_index = 0; y_index < g_ybuckets; y_index++) {
	    g_bboxes[x_index][y_index] = new bbox_t[g_zbuckets];
	    for (int z_index = 0; z_index < g_zbuckets; z_index++) {
		int ret;

		ret = fread((void *) GET_BBOX(x_index, y_index, z_index).min, 
			    sizeof(vec3_t), 1, fp); // bbox min
		if (ret == 0) {
		    WARN << "EOF encountered reading bbox file!" << endl;
		    ASSERT(0);
		}
		ret = fread((void *) GET_BBOX(x_index, y_index, z_index).max, 
			    sizeof(vec3_t), 1, fp); // bbox max
		if (ret == 0) {
		    WARN << "EOF encountered reading bbox file!" << endl;
		    ASSERT(0);
		}

	    }
	}
    }
}

//
// Load the bounding box file for use in sub computation
//
void  Quake3World::LoadBoundingBoxFile(const char *filename) {
#ifdef USE_SHARED_MEMORY
    LoadBBoxFromMmapedFile(filename);
#else
    FILE *fp = LoadBBoxFromNormalFile(filename);
#endif
	
    // ok to create!
    INFO << "g_xbuckets = " << g_xbuckets 
	 << " g_ybuckets = " << g_ybuckets 
	 << " g_zbuckets = " << g_zbuckets 
	 << " sizeof(bbox_t) = " << sizeof(bbox_t) << endl;

    g_total_vol = g_xbuckets*g_ybuckets*g_zbuckets*g_step*g_step*g_step;

#ifdef USE_SHARED_MEMORY
    ReadBBoxesFromMmapedFile();
    for (int i = 0 ; i < (int)(g_xbuckets * g_ybuckets * g_zbuckets * sizeof(bbox_t)) ; i++) {
	byte *mem = (byte *) g_bboxes;
	mem = mem + i;
	
	byte access = *mem;
    }
#else
    ReadBBoxesFromNormalFile(fp);
#endif


    for (int x_index = 0; x_index < g_xbuckets; x_index++) {
	for (int y_index = 0; y_index < g_ybuckets; y_index++) {
	    for (int z_index = 0; z_index < g_zbuckets; z_index++) {
		bbox_t *t = &(GET_BBOX(x_index, y_index, z_index));
		GET_FLAGS(*t) = 0;

		if (BBoxVolume(GET_BBOX(x_index, y_index, z_index).min,
			       GET_BBOX(x_index, y_index, z_index).max) <= 0) {
		    BBOX_SET_VALID(GET_BBOX(x_index, y_index, z_index), 0);
		} else {
		    BBOX_SET_VALID(GET_BBOX(x_index, y_index, z_index), 1);
		}
	    }
	}
    }

#ifndef USE_SHARED_MEMORY
    fclose(fp);
#endif

    GetPrecompWorldBBox(g_worldmin, g_worldmax);
}

///////////////////////////////////////////////////////////////////////////////

bool  Quake3World::HasPrecompBBoxes()
{
    return g_bboxes != NULL;
}

float  Quake3World::GetBoundingBox(vec3_t orig, vec3_t min, vec3_t max)
{
    if (g_bboxes != NULL) {
	// if we precomputed bboxes, use that
	return GetPrecompBoundingBox(orig, min, max);
    } else {
	// otherwise we have to compute it on the fly
	bboxer->ComputeVisibilityBox (orig, min, max);
	return 0;
    }
}

///////////////////////////////////////////////////////////////////////////////

void  Quake3World::ClearMarks(ivec3_t trunc, int width)
{
    for (int x = MAX(trunc[0], 0);
	 x <= MIN(trunc[0] + width, g_xbuckets-1); x++) {
	for (int y = MAX(trunc[1], 0);
	     y <= MIN(trunc[1] + width, g_ybuckets-1); y++) {
	    for (int z = MAX(trunc[2], 0);
               	 z <= MIN(trunc[2] + width, g_zbuckets-1); z++) {
		BBOX_SET_MARK( GET_BBOX(x, y, z), 0 );
	    }
	}
    }
}

//
// Perform an "expanding-ring" search around torig (which is assumed to be
// in the coordinate-space after translation by g_min) to find the closest
// valid point with a bbox.
//
// bbox_min - the resulting bbox_min point
// bbox_max - the resulting bbox_max point
// torig    - the origin point (in g_min translated space)
// width    - the number of g_step units for the diameter of the ring
// all      - combine the bboxes from all the points within width diameter
//            instead of finding the closest point (this might return 
//            the null box if nothing is in range)
// zlimit   - a HACK that allows us to limit the range search in the
//            z-direction to a given diameter. this is useful since most
//            rooms have ceilings and it makes no sense to have the search
//            space include the areas "above" the ceiling or "below" the
//            floors. (9999 ~ no limit)
//
// returns true if we found a bbox.
//
bool  Quake3World::ExpandingRingSearch(vec3_t bbox_min, vec3_t bbox_max,
				       vec3_t torig, int width, bool all, 
				       int zlimit)
{
    bool found = false;
    ivec3_t res = { 0, 0, 0 };
    vec3_t test;
    float mindist = INT_MAX;

    // Truncate the point to the closest "lower-left" corner bbox point
    ivec3_t trunc;
    trunc[0] = (int)(torig[0]/g_step) - width/2;
    trunc[1] = (int)(torig[1]/g_step) - width/2;
    trunc[2] = (int)(torig[2]/g_step) - MIN(width, zlimit)/2;

    /*
      INFO << "torig=" << torig << endl;
      INFO << "width=" << width << endl;
      INFO << "trunc=" << trunc << endl;
    */

    // Now sweep the box looking for valid points we haven't marked yet
    for (int x = MAX(trunc[0], 0); 
	 x <= MIN(trunc[0] + width, g_xbuckets-1); x++) {

	for (int y = MAX(trunc[1], 0); 
	     y <= MIN(trunc[1] + width, g_ybuckets-1); y++) {

	    for (int z = MAX(trunc[2], 0); 
		 z <= MIN(trunc[2] + MIN(width, zlimit), g_zbuckets-1); z++) {

		if (BBOX_IS_MARK( GET_BBOX(x, y, z) ) ||
		    ! BBOX_IS_VALID( GET_BBOX(x, y, z) ))
		    continue;
				
		test[0] = x*g_step;
		test[1] = y*g_step;
		test[2] = z*g_step;
				
		float dist = Distance(torig, test);
		if (dist <= (g_step*width)/2) {

		    // XXX We should also make sure the bbox point is
		    // visible from the origin point! E.G., we may be
		    // blocked by ceilings or go through a wall! But
		    // this requires doing a ray-trace, which is
		    // potentially expensive, so just hope we will find
		    // a close point fast for now so that is unnecessary
		    if (!all) {
			// mark it so we don't look at it again
			BBOX_SET_MARK( GET_BBOX(x, y, z), 1 );

			// we are just looking for the min point
			if (!found) {
			    found   = true;
			    mindist = dist;
			    res[0] = x; res[1] = y; res[2] = z;
			} else {
			    if (dist < mindist) {
				mindist = dist;
				res[0] = x; res[1] = y; res[2] = z;
			    }
			}
		    } else {
			// we are combining all the bboxes for all points
			if (!found) {
			    found = true;
			    VectorCopy(GET_BBOX(x, y, z).min, bbox_min);
			    VectorCopy(GET_BBOX(x, y, z).max, bbox_max);
			} else {
			    for (int i=0; i<3; i++) {
				bbox_min[i] = MIN(bbox_min[i], 
						  GET_BBOX(x, y, z).min[i]);
				bbox_max[i] = MAX(bbox_max[i], 
						  GET_BBOX(x, y, z).max[i]);
			    }
			}
		    }
		}
				
	    }
	}
    }

    // if we were getting the large combined bbox we are now done
    if (all) {
	return found;
    }

    // otherwise, we need to unclear the bits or possibly recurse if we
    // didn't find a valid point.
    if (found) {
	/*
	  if (all) {
	  INFO << "width=" << width << endl;
	  INFO << "found=" << found << endl;
	  INFO << bbox_min << " " << bbox_max << endl;
	  }
	*/

	// sweep across the search box and unclear the marked bits
	ClearMarks(trunc, width);
		
	VectorCopy(GET_BBOX(res[0], res[1], res[2]).min, bbox_min);
	VectorCopy(GET_BBOX(res[0], res[1], res[2]).max, bbox_max);

	return true;
    } else {
	// sanity check: if this happens then it implies that the bbox map
	// was malformed because there are no valid points in the search space
	if (width > g_xbuckets && width > g_ybuckets && width > g_zbuckets) {
	    // This seems to sometimes happen for missiles that fly out
	    // into space? Not sure, but don't crash because of it...
	    DB(1) << "ring search exhausted search space without a result!"
		  << endl;
	    ClearMarks(trunc, width);
	    bzero((byte *)bbox_min, sizeof(vec3_t));
	    bzero((byte *)bbox_min, sizeof(vec3_t));
	    return false;
	}

	// didn't find a valid box!, try again with larger diameter
	return ExpandingRingSearch(bbox_min, bbox_max, torig, 
				      width+1, all, zlimit);
    }
}

//
// Get the closest precomputed bounding box for a point in space.
//
float  Quake3World::GetPrecompBoundingBox(vec3_t orig, vec3_t bbox_min, 
					     vec3_t bbox_max, float diameter, 
					     float zlimit, float maxvol)
{
    ASSERT(g_bboxes != NULL);

    // check if quake put this object in some wierd place...
    for (int i=0; i<3; i++) {
	if (orig[i] < g_worldmin[i] ||
	    orig[i] > g_worldmax[i]) {
	    /*
	      INFO << " orig=" << orig << " min=" << g_worldmin
	      << " max=" << g_worldmax << endl;
	    */
	    return 0;
	}
    }

    // translate the origin to the coordinate space starting from the
    // "edge" of the spots for which we have bboxes for
    vec3_t torig;
    torig[0] = orig[0] - g_min[0];
    torig[1] = orig[1] - g_min[1];
    torig[2] = orig[2] - g_min[2];

    bool found = false;
    int zlim = (int)ceil( zlimit/g_step );

    if (diameter > 0) {
	// get the box of all potential points within this diameter!
	int width  = (int)ceil( diameter/g_step );

	START(ExpandingRingSearch);
		
	while (width > 0) {
	    found = ExpandingRingSearch(bbox_min, bbox_max, torig, 
					   width, true, zlim);
	    if (!found)
		break;
	    float vol = BBoxVolume(bbox_min, bbox_max)/g_total_vol;
	    if (vol <= maxvol)
		break;
	    else {
		DB(10) << "bbox with width=" << width << " too big! (diameter=" 
		    << diameter << " zlimit=" << zlimit << ")" 
		    << endl;
		// too big! try again with smaller diameter
		width -= 2;
	    }
	}
			
	STOP(ExpandingRingSearch);

	if (!found) {
	    START(ExpandingRingSearch);
	    // no points found in diameter! just find the closest then
	    found = ExpandingRingSearch(bbox_min, bbox_max, torig, 
					   1, false, zlim);
	    STOP(ExpandingRingSearch);
	}
	/* else {
	   ASSERT(bbox_max[0] - bbox_min[0] > 0 &&
	   bbox_max[1] - bbox_min[1] > 0 &&
	   bbox_max[2] - bbox_min[2] > 0);
	   }*/
    } else {
	// Now perform an "expanding-ring" search for the closest valid 
	// bbox point
	START(ExpandingRingSearch);
	found = ExpandingRingSearch(bbox_min, bbox_max, torig, 
				       1, false, zlim);
	STOP(ExpandingRingSearch);
    }

    /*
    // XXX
    vec3_t closest;
    IndexToPoint(closest, res);
	
    INFO << "origin =" << orig << endl;
    INFO << "closest=" << closest << endl;
    INFO << "dist   =" << Distance(orig, closest) << endl;
    */

    if (found)
	return BBoxVolume(bbox_min, bbox_max)/g_total_vol;
    else
	return 0;
}

void Quake3World::GetPrecompWorldBBox(vec3_t min, vec3_t max)
{
    ASSERT(g_bboxes != NULL);

    for (int i=0; i<3; i++) {
	min[i] = g_min[i];
    }

    max[0] = g_min[0] + g_xbuckets*g_step;
    max[1] = g_min[1] + g_ybuckets*g_step;
    max[2] = g_min[2] + g_zbuckets*g_step;

    INFO << "* computed world bbox as: " << min << " -> " << max << endl;
    INFO << " (params were: g_min=" << g_min << " g_step=" << g_step 
	 << " xbkt=" << g_xbuckets << " ybkt=" << g_ybuckets 
	 << " zbkt=" << g_zbuckets << endl;
}

void Quake3World::IndexToPoint(vec3_t out, ivec3_t index)
{
    for (int i=0; i<3; i++)
	out[i] = g_min[i] + g_step*index[i];
}

///////////////////////////////////////////////////////////////////////////////

void Quake3World::LoadDefaultWorld()
{
    // load world min/max from the map
    ASSERT(cm.numSubModels > 0);
    for (int i=0; i<3; i++) {
	g_worldmin[i] = cm.cmodels[0].mins[i];
	g_worldmax[i] = cm.cmodels[0].maxs[i];
    }
    
    INFO << "Default world: min=" << g_worldmin << " max=" << g_worldmax << endl;
    // no precomputed bboxes
    g_bboxes = NULL;
}

///////////////////////////////////////////////////////////////////////////////

Quake3World::Quake3World(const char *filename) 
#ifdef USE_SHARED_MEMORY
    : g_bbox_flags(NULL), g_shmaddress(NULL)
#endif
{ 

    if (filename) {
	LoadBoundingBoxFile(filename);
    } else {
	LoadDefaultWorld();
    }
    
    bboxer = new BBoxer (g_worldmin, g_worldmax, g_QuakePreferences.bbox_xystep, g_QuakePreferences.bbox_zstep);
    extent = BBox(Vec3(g_worldmin), Vec3(g_worldmax));
}

Quake3World::~Quake3World () 
{
    // XXX should probably unmap shared memory as well.
    delete bboxer;    
}

const BBox& Quake3World::GetExtent() {
    return extent;
}

// quake3 takes care of these things in its own way.  the bbox map
// don't really care if stuff is linked or not
void Quake3World::Link(GameObject *obj) {}
void Quake3World::Unlink(GameObject *obj) {}
void Quake3World::Update(GameObject *obj) {}

// does not takes into account prediction
uint32 Quake3World::GetAreaOfInterest(list<BBox> *toFill, 
				      DynamicGameObject *obj) { 
    uint32 ttl = 0;

    Quake3DynamicEntity *ent = dynamic_cast<Quake3DynamicEntity *>(obj);
    ASSERT (ent);
    ASSERT (!ent->IsReplica ());
    
    if (ent->GetClass() == Quake3DynamicEntity::PLAYER) {
	ttl = g_QuakePreferences.subttl_player;
    } else if (ent->GetClass() == Quake3DynamicEntity::MISSILE) {
	ttl = g_QuakePreferences.subttl_missile;
    } else {
	return 0;
    }

    // object is not in world!
    if (!extent.Contains(obj->GetOrigin()))
	return 0;

    BBox aoi = GetViewableBBox(ent);
    if (aoi.Volume() > 0) {
	toFill->push_back(aoi);
	return ttl;
    } else {
	return 0;
    }
}

// no prediction
BBox Quake3World::GetViewableBBox(Quake3DynamicEntity *ent) { 
    // object is not in world!
    if (!extent.Contains(ent->GetOrigin()))
	return BBox(Vec3::ZERO, Vec3::ZERO);

    Vec3 vorig = ent->GetOrigin();
    bbox_t bbox;
    vec3_t orig;
    for (int i=0; i<3; i++) {
	orig[i] = vorig[i];
    }

    if (HasPrecompBBoxes()) {
	float vol = 
	    GetPrecompBoundingBox(orig, bbox.min, bbox.max, 0, 1.0, 0.10);
	if (vol == 0)
	    return BBox(vorig, vorig);
    } else {
	// fallback to ray tracing if no bboxes
	GetBoundingBox(orig, bbox.min, bbox.max);
    }

    return BBox(Vec3(bbox.min), Vec3(bbox.max));
}

// with prediction
BBox Quake3World::GetPredictedBBox(Quake3DynamicEntity *ent,
				   float diameter, float zlimit) { 
    uint32 ttl = 0;
    float pred = 0;

    // if object is not in world, should not be here!
    Vec3 vorig = ent->GetOrigin();
    ASSERT(extent.Contains(vorig));

    bbox_t bbox;
    vec3_t orig;
    for (int i=0; i<3; i++) {
	orig[i] = vorig[i];
    }

    if (HasPrecompBBoxes()) {
	// XXX prediction should take into account velocity, not just speed
	// also, some objects have a hard stop after a given distance
	float max = 0;
	for (int i = 0; i < 3; i++) {
	    if ((extent.max [i] - extent.min [i]) > max)
		max = extent.max [i] - extent.min [i];
	}

	// just an early check. the later hysterisis check 
	// should restrict it further.
	if (diameter > max)
	    diameter = max;

	float vol = 
	    GetPrecompBoundingBox(orig, bbox.min, bbox.max, 
				  diameter, 
				  zlimit,
				  // hysterisies check to ensure the bbox
				  // isn't ridicuously large -- max sub
				  // to 50% of the volume of the world.
				  0.50);
	// no bbox found!
	if (vol == 0)
	    return BBox(vorig, vorig);
	
    } else {
	// fallback to ray tracing if no bboxes
	GetBoundingBox(orig, bbox.min, bbox.max);
    }

    ASSERT(bbox.min != bbox.max);
    return BBox(Vec3(bbox.min), Vec3(bbox.max));
}

const bool Quake3World::PublishDim(uint32 dim) { 
    return dim < 3;
}

BBox Quake3World::Clip(const BBox& box) {
    BBox ret;
    for (int i=0; i<3; i++) {
	ret.min[i] = MIN(MAX(box.min[i], extent.min[i]), extent.max[i]);
	ret.max[i] = MIN(MAX(box.max[i], extent.min[i]), extent.max[i]); 
    }
    return ret;
}

Vec3 Quake3World::Clip(const Vec3& clip) {
    Vec3 ret;
    for (int i=0; i<3; i++) 
	ret[i] = MIN(MAX(clip[i], extent.min[i]), extent.max[i]);
    
    return ret;
}

// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
