////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////
#ifndef QUAKE3_METADATA_H
#define QUAKE3_METADATA_H

#include <util/Packet.h>
#include <gameapi/GameObject.h>
#include "q_exports.h"
#include <hash_map.h>
#include "Options.h"

struct Quake3PointerInfo {
    int offset;
    const char *name;
};

extern Quake3PointerInfo g_EntityPointerOffsets[];
extern Quake3PointerInfo g_ClientPointerOffsets[];

// handle player and missile classes specially for now
//
// XXX eventually we want to be able to map pub/sub times on a per-class basis
extern const char *g_PlayerClassStrings[];
extern const char *g_MissileClassStrings[];
extern const char *g_ImmobileLongLiveClassStrings[];
extern const char *g_ImmobileShortLiveClassStrings[];
extern const char *g_MobileLongLiveClassStrings[];
extern const char *g_MobileShortLiveClassStrings[];

////////////////////////////////////////////////////////////////////////////
// infrastructure for semi-automatically gathering metadata about quake
// data-structures (gentity_t and gclient_t, basically) and serializing, 
// delta-encoding them.

struct hash_void_ptr { 
    size_t operator () (void *s) const { return (size_t) s; }
};
struct eq_void_ptr {
    bool operator () (void *s1, void *s2) const { return s1 == s2; }
};

struct hash_char_ptr { 
    size_t operator () (char *s) const { return hash<char *>()(s); }
};
struct eq_char_ptr { 
    bool operator () (char *s1, char *s2) const { return !strcmp (s1, s2); }
};
struct hash_const_char_ptr {
    size_t operator () (const char *s) const { return hash<const char *>()(s); }
};
struct eq_const_char_ptr { 
    bool operator () (const char *s1, const char *s2) const { return !strcmp (s1, s2); }
};

/**
 * Describes a sub-field in a C structure. The template parameter
 * represents the type of the field. Types like int,float,char,bool
 * double, etc. should work; also pointer types should work. 
 **/
class Field {
protected:
    const char *name;            // name of the field;   for example, s.pos.trBase
    int         offset;          // offset of the field inside the parent struct layout
    int         maskIndex;       // index assigned to this field in the delta-mask

    // array encoding; send only non-zero fields
    static const int ARR_MASK_MAXLEN = 256;
    static byte bitmask [ARR_MASK_MAXLEN];

public:     
    Field (const char *name, const int offset) :
	name (name), offset (offset), maskIndex (-1) {}
    virtual ~Field () {}

    void SetMaskIndex (int index) { 
	maskIndex = index;
	ASSERT (maskIndex >= 0);
    }

    const char *GetName () const { 
	return name;
    }
    virtual bool IsDirty (byte *prev, byte *curr, DeltaMask& mask) = 0;
    virtual void PackUpdate (byte *curr, Packet *pkt, const DeltaMask& mask) = 0;
    virtual void UnpackUpdate (byte *curr, Packet *pkt, const DeltaMask& mask) = 0;
    
    static void Initialize ();
    
    // for debugging
    static char *GetCharPointer (char *str) { 	
	int index = charPointerTable.ptrToIndex (str);
	return charPointerTable.indexToPtr (index);
    }

    // for debugging
    virtual int Size () = 0;

protected:
    // Perhaps this doesn't really belong to this class. Find a 
    // better position for this stuff. All of this stuff is 
    // basically metadata about Quake3. Perhaps, Quake3Entity is the
    // right place for this. - Ashwin [05/29/2006]

    template <typename T, class hashFunc, class eqFunc>
	class PointerTable {
	    typedef hash_map <T *, int, hashFunc, eqFunc> ptrHash;
	    
	    ptrHash        table;
	    vector <T *>   array;

	public:
	    PointerTable () {} 
	    ~PointerTable () {}

	    void add (T* ptr) { 
		if (table.find (ptr) != table.end ()) {
		    return;
		}
		int n = array.size ();		
		array.push_back (ptr);
		table.insert (typename ptrHash::value_type (ptr, n));
	    }

	    int ptrToIndex (T* ptr) { 
		typename ptrHash::iterator it = table.find (ptr);
		if (it == table.end ())
		    return -1;
		else 
		    return it->second;
	    }

	    T* indexToPtr (int index) { 
		if (index >= (int) array.size ()) {
		    WARN << "index (" << index << ") out of bounds " << endl;
		    return NULL;    
		}

		return array [index];
	    }

	    void writePointer (T** ptrToValuePtr, Packet *pkt) {
		ASSERT (ptrToValuePtr != NULL);
		
		if (!*ptrToValuePtr)
		    pkt->WriteInt ((uint32) -1);
		else {
		    int index = ptrToIndex (*ptrToValuePtr);
		    ASSERT (index >= 0);
		    pkt->WriteInt (index);
		}
	    }
	    void readPointer (T** ptrToValuePtr, Packet *pkt) {
		ASSERT (ptrToValuePtr != NULL);
		
		int index = (int) pkt->ReadInt ();
		if (index == -1) 
		    *ptrToValuePtr = NULL;
		else {
		    *ptrToValuePtr = indexToPtr (index);
		}
	    }
	};
    
    // 
    // These are pointers refering to addresses in the code-segment    
    // 
    static PointerTable<void, hash_void_ptr, eq_void_ptr> genericPointerTable;
    
    // 
    // Static strings, for example refer to addresses within the
    // data-segment which can be variable across different machines.
    // Hence, we make a table of the actual string values and use the
    // values for indexing.  (for the genericPointerTable, we use the
    // pointer addresses)
    // 
    static PointerTable<char, hash_char_ptr, eq_char_ptr> charPointerTable;
};

template<class T>
class TypedField : public Field {
public:
    TypedField (const char *name, const int offset) : 
	Field (name, offset) {}

    template<class U>
	friend ostream& operator<< (ostream& os, TypedField<U>* field);

    inline bool __isdirty (T& prev, T& curr) { 
	return prev != curr;
    }
    
    virtual bool IsDirty (byte *prev, byte *curr, DeltaMask& mask) { 
	if (!prev)
	    return true;

	ASSERT (curr != NULL);
	bool dirty = __isdirty (*(T *)(prev + offset), *(T *)(curr + offset));
	if (dirty)
	    mask.Set (maskIndex, 1);
	return dirty;
    }

    virtual void PackUpdate (byte *curr, Packet *pkt, const DeltaMask& mask) {
	if (!mask.IsSet (maskIndex))
	    return;

	DB (10) << "SERIALIZING field=" << name << endl;
	ASSERT (curr);
	Write (* (T *) (curr + offset), pkt);
    }

    virtual void UnpackUpdate (byte *curr, Packet *pkt, const DeltaMask& mask) {
	if (!mask.IsSet (maskIndex))
	    return;

	DB (10) << "DE-SERIALIZING field=" << name << endl;
	ASSERT (curr);
	Read (* (T *) (curr + offset), pkt);
    }
    
    // Default method; would work only for bool, char.
    // Other types will need endian-agnostic serializations

    inline void Write (T& value, Packet *pkt) { 
	pkt->WriteBuffer ((byte *) &value, sizeof (T));
    }

    inline void Read (T& value, Packet *pkt) { 
	pkt->ReadBuffer ((byte *) &value, sizeof (T));
    }

    // debugging
    virtual int Size () { 
	return sizeof (T);
    }
};

template<class T>
ostream& operator<<(ostream& os, TypedField<T>& field) { 
    return os << &field;
}

template<class T>
ostream& operator<<(ostream& os, TypedField<T>* field) { 
    return os << "name=" << field->name << " offset=" << field->offset 
	<< " maskindex=" << field->maskIndex;
}

// Dirty for char * pointers is different
template<>
inline bool TypedField<char *>::__isdirty (char *&prev, char *&curr) {
    if (prev == curr)
	return false;
    if (!prev || !curr)
	return true;
    return strcmp (prev, curr) != 0;
}

// gentity_t and gclient_t * -- they never get dirty, since they are handled at 
// the "top level"
template<>
inline bool TypedField<gentity_t *>::__isdirty (gentity_t *&prev, gentity_t *&curr) {
    return false;
}
template<>
inline bool TypedField<gclient_t *>::__isdirty (gclient_t *&prev, gclient_t *&curr) {
    return false;
}

// Template specializations to handle endian-ness

template<>
inline void TypedField<short>::Write (short& value, Packet *pkt) { 
    pkt->WriteShort (value);
}
template<>
inline void TypedField<short>::Read (short& value, Packet *pkt) { 
    value = pkt->ReadShort ();
}
template<>
inline void TypedField<int>::Write (int& value, Packet *pkt) { 
    pkt->WriteInt (value);
}
template<>
inline void TypedField<int>::Read (int& value, Packet *pkt) { 
    value = pkt->ReadInt ();
}

template<>
inline void TypedField<float>::Write (float& value, Packet *pkt) {
    pkt->WriteFloat (value);
}

template<>
inline void TypedField<float>::Read (float& value, Packet *pkt) {
    value = pkt->ReadFloat ();
}

// XXX Support a 'bit-stream' instead of a 'byte-stream' 
// with Merc::Packet someday, so we could write sth for `bool'

// Handle generic pointers (including gitem_t *?)

#if 0

// Full class specialization does seem to compile; so you may 
// want to refactor Read+Write into another class which you can 
// specialize wholesale.

/* Does not seem to compile :( */
template<class T> 
void TypedField<T *>::Write (T* &value, Packet *pkt) {
    if (!value) 
	pkt->WriteInt (-1);
    else {
	int index = GetIndexFromPointer (reinterpret_cast<void *> (value));
	ASSERT (index >= 0);
	pkt->WriteInt (index);
    }
}

template<class T> 
void TypedField<T *>::Read (T* &value, Packet *pkt) {
    int index = pkt->ReadInt ();
    if (index == -1) 
	value = NULL;
    else
	value = reinterpret_cast<T *> (GetPointerFromIndex (index));
}
#endif

// char *
template<>
inline void TypedField<char *>::Write (char* &value, Packet *pkt) {
    Field::charPointerTable.writePointer (&value, pkt);
}
template<>
inline void TypedField<char *>::Read (char* &value, Packet *pkt) {
    Field::charPointerTable.readPointer (&value, pkt);
}

// void *
template<>
inline void TypedField<void *>::Write (void* &value, Packet *pkt) {
    Field::genericPointerTable.writePointer (&value, pkt);
}
template<>
inline void TypedField<void *>::Read (void* &value, Packet *pkt) {
    Field::genericPointerTable.readPointer (&value, pkt);
}

// int *
template<>
inline void TypedField<int *>::Write (int* &value, Packet *pkt) {
    Field::genericPointerTable.writePointer ((void **) &value, pkt);
}
template<>
inline void TypedField<int *>::Read (int* &value, Packet *pkt) {
    Field::genericPointerTable.readPointer ((void **) &value, pkt);
}

// float *
template<>
inline void TypedField<float *>::Write (float* &value, Packet *pkt) {
    Field::genericPointerTable.writePointer ((void **) &value, pkt);
}
template<>
inline void TypedField<float *>::Read (float* &value, Packet *pkt) {
    Field::genericPointerTable.readPointer ((void **) &value, pkt);
}

// gitem_t *
template<>
inline void TypedField<gitem_t *>::Write (gitem_t* &value, Packet *pkt) {
    Field::genericPointerTable.writePointer ((void **) &value, pkt);
}
template<>
inline void TypedField<gitem_t *>::Read (gitem_t* &value, Packet *pkt) {
    Field::genericPointerTable.readPointer ((void **) &value, pkt);
}

// Do nothing for gentity_t and gclient_t pointers; they are serialized 
// at the top level explicitly

template<>
inline void TypedField<gentity_t *>::Write (gentity_t* &ent, Packet *pkt) {
};

template<>
inline void TypedField<gentity_t *>::Read (gentity_t* &ent, Packet *pkt) {
}

template<>
inline void TypedField<gclient_t *>::Write (gclient_t* &ent, Packet *pkt) {
};

template<>
inline void TypedField<gclient_t *>::Read (gclient_t* &ent, Packet *pkt) {
}

// Handle arrays 

// Hm.. newer gcc's have more stringent requirements about 
// dependent types. So, I have to qualify variables from 
// the parent classes. (using this->) - Ashwin [06/02/2006]

template<class T>
class TypedArray : public TypedField<T> {
    int    length;

    inline void initBitmask () { memset(Field::bitmask, 0, sizeof(byte) * Field::ARR_MASK_MAXLEN); }
    
    inline void setBitInsideByte (int j, int bit) { Field::bitmask[j] |= (0x1 << bit); }
    inline void setBit (int i) { setBitInsideByte ((i)/8, (i) % 8 ); } 
    
    inline bool isBitSetInsideByte (int j, int bit) { return Field::bitmask[j] & ((0x1 << bit)); }
    inline bool isBitSet (int i) { return isBitSetInsideByte ((i)/8, (i) % 8); } 

    inline int  maskLen (int x) { return (x / 8 + (x % 8 != 0 ? 1 : 0)); }    
public:
    TypedArray (const char *name, const int offset, const int len) : 
	TypedField<T> (name, offset), length (len) {}

    virtual bool IsDirty (byte *prev, byte *curr, DeltaMask& mask) { 
	if (!prev)
	    return true;
	ASSERT (curr != NULL);

	bool dirty = false;
	T *parr = (T *) (prev + this->offset), *carr = (T *) (curr + this->offset);

	for (int i = 0; i < length; i++) { 
	    if (parr [i] != carr [i]) {
		dirty = true;
		break;
	    }
	}
	if (dirty)
	    mask.Set (this->maskIndex, 1);
	return dirty;
    }
    
    virtual void PackUpdate (byte *curr, Packet *pkt, const DeltaMask& mask) {
	if (!mask.IsSet (this->maskIndex))
	    return;

	DB (10) << "SERIALIZING array=" << this->name << endl;
	ASSERT (curr);

	initBitmask ();
	T *array = (T *) (curr + this->offset);

	for (int i = 0; i < length; i++) {
	    if (array [i] != 0)  // should work for T = char,int,float,pointers
		setBit (i);
	}

	int bitmaskLen = maskLen (length);
	for (int i = 0; i < bitmaskLen; i++) 
	    pkt->WriteByte (Field::bitmask [i]);

	for (int i = 0; i < length; i++) { 
	    if (isBitSet (i)) 
		Write (array [i], pkt);
	}
    }

    virtual void UnpackUpdate (byte *curr, Packet *pkt, const DeltaMask& mask) {
	if (!mask.IsSet (this->maskIndex))
	    return;

	DB (10) << "DE-SERIALIZING array=" << this->name << endl;
	ASSERT (curr);

	initBitmask ();
	int bitmaskLen = maskLen (length);
	for (int i = 0; i < bitmaskLen; i++) 
	    Field::bitmask[i] = pkt->ReadByte();

	T *array = (T *) (curr + this->offset);
	for (int i = 0; i < length; i++) {
	    array [i] = 0; 
	    if (isBitSet (i))
		Read (array [i], pkt);
	}
    }

    // debugging; claim that, on average, half the elements are dirty
    virtual int Size () { 
	return sizeof (T) * (int) (length / 2); 
    }
};

////////////////////////////////////////////////////////////////////////////////////

// T would be one of gentity_t or gclient_t
//    g_GEntityTypeInfo = new Quake3TypeInfo<gentity_t> (g_GEntityFields);
//    g_GClientTypeInfo = new Quake3TypeInfo<gclient_t> (g_GClientFields);
//    
template <class T>
class Quake3TypeInfo { 
private:
    Field             **m_Fields;
    int                 m_NumFields;
    int                 m_MaskOffset;
public:
    Quake3TypeInfo (const Field *fields [], int numfields, int offset) :
	m_MaskOffset (offset)
    { 
	m_Fields = (Field **) fields;
	m_NumFields = numfields;

	for (int i = 0; i < m_NumFields; i++) {
	    m_Fields [i]->SetMaskIndex (offset + i);
	}
    }
    
    const int GetNumFields () const { 
	return m_NumFields; 
    }	

    const Field *GetField (int i) const {
	ASSERT (i < m_NumFields);
	return m_Fields [i];
    }

    bool IsDirty (T* prev, T* curr, DeltaMask& mask, list<string> *dirtylist = NULL) { 
	if (!prev) { 
	    for (int i = 0; i < m_NumFields; i++) {
		mask.Set (m_MaskOffset + i);
		if (dirtylist)
		    dirtylist->push_back (m_Fields [i]->GetName ());
	    }
	    return true;
	}

	bool ret = false;

	for (int i = 0; i < m_NumFields; i++) {
	    if (m_Fields [i]->IsDirty (reinterpret_cast<byte *> (prev), reinterpret_cast<byte *> (curr), mask)) {
		ret = true;
		if (dirtylist)
		    dirtylist->push_back (m_Fields [i]->GetName ());
	    }
	}
	return ret;
    }

    void GetDirtyFields (const DeltaMask& mask, list<string> *dirtylist) {
	for (int i = 0; i < m_NumFields; i++) {
	    if (mask.IsSet (m_MaskOffset + i))
		dirtylist->push_back (m_Fields [i]->GetName ());
	}
    }

    void PackUpdate (T *c, Packet *pkt, const DeltaMask& mask) {
	ASSERT (c != NULL);
	for (int i = 0; i < m_NumFields; i++) 
	    m_Fields [i]->PackUpdate (reinterpret_cast<byte *> (c), pkt, mask);
    }
    
    void UnpackUpdate (T *c, Packet *pkt, const DeltaMask& mask) {
	ASSERT (c != NULL);
	for (int i = 0; i < m_NumFields; i++)
	    m_Fields [i]->UnpackUpdate (reinterpret_cast<byte *> (c), pkt, mask);
    }
};

class Cluster {
    const char **m_Fields;
    int m_NumFields;
    int *m_FieldIndices;
    
public:
    Cluster (const char *fields [], int nfields) : 
	m_Fields ((const char **) fields), m_NumFields (nfields)
    {
	m_FieldIndices = new int [m_NumFields];
	for (int i = 0; i < m_NumFields; i++) 
	    m_FieldIndices [i] = -1;
    }
    
    ~Cluster () { 
	delete [] m_FieldIndices;
    }
    
    const int GetNumFields () const { 
	return m_NumFields; 
    }
    const char *GetField (int f) const { 
	ASSERT (f < m_NumFields);
	return m_Fields [f];
    }
    
    void SetFieldIndex (int f, int index) { 
	ASSERT (f < m_NumFields);
	m_FieldIndices [f] = index;
    }
    const int GetFieldIndex (int f) const {
	ASSERT (f < m_NumFields);
	return m_FieldIndices [f];
    }
};

class DeltaEncoder { 
    Quake3TypeInfo<gentity_t> *m_GEntityInfo;
    Quake3TypeInfo<gclient_t> *m_GClientInfo;

    int           m_Offset;               // reserve the first so many bits for stuff "outside" the delta-encoder
    Cluster     **m_Clusters;
    int           m_NumClusters;

    typedef map<int, int> FCMap;
    typedef hash_map<const char*, int, hash_const_char_ptr, eq_const_char_ptr> FIMap;    
    
    FCMap         m_FieldsToCluster;    
    FIMap         m_FieldsToIndex;
    
    DeltaEncoder (int offset);
protected:
    static DeltaEncoder *s_DeltaEncoder;
public:
    static DeltaEncoder *GetInstance (int offset) { 
	if (!s_DeltaEncoder)
	    s_DeltaEncoder = new DeltaEncoder (offset);
	return s_DeltaEncoder;
    }

    ~DeltaEncoder () {}

    uint32 GetDeltaMaskBits () { 
	if (g_QuakePreferences.noclusters) 
	    return m_GEntityInfo->GetNumFields () + m_GClientInfo->GetNumFields ();
	else
	    return m_NumClusters + 1;
    }

    void GetFieldList (list<string> *fieldlist);
    pair<bool, bool> RecordChanges (gentity_t *ent, gentity_t *nent, gclient_t *client, gclient_t *nclient, DeltaMask &mask, list<string> *dirtylist = NULL);
    void PackUpdate (gentity_t *ent, gclient_t *client, Packet *pkt, const DeltaMask& mask);
    void UnpackUpdate (gentity_t *ent, gclient_t *client, Packet *pkt, const DeltaMask& mask);

    // For debugging
    int GetClusterForField (const char *field) const;
    const Cluster *GetCluster (int i) const { 
	ASSERT (i >= 0);
	ASSERT (i < m_NumClusters);
	return m_Clusters [i];
    }
    const int GetNumClusters () const { 
	return m_NumClusters;
    }

    Quake3TypeInfo<gentity_t> * GetGEntityInfo () const {
	return m_GEntityInfo;
    }

    Quake3TypeInfo<gclient_t> * GetGClientInfo () const {
	return m_GClientInfo;
    }

private:
    const char *GetFieldName (int index) { 
	int entity_fields = m_GEntityInfo->GetNumFields ();
	if (index < entity_fields)
	    return m_GEntityInfo->GetField (index)->GetName ();
	else 
	    return m_GClientInfo->GetField (index - entity_fields)->GetName ();
    }

    // Used to indicate that there are fields not found in cluster definitions 
    inline int LastBit () { return m_NumClusters; }

private:
    DeltaMask MakeFieldMask (const DeltaMask& mask);
    void BuildFieldClusterMaps ();

};

void InitializeQuake3Metadata ();
extern DeltaEncoder *g_DeltaEncoder;

#endif

// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
