////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////
#include "Quake3Entity.h"
#include "Quake3Metadata.h"
#include "Options.h"

//
// location of gentity_t pointers in the gentity struct
//

#define GENTITY_FOFS(x) { (int)&(((gentity_t *)0)->x), #x }
#define GCLIENT_FOFS(x) { (int)&(((gclient_t *)0)->x), "client." #x }

Quake3PointerInfo g_EntityPointerOffsets[] = {
    GENTITY_FOFS(parent),
    GENTITY_FOFS(nextTrain),
    GENTITY_FOFS(prevTrain),
    GENTITY_FOFS(target_ent),
    GENTITY_FOFS(chain),
    /*
    GENTITY_FOFS(enemy),   // this pointer is manipulated by the AI, and the replicas dont need it, "I think" :P
    */
    GENTITY_FOFS(activator),
    /*
    GENTITY_FOFS(teamchain), // we make sure all items in the entity group are primaries on the same node.
    GENTITY_FOFS(teammaster),
    */
    GENTITY_FOFS(otherEntity),
    GENTITY_FOFS(otherEntity2),
    { -1, NULL }
};

Quake3PointerInfo g_ClientPointerOffsets[] = {
    GCLIENT_FOFS(hook),    
#ifdef MISSIONPACK
    GCLIENT_FOFS(persistantPowerup),
#endif
    { -1, NULL }
};

//
// classification of all dynamic classes based on quake3 classname
//

const char *g_PlayerClassStrings[] = { 
    "bot", 
    "player", 
    NULL,
};
const char *g_MissileClassStrings[] = {
    "plasma",
    "grenade",
    "hook",
    "nail",
    "prox mine",
    "rocket",
    "bfg",
    NULL,
};
const char *g_ImmobileLongLiveClassStrings[] = {
    // *** derived from game/bg_misc.c:bg_itemlist
    "item_armor_shard", 
    "item_armor_combat", 
    "item_armor_body", 
    "item_health_small",
    "item_health",
    "item_health_large",
    "item_health_mega",
    "weapon_gauntlet", 
    "weapon_shotgun", 
    "weapon_machinegun", 
    "weapon_grenadelauncher",
    "weapon_rocketlauncher",
    "weapon_lightning", 
    "weapon_railgun", 
    "weapon_plasmagun", 
    "weapon_bfg",
    "weapon_grapplinghook",
    "ammo_shells",
    "ammo_bullets",
    "ammo_grenades",
    "ammo_cells",
    "ammo_lightning",
    "ammo_rockets",
    "ammo_slugs",
    "ammo_bfg",
    "holdable_teleporter", 
    "holdable_medkit", 
    "item_quad", 
    "item_enviro",
    "item_haste",
    "item_invis",
    "item_regen",
    "item_flight",
    "team_CTF_redflag",
    "team_CTF_blueflag",
#ifdef MISSIONPACK
    "holdable_kamikaze", 
    "holdable_portal", 
    "holdable_invulnerability", 
    "ammo_nails",
    "ammo_mines",
    "ammo_belt",
    "item_scout",
    "item_guard",
    "item_doubler",
    "item_ammoregen",
    "team_CTF_neutralflag",
    "item_redcube",
    "item_bluecube",
    "weapon_nailgun", 
    "weapon_prox_launcher", 
    "weapon_chaingun", 
#endif

    // *** derived from game/g_spawn.c:spawns
    "func_plat",
    "func_button",
    "func_door",
    "func_static",
    "func_rotating",
    "func_bobbing",
    "func_pendulum",
    "func_train",
    "func_group",
    "func_timer",
    
    "trigger_always",
    "trigger_multiple",
    "trigger_push",
    "trigger_teleport",
    "trigger_hurt",
    
    /* jeffpang: some of these should be distributed...
    "target_give",
    "target_remove_powerups",
    "target_delay",
    "target_speaker",
    "target_print",
    "target_laser",
    "target_score",
    "target_teleporter",
    "target_relay",
    "target_kill",
    "target_position",
    "target_location",
    "target_push",
    */

    NULL,
};
const char *g_ImmobileShortLiveClassStrings[] = {
    /* jeffpang: need to handle this specially...
    "bodyque",
    */

    // jeffpang: I think these are for effects...
    "tempEntity",
    
    NULL
};
const char *g_MobileLongLiveClassStrings[] = {
    NULL,
};
const char *g_MobileShortLiveClassStrings[] = {
    NULL,
};

/////////////////////////////////////////////////////////////////////////
// infrastructure for semi-automatically gathering metadata about quake
// data-structures (gentity_t and gclient_t, basically) and serializing, 
// delta-encoding them.

byte Field::bitmask [Field::ARR_MASK_MAXLEN];
Field::PointerTable<void, hash_void_ptr, eq_void_ptr> Field::genericPointerTable;
Field::PointerTable<char, hash_char_ptr, eq_char_ptr> Field::charPointerTable;

// These are the F_LSTRING fields from the gentity_t structure 
// they are allocated by G_NewString - and, so they need to 
// be registered into the strings array...

Quake3PointerInfo g_EntityDynStringOffsets [] = {
    GENTITY_FOFS(classname),
    GENTITY_FOFS(model),
    GENTITY_FOFS(model2),
    GENTITY_FOFS(target),
    GENTITY_FOFS(targetname),
    GENTITY_FOFS(message),
    GENTITY_FOFS(team),
    GENTITY_FOFS(targetShaderName),
    GENTITY_FOFS(targetShaderNewName),
    { -1, NULL }
};

#include "ExtractedMetadata.cxx"
#include "ClusterTables.cxx"

#define ASIZE(arr)  (sizeof(arr)/sizeof(void *))

void Field::Initialize () {
    
    // add generic pointers
    for (int i = 0, n = ASIZE (g_FunctionPointers); i < n; i++)
	genericPointerTable.add (g_FunctionPointers [i]);   
    for (int i = 0, n = ASIZE (g_GlobalsPointers); i < n; i++)
	genericPointerTable.add (g_GlobalsPointers [i]); 
    
    // add strings
    for (int i = 0, n = ASIZE (g_StaticStrings); i < n; i++) 
	charPointerTable.add (g_StaticStrings [i]);
    
    gentity_t *ent = &g_entities [0];
    for (int i = 0; i < level.num_entities; i++, ent++) {
	if (!ent->inuse)
	    continue;
	
	for (Quake3PointerInfo *off = g_EntityDynStringOffsets; off->name != NULL; off++) {
	    char *ptr = * (char **) ((byte *) ent + off->offset);
	    if (ptr)
		charPointerTable.add (ptr);
	}
    }
}

DeltaEncoder *DeltaEncoder::s_DeltaEncoder = NULL;
DeltaEncoder *g_DeltaEncoder = NULL;
    
DeltaEncoder::DeltaEncoder (int offset) :  m_Offset (offset)
{
    Field::Initialize ();
    
    m_GEntityInfo = new Quake3TypeInfo<gentity_t> (g_GEntityFields, ASIZE (g_GEntityFields), 0);
    m_GClientInfo = new Quake3TypeInfo<gclient_t> (g_GClientFields, ASIZE (g_GClientFields), m_GEntityInfo->GetNumFields ());

    if (!g_QuakePreferences.noclusters) {
	m_Clusters = (Cluster **) g_FieldClusters;
	m_NumClusters = ASIZE (g_FieldClusters);

	BuildFieldClusterMaps ();
    }
}

void DeltaEncoder::GetFieldList (list<string> *fieldlist) 
{
    for (int i = 0, n = m_GEntityInfo->GetNumFields (); i < n; i++)
	fieldlist->push_back (m_GEntityInfo->GetField (i)->GetName ());
    for (int i = 0, n = m_GClientInfo->GetNumFields (); i < n; i++)
	fieldlist->push_back (m_GClientInfo->GetField (i)->GetName ());
}

pair<bool, bool> DeltaEncoder::RecordChanges (gentity_t *ent, gentity_t *nent, 
	gclient_t *client, gclient_t *nclient, 
	DeltaMask &mask, list<string> *dirtylist) 
{
    pair<bool, bool> dirty (false, false);
    DeltaMask pmask;

    dirty.first = m_GEntityInfo->IsDirty (ent, nent, pmask, dirtylist);
    
    if (client) {
	ASSERT (nclient);
	dirty.second = m_GClientInfo->IsDirty (client, nclient, pmask, dirtylist);
    }

    int total = m_GEntityInfo->GetNumFields () + m_GClientInfo->GetNumFields ();
    if (g_QuakePreferences.noclusters) {
	for (int i = 0; i < total; i++) {
	    if (pmask.IsSet (i)) 
		mask.Set (m_Offset + i);
	}
    }
    else { // map to cluster bits;
	for (int i = 0; i < total; i++) {
	    if (pmask.IsSet (i)) {
		FCMap::iterator it = m_FieldsToCluster.find (i);
		if (it == m_FieldsToCluster.end ()) {
		    WARN << "Dirty field " << GetFieldName (i) << " does not belong to any cluster " << endl;
		    mask.Set (m_Offset + LastBit ());
		}
		else {
		    mask.Set (m_Offset + it->second);
		}
	    }
	}
    }

    return dirty;
}

void DeltaEncoder::PackUpdate (gentity_t *ent, gclient_t *client, Packet *pkt, const DeltaMask& mask) 
{
    DeltaMask pmask = MakeFieldMask (mask);

    m_GEntityInfo->PackUpdate (ent, pkt, pmask);
    if (client)
	m_GClientInfo->PackUpdate (client, pkt, pmask);	
}

void DeltaEncoder::UnpackUpdate (gentity_t *ent, gclient_t *client, Packet *pkt, const DeltaMask& mask) 
{
    DeltaMask pmask = MakeFieldMask (mask);

    m_GEntityInfo->UnpackUpdate (ent, pkt, pmask);
    if (client)
	m_GClientInfo->UnpackUpdate (client, pkt, pmask);	
}

DeltaMask DeltaEncoder::MakeFieldMask (const DeltaMask& mask) 
{
    DeltaMask pmask;

    if (g_QuakePreferences.noclusters) { 
	for (int i = 0, m = m_GEntityInfo->GetNumFields () + m_GClientInfo->GetNumFields (); i < m; i++) {
	    if (mask.IsSet (m_Offset + i))
		pmask.Set (i);
	}
	return pmask;
    }
    
    int n = m_NumClusters;

    if (mask.IsSet (m_Offset + n)) { // Encode every damn thing!
	for (int i = 0, m = m_GEntityInfo->GetNumFields () + m_GClientInfo->GetNumFields (); i < m; i++)
	    pmask.Set (i);
    }
    else {
	for (int i = 0; i < n; i++) {
	    if (mask.IsSet (m_Offset + i)) {  // Cluster 'i' should be encoded
		Cluster *cl = m_Clusters [i];

		DB (10) << "encoding cluster [" << i << "]" << endl;
		for (int j = 0, nf = cl->GetNumFields (); j < nf; j++) { 
		    int fi = cl->GetFieldIndex (j);
		    pmask.Set (fi);
		}
	    }
	}
    }

    return pmask;
}
    
void DeltaEncoder::BuildFieldClusterMaps () 
{
    ASSERT (g_QuakePreferences.noclusters == false);    
    int entity_fields = m_GEntityInfo->GetNumFields ();

    for (int i = 0, n = entity_fields; i < n; i++) {
	const Field *f = m_GEntityInfo->GetField (i);
	m_FieldsToIndex.insert (FIMap::value_type (f->GetName (), i));
    }
    for (int i = 0, n = m_GClientInfo->GetNumFields (); i < n; i++) {
	const Field *f = m_GClientInfo->GetField (i);
	m_FieldsToIndex.insert (FIMap::value_type (f->GetName (), i + entity_fields));
    }

    for (int i = 0; i < m_NumClusters; i++) {	    
	Cluster *cl = m_Clusters [i];
	for (int j = 0, n = cl->GetNumFields (); j < n; j++) {
	    FIMap::iterator it = m_FieldsToIndex.find (cl->GetField (j));
	    ASSERT (it != m_FieldsToIndex.end ());
	    cl->SetFieldIndex (j, it->second);

	    m_FieldsToCluster.insert (FCMap::value_type (it->second, i));
	}		
    }
}

int DeltaEncoder::GetClusterForField (const char *field) const 
{
    ASSERT (g_QuakePreferences.noclusters == false);

    FIMap::const_iterator fit = m_FieldsToIndex.find (field);
    ASSERT (fit != m_FieldsToIndex.end ());
    
    DB (10) << "index for field `" << field << "' is " << fit->second << endl;
    FCMap::const_iterator it = m_FieldsToCluster.find (fit->second);
    
    if (it == m_FieldsToCluster.end ()) {
	WARN << "field does not map to a known cluster!" << endl;
	return -1;
    }
    else
	return it->second;
}

void InitializeQuake3Metadata () {
    int npointers = Quake3Entity::NumEntityPointers ();
    g_DeltaEncoder = DeltaEncoder::GetInstance (4 + npointers);
}

// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
