////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

#ifndef __QUAKE3_DYNAMIC_ENTITY_H__
#define __QUAKE3_DYNAMIC_ENTITY_H__

#include <gameapi/GameObject.h>
#include "Quake3Entity.h"
#include "q_exports.h"

/**
 * A dynamic gentity_t that has to be handled by Colyseus.
 */
class Quake3DynamicEntity : 
public DynamicGameObject, public Quake3Entity 
{
 public:
    
    typedef enum { INVALID, PLAYER, MISSILE, 
		   IMMOBILE_LONGLIVE, IMMOBILE_SHORTLIVE,
		   MOBILE_LONGLIVE, MOBILE_SHORTLIVE } Type;

 private:

    struct TypeClass {
	Type t;
	const char **c;
	TypeClass(Type t, const char **c) : t(t), c(c) {}
    };

    static const TypeClass mClassList[];
    static map<string, Type> mClassMap;

    // true when mEntityFields and mClientFields are initialized
    bool       mInitialized;
    // base delta mask for this entity (when "new")
    DeltaMask  mInitDeltaMask;
    // internal representation of fields 
    gentity_t *mEntityFields;
    // internal representation of fields
    gclient_t *mClientFields;
    // internal representation of client configstring
    char      *mConfigString;
    // colyseus representation of pointers to other entities, should only
    // be initialized once and then each entry mutated directly
    vector< gRef<Quake3Entity> > mPointers;

 public:
    static Type GetClass(const char *classname);
    static gentity_t *NumberToGEntity (int entnum);
    static int GEntityToNumber (gentity_t *ent);

    Quake3DynamicEntity(gentity_t *fields, GameManager *manager);
    Quake3DynamicEntity(GObjectInfoIface *info);
    virtual ~Quake3DynamicEntity();

    ///////////////////////////////////////////////////////////////////////////
    // IMPLEMENTS DynamicGameObject

    virtual Vec3 GetOrigin();
    virtual void SetOrigin(const Vec3& new_origin);

    virtual const DeltaMask& GetInitDeltaMask();
    virtual const uint32 GetDeltaMaskBits();
    virtual uint32 GetAreaOfInterest(list<BBox> *toFill, GameWorld *world);
    virtual uint32 IsInterested(const Vec3& pt, 
				gTypeCode t = GAMETYPE_UNKNOWN);
    virtual uint32 IsInterested(const BBox& vol,
				gTypeCode t = GAMETYPE_UNKNOWN);
    
    virtual void PackUpdate(Packet *buffer, const DeltaMask& mask);
    virtual void UnpackUpdate(Packet *buffer, const DeltaMask& mask, 
			      list<GameObjectRef *> *unresolved);

    virtual void CommitMigration(sid_t target);

    virtual void RunThink(GameManager *manager, real32 delta) {}

    virtual void PrintFields(ostream& out);

    ///////////////////////////////////////////////////////////////////////////

 protected:

    virtual void Tie(gentity_t *g_entities_ptr, gclient_t *g_client_ptr);

 private:

    void Initialize(gentity_t *fields);
    void Initialize(bool withclient);
    void SetClassParameters();
    void RecordDeltaFields (list<string>& flist, bool isNew = false);

 public:

    /**
     * Called when this Quake3DynamicEntity is not yet tied to an internal
     * gentity_t object. This method ties it to a new one.
     */
    void AllocateInternalEntity();
    /**
     * Called when this Quake3DynamicEntity is deleted by the GameManager.
     * This method unties the corresponding internal gentity_t object
     * and frees it.
     */
    void DeallocateInternalEntity();
    /**
     * Called at the beginning of each frame, signalling that the
     * GameManager may have applied changes to mEntityFields and
     * mClientFields that should be applied to the internal gentity_t
     * object.
     */
    void ApplyChangesToInternalEntity();
    /**
     * Called after quake runs its frame and changes are possibly made
     * to the internal gentity_t. This method applies those changes to
     * mEntityFields and mClientFields, setting the appropriate dirty
     * bits in the delta mask.
     */
    void ApplyChangesFromInternalEntity();

    void CheckInvariants();

    /**
     * Return object type for the visualizer 
     */
    virtual VisType GetVisualizerType ();

    /**
     * Classify entity type.
     */
    Type  GetClass();

    // For debugging
    const char *GetClassName ();

    /**
     * Distance moved each sec.
     */
    float GetSpeed();
};

#endif
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
