////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

#include <gameapi/GameManager.h>
#include <gameapi/GameModule.h>
#include "Quake3DynamicEntity.h"
#include "World.h"
#include "Options.h"
#include "Quake3Metadata.h"
#include "Logs.h"
#include <server/server.h>

extern void dumpObject (GObject *obj) { 
    Quake3DynamicEntity *ent = reinterpret_cast<Quake3DynamicEntity *> (obj);
    if (!ent)
	return;
    DB (-10) << ent << endl;
}

// Check Invariants if defined
#define CKINV() CheckInvariants()

gentity_t *Quake3DynamicEntity::NumberToGEntity (int entnum) {
    if (entnum < 0 || entnum > level.num_entities)
	return NULL;
    if (entnum == ENTITYNUM_NONE)
	return NULL;
    return g_entities + entnum;
}

int Quake3DynamicEntity::GEntityToNumber (gentity_t *ent) { 
    if (!ent)
	return ENTITYNUM_NONE;

    int num = ent - g_entities;
    ASSERT (num >= 0);
    ASSERT (num < level.num_entities);
    ASSERT (num != ENTITYNUM_NONE);

    return num;
}

///////////////////////////////////////////////////////////////////////////////

const Quake3DynamicEntity::TypeClass Quake3DynamicEntity::mClassList[] = {
    TypeClass(Quake3DynamicEntity::PLAYER, g_PlayerClassStrings),
    TypeClass(Quake3DynamicEntity::MISSILE, g_MissileClassStrings),
    TypeClass(Quake3DynamicEntity::IMMOBILE_LONGLIVE, g_ImmobileLongLiveClassStrings),
    TypeClass(Quake3DynamicEntity::IMMOBILE_SHORTLIVE, g_ImmobileShortLiveClassStrings),
    TypeClass(Quake3DynamicEntity::MOBILE_LONGLIVE, g_MobileLongLiveClassStrings),
    TypeClass(Quake3DynamicEntity::MOBILE_SHORTLIVE, g_MobileShortLiveClassStrings),
    TypeClass(Quake3DynamicEntity::INVALID, NULL)
};

map<string, Quake3DynamicEntity::Type> Quake3DynamicEntity::mClassMap;

Quake3DynamicEntity::Type Quake3DynamicEntity::GetClass(const char *classname) {
    if (mClassMap.size() == 0) {
	for (const TypeClass *tc = mClassList; tc->t != INVALID; tc++) {
	    const char **cls;
	    for (cls = tc->c; *cls != NULL; cls++) {
		mClassMap.insert(pair<string,Type>(string(*cls), tc->t));
	    }
	}
    }
    
    map<string,Type>::iterator it = mClassMap.find(string(classname));
    if (it == mClassMap.end())
	return INVALID;
    else
	return it->second;
}

///////////////////////////////////////////////////////////////////////////////

void Quake3DynamicEntity::Tie(gentity_t *g_entities_ptr, gclient_t *g_client_ptr)
{
    ASSERT(mInitialized);
    ASSERT(mEntityFields != NULL);
    Quake3Entity::Tie(g_entities_ptr, g_client_ptr);
    
    // must static_cast up because of multiple inheritance; 
    // otherwise the 'this' pointers will be different!
    mEntityFields->colyseusObject = static_cast<Quake3Entity *>(this);
}

void Quake3DynamicEntity::Initialize(gentity_t *internal_entity)
{
    ASSERT(internal_entity);

    mEntityFields = (gentity_t *)malloc(sizeof(gentity_t));
    memcpy(mEntityFields, internal_entity, sizeof(gentity_t));

    // * fix special fields otherEntity, otherEntity2, etc.
    mEntityFields->otherEntity = internal_entity->otherEntity = NumberToGEntity (internal_entity->s.otherEntityNum);
    mEntityFields->otherEntity2 = internal_entity->otherEntity2 = NumberToGEntity (internal_entity->s.otherEntityNum2);
    // cerr << "otherEntity=" <<  mEntityFields->otherEntity << endl;
    
    if (internal_entity->client) {
	mClientFields = (gclient_t *)malloc(sizeof(gclient_t));
	memcpy(mClientFields, internal_entity->client, sizeof(gclient_t));
	mEntityFields->client = mClientFields;

	int clNum = internal_entity - g_entities;
	ASSERT(clNum >= 0 && clNum < MAX_CLIENTS);
	mConfigString = new char[MAX_STRING_CHARS];
	SV_GetConfigstring(CS_PLAYERS+clNum, mConfigString, MAX_STRING_CHARS);
    } else
	mClientFields = NULL;

    // initialize the pointer vector
    vector<gentity_t **> ptrs = GetEntityPointers(internal_entity);
    for (uint32 i = 0; i < ptrs.size(); i++) {
	// XXX: could it be the case that some of the objects we are pointing
	// to do not yet exist in the colyseus store? ...
	gentity_t *ptr = *ptrs[i];
	gRef<Quake3Entity> ref( ptr == NULL ? NULL : reinterpret_cast<Quake3Entity *>(ptr->colyseusObject) );
	mPointers.push_back(ref);
    }
    // separate loop for registration required! else some optimization
    // uses the local ref variable instead of the internal vector one
    for (uint32 i = 0; i < mPointers.size(); i++) {
	RegisterRef(&mPointers[i]);
    }
    
    // typecode
    mInitDeltaMask.Set(0);

    // regular entity fields 
    mInitDeltaMask.Set(1);
    // regular client fields 
    if (mClientFields != NULL) {
	// ent->client
	mInitDeltaMask.Set(2);
	// configstring
	mInitDeltaMask.Set(3);
    }

    // pointers
    for (uint32 i = 0; i < mPointers.size(); i++) {
	if (mPointers[i] != NULL)
	    mInitDeltaMask.Set(4 + i);
    }
    
    // XXX: this '4' is fragile, and gets used in Quake3Metadata.cpp::InitializeQuake3Metadata() 
    // as well. Make this less fragile, so things get changed at one central place
    // somehow... - Ashwin [05/31/2006]
    // 
    int curr = 4 + mPointers.size ();
    int bits = g_DeltaEncoder->GetDeltaMaskBits ();
    
    // even though bits corresponding all clusters are set here,
    // the client part would not get encoded if the client pointer
    // is NULL
    // 
    for (int i = 0; i < bits; i++)
	mInitDeltaMask.Set (curr + i);

    mInitialized = true;
}

void Quake3DynamicEntity::Initialize(bool withclient)
{
    mEntityFields = (gentity_t *)malloc(sizeof(gentity_t));
    bzero(mEntityFields, sizeof(gentity_t));
    if (withclient) {
	mClientFields = (gclient_t *)malloc(sizeof(gclient_t));
	bzero(mClientFields, sizeof(gclient_t));
	mConfigString = new char[MAX_STRING_CHARS];
	bzero(mConfigString, MAX_STRING_CHARS);
    } else
	mClientFields = NULL;

    // don't have to maintain this pointer; this is just to let
    // GetEntityPointers know if there is a client struct or not
    mEntityFields->client = mClientFields;

    vector<gentity_t **> ptrs = GetEntityPointers(mEntityFields);
    for (uint32 i = 0; i < ptrs.size(); i++) {
	gRef<Quake3Entity> ref( (Quake3Entity *)NULL );
	mPointers.push_back(ref);
    }
    // separate loop for registration required! else some optimization
    // uses the local ref variable instead of the internal vector one
    for (uint32 i = 0; i < mPointers.size(); i++) {
	RegisterRef(&mPointers[i]);
    }

    mInitialized = true;
}

VisType Quake3DynamicEntity::GetVisualizerType () 
{
    switch (GetClass ()) {
    case PLAYER:
	if (mEntityFields->client) { 
	    if (strstr (mEntityFields->client->pers.netname, "Bot"))
		return VisType_Bot;
	    else
		return VisType_Player;
	}
	else
	    return VisType_Bot;
    case MISSILE:
	return VisType_Missile;
    default:
	return VisType_Other;
    }
}

void Quake3DynamicEntity::SetClassParameters()
{
    switch (GetClass()) {
    case PLAYER:
	SetPubInterval(g_QuakePreferences.pubttl_player);
	break;
    case MISSILE: 
	SetPubInterval(g_QuakePreferences.pubttl_missile);
	break;
    case IMMOBILE_LONGLIVE: 
	SetPubInterval(g_QuakePreferences.pubttl_immobile_longlive);
	break;
    case IMMOBILE_SHORTLIVE:
	SetPubInterval(g_QuakePreferences.pubttl_immobile_shortlive);
	break;
    case MOBILE_LONGLIVE:
	SetPubInterval(g_QuakePreferences.pubttl_mobile_longlive);
	break;
    case MOBILE_SHORTLIVE:
	SetPubInterval(g_QuakePreferences.pubttl_mobile_shortlive);
	break;
    default:
	WARN << "unknown quake3 class: " << GetClass() << endl;
    };
}

Quake3DynamicEntity::Quake3DynamicEntity(gentity_t *fields, GameManager *manager) :
    mInitialized(false), 
    mEntityFields(NULL), mClientFields(NULL), mConfigString(NULL),
    DynamicGameObject(GAMETYPE_QUAKE3_DYNAMIC, manager)
{
    ASSERT(fields);
    Initialize(fields);
    Tie(fields, fields->client);
    SetClassParameters();

    if (g_QuakePreferences.deltas) {
	list<string> fieldlist;
	g_DeltaEncoder->GetFieldList (&fieldlist);
	RecordDeltaFields (fieldlist, true);
    }

    CKINV();
}

Quake3DynamicEntity::Quake3DynamicEntity(GObjectInfoIface *info) :
    mInitialized(false), 
    mEntityFields(NULL), mClientFields(NULL), mConfigString(NULL),
    DynamicGameObject(GAMETYPE_QUAKE3_DYNAMIC, info)
{
}

Quake3DynamicEntity::~Quake3DynamicEntity()
{
    if (mInitialized) {
	if (mEntityFields) free(mEntityFields);
	if (mClientFields) free(mClientFields);
	if (mConfigString) delete[] mConfigString;
    }
}

Vec3 Quake3DynamicEntity::GetOrigin()
{
    ASSERT(mInitialized);

    vec3_t result;
    BG_EvaluateTrajectory(&mEntityFields->s.pos, level.time, result);
    if (mEntityFields && mEntityFields->classname) {
	DB(10) << "classname=" << mEntityFields->classname << " origin=" << result << endl;
    }
    return Vec3(result);
}

void Quake3DynamicEntity::SetOrigin(const Vec3& new_origin)
{
    // only quake should change the internal origin of the object
    ASSERT(0);
}

const DeltaMask& Quake3DynamicEntity::GetInitDeltaMask()
{
    return mInitDeltaMask;
}

const uint32 Quake3DynamicEntity::GetDeltaMaskBits()
{
    uint32 bits = 4 + NumEntityPointers () + g_DeltaEncoder->GetDeltaMaskBits ();
    return bits;
}

uint32 Quake3DynamicEntity::GetAreaOfInterest(list<BBox> *toFill, GameWorld *world)
{   
    return world->GetAreaOfInterest(toFill, this);
}

uint32 Quake3DynamicEntity::IsInterested(const Vec3& pt, gTypeCode t)
{
    Quake3World *w = dynamic_cast<Quake3World *>(GameManager::GetInstance()->GetInstance()->GetWorld());
    ASSERT(w);

    BBox aoi = w->GetViewableBBox(this);

    if (aoi.Contains(pt))
	return 1000;
    else
	return 0;
}

uint32 Quake3DynamicEntity::IsInterested(const BBox& vol, gTypeCode t)
{
    Quake3World *w = dynamic_cast<Quake3World *>(GameManager::GetInstance()->GetInstance()->GetWorld());
    ASSERT(w);

    BBox aoi = w->GetViewableBBox(this);

    if (aoi.Overlaps(vol))
	return 1000;
    else
	return 0;
}

void Quake3DynamicEntity::PackUpdate(Packet *buffer, const DeltaMask& mask)
{
    ASSERT(mInitialized);

    // override DynamicGameObject (it also sends origin)
    if (mask.IsSet(0)) {
	buffer->WriteInt(m_TypeCode);
    }

    // * pack normal gentity fields
    // * pack pointers from mPointers

    if (mask.IsSet (1) || mask.IsSet (2)) {
	if (mask.IsSet (2))
	    ASSERT (mClientFields != NULL);
	g_DeltaEncoder->PackUpdate (mEntityFields, mClientFields, buffer, mask);
    }

    // XXX: delta encode
    if (mask.IsSet(3)) {
	ASSERT(mConfigString != NULL);
	int len = strlen(mConfigString) + 1;
	buffer->WriteInt(len);
	buffer->WriteBuffer((byte *)mConfigString, len);
    }

    for (uint32 i = 0; i < mPointers.size(); i++) {
	if (mask.IsSet(4 + i)) {
	    mPointers[i].Serialize(buffer);
	}
    }

    CKINV();
}

void Quake3DynamicEntity::UnpackUpdate(Packet *buffer, const DeltaMask& mask, 
				list<GameObjectRef *> *unresolved)
{
    // override DynamicGameObject 
    if (mask.IsSet(0)) {
	gTypeCode type = buffer->ReadInt();
	ASSERT(type == m_TypeCode);
    }

    // * unpack normal gentity fields, into temp space if not initialized
    // * initialize if necessary
    // * unpack pointers to mPointers, overriding gentity corresponding fields
    // * if I am a primary, record mask as dirty bits

    if (!mInitialized) {
	Initialize (mask.IsSet (2));
	mInitDeltaMask = mask;
    }

    if (mask.IsSet (1) || mask.IsSet (2)) {
	if (mask.IsSet (2))
	    ASSERT (mClientFields != NULL);
	
	g_DeltaEncoder->UnpackUpdate (mEntityFields, mClientFields, buffer, mask);
	mEntityFields->colyseusObject = static_cast<Quake3Entity *>(this);
    }

    // XXX: delta encode
    if (mask.IsSet(3)) {
	ASSERT(mConfigString != NULL);
	int len = buffer->ReadInt();
	ASSERT(len <= MAX_STRING_CHARS);
	buffer->ReadBuffer((byte *)mConfigString, len);
    }

    for (uint32 i = 0; i < mPointers.size(); i++) {
	if (mask.IsSet(4 + i)) {
	    {
		mPointers[i] = gRef<Quake3Entity>(buffer);
	    }
	    if (!mPointers[i].IsResolved ()) {
		unresolved->push_back (&mPointers [i]);
	    }
	}
    }

    // if I am a primary, I need to transmit these changed bits to all
    // my replicas in the next frame
    if (!IsReplica()) {
	GetDeltaMask().Merge(mask);
    }

    CKINV();
}

void Quake3DynamicEntity::CommitMigration(sid_t target)
{   
    // called before IsReplica is toggled
    if (IsReplica())
	SetClassParameters();

    // If this wasn't allocated yet (i.e., if I am replicated and change
    // status in the same frame), then during the next frame, I will be
    // allocated properly.
}

void Quake3DynamicEntity::PrintFields(ostream& out)
{
    Quake3Entity::PrintFields(out);
}

///////////////////////////////////////////////////////////////////////////////

static void LinkEntity(gentity_t *ent) {
    SV_LinkEntity(reinterpret_cast<sharedEntity_t *>(ent));
}

static void UnlinkEntity(gentity_t *ent) {
    SV_UnlinkEntity(reinterpret_cast<sharedEntity_t *>(ent));
}

void Quake3DynamicEntity::AllocateInternalEntity()
{
    ASSERT(mInternalEntity == NULL);

    gentity_t *internal_entity = NULL;
    gclient_t *internal_client = NULL;

    // First, allocate the internal entity and client structures
    
    if (mClientFields != NULL) {
	// we need to allocate a "real" gclient_t slot for this object
	// and let the server data structures know about it
	//
	// *** derived from: server/g_bot.c:G_AddBot
	//                   game/g_client.c:ClientConnect
	
	// For now we don't handle real player migrations -- for that
	// we need some sort of connection migration also.
	
	int clientNum = SV_BotAllocateClient();
	
	if (clientNum == -1) {
	    // XXX: Either handle this more gracefully, or make sure this
	    // never happens!
	    WARN << "quake3 did not have a free client slot "
		 << "for a new replica or migrated client object!" << endl;
	    ASSERT(0);
	}
	
	internal_entity = &g_entities[ clientNum ];
	internal_client = level.clients + clientNum;
	
	ASSERTM(!internal_entity->inuse, 
		"internal entity for clientNum = %d is in use? isreplica=%d classname=%s userinfo=%s", clientNum, internal_entity->isReplica, 
	       (internal_entity->classname ? internal_entity->classname : "(null)"),
	       internal_client->pers.netname
	       );
	
	// we'll set the configstring in ApplyChangesToInternalEntity
	    
    } else {
	// Not a client entity, just allocate an ordinary gentity_t for it
	internal_entity = G_Spawn();
	
	if (internal_entity == NULL) {
	    WARN << "quake3 did not have any free entities "
		 << "to allocate for a new primary object!" << endl;
	    ASSERT(0);
	}
    }

    // Now we can tie them to this object and apply our initial changes
    Tie(internal_entity, internal_client);

    // changes to mEntityFields,mClientFields will be applied to
    // mInternalEntity afer every object has been allocated and linked
    // into the game world if necessary. don't apply them here because
    // pointer state may be inconsistent
}

void Quake3DynamicEntity::DeallocateInternalEntity()
{
    ASSERT(mInternalEntity != NULL);
    ASSERT(IsReplica()); // GameManager can not delete primaries

    UnlinkEntity(mInternalEntity);

    if (mClientFields) {
	// free clients entities this way
	ASSERT(mInternalEntity->client != NULL);
	int clientNum = mInternalEntity->client - level.clients;
	mInternalEntity->s.modelindex = 0;
	mInternalEntity->inuse = qfalse;
	mInternalEntity->classname = "disconnected";
	mInternalEntity->client->pers.connected = CON_DISCONNECTED;
	mInternalEntity->client->ps.persistant[PERS_TEAM] = TEAM_FREE;
	mInternalEntity->client->sess.sessionTeam = TEAM_FREE;
	mInternalEntity->colyseusObject = NULL;
	SV_BotFreeClient( clientNum );
	SV_SetConfigstring(CS_PLAYERS+clientNum, "");
    } else {
	// free all other entities this way
	G_FreeEntity(mInternalEntity);
    }

    mInternalEntity = NULL;
}

void Quake3DynamicEntity::ApplyChangesToInternalEntity()
{
    ASSERT(mInitialized);
    ASSERT(mInternalEntity != NULL);

    gclient_t *internal_client = mInternalEntity->client;
    if (internal_client != NULL) {
	ASSERT(mClientFields != NULL);
	memcpy(internal_client, mClientFields, sizeof(gclient_t));

	int clNum = mInternalEntity - g_entities;
	ASSERT(clNum >= 0 && clNum < MAX_CLIENTS);
	char old_configstring[MAX_STRING_CHARS];
	SV_GetConfigstring(CS_PLAYERS+clNum, 
			   old_configstring, 
			   MAX_STRING_CHARS);
	// check if configstring changed
	if (strcmp(old_configstring, mConfigString)) {
	    if (!IsReplica()) {
		WARN << "someone changed a primary's configstring! "
		     << this << " from=" << old_configstring
		     << " to=" << mConfigString << endl;
	    }

	    // XXX this causes a broadcast of the changes to all
	    // clients -- if the game world is large and we constantly
	    // get new replicas, this is very inefficient! at some
	    // point, refactor this to make it more efficient using
	    // caching, since these should change very rarely.
	    SV_SetConfigstring(CS_PLAYERS+clNum, mConfigString);
	}
    } else {
	ASSERT(mClientFields == NULL);
	ASSERT(mConfigString == NULL);
    }

    ASSERT(mEntityFields != NULL);
    ASSERT(mEntityFields->colyseusObject == static_cast<Quake3Entity *>(this));

    // some state needs to be fixed up for the local quake instance
    {
	qboolean prev_r_linked = mInternalEntity->r.linked;
	int      prev_linkcount = mInternalEntity->r.linkcount;
	int      curr_linkcount = mEntityFields->r.linkcount;
	
	memcpy(mInternalEntity, mEntityFields, sizeof(gentity_t));
	
	ASSERT(mInternalEntity->inuse);
	ASSERT(mInternalEntity->classname);
	
	mInternalEntity->s.number = mInternalEntity - g_entities;
	if (internal_client) {
	    int clNum = mInternalEntity - g_entities;
	    mInternalEntity->s.clientNum = clNum;
	    internal_client->ps.clientNum = clNum;
	}

	if (prev_r_linked != mInternalEntity->r.linked) {
	    // link/unlink if link state changed
	    mInternalEntity->r.linked = prev_r_linked;
	    if (!prev_r_linked)
		LinkEntity(mInternalEntity);
	    else
		UnlinkEntity(mInternalEntity);
	}
	else if (prev_linkcount != curr_linkcount) {
	    mInternalEntity->r.linkcount = curr_linkcount - 1;   // LinkEntity will bump it by 1
	    LinkEntity (mInternalEntity);
	}	    	    
    }

    mInternalEntity->client = internal_client;
    mInternalEntity->isReplica = (qboolean)IsReplica();

    // * apply changes to pointer fields
    vector<gentity_t **> ptrs = GetEntityPointers(mInternalEntity);
    for (uint32 i = 0; i < ptrs.size(); i++) {
	// if pointer is not resolved, just set it to NULL for now...
	if (mPointers[i].IsResolved())
	    *ptrs[i] = mPointers[i] == NULL ? NULL : mPointers[i]->GetInternalEntity();
	else
	    *ptrs[i] = NULL;
    }

    // * fix special fields like otherEntityNum(2)
    //
    // The pointers are now set up, put in the corresponding entity indices
    mEntityFields->s.otherEntityNum = mInternalEntity->s.otherEntityNum = GEntityToNumber (mInternalEntity->otherEntity);
    mEntityFields->s.otherEntityNum2 = mInternalEntity->s.otherEntityNum2 = GEntityToNumber (mInternalEntity->otherEntity2);
    
    CKINV();
}

// Mask bits refer to clusters now. In order to dump the actual fields, 
// we must another query. 
void Quake3DynamicEntity::RecordDeltaFields (list<string>& flist, bool isNew) 
{
    ASSERT (mInternalEntity);
    GUID g = GetGUID ();

    DeltaEncodingEntry e (isNew, IsReplica (), g,  mInternalEntity->classname);
    for (list<string>::iterator it = flist.begin (); it != flist.end (); ++it) {
	e.fields.push_back ((*it).c_str ()); // force a new copy?
    }

    LOG (DeltaEncodingLog, e);			    
}

void Quake3DynamicEntity::ApplyChangesFromInternalEntity()
{
    ASSERT(mInternalEntity != NULL);
    ASSERT(mEntityFields != NULL);
    
    // * apply changes from mInternalEntity and set dirty bits in mask

    list<string> dirtylist;
    list<string> *ptr = NULL;

    if (g_QuakePreferences.deltas) {
	dirtylist.clear ();
	ptr = &dirtylist;
    }
    
    pair<bool, bool> dirty = 
	g_DeltaEncoder->RecordChanges (mEntityFields, mInternalEntity, mClientFields, mInternalEntity->client, 
		GetDeltaMask (), ptr);
    
    if (dirty.first) {
	if (mInternalEntity->colyseusObject != static_cast<Quake3Entity *>(this)) {
	    WARN << "internal entity pointed to something else!" << endl;
	    WARN << "me=" << (void *)static_cast<Quake3Entity *>(this) << " " << static_cast<Quake3Entity *>(this) << endl;
	    WARN << "ptr=" << (void *)mInternalEntity->colyseusObject << " " << reinterpret_cast<Quake3Entity *>(mInternalEntity->colyseusObject) << endl;
	    ASSERT(0);
	}
	memcpy (mEntityFields, mInternalEntity, sizeof(gentity_t));
	SetDeltaMaskField (1);    // XXX not strictly necessary
    }

    if (mClientFields != NULL) {
	ASSERT (mInternalEntity->client);

	if (dirty.second) {
	    memcpy (mClientFields, mInternalEntity->client, sizeof(gclient_t));
	    SetDeltaMaskField (2);
	}

	ASSERT(mConfigString != NULL);
	int clNum = mInternalEntity - g_entities;
	ASSERT(clNum >= 0 && clNum < MAX_CLIENTS);
	char new_configstring[MAX_STRING_CHARS];
	SV_GetConfigstring(CS_PLAYERS+clNum, 
			   new_configstring, 
			   MAX_STRING_CHARS);
	// check if configstring changed
	if (strcmp(new_configstring, mConfigString)) {
	    strncpy(mConfigString, new_configstring, MAX_STRING_CHARS);
	    SetDeltaMaskField(3);
	}
    }

    if (g_QuakePreferences.deltas) // compute the delta-field distribution
	if (dirtylist.size () > 0)
	    RecordDeltaFields (dirtylist);

    // * fix special fields like otherEntity and otherEntity2
    //
    // If the pointers change, this change will then be reflected in 
    // GetEntityPointers() and the delta mask will be set appropriately
    //
    mEntityFields->otherEntity = mInternalEntity->otherEntity = NumberToGEntity (mInternalEntity->s.otherEntityNum);
    mEntityFields->otherEntity2 = mInternalEntity->otherEntity2 = NumberToGEntity (mInternalEntity->s.otherEntityNum2);
    
    // * fix the pointers that changed
    vector<gentity_t **> ptrs = GetEntityPointers(mInternalEntity);
    ASSERT(ptrs.size() == mPointers.size());
    for (uint32 i = 0; i < ptrs.size(); i++) {
	gRef<Quake3Entity>& ref = mPointers[i];
	
	Quake3Entity *target = NULL;
	if ( (*ptrs[i]) != NULL ) {
	    if ( ! (*ptrs[i])->inuse )
		// assume this means the target will become NULL
		target = NULL;
	    else {
		// all matching colyseus objs should be allocated at this point
		ASSERT( (*ptrs[i])->colyseusObject != NULL );
		target = reinterpret_cast<Quake3Entity *>((*ptrs[i])->colyseusObject);
	    }
	}
	
	// pointer changed if it was set/unset to NULL or if both were
	// non-NULL and was set to another object (check the GUID
	// rather than the pointer because it may not have been
	// resolved yet, in which case we can't dereference it)
	if ( target != NULL && ref.IsNull() ||
	     target == NULL && !ref.IsNull() ||
	     target != NULL && target->GetGUID() != ref.GetTarget() ) {
	    SetDeltaMaskField (4+i);
	    ref = target;
	}
    }

    CKINV();
}



///////////////////////////////////////////////////////////////////////////////

void Quake3DynamicEntity::CheckInvariants()
{
    if (mInitialized) {
	if (mInternalEntity) {
	    ASSERT(mEntityFields);
	    ASSERT(!mInternalEntity->client || mClientFields);
	}
	
	GUIDSet pset;
	for (uint32 i = 0; i<mPointers.size(); i++) {
	    if (IsDynamicGUID(mPointers[i].GetTarget())) {
		pset.insert(mPointers[i].GetTarget());
	    }
	}

	GObjectRefList tmp;
	GetAllRefs(&tmp);
	for (GObjectRefList::iterator it = tmp.begin(); it != tmp.end(); it++) {
	    GUID targ = (*it)->GetTarget();
	    ASSERTDO( targ == GUID_NONE || pset.find(targ) != pset.end(),
		    { WARN << "targ=" << targ << " pset.size=" << pset.size()
		      << " mPointers.size=" << mPointers.size() << endl; });
	}
    }
}

const char *Quake3DynamicEntity::GetClassName () 
{
    if (mEntityFields) 
	return mEntityFields->classname;
    return NULL;
}

Quake3DynamicEntity::Type Quake3DynamicEntity::GetClass()
{
    gentity_t *ent = mEntityFields;
    if ( ent->classname == NULL ) {
	return INVALID;
    }
    return GetClass(ent->classname);
}

float Quake3DynamicEntity::GetSpeed() {
    
    // *** derived from: game/bg_misc.c:BG_EvaluateTrajectory

    const trajectory_t *tr = &mEntityFields->s.pos;
    float		deltaTime;
    float		phase;

    switch( tr->trType ) {
    case TR_STATIONARY:
    case TR_INTERPOLATE: {
#if 0    
	float fps = 1000.0 / GameManager::GetInstance()->GetModule()->FrameInterval();
	// XXX Check that speed is actually once per frame!
	return mEntityFields->speed * fps;
#endif
	// speed is in units per second.  
	// level.time, trTime, trDuration, etc. is in `msecs'
	// given that one sees stuff in the code like `trTime = distance * 1000 / speed',
	// it implies that speed must in units/second

	return mEntityFields->speed;
    }
    case TR_LINEAR:
	return Vec3(tr->trDelta).Length();
    case TR_SINE:
	return Vec3(tr->trDelta).Length(); // mean speed...
	break;
    case TR_LINEAR_STOP:
	return Vec3(tr->trDelta).Length();
	break;
    case TR_GRAVITY: {
	vec3_t result;
	deltaTime = ( level.time - tr->trTime ) * 0.001;    // milliseconds to seconds			
	VectorMA( tr->trBase, 1.0, tr->trDelta, result );
	result[2] -= 0.5 * DEFAULT_GRAVITY * deltaTime * deltaTime;

	return (Vec3(result) - Vec3(tr->trBase)).Length();
    }
    default:
	WARN << "unknown trType: " << tr->trTime << endl;
	return 0;
    }
}

// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
