////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

#include <gameapi/GameManager.h>
#include <gameapi/GameStore.h>
#include <gameapi/GameObject.h>
#include <util/Benchmark.h>
#include "Module.h"
#include "Quake3DynamicEntity.h"
#include "Quake3StaticEntity.h"
#include "Options.h"
#include "q_exports.h"
#include <server/server.h>
#include "Quake3Metadata.h"
#include "Logs.h"
#include <om/GObjectInfo.h>

bool Quake3Module::m_Inited = false;

const uint32 Quake3Module::FrameInterval() { 
    return (const uint32)(1000 / sv_fps->integer);
}

const GUID& Quake3Module::AllocateStaticGUID()
{
    if (++m_LastStaticGUID.m_LocalOID == 0) {
	++m_LastStaticGUID.m_Port;
    }
    if (m_LastStaticGUID == GUID_NONE) {
	WARN << "Ran out of static guids to allocate!" << endl;
	ASSERT(0);
    }

    return m_LastStaticGUID;
}

bool Quake3Module::IsDynamicEntity(gentity_t *ent)
{
    if (ent->classname == NULL) return false;
    return Quake3DynamicEntity::GetClass(ent->classname) != 
	Quake3DynamicEntity::INVALID;
}

gRetCode Quake3Module::Init(GameManager *manager)
{
    ASSERT(!m_Inited);
    m_Inited = true;

    m_LastStaticGUID = GUID_NONE;

    // Disable cheats
    Cvar_Set( "sv_cheats", "0" );

    // Delta-encoding information log, etc.
    InitQuake3Logs ();

    // Run a couple frames to initialize the world
    for (int i = 0; i < INITIAL_GAME_FRAMES; i++) {
	RunGameFrame();
    }

    cerr << "* Initializing quake metadata " << endl;
    // Hopefully, dynamic strings will be loaded at this point
    InitializeQuake3Metadata ();

    // for each object in the internal g_entities array, initialize it
    // in the colyseus object store. we keep dynamic objects for which
    // are num % masterTotal == (masterIndex-1). 
    //
    // ASHWIN: we keep `multi-entity groups' which are linked together 
    // by `team' and `teamchain' together. disable replicating the 
    // `teamchain' attribute since hopefully only the think functions 
    // of one of these linked entities uses those attributes (and since 
    // they are all on one node, we should be fine.)- Ashwin [07/04/2006]
    //
    // assume no clients start out on the server yet.
    
    gentity_t *ent = &g_entities[0];
    for (int i=0 ; i<level.num_entities ; i++, ent++) {
	if ( !ent->inuse ) {
	    continue;
	}

	ASSERT( ent->client == NULL );

	GameObject *obj = NULL;

	if (IsDynamicEntity(ent)) {
	    // all entities with the same `team' attribute will get hashed to 
	    // the same value and thus will be situated on a single node.
	    int team_hash = 0;
	    if (ent->team) 
		team_hash = hash<char *> () (ent->team);
	    else
		team_hash = i;
	    
	    if (team_hash % g_QuakePreferences.masterTotal == (g_QuakePreferences.masterIndex - 1))
	    {
		if (team_hash != i) 
		    Com_DPrintf ("=======> Team (%s); ent (%d:%s) situated on masterIndex=%d\n", ent->team, i, ent->classname, g_QuakePreferences.masterIndex - 1);
		
		obj = new Quake3DynamicEntity(ent, manager);
		m_LastFrameObjects.insert(pair<GUID, gentity_t *>(obj->GetGUID(), ent));
	    } else {
		G_FreeEntity(ent);
		continue;
	    }
	} else {
	    obj = new Quake3StaticEntity(ent, AllocateStaticGUID());
	}

	manager->GetStore()->Add(obj);
    }

    // add initial bots
    for (int i = 0; i < g_QuakePreferences.nbots; i++) {
	char altname[64];
	char cmd[255];
	sprintf(altname, "[%s]Bot%d", 
		Manager::GetInstance()->GetSID().ToString(), i);
	sprintf(cmd, "addbot %s %d %s %d %s\n", 
		"sarge", 1, "''", 0, altname);
	Cbuf_AddText(cmd);
    }

    INFO << "Quake3Module initialized..." << endl;
    return GAME_OK;
}

// mostly copied from:
//
// * qcommon/common.c:Com_Frame
// * server/sv_main.c:SV_Frame
//
bool Quake3Module::RunGameFrame()
{

    ASSERT(com_sv_running->integer);
    ASSERT(com_dedicated->integer);

    // *** from Com_Frame

    // write config file if anything changed
    Com_WriteConfiguration(); 
    Com_EventLoop();
    Cbuf_Execute ();

    // *** from SV_Frame

    // the menu kills the server with this cvar
    if ( sv_killserver->integer ) {
	SV_Shutdown ("Server was killed.\n");
	Cvar_Set( "sv_killserver", "0" );
	return false;
    }

    // if it isn't time for the next frame, do nothing
    if ( sv_fps->integer < 1 ) {
	Cvar_Set( "sv_fps", "10" );
    }
    
    // XXX -- DO SOMETHING ABOUT THIS!!!  if time is about to hit the
    // 32nd bit, kick all clients and clear sv.time, rather than
    // checking for negative time wraparound everywhere.
    // 2giga-milliseconds = 23 days, so it won't be too often
    if ( svs.time > 0x70000000 ) {
	SV_Shutdown( "Restarting server due to time wrapping" );
	Cbuf_AddText( "vstr nextmap\n" );
	return false;
    }
    // this can happen considerably earlier when lots of clients play
    // and the map doesn't change
    if ( svs.nextSnapshotEntities >= 0x7FFFFFFE - svs.numSnapshotEntities ) {
	SV_Shutdown( "Restarting server due to numSnapshotEntities wrapping" );
	Cbuf_AddText( "vstr nextmap\n" );
	return false;
    }
    
    // update infostrings if anything has been changed
    if ( cvar_modifiedFlags & CVAR_SERVERINFO ) {
	SV_SetConfigstring( CS_SERVERINFO, Cvar_InfoString( CVAR_SERVERINFO ) );
	cvar_modifiedFlags &= ~CVAR_SERVERINFO;
    }
    if ( cvar_modifiedFlags & CVAR_SYSTEMINFO ) {
	SV_SetConfigstring( CS_SYSTEMINFO, Cvar_InfoString_Big( CVAR_SYSTEMINFO ) );
	cvar_modifiedFlags &= ~CVAR_SYSTEMINFO;
    }

    // update ping based on the all received frames
    SV_CalcPings();

    SV_BotFrame( svs.time );
    
    // we assume we only run 1 sim chunk at a time
    svs.time += FrameInterval();
    // let every primary in the world think and move
    VM_Call( gvm, GAME_RUN_FRAME, svs.time );

    return true;
}

void Quake3Module::_CheckSpecialInvariant (GameManager *manager) {
    GObjectInfoMap *oinfo = manager->GetManager()->GetObjectInfo ();
    ObjectStore *objStore = manager->GetObjectStore ();
    ObjectStore *penStore = manager->GetPendingStore ();

    penStore->Begin ();
    GObject *o;

    while ((o = penStore->Next())) {
	// ASSERT(guids.find (o->GetGUID ()) != guids.end ());
	ASSERT(penStore->Find(o->GetGUID ()) != NULL); // i dont trust Find!
	ASSERT(objStore->Find (o->GetGUID ()) == NULL);
	ASSERTDO (oinfo->find (o->GetGUID ()) != oinfo->end (),
		WARN << "guid=" << o->GetGUID() << " isreplica=" << o->IsReplica() << endl);
	// later_guids.insert (o->GetGUID ());
    }

}

gRetCode Quake3Module::RunFrame(GameManager *manager)
{
    DynamicGameObject *dgo;
    Quake3DynamicEntity *obj;
    GameStore *store = manager->GetStore();

    /* Catchup with stuff done by Colyseus after last frame finished */
    _CheckSpecialInvariant (manager);

#if 0
    // foreach (qobj in colyseus_objectstore, deleted since last frame) 
    //    remove qobj from g_entities;    /* this is quake's g_entities */
    for (EntityMapIter it = m_LastFrameObjects.begin();
	 it != m_LastFrameObjects.end(); it++) {
	if (store->Find(it->first) == NULL) {
	    gentity_t *ent = it->second;
	    ASSERT(ent != NULL);
	    ASSERT(ent->inuse);
	    Quake3Entity *q3e = 
		reinterpret_cast<Quake3Entity *>(ent->colyseusObject);
	    ASSERT(q3e != NULL);
	    ASSERT(q3e->IsDynamic());
	    obj = dynamic_cast<Quake3DynamicEntity *>(q3e);
	    ASSERT(obj->IsDeleted());
	    ASSERT(obj->IsReplica());

	    ASSERT(obj->GetGUID() == it->first);
	    ASSERT(obj->IsDeleted());
	    
	    // XXX: handle any pending last updates for clients
	    // (multicast events, etc.) in the object here before
	    // deleting it forever.

	    obj->DeallocateInternalEntity();
	    delete obj;
	}
    }
#endif

    // to get the Terminal to work correctly, we have to lock the
    // Manager data structures when we iterate over the object store
    //
    // however, be careful to release the lock before mutating the
    // object store because the Manager will acquire the lock itself
    Manager::GetInstance()->Lock();

    // We must allocate all objects before applying changes to them so
    // that pointers have a reference in both the real and colyseus store

    // foreach (qobj in colyseus_objectstore) { 
    //    if (qobj not_in g_entities) 
    //       add qobj.fields to g_entities;
    // }
    store->BeginDynamic();
    while ((dgo = store->NextDynamic()) != NULL) {
	obj = dynamic_cast<Quake3DynamicEntity *>(dgo);
	ASSERT(obj != NULL);
	
	if (m_LastFrameObjects.find(obj->GetGUID()) == 
	    m_LastFrameObjects.end() || obj->GetInternalEntity() == NULL) {
	    // If this object was not present during the last frame,
	    // that means it is a new replica or a new primary
	    // migrated to us.  We have to tie it to a internal quake
	    // entity in g_entities
	    //
	    // It could also be the case that an object that was present
	    // during the last frame was deleted and reinstantiated again,
	    // in which case, we have to allocate an internal entity for
	    // it again.
	    obj->AllocateInternalEntity();
	}
    }

    // foreach (qobj in colyseus_objectstore) { 
    //    actual = find (qobj, g_edicts);
    //    actual := qobj;    /* qobj.ApplyChangesTo (actual); */
    store->BeginDynamic();
    while ((dgo = store->NextDynamic()) != NULL) {
	obj = dynamic_cast<Quake3DynamicEntity *>(dgo);
	ASSERT(obj != NULL);
	ASSERT(obj->GetInternalEntity() != NULL);

	obj->ApplyChangesToInternalEntity();
    }

    Manager::GetInstance()->Unlock();

    /* Call the actual game dll RunFrame() */
    bool ok = RunGameFrame();
    if (!ok) {
	WARN << "RunGameFrame failed" << endl;
	return GAME_ERROR;
    }

    /* Propagate changes done by Quake to Colyseus now */

    // foreach (qobj in colyseus_objectstore && 
    //          not_in g_edicts & is_primary(qobj)) 
    //    delete qobj;

    list<Quake3DynamicEntity *> deleted;

    Manager::GetInstance()->Lock();

    store->BeginDynamic();
    while ((dgo = store->NextDynamic()) != NULL) {
	obj = dynamic_cast<Quake3DynamicEntity *>(dgo);
	ASSERT(obj != NULL);
	
	gentity_t *ent = obj->GetInternalEntity();
	ASSERT(ent != NULL);

	if (ent->colyseusObject == NULL) {
	    // this was deleted by quake (using G_FreeEntity)!
	    if (!obj->IsReplica()) {
		deleted.push_back(obj);
	    } else {
		// XXX: we may need to change G_FreeEntity to ensure
		// this does not happen. We can not just reinstantiate
		// the object because we need to ensure that its
		// "effects" get sent over to the primary. We should
		// add a special "deleted" flag that the primary knows
		// how to process
		//
		// jeffpang: I haven't seen quake delete a replica yet
		// so we'll ignore this for now
		WARN << "quake deleted a replica object! "
		     << "it should not do that!" << endl;
		ASSERT(0);
	    }
	}
    }

    Manager::GetInstance()->Unlock();

    for (list<Quake3DynamicEntity *>::iterator it = deleted.begin();
	 it != deleted.end(); it++) {
	GameObject *ret = store->Remove((*it)->GetGUID());
	ASSERT(ret != NULL);
	delete (*it);
    }

    // first construct all matching colyseus objects before applying
    // changes to ensure that all pointers have references in both

    // foreach (actual in g_edicts) {
    //    qobj = find(actual, colyseus_objectstore);
    //    if (qobj == null) 
    //       add actual to colyseus_objectstore;
    // }

    gentity_t *ent = &g_entities[0];
    for (int i = 0 ; i < level.num_entities ; i++, ent++) {
	if (!ent->inuse)
	    continue;

	if (IsDynamicEntity(ent)) {
	    if (ent->colyseusObject == NULL) {
		obj = new Quake3DynamicEntity(ent, manager);
		store->Add(obj);
	    }
	} else {
	    // static objects should always remain allocated and should not
	    // appear new during the game...
	    if (ent->colyseusObject == NULL) {
		WARN << "game constructed a new static object: " 
		     << ent->classname << endl;
		GameObject *go = 
		    new Quake3StaticEntity(ent, AllocateStaticGUID());
		store->Add(go);
	    }
	}
    }

    Manager::GetInstance()->Lock();

    // foreach (actual in colyseus_objectstore) {
    //    qobj := actual;     /* qobj.IntegrateAppChanges (actual); */
    store->BeginDynamic();
    while ((dgo = store->NextDynamic()) != NULL) {
	obj = dynamic_cast<Quake3DynamicEntity *>(dgo);
	obj->ApplyChangesFromInternalEntity();
    }

    // remember which objects exist at the end of this frame, so that if
    // colyseus deletes an object, we know about it in the next frame
    m_LastFrameObjects.clear();
    store->BeginDynamic();
    while ((dgo = store->NextDynamic()) != NULL) {
	obj = dynamic_cast<Quake3DynamicEntity *>(dgo);
	ASSERT(obj != NULL);
	ASSERT(obj->GetInternalEntity() != NULL);
	m_LastFrameObjects.insert(pair<GUID, gentity_t *>(obj->GetGUID(), obj->GetInternalEntity()));
    }

    Manager::GetInstance()->Unlock();

    /*
    TimeVal now;
    OS::GetCurrentTime(&now);
    PERIODIC2(5000, now, {
	cout << "GameStore: " << endl;
	GameObject *go;
	store->Begin();
	while ((go = store->Next()) != NULL) {
	    cout << go << endl;
	}
    });
    */

    return GAME_OK;
}

gRetCode Quake3Module::Shutdown(GameManager *manager)
{
    SV_Shutdown( "GameManager requested shutdown" );
    Sys_ConsoleInputShutdown();
    return GAME_OK;
}

DynamicGameObject *Quake3Module::ConstructObject(gTypeCode type,
						 GObjectInfoIface *info)
{
    ASSERT(type == GAMETYPE_QUAKE3_DYNAMIC);
    return new Quake3DynamicEntity(info);
}

GameObject *Quake3Module::ConstructObject(Packet *pkt)
{
    // unused
    ASSERT(0);
    return NULL;
}

void Quake3Module::DeleteObject(DynamicGameObject *dgo)
{
    Quake3DynamicEntity *obj = dynamic_cast<Quake3DynamicEntity *>(dgo);
    ASSERT(obj != NULL);

    // UPDATE: free the object here because otherwise it can result 
    // in race conditions when an object is deleted _and_ re-created 
    // in the same frame (by Colyseus); once during Maintenance, and 
    // later during 'Recv'... - Ashwin [10/24/2006]
    //
    // don't actually free the memory here; keep it around so that in
    // the next frame we can see that it was deleted by the manager
    // and perform any necessary final game actions on it before
    // actually garbage collecting it. The GameManager should have
    // removed it from the GameStore already, signalling it is
    // deleted.
    ASSERT(!obj->IsDeleted());
    obj->Delete();

    if (obj->GetInternalEntity() == NULL) {
	DBG << "deleting object not tied to a gentity_t: " << obj << endl;
	// never was tied to a internal gentity_t, so free it now
	delete obj;
    } else {
	ASSERT(obj->GetInternalEntity()->colyseusObject == static_cast<Quake3Entity *>(obj));

	gentity_t *ent = obj->GetInternalEntity ();
	ASSERT(ent != NULL);
	ASSERT(ent->inuse);
	ASSERT(obj->IsReplica());

	obj->DeallocateInternalEntity ();
	delete obj;
    }
}

///////////////////////////////////////////////////////////////////////////////

static void handleVersion(ostream& out, char **args);
static void handleGEdicts(ostream& out, char **args);
static void handleQuakeCommand(ostream& out, char **args);

/// XXX Gruesome hack for the terminal handlers here. I should 
//  really use 'callback' and 'wrap'

static Quake3Module *GetQ3Module () 
{
    GameModule *module = GameManager::GetInstance ()->GetModule ();
    return dynamic_cast<Quake3Module *> (module);
}
#define s_Module  (GetQ3Module())
#define s_Manager (GameManager::GetInstance())

static InputHandler g_QuakeHandlers[] = {
    { "QVER", "QV", handleVersion, "",
      "Show the version of quake3" },
    { "GEDICTS", "GE",  handleGEdicts,   "[{P|R|A} [{P|M|I}]]",
      "Show distributed entities in g_entities" },
    { "QCMD", "QC", handleQuakeCommand, "quake_console_command...",
      "Execute a quake console command on this server" },
    { NULL, NULL, NULL, NULL }
};

void Quake3Module::InstallTerminalHandlers (InputHandlers *tofill) 
{
    InputHandler *ih = g_QuakeHandlers;
    while (ih->cmd) {
        tofill->push_back(ih);
        ++ih;
    }
}

extern const char g_Quake3Version [];
extern const char g_Quake3BuildTime [];

void handleVersion(ostream& out, char **args) {
    out << "quake3 " << g_Quake3Version 
		<< " (" << g_Quake3BuildTime << ")" << "\n";
    return;
}

void handleGEdicts(ostream& out, char **args)
{
    bool showReplicas = true;
    bool showPrimaries = true;
    bool showPlayers = true;
    bool showMissiles = true;
    bool showItems   = true;
    int i;
    gentity_t *ent;

    if (args[1]) {
	if (!strcasecmp(args[1], "P")) {
	    showReplicas = false;
	} else if (!strcasecmp(args[1], "R")) {
	    showPrimaries = false;
	} else if (!strcasecmp(args[1], "A")) {
	    // already true
	} else {
	    out << "ERROR: unknown filter argument: " << args[1] << endl;
	    return;
	}

	if (args[2]) {
	    if (!strcasecmp(args[2], "P")) {
		showMissiles = false;
		showItems = false;
	    } else if (!strcasecmp(args[2], "M")) {
		showPlayers = false;
		showItems = false;
	    } else if (!strcasecmp(args[2], "I")) {
		showPlayers = false;
		showMissiles = false;
	    } else {
		out << "ERROR: unknown filter argument: " << args[2] << endl;
		return;
	    }
	}
    }

    GameStore *store = s_Manager->GetStore ();
    DynamicGameObject *dgo = NULL;

    store->BeginDynamic ();

    int rp = 0, rm = 0, ro = 0;
    int pp = 0, pm = 0, po = 0;

    while ((dgo = store->NextDynamic ()) != NULL) {
	
	Quake3DynamicEntity *q3e = dynamic_cast<Quake3DynamicEntity *> (dgo);
	Quake3DynamicEntity::Type t = q3e->GetClass ();
	
	bool isreplica = dgo->IsReplica ();
	
	switch (t) { 
	case Quake3DynamicEntity::INVALID:
	    continue;
	case Quake3DynamicEntity::PLAYER:
	    if (isreplica) rp++;
	    else pp++;

	    if (!showPlayers) 
		continue;
	    break;
	case Quake3DynamicEntity::MISSILE:
	    if (isreplica) rm++;
	    else pm++;

	    if (!showMissiles) 
		continue;
	    break;
	default:
	    if (isreplica) ro++;
	    else po++;
	    
	    if (!showItems) 
		continue;
	    break;
	}

	if (isreplica) {
	    if (!showReplicas)
		continue;
	}
	else {
	    if (!showPrimaries)
		continue;
	}

	ent = q3e->GetInternalEntity ();
	if (!ent)
	    continue;
	Vec3 origin = q3e->GetOrigin ();

	out << "[edict:"
	    << " guid=" << q3e->GetGUID()
	    << " class=" << (ent->classname ? ent->classname : "null")
	    << " origin=("
	    << origin[0] << ","
	    << origin[1] << ","
	    << origin[2] << ")";

	if (ent->client != NULL) {
	    out << " name=" << ent->client->pers.netname
		<< " score=" << ent->client->ps.persistant [PERS_SCORE]
		<< " health=" << ent->health;
	}
	out << "]" << endl;
    }
    
    int maxplayers = Cvar_VariableIntegerValue ("sv_maxclients");

    out << "maxplayers=" << maxplayers  << " "
	<< "numreplicas=" << rp << " " << rm << " " << ro << " "
	<< "numprimaries=" << pp << " " << pm << " " << po << " "
	<< endl;
}

void handleQuakeCommand(ostream& out, char **args)
{
    if (!args[1]) {
	out << "ERROR: expected quake command" << endl;
	return;
    }

    args++;

    char cmd[512];
    char *ptr = cmd;
    int len = 0;
    while (args[0] != NULL) {
	int thislen = strlen(args[0]);
	if (thislen >= 512-len-1) {
	    out << "ERROR: command too long" << endl;
	    return;
	}

	strcpy(ptr, args[0]);
	ptr += thislen;
	len += thislen;
	strcpy(ptr, " ");
	ptr += 1;
	len += 1;

	args++;
    }

    Cbuf_Init();
    Cbuf_AddText(cmd);
    Cbuf_Execute();

    out << "OK: " << cmd << endl;
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
