////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

using namespace std;

#include <iostream>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <gameapi/common.h>
#include <gameapi/GameManager.h>
#include <gameapi/GameDatabase.h>
#include <gameapi/GameStore.h>

#include "q_exports.h"
#include "InterestFactory.h"
#include "World.h"
#include "Module.h"
#include "ClientHandler.h"
#include "Options.h"

static uint32 g_RandSeed;
static bool   g_DisableColyseus;
static uint32 g_IdleTime;

static OptionType g_Options[] =
{
    { '#', "General params for colyquake3", OPT_SEP, "", NULL, "", NULL },
    { '/', "seed", OPT_INT, 
	"random seed", 
	&g_RandSeed, "42", NULL },
    { '/', "no-colyseus", OPT_BOOL|OPT_NOARG, 
	"disable colyseus (run game locally only)", 
	&g_DisableColyseus, "0", (void *)"1" },

    { '/', "bbox-file", OPT_STR, 
	"file to input precomputed bboxes", 
	g_QuakePreferences.bbox_file, "", NULL },
    { '/', "map", OPT_STR,
	"name of map (without .bsp)",
	g_QuakePreferences.mapname, "q3dm1", NULL },
    { '/', "bbox-xystep", OPT_FLT, 
	"the number of degrees to space each ray in xy", 
	&g_QuakePreferences.bbox_xystep, "10", NULL },
    { '/', "bbox-zstep", OPT_FLT, 
	"the number of degrees to space each ray in z", 
	&g_QuakePreferences.bbox_zstep, "10", NULL },

    { '/', "master-index", OPT_INT, 
	"is a master server i (0 if not master); keeps initial entities", 
	&g_QuakePreferences.masterIndex, "0", NULL },
    { '/', "master-servers", OPT_INT, 
	"number of master servers (initial entities split between them)", 
	&g_QuakePreferences.masterTotal, "1", NULL },

    { '/', "nbots", OPT_INT, 
	"number of bots to start on this server", 
	&g_QuakePreferences.nbots, "4", NULL },

    { '#', "Prediction and timers", OPT_SEP, "", NULL, "", NULL },
    { '/', "maxdislat", OPT_INT, 
	"max discovery latency for prediction fudge (msec)", 
	&g_QuakePreferences.maxdislat, "300", NULL },
    { '/', "pub-interval-player", OPT_INT, 
	"how often player objects should publish (msec)", 
	&g_QuakePreferences.pubttl_player, "1000", NULL },
    { '/', "pub-interval-missle", OPT_INT, 
	"how often missile objects should publish (msec)", 
	&g_QuakePreferences.pubttl_missile, "500", NULL },
    { '/', "pub-interval-immobile-longlive", OPT_INT, 
	"how often other immobile long lived objects should publish (msec)", 
	&g_QuakePreferences.pubttl_immobile_longlive, "30000", NULL },
    { '/', "pub-interval-immobile-shortlive", OPT_INT, 
	"how often other immobile short lived objects should publish (msec)", 
	&g_QuakePreferences.pubttl_immobile_shortlive, "500", NULL },
    { '/', "pub-interval-mobile-longlive", OPT_INT, 
	"how often other mobile long lived objects should publish (msec)", 
	&g_QuakePreferences.pubttl_mobile_longlive, "1000", NULL },
    { '/', "pub-interval-mobile-shortlive", OPT_INT, 
	"how often other mobile short lived objects should publish (msec)", 
	&g_QuakePreferences.pubttl_mobile_shortlive, "500", NULL },
    { '/', "sub-prediction-player", OPT_INT, 
	"how many msec to predict player subscriptions", 
	&g_QuakePreferences.subttl_player, "3000", NULL },
    { '/', "sub-prediction-missile", OPT_INT, 
	"how many msec to predict missile subscriptions", 
	&g_QuakePreferences.subttl_missile, "500", NULL },

    { '#', "Delta encoding", OPT_SEP, "", NULL, "", NULL },
    { '/', "deltas", OPT_NOARG | OPT_BOOL, "compute delta field distribution", 
	&(g_QuakePreferences.deltas),  "0", (void *)"1" },
    { '/', "no-clusters", OPT_NOARG | OPT_BOOL, "disable clustering",  
	&(g_QuakePreferences.noclusters),  "0", (void *)"1" },

    { '#', "Miscellaneous", OPT_SEP, "", NULL, "", NULL },
    { '/', "idletime", OPT_INT,
	"time to idle after hitting timelimit",
	&g_IdleTime, "120", NULL },
    { '/', "randomspawns", OPT_NOARG | OPT_BOOL,
	"make bots spawn at random locations",
	&g_QuakePreferences.randomspawns, "0", (void *) "1" },

    { 0, 0, 0, 0, 0, 0, 0 }
};

int main(int argc, char *argv[])
{
    Quake3Module *g;
    Quake3World  *w;
    Quake3ClientHandler *c;
    Quake3InterestFactory *f;

    GameManager *m = GameManager::GetInstance(&argc, argv, g_Options);

    srand(g_RandSeed);
    srand48(g_RandSeed);

    // Get quake to initialize itself normally
    Sys_Main(argc, argv);

    // ttyconsole doesn't work with nodelay disabled
    Cvar_Set( "ttycon", "0" );

    // make games last forever
    Cvar_Set ("fraglimit", "0");
    Cvar_Set ("timelimit", "0");

    // Load the quake map and startup the server
    // (constructing the world may require this step)
    SV_SpawnServer( g_QuakePreferences.mapname, qtrue );

    if (!strcmp(g_QuakePreferences.bbox_file, "")) {
	WARN << "no bbox-file specificied, loading default world" << endl;
	w = new Quake3World();
    } else {
	w = new Quake3World(g_QuakePreferences.bbox_file);
    }

    g = new Quake3Module();
    c = new Quake3ClientHandler();
    f = new Quake3InterestFactory();

    DBG_INIT(&g_LocalSID);

    cerr << "* Initializing the quake3 module " << endl;    
    m->Init(g, w, c, f, NULL, g_DisableColyseus);
    cerr << "* done! " << endl;

    GameStore *s = m->GetStore();

    cerr << "* Starting game frames " << endl;
    m->Run();

    cerr << "* idling for " << g_IdleTime << " seconds " << endl;
    m->Idle(g_IdleTime * 1000);
    cerr << "exiting gracefully" << endl;
    return 0;
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
