////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////
#ifndef __DG_BBOX__H
#define __DG_BBOX__H

#include <gameapi/common.h>
#include "q_exports.h"

///////////////////////////////////////////////////////////////////////////////
//
// Options for DG_ComputeVisibilityBox
//

#define XY_STEP 20 // degrees (default value)
#define  Z_STEP 90 // degrees (default value)

// how many degrees to "jitter" each ray. the intuition behind
// jittering the rays is that we will be more likely to eventually
// produce a sub to match some bounding box to capture all objects in
// the "real" box (since we stay interested in objects for longer than
// the lifetime of a sub, it doesn't matter than the subs die before
// new ones are generated)
//
// currently the jitter is a uniform offset; we might want to sample
// from a gaussian disk perpendicular to the ray, but that may be too
// expensive
#define XY_JITTER 0 // degrees
#define  Z_JITTER 0 // degrees

// make sure the bounding box is at least this wide/high around the ent
#define MIN_BBOX_WIDTH  0
#define MIN_BBOX_HEIGHT 0

///////////////////////////////////////////////////////////////////////////////
//
// Options for bounding box precomputation
//

// magic string that we begin bbox files with
#define MAGIC_STRING "q3_bbox_map"
// define if we output/input bbox file in binary
#define COMPACT_FORMAT
// allow for floating point correction
#define EPSILON 1e-6

ostream& operator<<(ostream& os, vec3_t v);

///////////////////////////////////////////////////////////////////////////////

// Use shared memory for loading the bboxes. This way, multiple
// virtual nodes on the same machine can use the same memory -- this
// is only there for experiments.  And, is quite hacky.
//
// Since the "flags" byte in the bbox_t structure can be mutated, we
// will use a separate per-process array for this (only if we are
// using shared memory).
// 
// #define USE_SHARED_MEMORY

typedef int ivec3_t[3];

typedef struct {
#ifndef USE_SHARED_MEMORY
	byte   flags;
#endif
	vec3_t min, max;
} bbox_t;

ostream& operator<<(ostream& os, bbox_t& v);

///////////////////////////////////////////////////////////////////////////////

#define RADIAN(x)     (M_PI * (float) (x) / (float) 180.0)
#define DEGREE(x)     (180.0 * (float) (x) / M_PI)

///////////////////////////////////////////////////////////////////////////////
//
// These are legacy definitions from quake2. expose BBox objects to gameapi
// and do not use bbox_t structs outside of legacy code.
//

bool BBoxContains(bbox_t *box1, bbox_t *box2);
bool BBoxOverlaps(bbox_t *box, vec3_t min, vec3_t max);
inline bool BBoxOverlaps(bbox_t *bbox1, bbox_t *bbox2) {
	return BBoxOverlaps(bbox1, bbox2->min, bbox2->max);
}
float Overlap(bbox_t *box1, bbox_t *box2, int dim);
void  Overlap(bbox_t *out, bbox_t *box1, bbox_t *box2);
float BBoxVolume(bbox_t *bbox);
float BBoxVolSubtract(bbox_t *bbox1, bbox_t *bbox2);

float BBoxVolume(vec3_t min, vec3_t max);
float BBoxVolSubtract(vec3_t min1, vec3_t max1, vec3_t min2, vec3_t max2);
float Distance(vec3_t from, vec3_t to);

void PointToBBox(bbox_t *bbox, vec3_t pt);
inline bool BBoxContains(bbox_t *bbox, vec3_t pt) {
    for (int i=0; i<3; i++) {
	if (bbox->min[i] > pt[i] || bbox->max[i] < pt[i])
	    return false;
    }
    return true;
}

float GetDiagonalLength(vec3_t min, vec3_t max);

///////////////////////////////////////////////////////////////////////////////
typedef enum {
    BBOX_INVALID,    
    BBOX_NORMAL,
    BBOX_POINTSIZE,
    BBOX_MAXWIDTH
} BBoxType;


class BBoxer {
    vec3_t   m_Worldmin, m_Worldmax;
    float    m_XYStep, m_ZStep;
    vector<float> m_Thetas, m_Phis;
    float    m_MaxRayLength;

public:
    BBoxer (vec3_t min, vec3_t max, float xy_step = XY_STEP, float z_step = Z_STEP);

    /** 
     * Compute the bounding box direction by ray tracing from the origin 
     */
    BBoxType ComputeVisibilityBox (vec3_t org, vec3_t bbox_min, vec3_t bbox_max);

private:
    void ShootRay (float r_theta, float r_phi, vec3_t org, trace_t *tr);
};

#endif

// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
