////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include "q_exports.h"
#include "BBox.h"
#include "Options.h"
#ifndef BENCHMARK_REQUIRED
#define BENCHMARK_REQUIRED
#endif
#include <util/Benchmark.h>
#include <om/Manager.h>

///////////////////////////////////////////////////////////////////////////////

ostream& operator<<(ostream& os, vec3_t v)
{
    os << merc_va("<%.2f %.2f %.2f>", v[0], v[1], v[2]);
    return os;
}

ostream& operator<<(ostream& os, ivec3_t v)
{
    os << merc_va("<%d %d %d>", v[0], v[1], v[2]);
    return os;
}

ostream& operator<<(ostream& os, bbox_t& v)
{
    os << "[" << v.min << " " << v.max << "]";
    return os;
}

///////////////////////////////////////////////////////////////////////////////

bool BBoxContains(bbox_t *box1, bbox_t *box2)
{
    for (int i=0; i<3; i++) {
	if (box1->min[i] > box2->min[i] || box1->max[i] < box2->max[i]) {
	    return false;
	}
    }

    return true;
}

static bool BBoxOverlapsHelper(vec3_t min1, vec3_t max1,
	vec3_t min2, vec3_t max2)
{
    for (int i=0; i<3; i++) {
	if (!( min1[i] <= min2[i] && min2[i] <= max1[i] || 
		    min1[i] <= max2[i] && max2[i] <= max1[i] )) {
	    return false;
	}
    }

    return true;
}

bool BBoxOverlaps(bbox_t *box, vec3_t min, vec3_t max)
{
    return BBoxOverlapsHelper(box->min, box->max, min, max) ||
	BBoxOverlapsHelper(min, max, box->min, box->max);
}

static float OverlapHelper(bbox_t *box1, bbox_t *box2, int i)
{
    if (box1->min[i] <= box2->min[i] && 
	    box2->min[i] <= box1->max[i]) {
	return MIN(box1->max[i], box2->max[i]) - box2->min[i];
    } else if (box1->min[i] <= box2->max[i] && 
	    box2->max[i] <= box1->max[i]) {
	return box2->max[i] - MAX(box1->min[i], box2->min[i]);
    }
    return 0;
}

float Overlap(bbox_t *box1, bbox_t *box2, int dim)
{
    float ret;

    ret = OverlapHelper(box1, box2, dim);
    if (ret > 0) return ret;
    ret = OverlapHelper(box2, box1, dim);
    return ret;
}

void Overlap(bbox_t *out, bbox_t *box1, bbox_t *box2)
{
    // This function already assumes the boxes overlap!

    for (int i=0; i<3; i++) {
	// The overlapping min point is larger of the two minimums
	if (box1->min[i] > box2->min[i]) {
	    out->min[i] = box1->min[i];
	} else {
	    out->min[i] = box2->min[i];
	}

	// The overlapping max point is the smaller of the two maximums
	if (box1->max[i] < box2->max[i]) {
	    out->max[i] = box1->max[i];
	} else {
	    out->max[i] = box2->max[i];
	}
    }

}

float BBoxVolume(bbox_t *bbox)
{
    return BBoxVolume(bbox->min, bbox->max);
}

float BBoxVolSubtract(bbox_t *bbox1, bbox_t *bbox2)
{
    return BBoxVolSubtract(bbox1->min, bbox1->max, bbox2->min, bbox2->max);
}

//
// Volume of a bbox
//
float BBoxVolume(vec3_t min, vec3_t max)
{
    return (max[0] - min[0])*(max[1] - min[1])*(max[2] - min[2]);
}

//
// Get the volume of (bbox1 - bbox2). That is, the volume of that stuff that
// is in bbox1 but not in bbox2 (hence the min is 0, if bbox1 == bbox2).
//
float BBoxVolSubtract(vec3_t min1, vec3_t max1, vec3_t min2, vec3_t max2)
{
    float bbox1_vol = BBoxVolume(min1, max1);

    vec3_t diffmin, diffmax;

    for (int i=0; i<3; i++) {
	diffmin[i] = MAX(min1[i], min2[i]);
	diffmax[i] = MIN(max1[i], max2[i]);

	if (diffmin[i] >= diffmax[i]) {
	    // no intersection of bboxes, difference is total
	    return bbox1_vol;
	}
    }

    return bbox1_vol - BBoxVolume(diffmin, diffmax);
}

//
// Distance between two points
//
/*
   float Distance(vec3_t from, vec3_t to)
   {
   float dx = from[0]-to[0];
   float dy = from[1]-to[1];
   float dz = from[2]-to[2];

   return sqrt(dx*dx+dy*dy+dz*dz);
   }
 */

void PointToBBox(bbox_t *bbox, vec3_t pt) {
    VectorCopy(pt, bbox->min);
    VectorCopy(pt, bbox->max);
}

float GetDiagonalLength (vec3_t min, vec3_t max) 
{
    double sq = 0.0;
    vec3_t diff;
    for (int i = 0; i < 3; i++)
	diff [i] = max [i] - min [i];

    for (int i = 0; i < 3; i++) 
	sq += (diff [i] * diff [i]);
    
    return sqrt (sq);
}

///////////////////////////////////////////////////////////////////////////////

BBoxer::BBoxer (vec3_t min, vec3_t max, float xy_step, float z_step) : 
    m_XYStep (xy_step), m_ZStep (z_step)
{
    for (int i = 0; i < 3; i++) {
	m_Worldmin [i] = min [i];
	m_Worldmax [i] = max [i];
    }

    m_MaxRayLength = GetDiagonalLength (m_Worldmin, m_Worldmax);
    INFO << "max_ray_len=" << m_MaxRayLength << endl;

    float theta, r_theta;
    float phi, r_phi;

    for (theta = 0; theta < 360; theta += xy_step) {
	r_theta = RADIAN(theta) + XY_JITTER - 2 * XY_JITTER * drand48();
	m_Thetas.push_back (r_theta);
    }
    for (phi = -90; phi <= 90; phi += z_step) {
	r_phi = RADIAN(phi) + Z_JITTER - 2 * Z_JITTER * drand48();
	m_Phis.push_back (r_phi);
    }

}

void BBoxer::ShootRay (float r_theta, float r_phi, vec3_t org, trace_t *tr) 
{
    vec3_t start, end, direction;

    direction[0] = cos(r_phi) * cos(r_theta);
    direction[1] = cos(r_phi) * sin(r_theta);
    direction[2] = sin(r_phi);

    VectorMA(org, 2, direction, start);
    VectorMA(org, m_MaxRayLength, direction, end);

    SV_Trace (tr, start, vec3_origin, vec3_origin, end,
	    0, MASK_OPAQUE, qfalse);
    //*tr = gi.trace( start, vec3_origin, vec3_origin, 
    //			   end, 0, MASK_OPAQUE );
}

#define TOTAL_RANDOM_RAYS 30

//
// Compute the bounding box for a point on the fly using ray-tracing
//
// org     - the origin point
// min     - the output min-corner of the bbox
// max     - the output max-corner of the bbox
//
BBoxType BBoxer::ComputeVisibilityBox (vec3_t org, vec3_t min, vec3_t max)
{
    trace_t tr;
    vec3_t dist;
    float r_phi, r_theta;

    // minimum bbox

    min[0] = org[0] - MIN_BBOX_WIDTH/2;
    min[1] = org[1] - MIN_BBOX_WIDTH/2;
    min[2] = org[2] - MIN_BBOX_HEIGHT/2;

    max[0] = org[0] + MIN_BBOX_WIDTH/2;
    max[1] = org[1] + MIN_BBOX_WIDTH/2;
    max[2] = org[2] + MIN_BBOX_HEIGHT/2;

    permute (m_Thetas);
    permute (m_Phis);

    int done = 0;
    int large = 0;

    // Go through thetas and phis in some random order. 
    for (int i = 0, len = m_Thetas.size (); i < len; i++) {
	r_theta = m_Thetas [i];
	for (int j = 0, len2 = m_Phis.size (); j < len2; j++) { 
	    r_phi = m_Phis [j];

	    ShootRay (r_theta, r_phi, org, &tr);

	    if ( tr.startsolid || tr.fraction >= 1.0 ) {
		// the origin was in a solid! that means bbox is null
		// so we can just return with the default bbox
		// OR
		// didn't hit anything! that means we are outside the
		// "game" area and can't possibly be here!
		//INFO << "here: min=" << min << " max=" << max << endl;
		VectorCopy(org, min);
		VectorCopy(org, max);
		return BBOX_POINTSIZE;
	    }

	    done += 1;

	    // check if the trace endpoint is too far away in any coordinate
	    if (g_QuakePreferences.open_map) {
		float max_coord_dist = -1;
		for (int k = 0; k < 3; k++) { 
		    float diff = fabs (tr.endpos [k] - org [k]);
		    if (diff > max_coord_dist)
			max_coord_dist = diff;
		}
		if (max_coord_dist > (g_QuakePreferences.max_bbox_width / 2)) {
		    large += 1;
		    // cerr << "Large dist: org=" << org << " tr.endpos=" << tr.endpos << endl;
		}


		/* if (done == TOTAL_RANDOM_RAYS)  */
		
		{ 
		    // if (((float) large / (float) done) > g_QuakePreferences.open_bbox_detection_threshold) 

		    /* forget about a threshold, even if 1 ray is long, we know the box is going
		     * to be very big! making it 2 just because i feel queasy with 1 :P */

		    if (large >= 2) 
		    {
			
			// return default max-sized box

			for (int x = 0; x < 2; x++) {
			    min [x] = org [x] - g_QuakePreferences.max_bbox_width / 2;
			    if (min [x] < m_Worldmin [x])
				min [x] = m_Worldmin [x];
			    max [x] = org [x] + g_QuakePreferences.max_bbox_width / 2;
			    if (max [x] > m_Worldmax [x])
				max [x] = m_Worldmax [x];
			}
			min [2] = org [2] - g_QuakePreferences.max_bbox_height / 2;
			if (min [2] < m_Worldmin [2])
			    min [2] = m_Worldmin [2];
			max [2] = org [2] + g_QuakePreferences.max_bbox_height / 2;
			if (max [2] > m_Worldmax [2])
			    max [2] = m_Worldmax [2];

			// cerr << "#large=" << large << " #done=" << done << " org=" << org << endl;
			return BBOX_MAXWIDTH;
		    }
		}
	    }

	    if ( tr.endpos[0] < min[0] ) min[0] = tr.endpos[0];
	    if ( tr.endpos[0] > max[0] ) max[0] = tr.endpos[0];
	    if ( tr.endpos[1] < min[1] ) min[1] = tr.endpos[1];
	    if ( tr.endpos[1] > max[1] ) max[1] = tr.endpos[1];
	    if ( tr.endpos[2] < min[2] ) min[2] = tr.endpos[2];
	    if ( tr.endpos[2] > max[2] ) max[2] = tr.endpos[2];

	    DB_DO(3) {
		vec3_t dist;
		float  mag_dist;

		VectorSubtract( tr.endpos, org, dist );
		mag_dist = sqrt(DotProduct(dist, dist));

		cerr << merc_va("phi: %.3f, theta: %.3f, dist: %.3f", DEGREE(r_phi), DEGREE(r_theta), mag_dist) << endl;
	    }

	}
    }

    return BBOX_NORMAL;
}

// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
