#!/usr/bin/perl
#
# Compute the average pub/sub lifetimes to estimate optimal TTL.
#
# Output is the median sub_ttl, pub_ttl
#
# -t        lifetimes for type {p,o,m,i}
# -v        verbose. print mean/median std dev.
# -d {p,s}  dump the {pub,sub} distribution.

use strict;
use Getopt::Std;
use Statistics::Descriptive;
use vars qw($opt_t $opt_v $opt_d);

getopts("t:vd:");

our %EXAMINE_TYPES;
if (defined $opt_t) {
    %EXAMINE_TYPES = map { $_ => 1 } split(/,/, $opt_t);
} else {
    %EXAMINE_TYPES = ( 'p' => 1,
		       'm' => 0,
		       'i' => 0 );
}

my $file;
if ($ARGV[0] =~ /\.gz$/) {
    $file = open(F, "gunzip -c $ARGV[0] |");
} else {
    $file = open(F, "<$ARGV[0]");
}

my $pub = new Statistics::Descriptive::Full();
my $sub = new Statistics::Descriptive::Full();

while (<F>) {
    chomp $_;
    my ($guid, $type, $class, $lifetime) = split(/\t/, $_);

    if (!$EXAMINE_TYPES{$type}) {
	next;
    }

    if ($class eq 'p') {
	if ($opt_d eq 'p') {
	    print "$lifetime\n";
	} else {
	    $pub->add_data( $lifetime );
	}
    } else {
	if ($opt_d eq 's') {
	    print "$lifetime\n";
	} else {
	    $sub->add_data( $lifetime );
	}
    }
}

close(F);

if ($opt_d) {
    exit(0);
}

if ($opt_v) {
    print sprintf("      \t%s\t%s\n", "pub", "sub");
    print sprintf("mean  \t%.5f\t%.5f\n", $pub->mean(), $sub->mean());
    print sprintf("stddev\t%.5f\t%.5f\n", $pub->standard_deviation(), 
		  $sub->standard_deviation());
    print sprintf("median\t%.5f\t%.5f\n", $pub->median(), $sub->median());
} else {
    print sprintf("%.5f\t%.5f\n", $pub->median(), $sub->median());
}
