#!/usr/bin/env python
'''Quake visualizer; monitor object stores for various
distributed quake servers
'''
# $Id: viz.py 1665 2005-03-11 19:06:07Z jeffpang $
# 
# this needs pygtk and gtk >= 2.2 which should be there on most redhat 9
# and all fedora machines. afaik, python >= 2.2 should be fine, as well.

import sys
import gobject
import pygtk
pygtk.require('2.0')
import gtk
import getopt

import socket
from struct import unpack
import time

VISUALIZER_PORT = 60000

STATUS_PRIMARY = 0x0
STATUS_REPLICA = 0x1

g_status =  { STATUS_PRIMARY: 'primary', STATUS_REPLICA: 'replica' }

CLASSNAME_MISSILE = 0x1 << 1
CLASSNAME_BOT  = 0x1 << 2
CLASSNAME_ITEM = 0x1 << 3
CLASSNAME_OTHER = 0x1 << 4

g_classnames = {
    CLASSNAME_OTHER: ('other', "#cc55cc", 6),
    CLASSNAME_ITEM: ('item', "#00cccc", 6),
    CLASSNAME_BOT: ('bot', "#cc0000", 10),
    CLASSNAME_MISSILE: ('missile', "#0000cc", 4)
    }

sentinel = time.time()       # python does not like large float values arising from time.time() * 1000.0

def timenow():
    return int((time.time() - sentinel) * 1000.0)

# bounding box and image information about the maps
g_maps = {
    "map.jpg" : (-1408, 896, -256, 704),
    "bigmap.jpg" : (-4096, 4096, -4096, 4096),
    "empty.jpg" : (-100, 100, -100, 100)
    }

# default BBox
class BBox:    
    minx = -1408
    maxx = 896
    miny = -256
    maxy = 704

# this is main display widget which shows the map image and moving entities
# we do not rely on gtk double buffering -- create our own off-screen pixmap
# to draw to...
class Canvas(gtk.DrawingArea):
    def __init__(self, mapfile):
        gtk.DrawingArea.__init__(self)
        self.gc = None  # initialized in realize-event handler
        self.set_double_buffered(False)

        self.image = gtk.Image()
        self.image.set_from_file(mapfile)
        self.pixbuf = self.image.get_pixbuf()
        self.render_pixbuf = self.pixbuf.copy()

        # dont let the user resize to less than the map-image size.
        # this isn't what you'd always want to do though :(
        self.set_size_request(self.pixbuf.get_width(), self.pixbuf.get_height())
        
        self.connect('size-allocate', self.on_size_allocate)
        self.connect('expose-event',  self.on_expose_event)
        self.connect('realize',       self.on_realize)

        self.localents = []
        self.remoteents = []

        self.__allocate_colors()

    def getcolors(self):    # allow my parent widget to create a legend
        return self.color
    
    def __allocate_darker(self, cmap, color, factor):
        assert factor < 1.0
        
        red = int(color.red * factor)
        blue = int(color.blue * factor)
        green = int(color.green * factor)
        return cmap.alloc_color(red=red, blue=blue, green=green)
    
    def __allocate_colors(self):
        """Create colors for all the types of objects we are going to render"""
        
        cmap = self.get_colormap()
        self.yellow = cmap.alloc_color("#ffff00")
        
        self.color = {}
        
        for type, info in g_classnames.iteritems():
            self.color[type] = [0, 0]
            c = cmap.alloc_color(info[1])
            self.color[type][STATUS_PRIMARY] = c
            self.color[type][STATUS_REPLICA] = self.__allocate_darker(cmap, c, 0.6)

    def on_realize(self, widget):
        """create the GC and off-screen pixmap when the parent window is first created"""
        self.gc = widget.window.new_gc()
        self.pixmap = gtk.gdk.Pixmap(self.window, self.pixbuf.get_width(), self.pixbuf.get_height())

    def on_size_allocate(self, widget, allocation):
        self.width = allocation.width
        self.height = allocation.height
        self.render_pixbuf = self.pixbuf.scale_simple(self.width, self.height, gtk.gdk.INTERP_NEAREST)

        # if the window has not been realized yet, the pixmap will not be created
        if self.window != None:
            self.pixmap = gtk.gdk.Pixmap(self.window, self.width, self.height)

    def draw_entity(self, drawable, ent, color=None):
        status = STATUS_PRIMARY
        if ent['mask'] & STATUS_REPLICA:
            status = STATUS_REPLICA

        type = 0
        for t in g_classnames.iterkeys():
            if ent['mask'] & t:
                type = t
                break

        origcolor = color
            
        x, y = ent['x'], ent['y']
        x = self.width * ((x - BBox.minx) / (BBox.maxx - BBox.minx))
        y = self.height * ((y - BBox.miny) / (BBox.maxy - BBox.miny))

        if color == None:
            color = self.color[type][status]

        # Jeff: this is actually the diameter, not the radius
        radius = g_classnames[t][2]
        self.gc.set_foreground(color)

        #print "min=(" + str(BBox.minx) + "," + str(BBox.miny) + ") max=(" + str(BBox.maxx) + "," + str(BBox.maxy) + ")"
        #print "(" + str(x) + "," + str(y) + ")"
        
        drawable.draw_arc(self.gc, True, int(x-radius/2), int(y-radius/2), radius, radius, 0, 64 * 360)

        # draw bounding box for primaries
        if (origcolor == None) and (status == STATUS_PRIMARY) and (ent['mask'] & CLASSNAME_BOT or ent['mask'] & CLASSNAME_MISSILE):

            pminx, pminy, pmaxx, pmaxy = ent['pminx'], ent['pminy'], ent['pmaxx'], ent['pmaxy']

            if pmaxx - pminx > 0 and pmaxx - pminx < BBox.maxx - BBox.minx and pmaxy - pminy > 0 and pmaxy - pminy < BBox.maxy - BBox.miny:
                self.gc.set_foreground(self.color[type][STATUS_REPLICA])
                width = self.width * (pmaxx - pminx) / (BBox.maxx - BBox.minx)
                height = self.height * (pmaxy - pminy) / (BBox.maxy - BBox.miny)
                mx = self.width * ((pminx - BBox.minx) / (BBox.maxx - BBox.minx))
                my = self.height * ((pminy - BBox.miny) / (BBox.maxy - BBox.miny))
                drawable.draw_rectangle(self.gc, False, int(mx), int(my), int(width), int(height))

            minx, miny, maxx, maxy = ent['minx'], ent['miny'], ent['maxx'], ent['maxy']

            if maxx - minx > 0 and maxx - minx < BBox.maxx - BBox.minx and maxy - miny > 0 and maxy - miny < BBox.maxy - BBox.miny:
                self.gc.set_foreground(color)
                width = self.width * (maxx - minx) / (BBox.maxx - BBox.minx)
                height = self.height * (maxy - miny) / (BBox.maxy - BBox.miny)
                mx = self.width * ((minx - BBox.minx) / (BBox.maxx - BBox.minx))
                my = self.height * ((miny - BBox.miny) / (BBox.maxy - BBox.miny))
                drawable.draw_rectangle(self.gc, False, int(mx), int(my), int(width), int(height))

    def __do_render(self):
        """This is where the drawing takes place"""
        drawable = self.pixmap     # replace with self.window to fall-back on gtk's double buffering
                                      
        
        # i have to use this deprecated method since redhat 9 seems
        # to have pygtk2.0 as the default!

        self.render_pixbuf.render_to_drawable(drawable, self.gc,
                                              0, 0, 0, 0,
                                              self.render_pixbuf.get_width(), self.render_pixbuf.get_height(),
                                              0, 0, 0)

        # this is the newer, more preferred method, sitting here ...
        
#        self.window.draw_pixbuf(self.gc, self.render_pixbuf,
#                                 0, 0, 0, 0,
#                                  self.render_pixbuf.get_width(), self.render_pixbuf.get_height(),
#                                  )

        for ent in self.remoteents:
            self.draw_entity(drawable, ent, color=self.yellow)

        for ent in self.localents:
            self.draw_entity(drawable, ent)

        if drawable != self.window:
            self.window.draw_drawable(self.gc, drawable,
                                      0, 0, 0, 0,
                                      self.render_pixbuf.get_width(), self.render_pixbuf.get_height())
        else:
            self.queue_draw()
        
    def render(self, localents, remoteents):
        self.localents = localents
        self.remoteents = remoteents

        def cmp_func(x, y):
            "sort entities based on their type; OTHER < ITEM < BOT < MISSILE"
            mx, my  = x['mask'] & ~0x1, y['mask'] & ~0x1        # clear the last bits
            return my - mx                                      # CLASSNAME_OTHER > CLASSNAME_ITEM...

        # sort so that the most important entities remain visible
        self.localents.sort(cmp_func)
        self.__do_render()
        
    def on_expose_event(self, widget, event):
        self.__do_render()

def get_server_string(addr, port):
    return "%s:%d" % (addr, port)

class ServerInfo:
    """Information about each server. includes last frame received and
    the entities in the last frame. this information is needed since only
    complete frames are rendered."""
    def __init__(self, addr, port):
        self.addr = addr
        self.port = port
        self.lastframe = -1
        self.curframe  = -1
        self.lastframeents = []
        self.curframeents  = []
    
class Visualizer(gtk.Window):
    """The top-level window. contains several smaller control widgets
    as well as the main Canvas (drawing area)"""
    def __init__(self, servers, mapfile, refresh_interval):
        # Create the toplevel window
        gtk.Window.__init__(self)
        self.connect('destroy', lambda *w: gtk.main_quit())

        self.refresh_interval = refresh_interval
        
        self.set_title("QuakeViz")
        self.set_border_width(8)
        vbox = gtk.VBox(False, 8)
        vbox.set_border_width(0)
        self.add(vbox)

        hbox = gtk.HBox()
        hbox.set_border_width(0)
        vbox.pack_start(hbox, False, False, 0)
        
        label = gtk.Label("Select server: ")
        label.set_use_underline(True)
        hbox.pack_start(label, False, False, 0)

        # Ideally, we would like to use ComboBox; but that requires GTK >= 2.4
        # which isn't so common (Redhat 9 still uses 2.2.1)

        self.active_server = servers[0]
        menu = gtk.Menu()                   # create a menu
        for opt in servers:                 # fill our menu with the options
            menu_item = gtk.MenuItem(opt)
            menu_item.connect("activate",    # attach "opt" to widget
                              self.server_changed, opt) 
            menu_item.show()                 # don't forget this here
            menu.append(menu_item)
            
        option_menu = gtk.OptionMenu()      # create the optionmenu
        option_menu.set_menu(menu)          # add the menu into the optionmenu

        label.set_mnemonic_widget(option_menu)

        # if you don't align, the option_menu will expand to fill up
        # the entire height of this hbox (which is the height of the largest
        # height widget inside the hbox)
        align = gtk.Alignment(0.0, .5, 0.0, 0.0)
        align.add(option_menu)
        hbox.pack_start(align, False, False, 0)

        self.canvas = Canvas(mapfile)

        legendwidget = self.make_legend(self.canvas.getcolors())
        hbox.pack_start(legendwidget, False, False, 5)

        qbutton = gtk.Button(label="Quit")
        qbutton.connect("clicked", self.quit_button_clicked)
        qbutton.set_relief(gtk.RELIEF_HALF)

        # again, dont allow the quit button to be expanded 
        align = gtk.Alignment(0.0, .5, 0.0, 0.0)
        align.add(qbutton)
        hbox.pack_start(align, False, False, 50)

        frame = gtk.Frame("Game map")
        frame.set_shadow_type(gtk.SHADOW_ETCHED_IN)
        frame.set_border_width(2)

        table = gtk.Table(1, 1, False)
        table.set_border_width(5)
        table.set_row_spacings(5)
        table.set_col_spacings(10)
        frame.add(table)

        vbox.pack_start(frame, True, True, 0)
        
        table.attach(self.canvas, 0, 1, 0, 1, gtk.EXPAND | gtk.FILL, gtk.EXPAND | gtk.FILL , 0, 0)

        self.connect("key-press-event", self.key_press_event)

        self.show_all()

        self.serverinfo = {}
        for srv in servers:
            host, port = srv.split(":")
            self.serverinfo[srv] = ServerInfo(host, port)

        # arrange for the render routine to be called every so often
        self.render()

    def make_legend(self, colorinfo):
        frame = gtk.Frame("Legend")
        # create a legend table

        table = gtk.Table(3, 5, False)
        table.set_border_width(2)
        table.set_row_spacings(2)
        table.set_col_spacings(10)
        frame.add(table)

        row = 1
        for status in g_status.iterkeys():
            l = gtk.Label(g_status[status])
            table.attach(l, 0, 1, row, row + 1, gtk.EXPAND | gtk.FILL, gtk.EXPAND | gtk.FILL, 0, 0)
            row += 1

        col = 1
        for type in g_classnames.iterkeys():
            l = gtk.Label(g_classnames[type][0])
            table.attach(l, col, col + 1, 0, 1, gtk.EXPAND | gtk.FILL, gtk.EXPAND | gtk.FILL, 0, 0)
            col += 1

        row = 1
        for status in g_status.iterkeys():
            col = 1
            for type in g_classnames.iterkeys():
                d_area = gtk.DrawingArea()
                d_area.set_size_request(10, 10)
                d_area.modify_bg(gtk.STATE_NORMAL, colorinfo[type][status])

                table.attach(d_area, col, col + 1, row, row + 1, 0, 0, 0, 0)
                col += 1
            row += 1

        return frame
    
    def server_changed(self, widget, opt):
        self.active_server = opt

    def quit(self):
        self.hide()
        self.destroy()
        gtk.main_quit()

    def quit_button_clicked(self, button):
        self.quit()

    # this is nice. a quick 'q' quits the application.
    def key_press_event(self, vizwin, event):
        if event.keyval in (gtk.keysyms.q, gtk.keysyms.Q):
            self.quit()

    def is_server_selected(self, addr):
        host, port = addr
        return get_server_string(host, port) == self.active_server
        
    def update_ents(self, addr, framenum, ents):
        try:
            inf = self.serverinfo[get_server_string(addr[0], addr[1])]
        except KeyError:
            return        # we dont want information about this server [possibly!]

        if framenum > inf.curframe:      # new frame!
            inf.lastframe = inf.curframe
            inf.lastframeents = inf.curframeents
            inf.curframeents = []

        inf.curframe = framenum
        inf.curframeents += ents        

    def render(self):
        start = timenow()
        # always render the lastframeents from everybody
        localents = []
        remoteents = []
        for srv, inf in self.serverinfo.iteritems():
            if self.active_server == srv:
                localents = inf.lastframeents
            else:
                remoteents += inf.lastframeents
                
        self.canvas.render(localents, remoteents)            
        end = timenow()

        elapsed = end - start
        sleeptime = self.refresh_interval - elapsed
        if sleeptime < 0:
            sleeptime = 0

        # call ourselves soon...
        gtk.timeout_add(sleeptime, self.render)


EDICT_DUMP_SIZE = 1 + 12 + 12 + 12 + 12 + 12

class UpdateReceiver:    
    def __init__(self, servers, update_ents):
        self.update_ents = update_ents
        self.servers = servers
        self.listen_for_data()

    def deserialize(self, update):
        t = unpack('B', update[0])
        mask = t[0]
        x, y, z = unpack('!fff', update[1:1+3*4])

        #print "bot=" + str(mask & CLASSNAME_BOT)
        #print "(" + str(x) + "," + str(y) + "," + str(z) + ")"
        #print "min=(" + str(BBox.minx) + "," + str(BBox.miny) + ") max=(" + str(BBox.maxx) + "," + str(BBox.maxy) + ")"
        
        minx, miny, minz, maxx, maxy, maxz = unpack('!ffffff', update[1+3*4:1+3*4+2*3*4])
        pminx, pminy, pminz, pmaxx, pmaxy, pmaxz = unpack('!ffffff', update[1+3*4+2*3*4:])
        
        if BBox.minx <= x <= BBox.maxx and BBox.miny <= y <= BBox.maxy:
            return { "mask":mask, "x":x, "y":y, "minx":max(minx,BBox.minx), "miny":max(miny,BBox.miny), "maxx":min(maxx,BBox.maxx), "maxy":min(maxy,BBox.maxy), "pminx":max(pminx,BBox.minx), "pminy":max(pminy,BBox.miny), "pmaxx":min(pmaxx,BBox.maxx), "pmaxy":min(pmaxy,BBox.maxy) }
        else:
            return None
        
    def on_data_receive(self, source, condition):
        data, addr = source.recvfrom(1500)
        if get_server_string(addr[0], addr[1]) not in self.servers:
            return True
        
        framenum = unpack("!i", data[0:4])
        framenum = framenum[0]
        data = data[4:]
        
        # make sure we got an integral number of entity updates        
        assert len(data) % EDICT_DUMP_SIZE == 0
        
        nents = len(data) / EDICT_DUMP_SIZE        
        ents = []
        for i in range(nents):
            #print i
            #print len(data)
            j = i * EDICT_DUMP_SIZE
            k = j + EDICT_DUMP_SIZE

            e = self.deserialize(data[j:k])
            if e != None:
                ents.append(e)

        self.update_ents(addr, framenum, ents)
        return True
        
    def listen_for_data(self):
        self.sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        self.ip = "0.0.0.0"
        self.port = VISUALIZER_PORT
        self.sock.bind((self.ip, self.port))

        gobject.io_add_watch(self.sock, gobject.IO_IN, self.on_data_receive)
            
def main():
    args = []
    try:
        opts, args = getopt.getopt(sys.argv[1:], "r:")
    except getopt.error, msg:
        print >>sys.stderr, err.msg
        print >>sys.stderr, "-r [refresh_interval]"
        return 2

    refresh_interval = 100
    for o, a in opts:
        if o == "-r":
            refresh_interval = int(a)
            
    if len(args) <= 0:
        print "No servers given! Exiting..."
        return 1

    servers = args
    print >>sys.stderr, "Performing DNS lookup on servers, please wait...",
    for i in range(len(servers)):
        host, port = servers[i].split(":")
        if not host or not port:
            raise ValueError, "Malformed host:port pair (%s)" % servers[i]
        servers[i] = "%s:%s" % (socket.gethostbyname(host), port)
        
    print >>sys.stderr, "done."

    #mapfile = "bigmap.jpg"
    mapfile = "empty.jpg"
    dim = g_maps[mapfile]
    BBox.minx = dim[0]
    BBox.maxx = dim[1]
    BBox.miny = dim[2]
    BBox.maxy = dim[3]
    
    vis = Visualizer(servers, mapfile, refresh_interval)
    ur = UpdateReceiver(servers, vis.update_ents)

    gtk.main()
    return 0

if __name__ == '__main__':
    sys.exit(main())
    
	
# vim:ts=4:noet:
