/*
 * $Id: QuakeViz.java 1456 2005-01-11 20:39:45Z jeffpang $
 *
 * Based on Sun Microsystem's Balls Demo Code:
 * http://java.sun.com/products/java-media/2D/samples/suite/Colors/Balls.java
 *
 * @(#)Balls.java	1.4  98/12/03
 *
 * Copyright 1998 by Sun Microsystems, Inc.,
 * 901 San Antonio Road, Palo Alto, California, 94303, U.S.A.
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of Sun Microsystems, Inc. ("Confidential Information").  You
 * shall not disclose such Confidential Information and shall use
 * it only in accordance with the terms of the license agreement
 * you entered into with Sun.
 */


import java.awt.*;
import java.awt.image.*;
import java.awt.event.*;
import java.awt.image.BufferedImage;
import javax.swing.*;
import java.io.*;
import java.util.*;
import java.util.regex.*;

public class QuakeViz extends JApplet {

    Demo demo;

    static final int PRIMARY = 0;
    static final int REPLICA = 1;
    static final int REMOTE  = 2;

    static final Color bgcolor = new Color(1,1,1,1);

    public void init(int minx, int maxx, int miny, int maxy, 
		     Image bgimage, String[] servers) throws Exception {
        getContentPane().add(demo = new Demo(minx, maxx, miny, maxy,
					     bgimage, servers));
        getContentPane().add("North", new DemoControls(demo));
    }

    public void start() {
        demo.start();
    }
  
    public void stop() {
        demo.stop();
    }


    /**
     * The Demo class performs the animation and painting.
     */
    static class Demo extends JPanel implements Runnable {

	static Bot[] bot = 
	{ new Bot(PRIMARY), new Bot(REPLICA), new Bot(REMOTE) }; 
	static Item[] item = 
	{ new Item(PRIMARY), new Item(REPLICA), new Item(REMOTE) }; 
	static Bolt[] bolt =
	{ new Bolt(PRIMARY), new Bolt(REPLICA), new Bolt(REMOTE) }; 

        private Thread thread; 
        private BufferedImage bimg;
        private long now, lasttime;

        protected Vector remoteEntities;
	protected Vector entities;

	public int           selected;
	public Peer          peers[];
	public ClientSession sessions[];
	private ParseThread[] threads;

	private int minx, maxx, miny, maxy;
	private int dimx, dimy;
	private Image background;
	private Image backgroundScaled;

	private Pattern entityRegex;
	
        public Demo(int minx, int maxx, int miny, int maxy, 
		    Image bgimage, String[] servers) throws Exception {
	    if (servers.length == 0)
		throw new RuntimeException("need at least 1 server");

	    this.minx = minx;
	    this.maxx = maxx;
	    this.miny = miny;
	    this.maxy = maxy;

	    this.dimx = 0;
	    this.dimy = 0;

	    background = bgimage;

	    peers    = new Peer[servers.length];
	    sessions = new ClientSession[servers.length];
	    selected = 0;

	    for (int i=0; i<servers.length; i++) {
		peers[i] = new Peer(servers[i]);
		sessions[i] = new ClientSession(peers[i], false);
	    }

	    threads = new ParseThread[sessions.length];

	    for (int i=0; i<sessions.length; i++) {
		boolean sel = i == selected;
		threads[i] = new ParseThread(null, sessions[i], false, 0, 0);
		threads[i].start();
	    }

	    entityRegex = Pattern.compile("status=([PR]) .* classname=(.+?) .* origin=\\(([\\-\\d\\.]+),([\\-\\d\\.]+),([\\-\\d\\.]+)\\)");
        }

        
        public void step(int w, int h) {
	    entities = null;
	    remoteEntities = null;
            
	    //System.gc();

	    entities = new Vector();
	    remoteEntities = new Vector();

	    long start = System.currentTimeMillis();

	    //ParseThread[] threads = new ParseThread[sessions.length];

	    for (int i=0; i<sessions.length; i++) {
		boolean sel = i == selected;
		threads[i].w = w;
		threads[i].h = h;
		threads[i].sel = sel;
		threads[i].entities = sel? entities : remoteEntities;
		threads[i].done = false;
	    }

	    while (true) {
		boolean done = true;
		for (int i=0; i<threads.length; i++) {
		    done = done && threads[i].done;
		}
		if (done)
		    break;
		try {
                    thread.sleep(25);
                } catch (InterruptedException e) { break; }
	    }

	    long time = System.currentTimeMillis() - start;
	    //CaltellaUtil.info("parse time: " + time);
        }

	
	class ParseThread extends Thread
	{
	    private ClientSession s;
	    public  boolean sel;
	    public  int w;
	    public  int h;
	    public Vector entities;

	    public boolean done;

	    public ParseThread(Vector entities, ClientSession s, 
			       boolean sel, int w, int h)
	    {
		this.entities = entities;
		this.s = s;
		this.sel = sel;
		this.w = w;
		this.h = h;
		done = true;
	    }
	    
	    public void run()
	    {
		// FIXME XXX: This does not work if a password is required!

		// read the initial welcome message banner
		try {
		    CaltellaUtil.readLine(s.in);
		} catch (Exception ex) {
		    CaltellaUtil.warn("Error while reading welcome banner!");
		}

		while (true) {
		    while (done)
			try {
			    Thread.sleep(10);
			} catch (InterruptedException e) { }

		    try {

			s.prt.print("STORE\n");
			
			String line = CaltellaUtil.readLine(s.in);
			if (! line.equals("==BEGIN_RESP") ) {
			    CaltellaUtil.warn("Bad response start: " + line);
			    return;
			}
			
			long start = System.currentTimeMillis();
			long parse = 0;
			long readln = 0;
			while ( true ) {
			    long readln_start = System.currentTimeMillis();
			    line = CaltellaUtil.readLine(s.in);
			    readln += System.currentTimeMillis() - readln_start;
			    
			    if (line.equals("==END_RESP"))
				break;
			    
			    long parse_start = System.currentTimeMillis();
			    Matcher matcher = entityRegex.matcher(line);
			    
			    if (matcher.find()) {
				String status = matcher.group(1);
				String classname = matcher.group(2);
				String xV = matcher.group(3);
				String yV = matcher.group(4);
				String zV = matcher.group(5);
				
				if (status == null ||
				    classname == null ||
				    xV == null ||
				    yV == null ||
				    zV == null) {
				    CaltellaUtil.warn("bad response line: " + line);
				    continue;
				}
				
				Ball b = null;
				int stat;

				if (!sel) {
				    stat = REMOTE;
				} else if (status.equals("R")) {
				    stat = REPLICA;
				} else {
				    stat = PRIMARY;
				}
				
				if (classname.equals("bot") ||
				    classname.equals("player")) {
				    b = bot[stat];
				} else if (classname.equals("bolt")) {
				    b = bolt[stat];
				} else {
				    // XXX FIXME: there are other stuff...
				    b = item[stat];
				}
				
				float x = Float.parseFloat(xV);
				float y = Float.parseFloat(yV);
				if (x < minx || x > maxx ||
				    y < miny || y > maxy)
				    continue;
				
				float xc = w * ( (x - minx)/(maxx - minx) );
				float yc = h * ( (y - miny)/(maxy - miny) );
				
				entities.add(new Entity(b, xc, yc));
			    } else {
				CaltellaUtil.warn("bad response line: " + line);
				continue;
			    }
			    parse += System.currentTimeMillis() - parse_start;
			}
			long time = System.currentTimeMillis() - start;
			//CaltellaUtil.info("while time: " + time);
			//CaltellaUtil.info("readl time: " + readln);
			//CaltellaUtil.info("regex time: " + parse);

		    } catch (Exception ex) {
			CaltellaUtil.warn("Exception parsing! " + ex.toString());
		    }

		    done = true;
		    entities = null;
		}
	    }
	}
    
    
        public void drawDemo(int w, int h, Graphics2D g2) {
	    for (int i = 0; i < remoteEntities.size(); i++) {
                Entity b = (Entity)remoteEntities.get(i);
		g2.drawImage(b.b.img, (int) b.x, (int) b.y, this);
	    }

            for (int i = 0; i < entities.size(); i++) {
                Entity b = (Entity)entities.get(i);
		if (b.b instanceof Item)
		    g2.drawImage(b.b.img, (int) b.x, (int) b.y, this);
	    }
	    for (int i = 0; i < entities.size(); i++) {
                Entity b = (Entity)entities.get(i);
		if (b.b instanceof Bot)
		    g2.drawImage(b.b.img, (int) b.x, (int) b.y, this);
	    }
	    for (int i = 0; i < entities.size(); i++) {
                Entity b = (Entity)entities.get(i);
		if (b.b instanceof Bolt)
		    g2.drawImage(b.b.img, (int) b.x, (int) b.y, this);
	    }
        }
    
    
        public Graphics2D createGraphics2D(int w, int h) {
            Graphics2D g2 = null;
            if (bimg == null || bimg.getWidth() != w || bimg.getHeight() != h) {
                bimg = (BufferedImage) createImage(w, h);
            } 
            g2 = bimg.createGraphics();
	    g2.setBackground(getBackground());
	    g2.clearRect(0, 0, w, h);
	    if (w != dimx || h != dimy || 
		backgroundScaled == null) {
		dimx = w;
		dimy = h;
		backgroundScaled = 
		    background.getScaledInstance(dimx,dimy,
						 Image.SCALE_FAST);
	    }
	    g2.drawImage(backgroundScaled, 0, 0, this);

            return g2;
        }
    
    
        public void paint(Graphics g) {
            Dimension d = getSize();
            step(d.width, d.height);
            Graphics2D g2 = createGraphics2D(d.width, d.height);
            drawDemo(d.width, d.height, g2);
            g2.dispose();
            g.drawImage(bimg, 0, 0, this);
        }
    
    
        public void start() {
            thread = new Thread(this);
            thread.setPriority(Thread.MIN_PRIORITY);
            thread.start();
        }
    
    
        public synchronized void stop() {
            thread = null;
        }
	
    
        public void run() {
            Thread me = Thread.currentThread();
            while (thread == me) {
		now = System.currentTimeMillis();
                if (now - lasttime < 100) {
		    try {
			thread.sleep(100 - (now-lasttime));
		    } catch (InterruptedException e) { }
		}
		repaint();
		lasttime = now;
            }
            thread = null;
        }

	static class Entity {
	    public Ball b;
	    public float x, y;

	    public Entity(Ball b, float x, float y) {
		this.b = b;
		this.x = x;
		this.y = y;
	    }
	}

        static class Ball {
        
	    public int status;
            public int bsize;
            public BufferedImage img;
            private Color origColor;
        
            public Ball(Color color, int bsize) {
                origColor = color;
		this.bsize = bsize;
                img = makeImage(color, bsize);
            }
        
	    public void SetColor(Color color)
	    {
		img = makeImage(color, bsize);
	    }

	    public static BufferedImage makeImage(Color color, int bsize) {
		BufferedImage img;

                int R = bsize;
                byte[] data = new byte[R * 2 * R * 2];
                int maxr = 0;
                for (int Y = 2 * R; --Y >= 0;) {
                    int x0 = (int) 
			(Math.sqrt(R * R - (Y - R) * (Y - R)) + 0.5);
                    int p = Y * (R * 2) + R - x0;
                    for (int X = -x0; X < x0; X++) {
                        int x = X + 15;
                        int y = Y - R + 15;
                        int r = (int) (Math.sqrt(x * x + y * y) + 0.5);
                        if (r > maxr) {
                            maxr = r;
                        }
                        data[p++] = r <= 0 ? 1 : (byte) r;
                    }
                }
        
                int bg = 255;
                byte red[] = new byte[256];
                red[0] = (byte) bg;
                byte green[] = new byte[256];
                green[0] = (byte) bg;
                byte blue[] = new byte[256];
                blue[0] = (byte) bg;

		for (int i = maxr; i >= 1; --i) {
		    float d = (float) i / maxr;
		    red[i] = (byte) 
			blend(blend(color.getRed(), 255, d), bg, 0.5f);
		    green[i] = (byte) 
			blend(blend(color.getGreen(), 255, d), bg, 0.5f);
		    blue[i] = (byte) 
			blend(blend(color.getBlue(), 255, d), bg, 0.5f);
		}
		IndexColorModel icm = new IndexColorModel(8, maxr + 1,
					  red, green, blue, 0);
		DataBufferByte dbb = new DataBufferByte(data, data.length);
		int bandOffsets[] = {0};
		WritableRaster wr = Raster.createInterleavedRaster(dbb,
					   R*2,R*2,R*2,1, bandOffsets,null);
		img = new BufferedImage(icm, wr, 
					icm.isAlphaPremultiplied(), null);

		return img;
	    }
        
	    // performs the blending of white and color
            private static final int blend(int fg, int bg, float fgfactor) {
                return (int) (bg + (fg - bg) * fgfactor);
            }

	    public void SetStatus(int status)
	    {
		this.status = status;

		switch(status) {
		case PRIMARY:
		    SetColor(origColor);
		    break;
		case REPLICA:
		    SetColor(origColor.darker().darker().darker().darker());
		    break;
		case REMOTE:
		    SetColor(Color.yellow);
		    break;
		default:
		    throw new RuntimeException("Unknown status " + status);
		}
	    }
        
        }  // End Ball class

	static class Bot extends Ball {
	    public Bot(int status) {
		super(Color.red, 8);
		SetStatus(status);
	    }
	}
	
	static class Item extends Ball {
	    public Item(int status) {
		super(Color.blue.brighter(), 5);
		SetStatus(status);
	    }
	}
	
	static class Bolt extends Ball {
	    public Bolt(int status) {
		super(Color.green, 3);
		SetStatus(status);
	    }
	}

    } // End Demo class 

    /**
     * The DemoControls class provides controls for choosing which colored
     * ball to display, the size of the balls and whether or not to clear
     * the drawing surface.
     */
    static class DemoControls extends JPanel implements ActionListener {

        Demo demo;
        JToolBar toolbar;

        public DemoControls(Demo demo) {
            this.demo = demo;
            setBackground(bgcolor);
            add(toolbar = new JToolBar());
            toolbar.setFloatable(false);
            addTool("Switch Server");
        }


        public void addTool(String str) {
            JButton b = (JButton) toolbar.add(new JButton(str));
            b.addActionListener(this);
        }


        public void actionPerformed(ActionEvent e) {
	    demo.selected = (demo.selected + 1) % demo.peers.length;
        }
    } // End DemoControls


    public static void main(String argv[]) throws Exception {
        final QuakeViz demo = new QuakeViz();
	
	Toolkit toolkit = Toolkit.getDefaultToolkit();
// Image img = toolkit.getImage("map.jpg");
	Image img = toolkit.getImage("bigmap.jpg");

//        demo.init(-1408, 896, -256, 704, img, argv);
        demo.init(-4096, 4096, -4096, 4096, img, argv);
        Frame f = new Frame("QuakeViz");
        f.addWindowListener(new WindowAdapter() {
            public void windowClosing(WindowEvent e) {System.exit(0);}
            public void windowDeiconified(WindowEvent e) { demo.start(); }
            public void windowIconified(WindowEvent e) { demo.stop(); }
        });
        f.add(demo);
        f.pack();
        f.setSize(new Dimension(640, 480));
        f.show();
        demo.start();
    }
}
