/*
 * Peer.java
 * Author: Jeffrey Pang <ee122-at>
 *
 * CVS: $Id: Peer.java 982 2004-07-02 00:23:31Z jeffpang $
 */

import java.io.*;
import java.util.StringTokenizer;

/**
 * Maintains static information about a peer.
 */
public class Peer {
  
    /**
     * Default port of server.
     */
    public static final int DEFAULT_PORT = 3030; // FIXME: should be from conf

    private String hostname;
    private int port;
    private int rtt; // -1 if unknown

    /**
     * Constructs the appropriate address string for a server running at
     * hostname on port.
     */
    public static String getAddress(String hostname, int port) {
	/*
	  if (port == DEFAULT_PORT) {
	  return hostname;
	  } else {
	  return hostname + ":" + port;
	  }
	*/
	// always contains port
	return hostname + ":" + port;
    }

    /**
     * Parse out the hostname part of an address
     */
    public static String parseHostname(String address) throws NumberFormatException {
	int sep = address.indexOf(':');
	if (sep > 0) {
	    return address.substring(0,sep);
	} else {
	    return address;
	}
    }

    /**
     * Parse out the port part of an address
     */
    public static int parsePort(String address) throws NumberFormatException {
	int sep = address.indexOf(':');
	if (sep > 0) {
	    return Integer.parseInt(address.substring(sep+1));
	} else {
	    return DEFAULT_PORT;
	}
    }

    /**
     * Create a new Peer.
     *
     * @param hostname the domain name or IP address of the host
     * @param port the port the server is running on
     * @param rtt RTT to the host. -1 if unknown.
     */
    public Peer(String hostname, int port, int rtt) {
	this.hostname = hostname;
	this.port = port;
	this.rtt = rtt;
    }

    /**
     * Create a new Peer.
     *
     * @param hostaddr host address in [hostname] or [hostname]:[port] format.
     * @param rtt RTT to the host. -1 if unknown.
     */
    public Peer(String hostaddr, int rtt) throws NumberFormatException {
	setHostAddress(hostaddr);
	this.rtt = rtt;
    }

    /**
     * Create a new Peer. RTT for this peer is unknown.
     *
     * @param hostaddr host address in [hostname] or [hostname]:[port] format.
     */
    public Peer(String hostaddr) throws NumberFormatException {
	this(hostaddr, -1);
    }

    /**
     * Create an empty peer entry.
     */
    public Peer() throws NumberFormatException {
	this("");
    }

    private void setHostAddress(String hostaddr) throws NumberFormatException {
	int sep = hostaddr.indexOf(':');
	if (sep > 0) {
	    this.hostname = hostaddr.substring(0,sep);
	    this.port = Integer.parseInt(hostaddr.substring(sep+1));
	} else {
	    this.hostname = hostaddr;
	    this.port = DEFAULT_PORT;
	}
    }

    /**
     * Return the hostname of this peer.
     */
    public String getHostname() {
	return hostname;
    }

    /**
     * Return the port this peer is running on.
     */
    public int getPort() {
	return port;
    }

    /**
     * Return the RTT to this peer. -1 if unknown.
     */
    public int getRTT() {
	return rtt;
    }

    /**
     * Set the RTT to this peer.
     */
    public void setRTT(int rtt) {
	this.rtt = rtt;
    }

    /**
     * Read in this peer's information from an input stream.
     */
    public void fromStream(InputStream is) 
	throws IOException, NumberFormatException 
    {
	int c;
	String line = CaltellaUtil.readLine(is);

	StringTokenizer tok = new StringTokenizer(line, "\t");
	hostname = tok.nextToken();
	port = Integer.parseInt(tok.nextToken());
	// Rtt is optional (-1 if none)
	if (tok.hasMoreTokens()) {
	    setRTT(Integer.parseInt(tok.nextToken()));
	} else {
	    setRTT(-1);
	}
    }
  
    /**
     * Writes this peer to an output stream appropriately formatted.
     */
    public void toStream(OutputStream os) {
	PrintStream out = new PrintStream(os);
	out.print(getHostname() + "\t" + getPort() + "\t" + getRTT() + "\n");
    }

    /**
     * As string
     */
    public String toString() {
	return getAddress(getHostname(), getPort());
    }

    /**
     * Two peers are equal iff they are on the same host and running on the
     * same port.
     */
    public boolean equals(Object obj) {
	Peer p2 = (Peer)obj;

	return p2.getHostname().equals(getHostname()) && 
	    p2.getPort() == getPort();
    }

    /**
     * Generate a hashcode.
     */
    public int hashCode() {
	return hostname.hashCode() + port;
    }

}
