/*
 * CaltellaManager.java
 * Author: Jeffrey Pang <ee122-at>
 *
 * CVS: $Id: CaltellaUtil.java 982 2004-07-02 00:23:31Z jeffpang $
 */

import java.io.*;
import java.util.*;
import java.text.*;
import java.security.*;

/**
 * Static utility functions for random stuff.
 */
public class CaltellaUtil {

    // ---------- Input / Output ----------
 
    /**
     * Wrapper over an input stream that causes all bytes read to be
     * spit out to System.err.
     */
    public static class DebugInputStream extends InputStream {
	private InputStream is;
	private OutputStream os;

	public DebugInputStream(InputStream is) {
	    this.is = is;
	    this.os = System.err;
	}

	public void close() throws IOException {
	    is.close();
	}

	public int read() throws IOException {
	    int c = is.read();
	    if (c != -1) {
		os.write(c);
		os.flush();
	    }
	    return c;
	}

	public int read(byte[] b) throws IOException {
	    int len = is.read(b);
	    if (len > 0) {
		os.write(b, 0, len);
		os.flush();
	    }
	    return len;
	}

	public int read(byte[] b, int off, int len) throws IOException {
	    int len2 = is.read(b, off, len);
	    if (len2 > 0) {
		os.write(b, off, len2);
		os.flush();
	    }
	    return len2;
	}

    }

    /**
     * Wrapper over an output stream that causes all bytes read to be
     * spit out to System.err.
     */
    public static class DebugOutputStream extends OutputStream {
	private OutputStream is;
	private OutputStream os;

	public DebugOutputStream(OutputStream is) {
	    this.is = is;
	    this.os = System.err;
	}

	public void close() throws IOException {
	    is.close();
	}

	public void flush() throws IOException {
	    is.flush();
	}

	public void write(int b) throws IOException {
	    is.write(b);
	    os.write(b);
	    os.flush();
	}

	public void write(byte[] b) throws IOException {
	    is.write(b);
	    os.write(b);
	    os.flush();
	}

	public void write(byte[] b, int off, int len) throws IOException {
	    is.write(b, off, len);
	    os.write(b, off, len);
	    os.flush();
	}

    }

    /**
     * Non-buffering read line. Treats \n (LF) and \r\n (CRLF) as newline.
     *
     * @param is input stream to read from
     * @return single line read from the stream (with newline stripped)
     */
    public static String readLine(InputStream is) throws IOException {
	StringBuffer s = new StringBuffer();
	int c;
	while ((c = is.read()) >= 0 && (char)c != '\n') {
	    if (c == '\r') {
		int d = is.read();
		if ((char)d == '\n') 
		    break;
		else if (d < 0) {
		    s.append((char)c);
		    c = d;
		    break;
		} else {
		    s.append((char)c);
		    s.append((char)d);
		}
	    } else {
		s.append((char)c);
	    }
	}
	return (c<0 && s.length()==0)? null : s.toString();
    }


    // ---------- Date / Time ----------

    // quoted date string format (only for formatting, not parsing)
    private static DateFormat dateformat =
	new SimpleDateFormat("'\"'d MMM yyyy HH:mm:ss z'\"'");

    /**
     * Get the current time string in RFC822 format.
     */
    public static String currentDate() {
	Calendar cal = Calendar.getInstance();
	return formatDate(cal.getTime());
    }
  
    /**
     * Format the java Date into RFC822 format.
     */
    public static String formatDate(Date date) {
	return dateformat.format(date);
    }

    // ---------- Exception Handling ----------

    /**
     * Extract the exception message from the exception ex.
     */
    public static String getErrorMessage(Exception ex) {
	String exmsg = ex.toString();
	return exmsg.substring(Math.min(exmsg.indexOf(" ")+1, exmsg.length()-1));
    }

    private static String toHexString(byte[] buf) {
	StringBuffer sb = new StringBuffer();
	for (int i=0; i<buf.length; i++) {
	    sb.append( "" + 
		       toHexChar((buf[i]>>4)&0x0f) + 
		       toHexChar(buf[i]&0x0f) );
	}
	return sb.toString();
    }
  
    private static char[] hex = 
	new char[] { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 
		     'A', 'B', 'C', 'D', 'E', 'F' };

    private static char toHexChar(int b) {
	return hex[b];
    }

    // ----------- Misc. -----------

    /**
     * Construct a "safe" iterator that will not disrupt the original
     * data structure.
     */
    public static Iterator safeIterator(Iterator orig) {
	LinkedList list = new LinkedList();
	while (orig.hasNext()) {
	    list.add(orig.next());
	}
	return list.iterator();
    }
  
    /**
     * Print error message.
     */
    public static void warn(String msg) {
	System.out.println("Warning: " + msg);
    }
  
    /**
     * Print information message.
     */
    public static void info(String msg) {
	System.out.println(msg);
    }

    public static void die(String msg) {
	warn(msg);
	System.exit(1);
    }

}
