/* vim: set sw=4 ts=4 noet: -*- Mode:c++; c-basic-offset:4; tab-width:4; indent-tabs-mode:t -*- */

/* $Id: dg_viz.cpp 2380 2005-11-03 22:05:50Z jeffpang $ */

#include <game/g_local.h>
#include "dg.h"
#include <om/Manager.h>
#include <wan-env/RealNet.h>

struct sockaddr_in g_VisAddr;

static void __initialize_socket(int &udp_sock, struct sockaddr_in& vis_addr)
{
	if (OS::CreateSocket(&udp_sock, PROTO_UDP) < 0) 
		Debug::die("Could not create udp socket for the visualizer!");

	struct sockaddr_in	server_address;
	memset((char *)&server_address, 0, sizeof(server_address));

	server_address.sin_family = AF_INET;
	server_address.sin_addr.s_addr = g_LocalSID.GetIP();
	server_address.sin_port = htons(g_Preferences.port + 10000);  // From the Terminal code

	// Bind so that the visualizer can figure out who we are!
	if (bind(udp_sock, (struct sockaddr *) &server_address, sizeof(server_address)) < 0) {
		perror ("bind");
		Debug::die ("could not bind the udp socket for sending to the visualizer!");
	}

	IPEndPoint visIP(g_QuakePreferences.visualizer_ip, VISUALIZER_PORT);

	memset((char *) &vis_addr, 0, sizeof(vis_addr));
	vis_addr.sin_family = AF_INET;
	vis_addr.sin_addr.s_addr = visIP.GetIP();
	vis_addr.sin_port = htons(visIP.GetPort());
}

#define UDP_MAXLEN 1470 /// just to be safe; on linux, the max is 64K
static byte sg_Buffer[UDP_MAXLEN];
static int sg_BufLen = 0;

static void __send_data(int udp_sock, struct sockaddr_in &vis_addr)
{
	if (sendto(udp_sock, sg_Buffer, sg_BufLen, 0, 
				(struct sockaddr *) &vis_addr, sizeof(vis_addr)) < 0) 
	{
		DBG_DO {
			perror("sendto");
			Debug::warn("sendto call failed");
		}
	}
}

#define STATUS_PRIMARY 0x0
#define STATUS_REPLICA 0x1

#define CLASSNAME_MISSILE (0x1 << 1)
#define CLASSNAME_BOT  (0x1 << 2)
#define CLASSNAME_ITEM (0x1 << 3)
#define CLASSNAME_OTHER (0x1 << 4)

#define EDICT_DUMP_SIZE 1 /* status + classname byte */ + 12 /* origin[0-2] */ + 12 + 12 /* bbox min, max */ + 12 + 12 /* pred bbox min, max */

static void __write_float(float f)
{
	/* forget endian-ness. we are all x86 people here! 
	union
	{
		float f;
		int	l;
	} dat;
		
	dat.f = f;
	dat.l = LittleLong (dat.l);
	
	memcpy(sg_Buffer + sg_BufLen, &dat.l, 4);
	*/
	uint32 l = htonl( *(uint32 *)&f );
	memcpy(sg_Buffer + sg_BufLen, &l, 4);
	sg_BufLen += 4;
}

static inline void __write_byte(byte b)
{
	sg_Buffer[sg_BufLen++] = b;
}

static inline void __write_int(int i)
{
	uint32 l = htonl(i);
	memcpy(sg_Buffer + sg_BufLen, &l, 4);
	sg_BufLen += 4;
}

static void __dump_dg_edict(GObject *obj)
{
	byte mask = 0x0;
	if (obj->IsReplica()) 
		mask |= STATUS_REPLICA;
	
	edict_t *ent = ((DG_Edict *) obj)->GetEntity();
	if (strstr(ent->classname, "bot") || strstr(ent->classname, "player")) {
		// fprintf(stderr, "bot: ");
		mask |= CLASSNAME_BOT;
	}
	else if (streq(ent->classname, "bolt") ||
			 streq(ent->classname, "rocket") ||
			 streq(ent->classname, "grenade") ||
			 streq(ent->classname, "bfg blast")) {
		// fprintf(stderr, "bolt: ");
		mask |= CLASSNAME_MISSILE;
	}
	else if (strstr(ent->classname, "health") ||
			 strstr(ent->classname, "weapon") ||
			 strstr(ent->classname, "ammo")) {
		// fprintf(stderr, "item: ");
		mask |= CLASSNAME_ITEM;
	}
	else {
		// fprintf(stderr, "other: ");
		mask |= CLASSNAME_OTHER;
	}
	// fprintf(stderr, "mask = %d\n", mask);

	__write_byte(mask);
	for (int i = 0; i < 3; i++)
		__write_float(ent->s.origin[i]);

	for (int i = 0; i < 3; i++)
		__write_float(((DG_Edict *)obj)->m_CurrBBox.min[i]);
	for (int i = 0; i < 3; i++)
		__write_float(((DG_Edict *)obj)->m_CurrBBox.max[i]);

	for (int i = 0; i < 3; i++)
		__write_float(((DG_Edict *)obj)->m_LastPredBBox.min[i]);
	for (int i = 0; i < 3; i++)
		__write_float(((DG_Edict *)obj)->m_LastPredBBox.max[i]);
}

void DG_SendVisualizerUpdates()
{
	static bool inited = false;
	static int  udp_sock = -1;

	if (!inited) {
		__initialize_socket(udp_sock, g_VisAddr);
		inited = true;
	}
	
	// INFO << "sending visualizer updates" << endl;

	GObject *obj;
	g_ObjStore->Begin();
	sg_BufLen = 0;
	
	__write_byte(MSGTYPE_ENTITIES);
	__write_int(level.framenum);
	while ((obj = g_ObjStore->Next()) != NULL) {
		if (sg_BufLen + EDICT_DUMP_SIZE > UDP_MAXLEN) {
			__send_data(udp_sock, g_VisAddr);			
			sg_BufLen = 0;
			__write_byte(MSGTYPE_ENTITIES);
			__write_int(level.framenum);
		}

		__dump_dg_edict(obj);
	}

	if (sg_BufLen > 0) 
		__send_data(udp_sock, g_VisAddr);
}

