#define DG_INCLUDE
#include <game/g_local.h>
#include "dg.h"
#include <game/inc_registry.h>
#include <new>
#include <util/Benchmark.h>

#define ENABLE_ARRAY_ENCODING_HACK

#ifdef ENABLE_ARRAY_ENCODING_HACK
#define DGENC_BITMASK_MAXLEN  256

static byte dg_enc_bitmask[DGENC_BITMASK_MAXLEN];
#define INIT_BITMASK()      do { memset(dg_enc_bitmask, 0, sizeof(byte) * DGENC_BITMASK_MAXLEN); } while(0)
#define DGENC_SET_BIT_INSIDE_BYTE(j, bit)   do { dg_enc_bitmask[j] |= (0x1 << bit); } while (0)
#define DGENC_SET_BIT(i)   do { DGENC_SET_BIT_INSIDE_BYTE( (i)/8, (i) % 8 ); } while (0)

#define DGENC_IS_BIT_SET_INSIDE_BYTE(j, bit)   (dg_enc_bitmask[j] & ((0x1 << bit)))
#define DGENC_IS_BIT_SET(i)   (DGENC_IS_BIT_SET_INSIDE_BYTE( (i)/8, (i) % 8))

#define DGENC_MASKLEN(x)          ((x) / 8 + ((x) % 8 != 0 ? 1 : 0))

bool __DG_IsNonZero(dg_field_t *field, byte *base, int i)
{
    ASSERT(i >= 0 && i < field->alen);
    if (field->type == DF_INT_ARRAY) 
	return  *((int *) base + i) != 0x0;
    else if (field->type == DF_CHAR_ARRAY)
	return  *((byte *) base + i) != 0x0;
    else if (field->type == DF_SHORT_ARRAY)
	return  *((short *) base + i) != 0x0;
}

void __DG_WriteField(dg_field_t *field, byte *base, int i, Packet *pkt)
{
    if (field->type == DF_INT_ARRAY) 
	pkt->WriteInt( *( ((int *) base) + i) );
    else if (field->type == DF_CHAR_ARRAY)
	pkt->WriteByte( *( ((byte *) base) + i) );
    else if (field->type == DF_SHORT_ARRAY)
	pkt->WriteShort( *( ((short *) base) + i) );
}

void __DG_ReadField(dg_field_t *field, byte *base, int i, Packet *pkt)
{
    if (field->type == DF_INT_ARRAY) 
	*( ((int *) base) + i) = pkt->ReadInt();
    else if (field->type == DF_CHAR_ARRAY)
	*( ((byte *) base) + i) = pkt->ReadByte();
    else if (field->type == DF_SHORT_ARRAY)
	*( ((short *) base) + i) = pkt->ReadShort();
}

void __DG_PrintField(dg_field_t *field, byte *base, int i)
{
    if (field->type == DF_INT_ARRAY) 
	cerr << *( ((int *) base) + i)  << " ";
    else if (field->type == DF_CHAR_ARRAY)
	cerr << *( ((byte *) base) + i) << " ";
    else if (field->type == DF_SHORT_ARRAY)
	cerr << *( ((short *) base) + i) << " ";
}

int  __DG_GetFieldSize(dg_field_t *field)
{
    if (field->type == DF_INT_ARRAY) 
	return sizeof(int);
    else if (field->type == DF_CHAR_ARRAY)
	return 1;
    else if (field->type == DF_SHORT_ARRAY)
	return sizeof(short);
}

int  DG_ArrayLength(dg_field_t *field, byte *base)
{
    int nonzero = 0;
    
    for (int i = 0; i < field->alen; i++) {
	if (__DG_IsNonZero(field, base, i))
	    nonzero++;
    }

    int bitmasklen = DGENC_MASKLEN(field->alen);
    DB(10) << merc_va("field=%s alen=%d bitmasklen=%d nonzero=%d serlen=%d", field->name, field->alen, bitmasklen, nonzero,
	    bitmasklen + nonzero * __DG_GetFieldSize(field)) << endl;
    return bitmasklen + nonzero * __DG_GetFieldSize(field);
}

void DG_ArraySerialize(dg_field_t *field, byte *base, Packet *pkt)
{
    INIT_BITMASK();
    int nonzero = 0;
    
    for (int i = 0; i < field->alen; i++) {
	if (__DG_IsNonZero(field, base, i)) {
	    DGENC_SET_BIT(i);
	    nonzero++;
	}
    }

    int bitmasklen = DGENC_MASKLEN(field->alen);
    for (int i = 0; i < bitmasklen; i++)  {
	// DB_DO(1) { fprintf(stderr, "%02x ", (byte) dg_enc_bitmask[i]); }
	pkt->WriteByte(dg_enc_bitmask[i]);
    }
    // DB_DO(1) { fprintf(stderr, "]\n"); }
    
    DB(10) << "serializing field=" << field->name << " alen=" << field->alen << " nonzero=" << nonzero << endl;
    DB(10) << merc_va("bitmask=[%02x %02x %02x %02x %02x %02x %02x %02x]", 
	    dg_enc_bitmask[0], dg_enc_bitmask[1], dg_enc_bitmask[2], dg_enc_bitmask[3], 
	    dg_enc_bitmask[4], dg_enc_bitmask[5], dg_enc_bitmask[6], dg_enc_bitmask[7]) << endl;
    DB(10) << "Wrote the following - " << endl;
    for (int i = 0; i < field->alen; i++)  {
	if (DGENC_IS_BIT_SET(i)) {
	    __DG_WriteField(field, base, i, pkt);
	    DB_DO(10) { __DG_PrintField(field, base, i); }
	}
    }
    DB(10) << "---" << endl;

}

void DG_ArrayDeserialize(dg_field_t *field, byte *base, Packet *pkt)
{
    INIT_BITMASK();
    int bitmasklen = DGENC_MASKLEN(field->alen);
    for (int i = 0; i < bitmasklen; i++) {
	dg_enc_bitmask[i] = pkt->ReadByte();
    }

    
    DB(10) << "DEserializing field=" << field->name << " alen=" << field->alen << endl;
    DB(10) << merc_va("READ bitmask=[%02x %02x %02x %02x %02x %02x %02x %02x]", 
	    dg_enc_bitmask[0], dg_enc_bitmask[1], dg_enc_bitmask[2], dg_enc_bitmask[3], 
	    dg_enc_bitmask[4], dg_enc_bitmask[5], dg_enc_bitmask[6], dg_enc_bitmask[7]) << endl;
    
    memset(base, 0, field->alen * __DG_GetFieldSize(field));

    DB(10) << "Read the following - " << endl;
    for (int i = 0; i < field->alen; i++) {
	if (DGENC_IS_BIT_SET(i)) {
	    __DG_ReadField(field, base, i, pkt);
	    DB_DO(10) { __DG_PrintField(field, base, i); }
	}
    }
    DB(10) << "---" << endl;
    
    
}
#endif


dg_field_t sentinel[] = { {0, 0, DF_INVALID, 0 } };
char **dynamic_strings;
int len_static_strings, len_dynamic_strings, len_func_pointers, len_globals_pointers;

/* Must be in the same order as the enum typedef for dg_fieldtype_t */
typeinfo_t typeinfo[] = 
{
    { DF_BOOL, null_init, bool_print, bool_length, bool_is_dirty, bool_serialize, bool_deserialize, sentinel },
    { DF_INT, null_init, int_print, int_length, int_is_dirty, int_serialize, int_deserialize, sentinel },
    { DF_INT_ARRAY, null_init, intarray_print, intarray_length, intarray_is_dirty, intarray_serialize, intarray_deserialize, sentinel },
    { DF_CHAR, null_init, char_print, char_length, char_is_dirty, char_serialize, char_deserialize, sentinel },
    { DF_CHAR_ARRAY, null_init, chararray_print, chararray_length, chararray_is_dirty, chararray_serialize, chararray_deserialize, sentinel },
    { DF_SHORT, null_init, short_print, short_length, short_is_dirty, short_serialize, short_deserialize, sentinel },
    { DF_SHORT_ARRAY, null_init, shortarray_print, shortarray_length, shortarray_is_dirty, shortarray_serialize, shortarray_deserialize, sentinel },
    { DF_FLOAT, null_init, float_print, float_length, float_is_dirty, float_serialize, float_deserialize, sentinel },
    { DF_FLOAT_ARRAY, null_init, floatarray_print, floatarray_length, floatarray_is_dirty, floatarray_serialize, floatarray_deserialize, sentinel },
    { DF_VECTOR, null_init, vector_print, vector_length, vector_is_dirty, vector_serialize, vector_deserialize, sentinel },
    { DF_GUID, null_init, guid_print, guid_length, guid_is_dirty, guid_serialize, guid_deserialize, sentinel },

    // changed to generic pointer [3/29/2004]
    { DF_CHAR_PTR, null_init, genericptr_print, genericptr_length, genericptr_is_dirty, genericptr_serialize, genericptr_deserialize, sentinel },

    { DF_EDICT_PTR, null_init, edictptr_print, edictptr_length, edictptr_is_dirty, edictptr_serialize, edictptr_deserialize, edict_fields },

    { DF_FUNC_PTR, null_init, genericptr_print, genericptr_length, genericptr_is_dirty, genericptr_serialize, genericptr_deserialize, sentinel },
    { DF_ITEM_PTR, null_init, genericptr_print, genericptr_length, genericptr_is_dirty, genericptr_serialize, genericptr_deserialize, sentinel },
    { DF_ITEM_ARMOR_PTR, null_init, genericptr_print, genericptr_length, genericptr_is_dirty, genericptr_serialize, genericptr_deserialize, sentinel },
    { DF_MMOVE_PTR, null_init, genericptr_print, genericptr_length, genericptr_is_dirty, genericptr_serialize, genericptr_deserialize, sentinel },
    { DF_MFRAME_PTR, null_init, genericptr_print, genericptr_length, genericptr_is_dirty, genericptr_serialize, genericptr_deserialize, sentinel },

    { DF_CLIENT_PTR, object_init, object_print, object_length, object_is_dirty, object_serialize, object_deserialize, gclient_fields }, 
    { DF_ENTITYSTATE, object_init, object_print, object_length, object_is_dirty, object_serialize, object_deserialize, entitystate_fields }, 
    { DF_MOVEINFO, object_init, object_print, object_length, object_is_dirty, object_serialize, object_deserialize, moveinfo_fields }, 
    { DF_MONSTERINFO, object_init, object_print, object_length, object_is_dirty, object_serialize, object_deserialize, monsterinfo_fields }, 
    { DF_PLAYER_STATE, object_init, object_print, object_length, object_is_dirty, object_serialize, object_deserialize, playerstate_fields }, 
    { DF_PMOVE_STATE, object_init, object_print, object_length, object_is_dirty, object_serialize, object_deserialize, pmovestate_fields }, 
    { DF_CLIENT_PERSISTANT, object_init, object_print, object_length, object_is_dirty, object_serialize, object_deserialize, clientpersistant_fields }, 
    { DF_CLIENT_RESPAWN, object_init, object_print, object_length, object_is_dirty, object_serialize, object_deserialize, clientrespawn_fields },

    { DF_MULTICAST_LIST, null_init, multicast_list_print, multicast_list_length, multicast_list_is_dirty, multicast_list_serialize, multicast_list_deserialize, sentinel }
};

///////////////////////////////////////////////////////////////////////////////

void null_init(dg_field_stack_t *, dg_field_t *)
{
}

///////////////////////////////////////////////////////////////////////////////
// DF_BOOL

void bool_print(int indent, ostream& out, dg_field_stack_t * prefix, dg_fields *toprint, dg_field_t *field, byte *from)
{
    out << *(qboolean *)from;
}

qboolean bool_is_dirty(dg_field_stack_t * prefix, dg_fields *isdirty, dg_field_t *, byte *from, byte *to)
{
    return (*(qboolean *) from != *(qboolean *) to);
}

int bool_length(dg_field_stack_t * prefix, dg_fields *encoded, dg_field_t *, byte *base) {
    return 1;
}

void bool_serialize(dg_field_stack_t * prefix, dg_fields *encoded, DG_Edict *obj, dg_field_t *, byte *from, Packet *pkt)
{
    qboolean bit = *(qboolean *) from;
    if (bit)
        pkt->WriteByte( 0x1);
    else
        pkt->WriteByte( 0x0);
}

void bool_deserialize(dg_field_stack_t * prefix, dg_fields *encoded, DG_Edict *obj, dg_field_t *, Packet *pkt, byte *ptr_object, dg_fieldtype_t container, int ofs)
{
    byte c = pkt->ReadByte();
    if (c == 0x0) {
        *(qboolean *) ptr_object = false;
    }
    else {
        *(qboolean *) ptr_object = true;
    }
}

///////////////////////////////////////////////////////////////////////////////
// DF_INT

void int_print(int indent, ostream& out, dg_field_stack_t * prefix, dg_fields *toprint, dg_field_t *field, byte *from)
{
    out << *(int *)from;
}

qboolean int_is_dirty(dg_field_stack_t * prefix, dg_fields *isdirty, dg_field_t *, byte *from, byte *to)
{
    return (*(int *) from != *(int *) to);
}

int int_length(dg_field_stack_t * prefix, dg_fields *encoded, dg_field_t *field, byte *base) {
    return 4;
}

void int_serialize(dg_field_stack_t * prefix, dg_fields *encoded, DG_Edict *obj, dg_field_t *field, byte *from, Packet *pkt)
{
    // cast as unsigned, otherwise negatives will be serialized incorrectly
    uint32 v = *(uint32 *) from;
     
    pkt->WriteInt( v );
}

void int_deserialize(dg_field_stack_t * prefix, dg_fields *encoded, DG_Edict *obj, dg_field_t *field, Packet *pkt, byte *ptr_object, dg_fieldtype_t container, int ofs)
{
    uint32 v = pkt->ReadInt();

    *(uint32 *) ptr_object = v;
}

///////////////////////////////////////////////////////////////////////////////
// DF_CHAR

void char_print(int indent, ostream& out, dg_field_stack_t * prefix, dg_fields *toprint, dg_field_t *field, byte *from)
{
    out << *(char *)from;
}

qboolean char_is_dirty(dg_field_stack_t * prefix, dg_fields *isdirty, dg_field_t *field, byte *from, byte *to)
{
    return (*(char *) from != *(char *) to);
}

int char_length(dg_field_stack_t * prefix, dg_fields *encoded, dg_field_t *field, byte *base) {
    return 1;
}

void char_serialize(dg_field_stack_t * prefix, dg_fields *encoded, DG_Edict *obj, dg_field_t *field, byte *from, Packet *pkt)
{
    pkt->WriteByte( *(char *) from);
}

void char_deserialize(dg_field_stack_t * prefix, dg_fields *encoded, DG_Edict *obj, dg_field_t *field, Packet *pkt, byte *ptr_object, dg_fieldtype_t container, int ofs)
{
    *(char *) ptr_object = pkt->ReadByte();
}

///////////////////////////////////////////////////////////////////////////////
// DF_SHORT

void short_print(int indent, ostream& out, dg_field_stack_t * prefix, dg_fields *toprint, dg_field_t *field, byte *from)
{
    out << *(short *)from;
}

qboolean short_is_dirty(dg_field_stack_t * prefix, dg_fields *isdirty, dg_field_t *field, byte *from, byte *to)
{
    return (*(short *) from != *(short *) to);
}

int short_length(dg_field_stack_t * prefix, dg_fields *encoded, dg_field_t *field, byte *base) {
    return 2;
}

void short_serialize(dg_field_stack_t * prefix, dg_fields *encoded, DG_Edict *obj, dg_field_t *field, byte *from, Packet *pkt)
{
    // cast as unsigned, otherwise negatives will be serialized incorrectly
    pkt->WriteShort( *(uint16 *) from);
}

void short_deserialize(dg_field_stack_t * prefix, dg_fields *encoded, DG_Edict *obj, dg_field_t *field, Packet *pkt, byte *ptr_object, dg_fieldtype_t container, int ofs)
{
    *(uint16 *) ptr_object = pkt->ReadShort();
}

///////////////////////////////////////////////////////////////////////////////
// DF_FLOAT

void float_print(int indent, ostream& out, dg_field_stack_t * prefix, dg_fields *toprint, dg_field_t *field, byte *from)
{
    out << *(float *)from;
}

qboolean float_is_dirty(dg_field_stack_t * prefix, dg_fields *isdirty, dg_field_t *field, byte *from, byte *to)
{
    return (*(float *) from != *(float *) to);
}

int float_length(dg_field_stack_t * prefix, dg_fields *encoded, dg_field_t *field, byte *base) {
    return 4;
}

void float_serialize(dg_field_stack_t * prefix, dg_fields *encoded, DG_Edict *obj, dg_field_t *field, byte *from, Packet *pkt)
{
    pkt->WriteFloat( *(float *) from);
}

void float_deserialize(dg_field_stack_t * prefix, dg_fields *encoded, DG_Edict *obj, dg_field_t *field, Packet *pkt, byte *ptr_object, dg_fieldtype_t container, int ofs)
{
    *(float *) ptr_object = pkt->ReadFloat();
}

///////////////////////////////////////////////////////////////////////////////
// DF_INT_ARRAY

void intarray_print(int indent, ostream& out, dg_field_stack_t * prefix, dg_fields *toprint, dg_field_t *field, byte *from)
{
    out << "[";
    for (int i = 0; i < field->alen; i++) {
	if (i != 0) out << ",";
	out << *((int *)from + i);
    }
    out << "]";
}

qboolean intarray_is_dirty(dg_field_stack_t * prefix, dg_fields *isdirty, dg_field_t *field, byte *from, byte *to)
{
    int i;

    for (i = 0; i < field->alen; i++)
        if (*((int *) from + i) != *((int *) to + i) )
            return true;
    return false;
}

int intarray_length(dg_field_stack_t * prefix, dg_fields *encoded, dg_field_t *field, byte *base) {
#ifdef ENABLE_ARRAY_ENCODING_HACK
    return DG_ArrayLength(field, base);
#else
    return sizeof(int) * field->alen;
#endif
}

void intarray_serialize(dg_field_stack_t * prefix, dg_fields *encoded, DG_Edict *obj, dg_field_t *field, byte *from, Packet *pkt)
{
#ifdef ENABLE_ARRAY_ENCODING_HACK
    DG_ArraySerialize(field, from, pkt);
#else
    int i;
    for (i = 0; i < field->alen; i++)
        pkt->WriteInt( *((int *) from + i));
#endif
}

void intarray_deserialize(dg_field_stack_t * prefix, dg_fields *encoded, DG_Edict *obj, dg_field_t *field, Packet *pkt, byte *ptr_object, dg_fieldtype_t container, int ofs)
{
#ifdef ENABLE_ARRAY_ENCODING_HACK
    DG_ArrayDeserialize(field, ptr_object, pkt);
#else
    int *p_arr = (int *) ptr_object;
    for (int i = 0; i < field->alen; i++) {
	p_arr[i] = pkt->ReadInt();
    }
#endif
}

///////////////////////////////////////////////////////////////////////////////
// DF_FLOAT_ARRAY

void floatarray_print(int indent, ostream& out, dg_field_stack_t * prefix, dg_fields *toprint, dg_field_t *field, byte *from)
{
    out << "[";
    for (int i = 0; i < field->alen; i++) {
	if (i != 0) out << ",";
	out << *((float *)from + i);
    }
    out << "]";
}

qboolean floatarray_is_dirty(dg_field_stack_t * prefix, dg_fields *isdirty, dg_field_t *field, byte *from, byte *to)
{
    int i;

    for (i = 0; i < field->alen; i++)
        if (*((float *) from + i) != *((float *) to + i) )
            return true;
    return false;
}

int floatarray_length(dg_field_stack_t * prefix, dg_fields *encoded, dg_field_t *field, byte *base) {
    return 4 * field->alen;
}

void floatarray_serialize(dg_field_stack_t * prefix, dg_fields *encoded, DG_Edict *obj, dg_field_t *field, byte *from, Packet *pkt)
{
    int i;
    for (i = 0; i < field->alen; i++)
        pkt->WriteFloat( *((float *) from + i));
}

void floatarray_deserialize(dg_field_stack_t * prefix, dg_fields *encoded, DG_Edict *obj, dg_field_t *field, Packet *pkt, byte *ptr_object, dg_fieldtype_t container, int ofs)
{
    int i;
    for (i = 0; i < field->alen; i++)
        *((float *) ptr_object + i) = pkt->ReadFloat();
}

///////////////////////////////////////////////////////////////////////////////
// DF_byte_ARRAY

void chararray_print(int indent, ostream& out, dg_field_stack_t * prefix, dg_fields *toprint, dg_field_t *field, byte *from)
{
    out << "[";
    for (int i = 0; i < field->alen; i++) {
	if (i != 0) out << ",";
	out << *((byte *)from + i);
    }
    out << "]";
}

qboolean chararray_is_dirty(dg_field_stack_t * prefix, dg_fields *isdirty, dg_field_t *field, byte *from, byte *to)
{
    int i;

    for (i = 0; i < field->alen; i++)
        if (*((byte *) from + i) != *((byte *) to + i) )
            return true;
    return false;
}

int chararray_length(dg_field_stack_t * prefix, dg_fields *encoded, dg_field_t *field, byte *base) {
#ifdef ENABLE_ARRAY_ENCODING_HACK
    return DG_ArrayLength(field, base);
#else
    return 1 * field->alen;
#endif
}

void chararray_serialize(dg_field_stack_t * prefix, dg_fields *encoded, DG_Edict *obj, dg_field_t *field, byte *from, Packet *pkt)
{
#ifdef ENABLE_ARRAY_ENCODING_HACK
    DG_ArraySerialize(field, from, pkt);
#else
    int i;
    for (i = 0; i < field->alen; i++)
        pkt->WriteByte( *((byte *) from + i));
#endif
}

void chararray_deserialize(dg_field_stack_t * prefix, dg_fields *encoded, DG_Edict *obj, dg_field_t *field, Packet *pkt, byte *ptr_object, dg_fieldtype_t container, int ofs)
{
#ifdef ENABLE_ARRAY_ENCODING_HACK
    DG_ArrayDeserialize(field, ptr_object, pkt);
#else
    int i;
    for (i = 0; i < field->alen; i++)
        *((byte *) ptr_object + i) = pkt->ReadByte();
#endif
}

///////////////////////////////////////////////////////////////////////////////
// DF_SHORT_ARRAY

void shortarray_print(int indent, ostream& out, dg_field_stack_t * prefix, dg_fields *toprint, dg_field_t *field, byte *from)
{
    out << "[";
    for (int i = 0; i < field->alen; i++) {
	if (i != 0) out << ",";
	out << *((short *)from + i);
    }
    out << "]";
}

qboolean shortarray_is_dirty(dg_field_stack_t * prefix, dg_fields *isdirty, dg_field_t *field, byte *from, byte *to)
{
    int i;

    for (i = 0; i < field->alen; i++)
        if (*((short *) from + i) != *((short *) to + i) )
            return true;
    return false;
}

int shortarray_length(dg_field_stack_t * prefix, dg_fields *encoded, dg_field_t *field, byte *base) {
#ifdef ENABLE_ARRAY_ENCODING_HACK
    return DG_ArrayLength(field, base);
#else
    return 2 * field->alen;
#endif
}

void shortarray_serialize(dg_field_stack_t * prefix, dg_fields *encoded, DG_Edict *obj, dg_field_t *field, byte *from, Packet *pkt)
{
#ifdef ENABLE_ARRAY_ENCODING_HACK
    DG_ArraySerialize(field, from, pkt);
#else
    int i;
    for (i = 0; i < field->alen; i++)
        pkt->WriteShort( *((short *) from + i));
#endif
}

void shortarray_deserialize(dg_field_stack_t * prefix, dg_fields *encoded, DG_Edict *obj, dg_field_t *field, Packet *pkt, byte *ptr_object, dg_fieldtype_t container, int ofs)
{
#ifdef ENABLE_ARRAY_ENCODING_HACK
    DG_ArrayDeserialize(field, ptr_object, pkt);
#else
    int i;
    for (i = 0; i < field->alen; i++)
        *((short *) ptr_object + i) = pkt->ReadShort();
#endif
}

///////////////////////////////////////////////////////////////////////////////
// DF_VECTOR

void vector_print(int indent, ostream& out, dg_field_stack_t * prefix, dg_fields *toprint, dg_field_t *field, byte *from)
{
    out << "(";
    for (short i = 0; i < 3; i++) {
	if (i != 0) out << ",";
	out << *((float *)from + i);
    }
    out << ")";
}

qboolean vector_is_dirty(dg_field_stack_t * prefix, dg_fields *isdirty, dg_field_t *field, byte *from, byte *to)
{
    int i;

    for (i = 0; i < 3; i++)
        if (*((float *) from + i) != *((float *) to + i) )
            return true;
    return false;
}

int vector_length(dg_field_stack_t * prefix, dg_fields *encoded, dg_field_t *field, byte *base) {
    return 4 * 3;
}

void vector_serialize(dg_field_stack_t * prefix, dg_fields *encoded, DG_Edict *obj, dg_field_t *field, byte *from, Packet *pkt)
{
    int i;
    for (i = 0; i < 3; i++)
        pkt->WriteFloat( *((float *) from + i));
}

void vector_deserialize(dg_field_stack_t * prefix, dg_fields *encoded, DG_Edict *obj, dg_field_t *field, Packet *pkt, byte *ptr_object, dg_fieldtype_t container, int ofs)
{
    int i;
    for (i = 0; i < 3; i++) 
        *((float *) ptr_object + i) = pkt->ReadFloat();
}

///////////////////////////////////////////////////////////////////////////////
// DF_GUID

void guid_print(int indent, ostream& out, dg_field_stack_t * prefix, dg_fields *toprint, dg_field_t *field, byte *from)
{
    out << *(guid_t *)from;
}

qboolean guid_is_dirty(dg_field_stack_t * prefix, dg_fields *isdirty, dg_field_t *field, byte *from, byte *to)
{
    guid_t *g1 = (guid_t *) from;
    guid_t *g2 = (guid_t *) to;

    return (*g1 != *g2);
}

int guid_length(dg_field_stack_t * prefix, dg_fields *encoded, dg_field_t *field, byte *base) {
    return zero_guid.GetLength();
}

// Ashwin [04/23/2004]
// XXX: so, if I say
//     guid_t *g = (guid_t *) from;
// and g->Serialize(pkt);
//
// it dumps core. WHY? I don't understand C++
//
void guid_serialize(dg_field_stack_t * prefix, dg_fields *encoded, DG_Edict *obj, dg_field_t *field, byte *from, Packet *pkt)
{
    guid_t g = * (guid_t *) from;
    g.Serialize(pkt);
}

// My first use of the "placement new" operator - Ashwin [4/6/2004]
void guid_deserialize(dg_field_stack_t * prefix, dg_fields *encoded, DG_Edict *obj, dg_field_t *field, Packet *pkt, byte *ptr_object, dg_fieldtype_t container, int ofs)
{
    (void) new(ptr_object) guid_t(pkt);
}


///////////////////////////////////////////////////////////////////////////////
// DF_MULTICAST_LIST

void multicast_list_print(int indent, ostream& out, dg_field_stack_t * prefix, dg_fields *toprint, dg_field_t *field, byte *from)
{
    DG_MulticastEvent *list = *(DG_MulticastEvent **)from;

    if (!list)
	out << "<NULL>";
    else 
	out << "<multicastEvents...>";
}

qboolean multicast_list_is_dirty(dg_field_stack_t * prefix, dg_fields *isdirty, dg_field_t *field, byte *from, byte *to)
{
    // new multicast_lists are allocated each frame, so if they differ, then
    // they are dirty and otherwise not
    return *(DG_MulticastEvent **) from  != *(DG_MulticastEvent **) to;
}

int multicast_list_length(dg_field_stack_t * prefix, dg_fields *encoded, dg_field_t *field, byte *base) {
    DG_MulticastEvent *list = *(DG_MulticastEvent **)base;
    if (!list) return 1;

    return 1 + list->GetLength();
}

void multicast_list_serialize(dg_field_stack_t * prefix, dg_fields *encoded, DG_Edict *obj, dg_field_t *field, byte *from, Packet *pkt)
{
    DG_MulticastEvent *list = *(DG_MulticastEvent **)from;

    if (list) {
	pkt->WriteByte(1);
	list->Serialize(pkt);
    } else {
	pkt->WriteByte(0);
    }
}

void multicast_list_deserialize(dg_field_stack_t * prefix, dg_fields *encoded, DG_Edict *obj, dg_field_t *field, Packet *pkt, byte *ptr_object, dg_fieldtype_t container, int ofs)
{
    bool hasList = (bool)pkt->ReadByte();

    if (hasList) {
	*(DG_MulticastEvent **)ptr_object = new DG_MulticastEvent(pkt);
    } else
	*(DG_MulticastEvent **)ptr_object = NULL;
}


///////////////////////////////////////////////////////////////////////////////
// DF_EDICT_PTR

void edictptr_print(int indent, ostream& out, dg_field_stack_t * prefix, dg_fields *toprint, dg_field_t *field, byte *from)
{
    edict_t *ent = *(edict_t **)from;
    
    if (!ent || !ent->inuse)
	out << "<NULL>";
    else 
	out << "<" << ent->guid << ">";
}

qboolean edictptr_is_dirty(dg_field_stack_t * prefix, dg_fields *isdirty, dg_field_t *field, byte *from, byte *to)
{
    return ( *(edict_t **) from  != *(edict_t **) to);
}

int edictptr_length(dg_field_stack_t * prefix, dg_fields *encoded, dg_field_t *field, byte *base) {
    return zero_guid.GetLength() + zero_sid.GetLength();
}

void edictptr_serialize(dg_field_stack_t * prefix, dg_fields *encoded, DG_Edict *obj, dg_field_t *field, byte *from, Packet *pkt)
{
    int i;
    edict_t *ent = (edict_t *) *(edict_t **) from;

    if (!ent || !ent->inuse) {
        zero_guid.Serialize(pkt);
	zero_sid.Serialize(pkt); // XXX: don't need to send this do we?
        return;
    }

    if (ent->is_distributed) {
	ent->guid.Serialize(pkt);
	DG_Edict *oobj = (DG_Edict *) g_ObjStore->Find(ent->guid);
	ASSERT(oobj);
	SID owner = oobj->GetSID();
	owner.Serialize(pkt);
    } else {
	ASSERT(DG_IsNonDistributed(ent->guid));
	ent->guid.Serialize(pkt);
	zero_sid.Serialize(pkt); // XXX: don't need to send this do we?
    }
}

void edictptr_deserialize(dg_field_stack_t * prefix, dg_fields *encoded, DG_Edict *obj, dg_field_t *field, Packet *pkt, byte *ptr_object, dg_fieldtype_t container, int ofs)
{
    guid_t guid(pkt);
    IPEndPoint owner(pkt);
    edict_t *ent;
    qboolean missing = true;

    // INVARIANT: edict pointers should only point to objects in the 
    // ObjectStore (i.e., g_edicts), not the PendingStore. See
    // DG_Edict::ResolveRef for an explanation.

    if (guid == zero_guid) {
        *(edict_t **) ptr_object = NULL;
        missing = false;
    }
    else {
        ent = DG_ExistsEntity(guid);
	
	if (ent) {
	    ASSERT((uint32)ent >= (uint32)g_edicts);
	    ASSERT((uint32)ent < (uint32)(g_edicts + game.maxentities));

            *(edict_t **) ptr_object = ent;
            missing = false;
        }
        else {
	    // Can't be missing pointers to non-distributed objects!
	    ASSERT(!DG_IsNonDistributed(guid));
	    
	    // Jeff 7/9/04: Just assign NULL and assume quake does the
	    // "right thing"
            *(edict_t **) ptr_object = (edict_t *) NULL; // 0x01;
        }
    }
    
    obj->RegisterRef(guid, owner, container == DF_CLIENT_PTR, ofs, missing);
}

///////////////////////////////////////////////////////////////////////////////
// DF_FUNC_PTR, DF_ITEM_PTR, DF_ITEM_ARMOR_PTR, DF_MMOVE_PTR, DF_MFRAME_PTR:

void genericptr_print(int indent, ostream& out, dg_field_stack_t * prefix, dg_fields *toprint, dg_field_t *field, byte *from)
{
    int i;
    void *ptr = (void *) *(byte **) from;
    if (!ptr) {
        out << "<NULL>";
        return;
    }

    switch (field->type) {
        case DF_FUNC_PTR:
            i = search_pointer(func_pointers, len_func_pointers, ptr);
	    out << "<FUNC:" << i << ">" << endl;
            break;
        case DF_CHAR_PTR:
            out << "\"" << *(char **)from << "\"";
            break;
        default:
            i = search_pointer(globals_pointers, len_globals_pointers, ptr);
	    out << "<PTR:" << i << ">" << endl;
            break;
    }
}

qboolean genericptr_is_dirty(dg_field_stack_t * prefix, dg_fields *isdirty, dg_field_t *field, byte *from, byte *to)
{
    return ( *(byte **) from  != *(byte **) to);
}
int genericptr_length(dg_field_stack_t * prefix, dg_fields *encoded, dg_field_t *field, byte *base) {
    return 4;
}
void genericptr_serialize(dg_field_stack_t * prefix, dg_fields *encoded, DG_Edict *obj, dg_field_t *field, byte *from, Packet *pkt)
{
    int i;
    void *ptr = (void *) *(byte **) from;
    if (!ptr) {
        pkt->WriteInt( -1);
        return;
    }

    switch (field->type) {
        case DF_FUNC_PTR:
            i = search_pointer(func_pointers, len_func_pointers, ptr);
            break;
        case DF_CHAR_PTR:
//	    printf("searching for %s\n", (char *) ptr);
            i = search_char_pointer((void **) static_strings, len_static_strings, ptr);
            if (i == -1) {
                i = search_char_pointer((void **) dynamic_strings, len_dynamic_strings, ptr);
                if (i != -1) {
                    i += len_static_strings;   // yet another HACK! :)
                }
            }
            break;
        default:
            i = search_pointer(globals_pointers, len_globals_pointers, ptr);
            break;
    }

    ASSERT(i >= 0);
    if (i < 0) {
        gi.dprintf("Warning: POINTER NOT FOUND while serializing...\n");
    }
    else {
        pkt->WriteInt( i);
    }
}
void genericptr_deserialize(dg_field_stack_t * prefix, dg_fields *encoded, DG_Edict *obj, dg_field_t *field, Packet *pkt, byte *ptr_object, dg_fieldtype_t container, int ofs)
{
    int i;
    i = pkt->ReadInt();
    if (i == -1) {
        *(void **) ptr_object = NULL;
        return;
    }

    switch (field->type) {
    case DF_FUNC_PTR:
        *(void **) ptr_object = func_pointers[i];
        break;
    case DF_CHAR_PTR:
        if (i < len_static_strings) {
            *(void **) ptr_object = static_strings[i];
        }
        else {
            *(void **) ptr_object = dynamic_strings[i - len_static_strings];
        }
    	break;
    default:
        *(void **) ptr_object = globals_pointers[i];
    }
}


///////////////////////////////////////////////////////////////////////////////
//
// Careful here..
//
//  case DF_CLIENT_PTR: DF_MOVEINFO: DF_MONSTERINFO: DF_PLAYER_STATE: DF_PMOVE_STATE:
//  case DF_CLIENT_PERSISTANT: DF_CLIENT_RESPAWN:

void object_init(dg_field_stack_t *s, dg_field_t *field)
{
    dg_field_t *start = typeinfo[field->type].subfield_list;
    dg_field_t *sub_field;

    int index = DG_GetFieldStackIndex(s);

    for (sub_field = start; sub_field->name ; sub_field++)
    {
	s->parts[index] = sub_field->name;

	string fieldname("");
	for (int i=0; i<=index; i++) {
	    if (i != 0) fieldname.append(".");
	    fieldname.append(s->parts[i]);
	}

	uint32 key = DG_FieldStackToKey(s);

	dg_cluster_t *cl;

	if (_dg_field_encoding.find(fieldname) == 
	    _dg_field_encoding.end()) {
	    //DB(0) << "unknown field: key=" << key << " " << fieldname << endl;
	    cl = NULL;
	} else {
	    cl = _dg_field_encoding[fieldname];
	    ASSERT(cl);

	    bool found = false;
	    for (int j=0; j < cl->num_fields; j++) {
		if (fieldname == string(cl->fields[j].name)) {
		    // remember the key for this fieldname
		    cl->fields[j].key = key;
		    found = true;
		    break;
		}
	    }
	    ASSERT(found);
	}
	
	dg_field_encoding[key] = cl;

	// Record the key => name mapping for debugging if necessary
	if (dg_fieldname_map.find(key) != dg_fieldname_map.end()) {
	    WARN << "two object fields map to the same key! key=" << key
		 << " field1=" << dg_fieldname_map[key]
		 << " field2=" << fieldname << endl;
	    ASSERT(0);
	}
	dg_fieldname_map[key] = fieldname;

        typeinfo[sub_field->type].init(s, sub_field);

	s->parts[index] = NULL;
    }
}

void object_print(int indent, ostream& out, dg_field_stack_t * prefix, dg_fields *toprint, dg_field_t *field, byte *from)
{
    dg_field_t *start = typeinfo[field->type].subfield_list;
    dg_field_t *sub_field;
    qboolean changed = false;

    // if there's a pointer at this location => follow the pointer...
    if (field->type == DF_CLIENT_PTR) {
        from = *(byte **) from;

        if (!from) {
            out << "<NULL>";
	    return;
	}
    }

    out << endl;
    for (int i=0; i<indent; i++) out << " ";
    out << "{" << endl;

    int index = DG_GetFieldStackIndex(prefix);

    indent += 2;
    for (sub_field = start; sub_field->name ; sub_field++)
    {
        byte *loc = from + sub_field->ofs;

	prefix->parts[index] = sub_field->name;
	uint32 key = DG_FieldStackToKey(prefix);

	string fieldname("");
	for (int i=0; i<=index; i++) {
	    if (i != 0) fieldname.append(".");
	    fieldname.append(prefix->parts[i]);
	}

	if (toprint && toprint->size() != 0 && 
	    toprint->find(key) == toprint->end())
	    continue;
    
	for (int i=0; i<indent; i++) out << " ";
	out << fieldname << "=";
        typeinfo[sub_field->type].print(indent, out, prefix, toprint, sub_field, loc);
	out << endl;
    }
    prefix->parts[index] = NULL;
    indent -= 2;

    for (int i=0; i<indent; i++) out << " ";
    out << "}";
}

qboolean object_is_dirty(dg_field_stack_t * prefix, dg_fields *isdirty, dg_field_t *field, byte *from, byte *to)
{
    dg_field_t *start = typeinfo[field->type].subfield_list;
    dg_field_t *sub_field;
    qboolean changed = false;

    // if there's a pointer at this location => follow the pointer...
    if (field->type == DF_CLIENT_PTR) {
        from = *(byte **) from;
        to = *(byte **) to;

	// this is bad... it means everything for the client is dirty, but
	// we can't really traverse it because one pointer is NULL...
        if (!from && to || from && !to)
            ASSERT(0);
	else if (!from || !to)
	    return false;
    }

    int index = DG_GetFieldStackIndex(prefix);

    for (sub_field = start; sub_field->name ; sub_field++)
    {
        byte *loc1 = from + sub_field->ofs, *loc2 = to + sub_field->ofs;

	prefix->parts[index] = sub_field->name;
	uint32 key = DG_FieldStackToKey(prefix);
    
        if (typeinfo[sub_field->type].is_dirty(prefix, isdirty, sub_field, loc1, loc2))
	{
	    //INFO << "isdirty: " << sub_field->name << endl;
	    isdirty->insert(key);
            changed = true;
        }
    }

    prefix->parts[index] = NULL;

    return changed;
}

int object_length(dg_field_stack_t * prefix, dg_fields *encoded, dg_field_t *field, byte *base) {
    dg_field_t *start = typeinfo[field->type].subfield_list;
    dg_field_t *sub_field;
    int length = 0;

    if (field->type == DF_CLIENT_PTR) {
        // if there's a pointer at this location => follow the pointer...
        base = *(byte **) base;
	length += sizeof(byte);
        if (!base) {
            return length;
	}
    }

    int index = DG_GetFieldStackIndex(prefix);

    for (sub_field = start; sub_field->name ; sub_field++)
    {
        byte *loc1 = base + sub_field->ofs;

	prefix->parts[index] = sub_field->name;
	uint32 key = DG_FieldStackToKey(prefix);

	if (encoded->size() != 0 && encoded->find(key) == encoded->end())
	    continue;
	
        length += typeinfo[sub_field->type].length(prefix, encoded, sub_field, loc1);
    }

    prefix->parts[index] = NULL;

    return length;
}
void object_serialize(dg_field_stack_t * prefix, dg_fields *encoded, DG_Edict *obj, dg_field_t *field, byte *from, Packet *pkt)
{
    dg_field_t *start = typeinfo[field->type].subfield_list;
    dg_field_t *sub_field;
    int i = 0;
    
    // if there's a pointer at this location => follow the pointer...
    if (field->type == DF_CLIENT_PTR) {
        guid_t guid;
	byte *p_client;
	
        p_client = *(byte **) from;
	/* Jeff 7/14/2004: Why are we using a guid here?
        if (!p_client) {
            zero_guid.Serialize(pkt); 
	    
            return;
        }
        else {
            DG_GetClientGUID(from, &guid);
            guid.Serialize(pkt);
	    from = p_client; // we dereference...
        }
	*/

	if (!p_client) {
            pkt->WriteByte((byte)0);
            return;
        }
        else {
	    pkt->WriteByte((byte)1);
	    from = p_client; // we dereference...
        }
    }

    int index = DG_GetFieldStackIndex(prefix);

    for (sub_field = start; sub_field->name ; sub_field++)
    {
	prefix->parts[index] = sub_field->name;
	uint32 key = DG_FieldStackToKey(prefix);

	if (encoded->size() != 0 && encoded->find(key) == encoded->end())
	    continue;

	/*
	if (obj->m_Entity && strstr(obj->m_Entity->classname, "item")) {
	    string fieldname("");
	    for (int i=0; i<=index; i++) {
		if (i != 0) fieldname.append(".");
		fieldname.append(prefix->parts[i]);
	    }
	    DB(0) << obj->GetGUID() << ": item=" << obj->m_Entity->classname 
		  << " encoding: " << fieldname << endl;
	}
	*/

	//INFO << "encoding: " << sub_field->name << endl;
        byte *loc1 = from + sub_field->ofs;
        typeinfo[sub_field->type].serialize(prefix, encoded, obj, sub_field, loc1, pkt);
    }

    prefix->parts[index] = NULL;
}

// For an edict, the location of the edict will be passed as the last parameter
void object_deserialize(dg_field_stack_t * prefix, dg_fields *encoded, DG_Edict *obj, dg_field_t *field, Packet *pkt, byte *ptr_object, dg_fieldtype_t container, int ofs)
{
    dg_field_t *start = typeinfo[field->type].subfield_list;
    dg_field_t *sub_field;
    guid_t guid;
    int length = 0;

    if (field->type == DF_EDICT_PTR || field->type == DF_CLIENT_PTR) {
	// now inside a different container, reset
	container = field->type;
	ofs = 0;
    }

    if (field->type == DF_CLIENT_PTR)  {
        // Jeff 7/14/2004: I don't think we need a guid here...
	/*
	// Another use of the placement new! :)
        (void) new(&guid) guid_t(pkt);

        if (guid == zero_guid) {
            *(gclient_t **) ptr_object = NULL;
            return; 
        }
        else {
	    // tricky. Before we call object_deserialize, this pointer has been
	    // set by the DG_ReceivePackets method. So, we can safely 
	    // dereference it...
            ptr_object = *(byte **) ptr_object; // dereference the pointer.
        }
	*/

	bool client = (bool)pkt->ReadByte();

	if (!client) {
            *(gclient_t **) ptr_object = NULL;
            return; 
        }
        else {
	    // tricky. Before we call object_deserialize, this pointer has been
	    // set by the DG_ReceivePackets method. So, we can safely 
	    // dereference it...
            ptr_object = *(byte **) ptr_object; // dereference the pointer.
        }
    }

    int index = DG_GetFieldStackIndex(prefix);

    for (sub_field = start; sub_field->name ; sub_field++)
    {
	prefix->parts[index] = sub_field->name;
	uint32 key = DG_FieldStackToKey(prefix);

	if (encoded->size() != 0 && encoded->find(key) == encoded->end())
	    continue;

	//INFO << "decoding: " << sub_field->name << endl;
        byte *loc1 = ptr_object + sub_field->ofs;
        typeinfo[sub_field->type].deserialize(prefix, encoded, obj, sub_field, pkt, loc1, container, ofs + sub_field->ofs);
    }

    prefix->parts[index] = NULL;
}
