/* vim: set sw=4 ts=4 noet: -*- Mode:c++; c-basic-offset:4; tab-width:4; indent-tabs-mode:t -*- */

#include <map>
#include <om/common.h>
#include <mercury/ID.h>
#include <util/ExpLog.h>
#include "dg.h"
#include "dg_bbox.h"
#include "dg_pubsubsim.h"

enum { CLS_PUB, CLS_SUB };
enum { OP_REG, OP_UNREG, OP_MATCH };
enum { TYPE_PLAYER, TYPE_MONSTER, TYPE_MISSILE, TYPE_ITEM };

// an "interest" of an object
struct bbox {
    vec3_t min, max;
    int initframe;

    bbox() : initframe(-1) {}
};

// the pub and the sub of an object
struct pubsub {
    bbox pub, sub;
    GUID guid, owner;
	int type;
    bool mark;

    pubsub() : guid(GUID_NONE), type(-1), mark(false) {}
    pubsub(GUID guid, GUID owner, int type) : 
		guid(guid), owner(owner), type(type), mark(false) {}
};

typedef map<GUID, pubsub, less_GUID> BBoxMap;
typedef BBoxMap::iterator BBoxMapIter;

struct PubSubCostEntry : public ExpLogEntry
{
    int frameno;
	uint32 objid;
	uint32 ownerid;
	char type, cls, op;
    vec3_t min, max;
	int lifetime;
	
	uint32 match_objid;
	char match_type;

	static char TypeToChar(int t) {
		int type;

		switch(t) {
		case TYPE_PLAYER:  type = 'p'; break;
		case TYPE_MONSTER: type = 'o'; break;
		case TYPE_MISSILE: type = 'm'; break;
		case TYPE_ITEM: type = 'i'; break;
		default: ASSERT(0);
		}

		return type;
	}
	
    PubSubCostEntry(int frameno, GUID& guid, GUID& owner, int t, int c, int o,
					vec3_t min, vec3_t max, int lifetime = 0,
					GUID match_guid = GUID_NONE, int mt = 0) :
		frameno(frameno), lifetime(lifetime) {

		type = TypeToChar(t);

		switch(c) {
		case CLS_PUB: cls = 'p'; break;
		case CLS_SUB: cls = 's'; break;
		default: ASSERT(0);
		}

		switch(o) {
		case OP_REG: op = 'r'; break;
		case OP_UNREG: op = 'u'; break;
		case OP_MATCH: op = 'm'; break;
		default: ASSERT(0);
		}

		VectorCopy(min, this->min);
		VectorCopy(max, this->max);

		objid = guid.GetLocalOID();
		ownerid = owner.GetLocalOID();
		if (op == 'm') {
			match_objid = match_guid.GetLocalOID();
			match_type  = TypeToChar(mt);
		}
			
	}
    PubSubCostEntry() {}
    virtual ~PubSubCostEntry() {}
    
    uint32 Dump(FILE *fp) {
		int sz = 0;
		sz += fwrite((void *)&frameno, sizeof(int), 1, fp);
		sz += fwrite((void *)&objid, sizeof(int), 1, fp);
		sz += fwrite((void *)&ownerid, sizeof(int), 1, fp);
		sz += fwrite((void *)&type, sizeof(char), 1, fp);
		sz += fwrite((void *)&cls, sizeof(char), 1, fp);
		sz += fwrite((void *)&op, sizeof(char), 1, fp);
		sz += fwrite((void *)&min, sizeof(vec3_t), 1, fp);
		sz += fwrite((void *)&max, sizeof(vec3_t), 1, fp);
		if (op == 'u' || op == 'm') {
			sz +=  fwrite((void *)&lifetime, sizeof(int), 1, fp);
		}
		if (op == 'm') {
			sz += fwrite((void *)&match_objid, sizeof(int), 1, fp);
			sz += fwrite((void *)&match_type, sizeof(char), 1, fp);
		}
		/*
		sz += fprintf(fp, "%d\t%d\t%c\t%c\t%c\t%.5f\t%.5f",
					  frameno, objid, type, cls, op, xrange, yrange);
		if (op == 'u' || op == 'm')
			sz += fprintf(fp, "\t%d", lifetime);
		if (op == 'm')
			sz += fprintf(fp, "\t%d\t%c", match_objid, match_type);
		sz += fprintf(fp, "\n");
		*/
		return sz;
    }
};

struct PubSubLifetimeEntry : public ExpLogEntry
{
	uint32 objid;
    char type, cls;
	int life;

    PubSubLifetimeEntry(GUID& guid, int t, int c, int life) : life(life) {
		objid = guid.GetLocalOID();

		switch(t) {
		case TYPE_PLAYER: type = 'p'; break;
		case TYPE_MONSTER: type = 'o'; break;
		case TYPE_MISSILE: type = 'm'; break;
		case TYPE_ITEM: type = 'i'; break;
		default: ASSERT(0);
		}
		
		switch(c) {
		case CLS_PUB: cls = 'p'; break;
		case CLS_SUB: cls = 's'; break;
		default: ASSERT(0);
		}
	}
    PubSubLifetimeEntry() {}
    virtual ~PubSubLifetimeEntry() {}
    
    uint32 Dump(FILE *fp) {
		return fprintf(fp, "%d\t%c\t%c\t%d\n", objid, type, cls, life);
    }
};

// parameters
static float subDistPredict; // how far to predict subscriptions
static float pubDistPredict; // how far to predict publications
static bool  ignoreSelfPubs; // ignore pubs that we generated ourselves

// local variables
IMPLEMENT_EXPLOG(PubSubCostsLog, PubSubCostEntry);
IMPLEMENT_EXPLOG(PubSubLifetimeLog, PubSubLifetimeEntry);
static bool started = false;
static BBoxMap imap;
static int frameno;

static vec3_t worldmin, worldmax, worldrange;

///////////////////////////////////////////////////////////////////////////////

// get the type of an entity
static int EntityType(edict_t *ent);
// log that a particular cost was incurred
static void LogCost(int frameno, GUID& guid, GUID& owner, 
					int type, int cls, int op,
					vec3_t min, vec3_t max,
					int lifetime = 0, GUID match_guid = GUID_NONE, 
					int match_type = -1);
// log the lifetime of a particular cls instance in #frames (pub, sub)
static void LogLife(GUID& guid, int type, int cls, int life);

// true if box completely contains (min, max)
static bool BBoxContains(bbox *box, vec3_t min, vec3_t max);
// true if box and (min, max) overlap (or touch)
static bool BBoxOverlaps(bbox *box, vec3_t min, vec3_t max);
// return the amount by which box1 and box2 overlap along a dimension
// (0=x, 1=y, 2=z) assumes BBoxOverlaps(bbox1, bbox2)
static float Overlap(bbox *box1, bbox *box2, int index);
// return the bbox (min, max) that contains the overlapping portion
// of bbox1 and bbox2
static float Overlap(vec3_t min, vec3_t max, bbox *box1, bbox *box2);

// submit a pub or sub to the virtual system and log its cost
//
// cls     = { CLS_PUB, CLS_SUB }
// ent     = the object sending the xub
// xub     = the bbox pub or sub datastructure to modify
// predict = the radius to predict that the object might move
static void Submit(int cls, edict_t *ent, bbox *xub, float predict);

///////////////////////////////////////////////////////////////////////////////

static void Init()
{
    if (started) return;
    started = true;
	INIT_EXPLOG(PubSubCostsLog, g_LocalSID);
	INIT_EXPLOG(PubSubLifetimeLog, g_LocalSID);

    subDistPredict = g_QuakePreferences.pubsubsim_subdist;
    pubDistPredict = g_QuakePreferences.pubsubsim_pubdist;
	ignoreSelfPubs = g_QuakePreferences.pubsubsim_noselfpub;

    DG_GetPrecompWorldBBox(worldmin, worldmax);
    worldrange[0] = worldmax[0] - worldmin[0];
    worldrange[1] = worldmax[1] - worldmin[1];
	worldrange[2] = worldmax[2] - worldmin[2];

	/*
	// XXX TEST

	bbox box1, box2;

#define s(b, x, y, z) { \
 b[0] = (x); \
 b[1] = (y); \
 b[2] = (z); \
}

	s(box1.min, 0, 0, 0);
	s(box1.max, 0, 0, 0);

	s(box2.min, -2, -2, -2);
	s(box2.max, 2, 2, 1);

	DBG << "1 contains 2? " << BBoxContains(&box1, box2.min, box2.max) << endl;
	DBG << "1 overlaps 2? " << BBoxOverlaps(&box1, box2.min, box2.max) << endl;
	DBG << "overlap x=" << Overlap(&box1, &box2, 0) << endl;
	DBG << "overlap y=" << Overlap(&box1, &box2, 1) << endl;
	exit(0);
	*/
}

void DG_EstimatePubSubCostsHalt()
{
	//
    // dump the lifetimes of remaining pubs/subs
    //
    for (BBoxMapIter it = imap.begin(); it != imap.end(); it ++) {
		int type = it->second.type;
		GUID guid = it->second.guid;
		if (it->second.pub.initframe > 0) {
			bbox *pub = &(it->second.pub);
			LogLife(guid, type, CLS_PUB, frameno - pub->initframe);
		}
		if (it->second.sub.initframe > 0) {
			bbox *sub = &(it->second.sub);
			LogLife(guid, type, CLS_SUB, frameno - sub->initframe);
		}
    }

	FLUSH_ALL();
}

void DG_EstimatePubSubCosts(int fno)
{
    Init();

	//DBG << "called " << fno << endl;
    
    int i;
	edict_t *ent;
    frameno = fno;
    
    // unmark "alive objects"
    for (BBoxMapIter it = imap.begin(); it != imap.end(); it++) {
		it->second.mark = false;
    }
    
    ent = &g_edicts[0];
    for (i = 0; i < game.maxentities; i++, ent++) {
		// don't care about non-distributed objects
		if ( !ent->inuse || DG_IsNonDistributed(ent->guid) ) {
			continue;
		}
	
		GUID guid = ent->guid;
	
		//
		// figure out the new objects
		//
		if (imap.find(guid) == imap.end()) {
			// new object
			imap[guid] = pubsub(ent->guid, 
								ent->owner ? ent->owner->guid : GUID_NONE,
								EntityType(ent));
		}

		//DBG << "adding guid: " << guid << endl;
	
		imap[guid].mark = true;
    }

    //
    // delete objects that no longer exist
    //
    for (BBoxMapIter it = imap.begin(); it != imap.end(); /* !!! */) {
		BBoxMapIter oit = it;
		oit++;

		if (! it->second.mark ) {
			imap.erase(it);
			GUID guid  = it->second.guid;
			GUID owner = it->second.owner;
			int type   = it->second.type;
			if (it->second.pub.initframe >= 0) {
				bbox *pub = &(it->second.pub);
				LogCost(fno, guid, owner, type, CLS_PUB, OP_UNREG, 
						pub->min, pub->max,
						frameno - it->second.pub.initframe);
				LogLife(guid, type, CLS_PUB, frameno - pub->initframe);
			}
			if (it->second.sub.initframe >= 0) {
				bbox *sub = &(it->second.sub);
				LogCost(fno, guid, owner, type, CLS_SUB, OP_UNREG, 
						sub->min, sub->max,
						frameno - it->second.sub.initframe);
				LogLife(guid, type, CLS_SUB, frameno - sub->initframe);
			}

			it = oit;
		} else {
			it++;
		}
    }

    //
    // figure out who needs to submit a new pub, sub
    //
	ent = &g_edicts[0];
    for (i = 0; i < game.maxentities; i++, ent++) {
		// don't care about non-distributed objects
		if ( !ent->inuse || DG_IsNonDistributed(ent->guid) ) {
			continue;
		}

		GUID guid = ent->guid;

		//DBG << "looking up guid: " << guid << endl;

		BBoxMapIter p = imap.find(guid);
		ASSERT(p != imap.end());

		bbox *pub = &(p->second.pub);
		bbox *sub = &(p->second.sub);

		int type = EntityType(ent);

		if (pub->initframe < 0 ||
			!BBoxContains(pub, ent->s.origin, ent->s.origin)) {
			float predict;

			/*
			DBG << "class=" << ent->classname
				<< " pubdiff old=" << pub->max << " " << pub->min
				<< " new=" << ent->s.origin << endl;
			*/

			if (type == TYPE_PLAYER || 
				type == TYPE_MONSTER ||
				type == TYPE_MISSILE) {
				predict = pubDistPredict;
			} else {
				// don't predict for items, they don't change
				predict = 0;
			}

			Submit( CLS_PUB, ent, pub, predict);
		}

		if (type != TYPE_ITEM) {

			vec3_t min, max;
			DG_GetBoundingBox(ent->s.origin, min, max);

			if (sub->initframe < 0 ||
				!BBoxContains(sub, min, max)) {
				float predict;
				
				if (type == TYPE_PLAYER || type == TYPE_MONSTER) {
					predict = subDistPredict;
				} else {
					// don't predict for missiles/items
					predict = 0;
				}
				
				Submit( CLS_SUB, ent, sub, predict );
			}

		}
    }
}

///////////////////////////////////////////////////////////////////////////////

void Submit(int cls, edict_t *ent, bbox *xub, float predict)
{
    vec3_t pmin, pmax;

	GUID guid   = ent->guid;
	GUID owner  = ent->owner ? ent->owner->guid : GUID_NONE;
	int type    = EntityType(ent);
	vec3_t orig; 
	VectorCopy(ent->s.origin, orig);

    // calculate new xub

#define ERR_EPSILON 128  // (some items are a little "in the floor")

    if (cls == CLS_PUB) {
		for (int i=0; i<2; i++) {
			pmin[i] = MAX( orig[i] - predict, worldmin[i]-ERR_EPSILON );
			pmax[i] = MIN( orig[i] + predict, worldmax[i]+ERR_EPSILON );
		}
		// don't predict too far along-z
		pmin[2] = MAX( orig[2] - MIN(predict, 32), worldmin[2]-ERR_EPSILON );
		pmax[2] = MIN( orig[2] + MIN(predict, 32), worldmax[2]+ERR_EPSILON );

		/*
		if (type == TYPE_ITEM) {
			DBG << "min=" << pmin << " max=" << pmax << " orig=" << orig
				<< " predict=" << predict 
				<< " max[0]-min[0]=" << (pmax[0]-pmin[0]) << endl;
		}
		*/
		
    } else if (cls == CLS_SUB) {
		// calculate new sub
		DG_GetPrecompBoundingBox(orig, pmin, pmax, 
								 // grow the predicted sub location by 
								 // this much (x2 because its a diameter)
								 2*predict, 
								 // limit the bbox growth in the z-dimension
								 // so we don't go through ceilings or floors
								 32.0);
    } else {
		ASSERT(0);
	}

    // log the lifetime of the previous xub
    if (xub->initframe >= 0) {
		LogLife(guid, type, cls, frameno - xub->initframe);
    }

    // log that we had to unregister the old xublication
    if (xub->initframe >= 0) {
		LogCost(frameno, guid, owner, type, cls, OP_UNREG, 
				xub->min, xub->max,
				frameno - xub->initframe);
    }
	
    VectorCopy(pmin, xub->min);
    VectorCopy(pmax, xub->max);
    xub->initframe = frameno;
    
    // log that we are submitting a new xublication
    LogCost(frameno, guid, owner, type, cls, OP_REG, pmin, pmax);

    // now log each opposite cls that it matches...
    for (BBoxMapIter it = imap.begin(); it != imap.end(); it++) {
		bbox *other;
		GUID recvguid, sendguid;
		GUID recvowner, sendowner;
		int recvtype, sendtype;
		if (cls == CLS_PUB) {
			other = &(it->second.sub);
			recvtype = it->second.type;
			recvguid = it->second.guid;
			recvowner = it->second.owner;

			sendtype = type;
			sendguid = guid;
			sendowner = owner;
		} else {
			other = &(it->second.pub);
			recvtype = type;
			recvguid = guid;
			recvowner = owner;

			sendtype = it->second.type;
			sendguid = it->second.guid;
			sendowner = it->second.owner;
		}
		
		// hasn't sent anything yet
		if (other->initframe < 0)
			continue;
		// don't match pubs that we send ourselves
		if (ignoreSelfPubs && guid == it->second.guid)
			continue;
	
		if (BBoxOverlaps(other, xub->min, xub->max)) {
			// the cost of the match is proportional to how many matchers
			// the xub reached...
			//
			// the "recv*" logged is always that of the object receiving
			// the pub. the "send*" is that of the object that generated
			// the matching sub.
			//
			// the "class" is the thing that initiated the match
			//
			// we record the lifetime of the thing that did the match
			// (since the lifetime of the current thing is always 0)
			vec3_t overlap_min, overlap_max;
			Overlap(overlap_min, overlap_max, other, xub);

			LogCost(frameno, recvguid, recvowner, recvtype, cls, OP_MATCH,
					overlap_min, overlap_max,
					frameno - other->initframe,
					sendguid, sendtype);
		}
    }
}

///////////////////////////////////////////////////////////////////////////////

int EntityType(edict_t *ent)
{
    if (streq(ent->classname, "bot") ||
		streq(ent->classname, "player") ) {
		if (ent->client->is_monster)
			return TYPE_MONSTER;
		else
			return TYPE_PLAYER;
    } else if ( streq(ent->classname, "bolt") ||
				streq(ent->classname, "grenade") ||
				streq(ent->classname, "hgrenade") ||
				streq(ent->classname, "rocket") ||
				streq(ent->classname, "bfg blast") ) {
		return TYPE_MISSILE;
    } else {
		return TYPE_ITEM;
    }
}

extern bool DG_AllBotsSpawned();

void LogCost(int frameno, GUID& guid, GUID& owner, int type, int cls, int op,
			 vec3_t min, vec3_t max,
			 int lifetime, GUID match_guid, int match_type)
{
	// wait until everyone has joined before logging
	if (!DG_AllBotsSpawned()) return;

	for (int i=0; i<3; i++) {
		// in units from beginning
		min[i] -= worldmin[i];
		max[i] -= worldmin[i];
		// normalize to 1
		min[i] /= worldrange[i];
		min[i] = MAX(0, min[i]);
		min[i] = MIN(1, min[i]);
		max[i] /= worldrange[i];
		max[i] = MAX(0, max[i]);
		max[i] = MIN(1, max[i]);
	}

    PubSubCostEntry e(frameno, guid, owner, type, cls, op, 
					  min, max, 
					  lifetime, match_guid, match_type);

	LOG(PubSubCostsLog, e);
}

void LogLife(GUID& guid, int type, int cls, int life)
{
	// wait until everyone has joined before logging
	if (!DG_AllBotsSpawned()) return;

    PubSubLifetimeEntry e(guid, type, cls, life);

	LOG(PubSubLifetimeLog, e);
}

///////////////////////////////////////////////////////////////////////////////

bool BBoxContains(bbox *box, vec3_t min, vec3_t max)
{
    for (int i=0; i<3; i++) {
		if (box->min[i] > min[i] || box->max[i] < max[i]) {
			return false;
		}
    }

    return true;
}

static bool BBoxOverlapsHelper(vec3_t min1, vec3_t max1,
							   vec3_t min2, vec3_t max2)
{
	for (int i=0; i<3; i++) {
		if (!( min1[i] <= min2[i] && min2[i] <= max1[i] || 
			   min1[i] <= max2[i] && max2[i] <= max1[i] )) {
			return false;
		}
    }

	return true;
}

bool BBoxOverlaps(bbox *box, vec3_t min, vec3_t max)
{
    return BBoxOverlapsHelper(box->min, box->max, min, max) ||
		BBoxOverlapsHelper(min, max, box->min, box->max);
}

static float OverlapHelper(bbox *box1, bbox *box2, int i)
{
    if (box1->min[i] <= box2->min[i] && 
		box2->min[i] <= box1->max[i]) {
		return MIN(box1->max[i], box2->max[i]) - box2->min[i];
    } else if (box1->min[i] <= box2->max[i] && 
			   box2->max[i] <= box1->max[i]) {
		return box2->max[i] - MAX(box1->min[i], box2->min[i]);
    }
    return 0;
}

float Overlap(bbox *box1, bbox *box2, int index)
{
    float ret;

    ret = OverlapHelper(box1, box2, index);
    if (ret > 0) return ret;
    ret = OverlapHelper(box2, box1, index);
    return ret;
}

float Overlap(vec3_t min, vec3_t max, bbox *box1, bbox *box2)
{
	// This function already assumes the boxes overlap!

	for (int i=0; i<3; i++) {
		// The overlapping min point is larger of the two minimums
		if (box1->min[i] > box2->min[i]) {
			min[i] = box1->min[i];
		} else {
			min[i] = box2->min[i];
		}

		// The overlapping max point is the smaller of the two maximums
		if (box1->max[i] < box2->max[i]) {
			max[i] = box1->max[i];
		} else {
			max[i] = box2->max[i];
		}
	}

}
