/* vim: set sw=4 ts=4 noet: -*- Mode:c++; c-basic-offset:4; tab-width:4; indent-tabs-mode:t -*- */

#include <map>
#include <om/common.h>
#include <mercury/ID.h>
#include <util/ExpLog.h>
#include "dg.h"
#include "dg_bbox.h"
#include "dg_playablemaparea.h"

const bbox_t worldvol = {
#ifndef USE_SHARED_MEMORY
    0,
#endif
    { -4096, -4096, -4096 },
    { 4096, 4096, 4096 }
};

#define CLOSE_TO_GROUND 512 //10

struct PlayableAreaEntry : public ExpLogEntry
{
    vec3_t pt;
    bool v;

    PlayableAreaEntry() { }
    virtual ~PlayableAreaEntry() {}
    
    uint32 Dump(FILE *fp) {
	//return fprintf(fp, "%.2f,%.2f,%.2f\t%d\n", pt[0], pt[1], pt[2], v?1:0);
	return fprintf(fp, "%d\n", v?1:0);
    }
};

struct VisibleAreaEntry : public ExpLogEntry
{
    float uvol;
    float vol;

    VisibleAreaEntry() {}
    virtual ~VisibleAreaEntry() {}

    uint32 Dump(FILE *fp) {
	return fprintf(fp, "%f\t%f\n", vol, uvol);
    }
};

DECLARE_EXPLOG(PlayableAreaLog, PlayableAreaEntry);
IMPLEMENT_EXPLOG(PlayableAreaLog, PlayableAreaEntry);

DECLARE_EXPLOG(VisibleAreaLog, VisibleAreaEntry);
IMPLEMENT_EXPLOG(VisibleAreaLog, VisibleAreaEntry);

static bool IsPlayable(vec3_t pt);

void DG_CalcPlayableMapArea(int samples)
{
    vec3_t range;
    for (int i=0; i<3; i++)
	range[i] = worldvol.max[i] - worldvol.min[i];
    
    INIT_EXPLOG(PlayableAreaLog, g_LocalSID);

    int hits = 0;

    for (int i=0; i<samples; i++) {
	PlayableAreaEntry e;
	for (int j=0; j<3; j++)
	   e.pt[j] = drand48()*range[j] + worldvol.min[j];

	e.v = IsPlayable(e.pt);
	hits += e.v;

	if (i % 10000 == 0)
	    cerr << "+";
	LOG(PlayableAreaLog, e);
    }

    cerr << endl;

    FLUSH(PlayableAreaLog);

    float estvol = range[0]*range[1]*range[2]*hits/samples;

    cerr << "hits: " << hits << "/" << samples 
	 << "  est. vol: " << estvol << endl;

    //////////

    INIT_EXPLOG(VisibleAreaLog, g_LocalSID);
    
    for (int i=0; i<10000/*XXX: config*/; i++) {
	vec3_t pt;
	bbox_t bbox;
	for (int j=0; j<3; j++)
	    pt[j] = drand48()*range[j] + worldvol.min[j];
	VisibleAreaEntry e;

	DG_GetBoundingBox(pt, bbox.min, bbox.max);
	e.uvol = BBoxVolume(&bbox); 
	e.vol = e.uvol/estvol;
	
	if (e.vol > 0) {
	    if (i % 100 == 0)
		cerr << ".";
	    
	    LOG(VisibleAreaLog, e);
	} else {
	    i--; // bad point, try again
	}
    }

    cerr << endl;

    FLUSH(VisibleAreaLog);

}

/**
 * A point is playable if it is in the map and is at most CLOSE_TO_GROUND
 * units away from the ground.
 */
bool IsPlayable(vec3_t pt)
{
    static vec3_t down = { RADIAN(0), RADIAN(0), RADIAN(-90) };
    vec3_t end;
    trace_t tr;
    
    VectorMA(pt, CLOSE_TO_GROUND+0.0001, down, end);
    tr = gi.trace( pt, vec3_origin, vec3_origin, end, 0, MASK_OPAQUE );

    if ( tr.startsolid || tr.fraction >= 1.0 ) {
	return false;
    }

    return true;
}
