/* vim: set sw=4 ts=8 sts=4 noet filetype=cpp: -*- Mode:c++; c-basic-offset:4; tab-width:8; indent-tabs-mode:t -*- */

#include <qcommon/qcommon.h>
#include <game/g_local.h>
#include "dg.h"
#include "dg_logs.h"
#include "dg_bbox.h"
#include "dg_pubsubsim.h"
#include "dg_playablemaparea.h"
#ifdef DEBUG
#include "dg_test.h"
#endif
#include "DG_DummyManager.h"
#include <game/inc_registry.h>
#include <util/Benchmark.h>
#include <om/Manager.h>
#include <mercury/Message.h>
#include <om/OMMessage.h>
#include <wan-env/WANScheduler.h>
#include <wan-env/RealNet.h>

#pragma warning(once: 4133)

#define CHECK_INVARIANTS() \
if (ManagerParams::CHECK_INVARIANTS) { g_QuakeAdaptor->CheckInvariants(); }

extern void SV_DemoInit (char *name);
extern void SV_ServerStop_f ();

// counter for assigning localOIDs to nondistributed GUIDs
unsigned long g_NonDistLocalOIDCounter = 1;
// average interval between frames, updated dynamically
double g_AvgFrameInterval = 100;
// index of the hub to be used for DHT
int g_DHTHubIndex;
// index of the hub used for striping
int g_StripeHubIndex;

// Jeff 6/30/2004:
// Ack. This code is turning into (more) spagetti. :P We need to componentize
// all the DG stuff badly. I can hardly understand anything that is going on
// with all the hacky global variables that are touched everywhere now --
// in the DG code, in the acebot code, in the game DLL code, *and* in the
// server code.

//  Ashwin [2/21/2004]
// Shadow array of entities in order to figure out "writes" to the objects
// We need this because we don't want to change every line of quake code for 
// detecting the writes/changes to the individual objects.
//
// Also need shadow array of clients.
//

edict_t *g_shadow_edicts;
gclient_t *g_shadow_clients;

Manager *g_Manager;
QuakeAdaptor *g_QuakeAdaptor;
DG_ObjStore *g_ObjStore;

/* 
 * Hack for now. Should implement some data structure which allows 
 * for fast queries and reasonable add/delete costs 
 */
#define MAX_REMOTE_CLIENTS 1024

typedef struct {
    gclient_t client;
    guid_t    guid;
    qboolean  inuse;
} r_gclient_t;

r_gclient_t dg_remote_clients[MAX_REMOTE_CLIENTS];

extern bool DG_AllBotsSpawned();
static void DG_DisplayLookingAt();
static void DG_RecordObjectInterests(bool include_items = false);
static void DG_CheckBBoxAccuracy();

///////////////////////////////////////////////////////////////////////////////

// hacky "master" server stuff for bootstrapping..
bool inited = false, done_migration = false;
typedef map<SID, bool, less_SID> EPMap;
typedef EPMap::iterator EPMapIter;
EPMap g_SlaveMap;


WANScheduler *g_Scheduler = 0;
RealNet *g_Network = 0;
IPEndPoint g_MasterIP;

bool DG_AllSlavesJoined()
{
    return (g_SlaveMap.size() >= g_QuakePreferences.nservers - 1);
}

void master_init() {
    g_MasterIP = IPEndPoint(g_Preferences.hostname, 50000);
    g_Scheduler = new WANScheduler ();
    g_Network = new RealNet (g_Scheduler, g_MasterIP);
    g_Network->StartListening (PROTO_TCP);
}

void master_check_data()
{
    bool nomsg = false;
    IPEndPoint from((uint32)0, 0);
    Message *msg = 0;

    for (int i = 0; i < 10; i++) {
	ConnStatusType status = g_Network->GetNextMessage(&from, &msg);

	switch (status) {
	case CONN_NEWINCOMING:
	case CONN_OK:
	    {
		ASSERT(msg->GetType () == MSG_SIDLIST);
		MsgSIDList *ping = (MsgSIDList *)msg;

		for (SIDSetIter it = ping->sidlist.begin();
			it != ping->sidlist.end(); it++) {
		    EPMapIter p = g_SlaveMap.find(*it);
		    if (p == g_SlaveMap.end()) {
			DB(-2) << "received slave message from " 
			    << *it << endl;
			g_SlaveMap.insert(EPMap::value_type(*it, true));
		    }
		    else {
			WARN << "received a duplicate announcement for "
			    << "the slave " << *it << " from " 
			    << from << endl;
		    }
		}

		g_Network->FreeMessage(msg);
	    }
	    break;
	case CONN_CLOSED:
	case CONN_ERROR:
	    break;

	case CONN_NOMSG:
	    nomsg = true;
	    break;
	default:
	    WARN << "BUGBUG: Hmm... got weird connection status." << endl;
	    break;
	}

	if (nomsg)
	    return;
    }
}

DG_Edict *master_get_entity(int *start) {
    int i = *start;
    edict_t *ent;
    DG_Edict *dg_obj;

    do {
	if (i >= game.maxentities) return NULL;
	ent = &g_edicts[i];
	i++;
	*start = i;
	dg_obj = (DG_Edict *) g_ObjStore->Find(ent->guid);
    } while (!ent->inuse || !dg_obj || dg_obj->IsReplica());

    return dg_obj;
}

void DG_PerformRegionBasedMigration()
{
    sid_t target;

    // Chop the world to be partitioned into all the other slave servers
    int boxes[3];
    DG_ChopWorld(g_QuakePreferences.nservers - 1, boxes);
    INFO << "World partition for items stats - xslices=" << boxes[0] << " yslices=" << boxes[1] << " zslices=" << boxes[2] << endl;

    // assign boxes to various servers first. 
    int nboxes = boxes[0] * boxes[1] * boxes[2];
    map<int, int>  boxToServerMap;

    for (int i = 0; i < nboxes; i++) {
	int serverToAssignTo = rand() % (g_QuakePreferences.nservers - 1);

	// INFO << "assigned server " << serverToAssignTo << " for box " << i << endl;
	boxToServerMap.insert(map<int, int>::value_type(i, serverToAssignTo));
    }

    // create the list of objects for each server
    typedef list<DG_Edict *> ObjList;
    typedef ObjList::iterator ObjListIter;

    ObjList *slaveObjects = new ObjList[g_QuakePreferences.nservers - 1];

    int i = 1;
    while (true) { 
	DG_Edict *dg_obj = master_get_entity(&i);
	if (!dg_obj) 
	    break;

	edict_t *ent = dg_obj->GetEntity();
	int boxID = DG_GetChoppedWorldBox(ent->s.origin, boxes);

	if (boxID >= nboxes || boxID < 0) {
	    // some objects are created in wierd locations... just leave
	    // them here...
	    continue;
	}

	// INFO << "Entity[" << ent->s.number << "] " << (ent->classname ? ent->classname : "(null)") << 
	// 	" belongs to box " << boxID << endl;
	ASSERT(boxID < nboxes);
	ASSERT(boxToServerMap.find(boxID) != boxToServerMap.end());
	ASSERT(boxToServerMap[boxID] < g_QuakePreferences.nservers - 1);

	slaveObjects[ boxToServerMap[boxID] ].push_back(dg_obj);
    }

    DB_DO(1) {
	for (int i = 0; i < g_QuakePreferences.nservers - 1; i++) {
	    //DB(1) << "TO slave " << i << ": objects --- " ;
	    for (ObjListIter it = slaveObjects[i].begin(); it != slaveObjects[i].end(); it++)
	    {
		edict_t *ent = (*it)->GetEntity();
		// cerr << merc_va("%d;%s ", ent->s.number, ent->classname ? ent->classname : "(null)");
	    }
	    // cerr << endl;
	}
    }

    // Finally, perform migration for everybody
    int serverIndex = 0;
    for (EPMapIter it = g_SlaveMap.begin(); it != g_SlaveMap.end(); it++, serverIndex++) {
	target = it->first;
	ASSERT(target != g_LocalSID);

	//			DB(1) << "migrating obj: " << dg_obj->GetGUID() << endl;
	for (ObjListIter oit = slaveObjects[serverIndex].begin(); oit != slaveObjects[serverIndex].end(); oit++)
	{
	    edict_t *ent = (*oit)->GetEntity();
	    // INFO << "migrating object[" << ent->s.number << "] " << (ent->classname ? ent->classname : "(null)")
	    // 	<< " to target " << target << endl;
	    g_Manager->Migrate(*oit, target);
	}
    }

    boxToServerMap.clear();
    for (int i = 0; i < g_QuakePreferences.nservers - 1; i++) 
	slaveObjects[i].clear();
    delete[] slaveObjects;
}

void master_do_migration()
{
    sid_t target;
    int i = 1;

    if (g_QuakePreferences.item_partition_method == 'b') {
	DG_PerformRegionBasedMigration();
    }
    else if (g_QuakePreferences.item_partition_method == 'r') {

	while (true) {
	    DG_Edict *dg_obj = master_get_entity(&i);

	    // assign this entity to me; so don't do anything!		
	    for (EPMapIter it = g_SlaveMap.begin(); it != g_SlaveMap.end(); it++) {
		dg_obj = master_get_entity(&i);
		if (!dg_obj) break;
		target = it->first;

		ASSERT(target != g_LocalSID);
		//			DB(1) << "migrating obj: " << dg_obj->GetGUID() << endl;
		g_Manager->Migrate(dg_obj, target);
	    }
	    if (!dg_obj) break;
	}
    }
}

// send a nonce to the stupid master
void slave_do_once()
{
    g_Scheduler = new WANScheduler ();
    g_Network = new RealNet (g_Scheduler, g_LocalSID /* dummy */);
    g_Network->StartListening (PROTO_TCP_PASSIVE);

    g_MasterIP = IPEndPoint(g_QuakePreferences.master_ip, 50000);
    if (g_MasterIP == SID_NONE)
	Debug::die("bad master-ip: %s", g_QuakePreferences.master_ip);

    INFO << "setting master ip = " << g_MasterIP << endl;

    MsgSIDList ping (g_LocalSID);
    ping.sidlist.insert (g_LocalSID);
    g_Network->SendMessage (&ping, &g_MasterIP, PROTO_TCP_PASSIVE);

    g_Network->StopListening();
    delete g_Network;
}

///////////////////////////////////////////////////////////////////////////////

extern int SV_GetCurrentFrame();

void DumpObjectStoreDetails(TimeVal now)
{
    DGStoreLogEntry e;

    edict_t *ent;

    GObject *obj;

    g_Manager->Lock(); // XXX HACK
    g_ObjStore->Begin();
    while ((obj = g_ObjStore->Next())) {
	e.no++;
	ent = debug_get_entity(obj);

	if (!obj->IsReplica()) {
	    e.np++;
	    if (!ent->classname) continue;
	    if (strstr(ent->classname, "bot") ||
		    streq(ent->classname, "player"))
		if (ent->client->is_monster)
		    e.p_mo++;
		else
		    e.p_pl++;
	    else if (strstr(ent->classname, "item") ||
		    strstr(ent->classname, "ammo") ||
		    strstr(ent->classname, "weapon"))
		e.p_it++;
	    else if (strstr(ent->classname, "missile") ||
		    strstr(ent->classname, "bolt"))
		e.p_mi++;
	    else
		e.p_ot++;
	}
	else {
	    e.nr++;
	    if (!ent->classname) continue;
	    if (strstr(ent->classname, "bot") ||
		    streq(ent->classname, "player"))
		if (ent->client->is_monster)
		    e.r_mo++;
		else
		    e.r_pl++;
	    else if (strstr(ent->classname, "item") ||
		    strstr(ent->classname, "ammo") ||
		    strstr(ent->classname, "weapon"))
		e.r_it++;
	    else if (strstr(ent->classname, "missile") ||
		    strstr(ent->classname, "bolt"))
		e.r_mi++;
	    else
		e.r_ot++;
	}
    }
    g_Manager->Unlock();

    e.frameno = SV_GetCurrentFrame();

    LOG(DGStoreLog, e);
}

///////////////////////////////////////////////////////////////////////////////

void DG_GameFunc(int cmd)
{

    // This shouldn't run on clients
    if (g_QuakePreferences.client)
	return;

    CHECK_INVARIANTS();

    START(DGMAIN);

    START(DGMAIN::MasterSlaveCrap);

    // XXX HACK: run a couple frames for the world to startup before doing
    // the migration...
    static uint32 init_frames = 10;
    if (!g_QuakePreferences.disable_colyseus &&
	    init_frames <= SV_GetCurrentFrame()) {

	//
	// Master/Slave Initialization Code
	//
	if (g_QuakePreferences.is_master) {
	    if (g_QuakePreferences.nservers == 1) {
		done_migration = true;
	    }

	    if (!done_migration) {
		if (!inited) {
		    master_init();
		    inited = true;
		}

		master_check_data();
                TimeVal now;
	        OS::GetCurrentTime(&now);
	        PERIODIC2(5000, now, {
		INFO << " nservers = " << g_QuakePreferences.nservers 
		    << " slaveMap.size = " << g_SlaveMap.size() << endl;
	        });
		if (g_SlaveMap.size() >= g_QuakePreferences.nservers - 1) {
		    master_do_migration();
		    done_migration = true;
		    INFO << "Master migration done!" << endl;
		}
	    }
	}
	else {
	    if (!inited) {
		slave_do_once();
		inited = true;
	    }
	}
    }

    STOP(DGMAIN::MasterSlaveCrap);

    CHECK_INVARIANTS();

    ///////////////////////////////////////////////////////////////////////////

    TimeVal now;
    gettimeofday(&now, NULL);

    //
    // Execute Command
    //
    switch (cmd) {
    case DG_CMD_DO_MAINTENANCE:
	g_Manager->Maintenance();   // hm. what timeout is appropriate here?? 
	break;
    case DG_CMD_DO_RECV:
	g_Manager->Recv(100 /* timeout in milliseconds */, false /* doMaintenance */);  
	CHECK_INVARIANTS();
	break;
    case DG_RESOLVE_CONFLICTS:
	DG_ResolveConflicts();
	CHECK_INVARIANTS();
	break;
    case DG_PRECOMPUTE_INFO:
	DG_PrecomputePerFrameInfo();
	CHECK_INVARIANTS();
	break;
    case DG_CMD_SEND_UPDATES:
	START(DGMAIN::RecordObjectChanges);
	DG_RecordObjectChanges();
	STOP(DGMAIN::RecordObjectChanges);

	CHECK_INVARIANTS();

	START(DGMAIN::Manager::Send);
	g_Manager->Send();
	STOP(DGMAIN::Manager::Send);

	CHECK_INVARIANTS();

	START(DGMAIN::ProcessRemoteStateless);
	DG_ProcessRemoteStatelessEvents();
	STOP(DGMAIN::ProcessRemoteStateless);

	CHECK_INVARIANTS();
	break;
    case DG_CMD_CHECKPOINT_PRIMARIES:
	DG_CheckPointObjects(/* is_replica= */ false);
	CHECK_INVARIANTS();
	break;
    case DG_CMD_CHECKPOINT_REPLICAS:
	DG_CheckPointObjects(/* is_replica= */ true);
	CHECK_INVARIANTS();
	break;
    case DG_GARBAGE_COLLECT:
	DG_GarbageCollect();
	// XXX -- invariants are broken here because we free some edicts
	// which are still in our object store. (the reason we do this is
	// because we want them to be removed from the object store *next*
	// time)
	break;
    default:
	Debug::die("Unknown command: %d", cmd);
    }

    ///////////////////////////////////////////////////////////////////////////

    //
    // Extra Stuff
    //

    ///// BEGIN_SD_TEST
#ifdef DG_TEST_SERIALIZE_DESERIALIZE
    //START(DGMAIN::SerializeTest);
    DG_test_serialize_deserialize(g_edicts, game.maxentities);
    //STOP(DGMAIN::SerializeTest);
#endif
    ///// END_SD_TEST

    //
    // Check Invariants
    //
    if (ManagerParams::CHECK_INVARIANTS) { g_QuakeAdaptor->CheckInvariants(); }

    //START(DGMAIN::DumpObjStoreDetails);

    //
    // Measurement
    //
    if (g_MeasurementParams.enabled) {
	static TimeVal last = TIME_NONE;

	// loosely synchronize the dump to occur on the 500 mark so that
	// all nodes dump their stores somewhat at the same time
	now.tv_usec = 500*USEC_IN_MSEC*(now.tv_usec/(500*USEC_IN_MSEC));

	if (now >= last + 500) {
	    DumpObjectStoreDetails(now);
	    last = now;
	}
    }

    //STOP(DGMAIN::DumpObjStoreDetails);

    //START(DGMAIN::RecordOtherJunk);

    // These are called only once per frame
    if (cmd == DG_CMD_SEND_UPDATES) {

	//
	// Possibly display what players are looking at
	//
	if (g_QuakePreferences.enable_looking_at) {
	    DG_DisplayLookingAt();
	}

	//
	// Possibly check accuracy of predicted bboxes
	//
	if (g_QuakePreferences.bbox_acc_test) {
	    DG_CheckBBoxAccuracy();
	}

	//
	// Possibly run the pub-sub simlator
	//
	if (g_QuakePreferences.pubsubsim) {
	    DG_EstimatePubSubCosts(SV_GetCurrentFrame());
	}

	//
	// Possible dump of initial object positions (right before migration)
	//
	if ( g_QuakePreferences.dump_initial && 
		init_frames-1 == SV_GetCurrentFrame() ) {
	    INFO << "Dumping initial object positions and area of items" << endl;
	    DG_RecordObjectInterests(true);
	    g_ObjectInterestLog->Flush();
	    exit(0);
	}

	//
	// Possible dump of initial object positions (right before migration)
	//
	if ( g_QuakePreferences.dump_initial && 
		init_frames-1 == SV_GetCurrentFrame() ) {
	    INFO << "Dumping initial object positions and area of items" << endl;
	    DG_RecordObjectInterests(true);
	    g_ObjectInterestLog->Flush();
	    exit(0);
	}

	//
	// Possible calculation of playable map area
	//
	if ( g_QuakePreferences.calc_playable_area > 0 &&
		init_frames-1 == SV_GetCurrentFrame() ) {
	    INFO << "Calculating playable map area" << endl;
	    DG_CalcPlayableMapArea(g_QuakePreferences.calc_playable_area);
	    exit(0);
	}

	if ( g_QuakePreferences.record_obj_interests &&
		init_frames-1 <= SV_GetCurrentFrame() ) {
	    DG_RecordObjectInterests(true);
	}

	//
	// Record the object locations and bboxes
	//
	if (g_QuakePreferences.record_obj_interests) {
	    DG_RecordObjectInterests();
	}

	//
	// Send updates to the visualizer if one has been specified
	//
	if (g_QuakePreferences.visualizer_ip[0]) {
	    DG_SendVisualizerUpdates();
	}
    }

    //STOP(DGMAIN::RecordOtherJunk);

    STOP(DGMAIN);
}

///////////////////////////////////////////////////////////////////////////////

int compare_pointers(const void *a, const void *b) 
{
    void *sa = *(void **) a;
    void *sb = *(void **) b;
    if (sa < sb) 
	return -1;
    else {
	if (sa > sb)
	    return 1;
	else
	    return 0;
    }
}

int compare_char_pointers(const void *a, const void *b) {
    char *sa = *(char **) a;
    char *sb = *(char **) b;
    return strcmp((const char *) sa, (const char *) sb);
}

int _DG_get_ptrarray_length(void **array)
{
    int i = 0;
    void *ptr;

    do {
	ptr = array[i];
	i++;
    } while (ptr != NULL);

    return (i - 1);
}

void _DG_sort_pointer_arrays()
{
    void *ptr;

    len_func_pointers = _DG_get_ptrarray_length(func_pointers);
    len_globals_pointers = _DG_get_ptrarray_length(globals_pointers);

    qsort(func_pointers, len_func_pointers, sizeof(void *), &compare_pointers);
    qsort(globals_pointers, len_globals_pointers, sizeof(void *), &compare_pointers);
}

#define EDICT_FOFS(x) (int)&(((edict_t *)0)->x)
//
//  Ashwin [3/29/2004]
// These are the F_LSTRING fields from the edict structure 
// they are allocated by ED_NewString - and, so they need to 
// be registered into the strings array...
//
int special_field_offsets[] = 
{
    EDICT_FOFS(classname),
    EDICT_FOFS(model), 
    EDICT_FOFS(target), 
    EDICT_FOFS(targetname), 
    EDICT_FOFS(pathtarget), 
    EDICT_FOFS(deathtarget), 
    EDICT_FOFS(killtarget), 
    EDICT_FOFS(combattarget),
    EDICT_FOFS(message), 
    EDICT_FOFS(team), 
    EDICT_FOFS(map)
};
int len_special_field_offsets = 0;

void _DG_build_string_arrays()
{
    int i, j, k, n_strs = 0;
    edict_t *ent;

    len_special_field_offsets = sizeof(special_field_offsets)/sizeof(int);

    ent = &g_edicts[0];
    for (i = 0; i < game.maxentities; i++, ent++) {
	int j;
	if (!ent->inuse)
	    continue;

	for (j = 0; j < len_special_field_offsets; j++) {
	    char *str = *(char **) ((byte *) ent + special_field_offsets[j]);
	    if (!str)
		continue;

	    n_strs++;
	}
    }

    // XXX: This TAG business has to be sorted out...
    dynamic_strings = (char **) ALLOC_MEM(n_strs * sizeof(char *));
    memset(dynamic_strings, 0, n_strs * sizeof(char *));

    // set the lengths
    len_dynamic_strings = n_strs;
    len_static_strings  = _DG_get_ptrarray_length((void **) static_strings);

    gi.dprintf("Initialized string arrays... static{%d}, dynamic{%d}\n", len_static_strings, len_dynamic_strings);

    ent = &g_edicts[0];
    k = 0;
    for (i = 0; i < game.maxentities; i++, ent++) {
	int j;
	if (!ent->inuse)
	    continue;

	for (j = 0; j < len_special_field_offsets; j++) {
	    char *str = *(char **) ((byte *) ent + special_field_offsets[j]);
	    if (!str)
		continue;

	    dynamic_strings[k] = str;
	    k++;
	}
    }

    qsort(static_strings, len_static_strings, sizeof(char *), &compare_char_pointers);
    qsort(dynamic_strings, len_dynamic_strings, sizeof(char *), &compare_char_pointers);

    /*
       printf("static strings\n");
       for (i= 0; i < len_static_strings; i++) {
       printf("\t=%s=\n", static_strings[i]);
       }
       printf("dynamic strings\n");
       for (i= 0; i < len_dynamic_strings; i++) {
       printf("\t:%s:\n", dynamic_strings[i]);
       }
     */
}

typedef struct {
    char *name;
    float cost;
} cost_t;

static cost_t s_DeltaCostsArray[] =
{
    { "ammo_bullets", 10.47 },
    { "item_quad", 12.48 },
    { "item_invulnerability", 10.28 },
    { "weapon_machinegun", 92.05 },
    { "func_timer", 3.15 },
    { "bot", 39.13 },
    { "item_health_large", 9.81 },
    { "item_pack", 12.00 },
    { "bolt", 50.46 },
    { "weapon_grenadelauncher", 61.50 },
    { "misc_teleporter_dest", 2.82 },
    { "item_power_shield", 24.67 },
    { "weapon_supershotgun", 69.94 },
    { "grenade", 140.13 },
    { "trigger_hurt", 4.00 },
    { "ammo_rockets", 8.11 },
    { "item_health", 9.86 },
    { "weapon_rocketlauncher", 39.25 },
    { "weapon_hyperblaster", 41.03 },
    { "info_player_deathmatch", 2.88 },
    { "weapon_railgun", 49.89 },
    { "item_health_small", 12.17 },
    { "info_notnull", 6.00 },
    { "misc_teleporter", 2.81 },
    { "ammo_cells", 9.24 },
    { "bodyque", 29.07 },
    { "item_armor_jacket", 10.70 },
    { "trigger_multiple", 3.91 },
    { "ammo_grenades", 18.19 },
    { "item_armor_body", 10.69 },
    { "ammo_shells", 10.77 },
    { "info_player_start", 6.00 },
    { "item_armor_shard", 10.00 },
    { "weapon_shotgun", 92.40 },
    { "item_armor_combat", 10.73 },
    { "rocket", 135.61 },
    { "item_breather", 10.21 },
    { "func_door", 7.59 },
    { "info_player_intermission", 8.00 },
    { "weapon_chaingun", 18.67 },
    { "noclass", 18.00 },
    { "DelayedUse", 20.00 },      // changed from 952.00
    { "weapon_bfg", 14.00 },
    { "item_bandolier", 8.73 },
    { "ammo_slugs", 8.21 },
    { "func_plat", 9.30 },
    { "item_health_mega", 9.22 }
};
CostMap g_CostMap;

#if 0
static int HubToIndex (char *str)
{
    if (streq (str, "x"))
	return 0;
    else if (streq (str, "y"))
	return 1;
    else if (streq (str, "z"))
	return 2;
    else {
	ASSERT (0);
	return -1;
    }
}
#endif

int *g_CoordIndices;

static void InitializeHubIndices ()
{
    enum { X_INDEX = 0, Y_INDEX, Z_INDEX };
    const int NCOORDS = 3;
    const char *coordNames[NCOORDS] = { "x", "y", "z" };
    
    g_CoordIndices = new int[NCOORDS];

    for (int i = 0; i < NCOORDS; i++) 
	g_CoordIndices[i] = -1;
    
    // if we are using DHT, we use the first attribute for publishing the key 
    if (g_QuakePreferences.dht || g_QuakePreferences.stripe_dim >= 0) {
	g_DHTHubIndex = 0;
	g_StripeHubIndex = 0;

	for (int i = 0; i < NCOORDS; i++) 
	    g_CoordIndices[i] = i + 1;
	return;
    }
    
    // hash table recording name -> index mapping for coordinates
    hash_map<const char*, int, hash<const char*>, eqstr> ctab;	    	  
    for (int i = 0; i < NCOORDS; i++) 
	ctab.insert (pair<const char *, int> (coordNames[i], i));
    
    // make hash table of which routing hubs exist
    map<int, bool, less<int> > rhtab;	    	  

    vector<string> rhubs;
    tokenizer<comma_sep>::tokenize (rhubs, g_QuakePreferences.routing_hubs); 
    for (int i = 0, n = rhubs.size (); i < n; i++) {
	const char *hn = rhubs[i].c_str ();
	
	ASSERT (ctab.find (hn) != ctab.end ());
	
	rhtab.insert (pair<int, bool> (i, true));
	g_CoordIndices[ ctab[hn] ] = i;
    }

    for (int i = 0; i < NCOORDS; i++) {
	// foreach coordinate index; check if it is assigned
	if (g_CoordIndices[i] != -1) 
	    continue;

	// otherwise look for an index which has not been assigned
	for (int j = 0; j < NCOORDS; j++) {
	    if (rhtab.find (j) == rhtab.end ()) {
		g_CoordIndices[i] = j; 
		rhtab.insert (pair<int, bool> (j, true));
		break;
	    }
	}
    }

    INFO << " assigned coordinates indices as follows: " << endl;
    for (int i = 0; i < NCOORDS; i++) { 
	INFO << "\t" << coordNames[i] << " -> " << g_CoordIndices[i] << endl;	    
    }
    /* 
    if (g_QuakePreferences.dht && g_QuakePreferences.dht_hub) 
	g_DHTHubIndex = HubToIndex (g_QuakePreferences.dht_hub);
    */
}

///////////////////////////////////////////////////////////////////////////////

// XXX Jeff 9/16: Why is this separate from DG_Init()?
void SVDG_Init()
{

    g_QuakeAdaptor = new QuakeAdaptor();
    g_ObjStore = (DG_ObjStore *) g_QuakeAdaptor->GetObjectStore();

    // (1) Start up Manager and Mercury if enabled.
    //     This call will block until everyone is joined to the network
    //     (assuming ManagerParams::WAIT_ALL_JOIN is enabled)
    if (!g_QuakePreferences.disable_colyseus)
	g_Manager = Manager::GetInstance(g_QuakeAdaptor);
    else
	g_Manager = new DG_DummyManager();

    // (1.5) initialize debugging
    DBG_INIT(&g_LocalSID);

    // XXX REDO THIS!!! -- compute the estimated cost of each object
    for (int i = 0, len = sizeof(s_DeltaCostsArray)/sizeof(cost_t); 
	    i < len; i++) {
	g_CostMap[s_DeltaCostsArray[i].name] = 10 * s_DeltaCostsArray[i].cost; 
	// multiply by 10 because we want bandwidth in bytes per second; 
	// the array gives bytes per frame
    }

    // (2) Enable Quake Measurement Logs
    if (g_MeasurementParams.enabled) {
	DG_InitLogs();
    }

    if (!g_QuakePreferences.disable_colyseus)
	// use this instead of hooky_wait_timer :)
	ASSERT(ManagerParams::WAIT_ALL_JOIN);

    InitializeHubIndices ();

    if (ManagerParams::CHECK_INVARIANTS) { g_QuakeAdaptor->CheckInvariants(); }

    INFO << "finished SVDG_Init()" << endl;
}

//
// This is called from SpawnEntities after all the entities have been spawned
// We make a complete copy of the entities, bit-by-bit.
//
void DG_Init() {
    int i;
    edict_t *ent;

    //
    // Possible bounding box precomputation. Once done, just exit.
    //
    if (g_QuakePreferences.bbox) {
	DG_ExportBoundingBoxes();
	INFO << " ---- done with bbox computation! ----" << endl;
	exit(0);
    } else {
	// otherwise try to load the bbox map file
	if ( strcmp(g_QuakePreferences.bbox_file, "") ) {
	    INFO << "loading precomputed bboxes from: "
		<< g_QuakePreferences.bbox_file << endl;
	    DG_LoadBoundingBoxFile();
	} else {
	    INFO << "no bbox file specified, will calculate bboxes manually"
		<< endl;
	}
    }

    //
    // Possible recoding of delta encoding info
    //
    if (g_QuakePreferences.deltas) {
	INIT_EXPLOG(DeltaEncodingLog, g_LocalSID);
    }

    //
    // Possible test of bbox accuracy (accuracy logged)
    //
    if (g_QuakePreferences.bbox_acc_test) {
	INIT_EXPLOG(BBoxErrorLog, g_LocalSID);
    }

    //
    // Possibly initialize the DHT buckets
    //
    if (g_QuakePreferences.dht) {
	INFO << "Initializing DHT with " << g_QuakePreferences.dht_buckets
	    << " on hub = " << g_QuakePreferences.dht_hub << endl;
	DG_ChopWorldForDHT(g_QuakePreferences.dht_buckets);
    }

    //
    // Initialize static and dynamic string tables
    //
    _DG_sort_pointer_arrays();      // func_pointers and global_pointers
    _DG_build_string_arrays();      // string pointers

    //
    // Initialize delta-encoding tables
    //
    DG_InitDeltaTables();

    //
    // Ashwin [05/20] -- moved this from g_spawn:SpawnEntities, since we need to build the "dynamic strings" table correctly
    // before embarking on any free_edict'ing
    //
    // - Ashwin [04/25/04]
    // so you spawned all entities you could. nice. but i have to give you ownership for
    // only some of the entities. So, I am gonna be FreeEdict'ing a few of them 
    // using the hacky g_QuakePreferences.nserver option.
    //
    // do i need to always keep some "special" entities around like "world" for example?
    // yeah, i need it. thankfully, g_edicts[0] never seems to think. good for him (i am sexist, yeah.) 

    //
    // - Ashwin [05/19/04] Okay, now we change things slightly 
    //   if i am master, i keep everything and then i migrate when i am told to ... 

    if (!g_QuakePreferences.is_master) {
	//
	// Free those entities which are going to be in the OM layer
	// The others are just normal objects (not distributed)
	//
	for (i = 0, ent = &g_edicts[0]; i < globals.num_edicts; i++, ent++) {
	    if (!ent->inuse)
		continue;

	    // free if the object is distributed; the master is gonna add exactly these entities to the object store 
	    if (DG_is_object_distributed(ent)) {
		if (!ent->is_bot)  // bots are special, since only the slaves spawn them!
		    G_FreeEdict(ent);
	    }

#if 0			
	    if (ent->classname && strstr(ent->classname, "trigger_once")) {
		DB(1) << " \n\n\ntrigger_once guid: " << ent->guid << " and " <<
		    (ent->is_distributed ? " is " : " is NOT ") << " distributed \n\n\n" << endl;
	    }
#endif
	}
    }

    //
    // - Ashwin [05/21]
    //
    // For the slave: the only distributed objects remaining now are the bots
    // For the master: the only distributed objects remaining are whatever returned DG_is_object_distributed(ent) true
    //
    for (i = 0, ent = &g_edicts[0]; i < game.maxentities; i++, ent++) {
	if (!ent->inuse) 
	    continue;

	//ent->primary_owner = g_Manager->GetSID();
	//ent->is_replica = false;

	// XXX: i == 0 -> special entity we don't want the manager to bother about
	//      however, we have to generate the GUID for transmitting pointers.
	//      These are default-replicated objects, basically.
	//
	//      there's a corresponding hack in ExistsEntity...
	//      - Ashwin [05/17]
	//
	// Only add objects to the object store which are going to be "distributed"
	// Other objects saty as normal "edicts" having nothing do with the manager
	// or any of our layers...
	//
	// The Shadow/Restore routines should also be modified accordingly
	// 
	if (DG_is_object_distributed(ent)) {
	    DG_RecordNewObject(ent);
	}
    }

    g_shadow_edicts = (edict_t *) ALLOC_MEM(game.maxentities * sizeof(g_edicts[0]));
    memmove(g_shadow_edicts, g_edicts, game.maxentities * sizeof(g_edicts[0]));

    // XXX - 12/17/2004: Jeff -- on second thought, we should do something
    // smarter. This is a horribly large waste of memory. :)
    // 6/30/2004: Jeff
    // Screw that. Just allocate a shadow client for each edict.
    // NO We'll keep this shadow array only for our "local" clients.
    // NO For the foreign clients, whenever we get a serialized client entity, 
    // NO we will malloc a new object.
    g_shadow_clients = (gclient_t *) ALLOC_MEM(game.maxentities * 
	    sizeof(game.clients[0]));
    memset(g_shadow_clients, 0, game.maxentities * sizeof(game.clients[0]));

    for (i = 0; i < game.maxentities; i++) {
	if (g_shadow_edicts[i].client) {
	    g_shadow_edicts[i].client = g_shadow_clients + i;
	    memmove(g_shadow_edicts[i].client, g_edicts[i].client, sizeof(gclient_t));
	}
    }

    memset(dg_remote_clients, 0, MAX_REMOTE_CLIENTS * sizeof(r_gclient_t));
    for (i = 0; i < MAX_REMOTE_CLIENTS; i++) {
	dg_remote_clients[i].inuse = false;
    }

    if (ManagerParams::CHECK_INVARIANTS) { g_QuakeAdaptor->CheckInvariants(); }
}

// called when we exit
void DG_Halt()
{
    //
    // Tell simulator to halt and flush logs
    //
    if (g_QuakePreferences.pubsubsim) {
	DG_EstimatePubSubCostsHalt();
    }

    //
    // Flush all outstanding log entries to disk
    //
    FLUSH_ALL();
}

///////////////////////////////////////////////////////////////////////////////

bool DG_is_object_distributed(edict_t *ent)
{
    ASSERT(ent->inuse);

    if (ent->s.number == 0) { // special case
	ent->is_distributed = false;
	ASSERT(DG_IsNonDistributed(ent->guid));
    }
    else if (ent->classname && strstr(ent->classname, "func_plat")
	     || ent->classname && strstr(ent->classname, "trigger")
	     || ent->classname && strstr(ent->classname, "door")
	     )
	ent->is_distributed = false;
    else if (ent->s.number <= game.maxclients                     // a bot or a client
	    || ent->classname && strstr(ent->classname, "item")  // an item
	    || ent->classname && strstr(ent->classname, "missile") // missiles, weapons, guns
	    || ent->classname && strstr(ent->classname, "weapon")
	    || ent->classname && strstr(ent->classname, "ammo")
	    || ent->classname && strstr(ent->classname, "gun")
	    || ent->classname && strstr(ent->classname, "rocket")
	    || ent->classname && strstr(ent->classname, "gib")
	    //			 || ent->classname && strstr(ent->classname, "func_plat")
	    //			 || ent->classname && strstr(ent->classname, "trigger")
	    //			 || ent->classname && strstr(ent->classname, "door")
	    || ent->movetype == MOVETYPE_FLYMISSILE                 
	    || ent->movetype == MOVETYPE_TOSS
	    || ent->movetype == MOVETYPE_FLY
	    || ent->movetype == MOVETYPE_BOUNCE
	    )
    {
	ent->is_distributed = true;
    }
    else
    {
	ent->is_distributed = false;
	// Jeff: "bodyqueue" edicts are a special preallocated region of
	// the edict array used for allocation of dead bodies... in the
	// beginning all these edicts are actually "unused" so there is
	// no reason to distribute them.
	// XXX Need to check that when a dead body is created that it
	// is properly distributed...
	ASSERT(!strcmp(ent->classname, "bodyque") ||
		DG_IsNonDistributed(ent->guid));
    }

    return ent->is_distributed;
}

int search_pointer(void **ptr_array, int len, void *ptr)
{
    // binary search implementation
    int lower = 0, upper = len - 1;
    int cur = 0;
    int comp;

    if (len <= 0)
	return -1;

    while (true)
    {
	if (lower > upper)
	    return -1;

	cur = (lower + upper) / 2;

	comp = compare_pointers(&(ptr_array[cur]), &ptr);
	if (comp == 0) // pointer are equal
	    return cur;
	else {
	    if (comp > 0) // current pointer is greater than the target
		upper = cur - 1;
	    else // current pointer is less than the target
		lower = cur + 1;
	}
    }
}

int search_char_pointer(void **ptr_array, int len, void *ptr)
{
    // binary search implementation
    int lower = 0, upper = len - 1;
    int cur = 0;
    int comp;

    if (len <= 0)
	return -1;

    while (true)
    {
	if (lower > upper)
	    return -1;

	cur = (lower + upper) / 2;

	comp = compare_char_pointers(&(ptr_array[cur]), &ptr);
	if (comp == 0) // pointer are equal
	    return cur;
	else {
	    if (comp > 0) // current pointer is greater than the target
		upper = cur - 1;
	    else // current pointer is less than the target
		lower = cur + 1;
	}
    }
}

#define EDICT_FOFS(x) (int)&(((edict_t *)0)->x)

//
// pp_client is the *address* of the client pointer within the edict
// structure.
//
void DG_GetClientGUID(byte *pp_client, guid_t *guid)
{
    edict_t *ent = (edict_t *) (pp_client - EDICT_FOFS(client));
    *guid = ent->guid; // could have used the placement new! :)
}

bool DG_IsClient(int edict_num)
{
    ASSERT(edict_num < game.maxentities);
    edict_t *ent = &g_edicts[edict_num];

    return ent->inuse && ent->client;
}

///////////////////////////////////////////////////////////////////////////////

void DG_PrecomputePerFrameInfo()
{
    DG_Edict *obj;

    g_Manager->Lock(); // XXX HACK
    g_ObjStore->Begin();
    while ((obj = (DG_Edict *)g_ObjStore->Next())) {
	if (!obj->IsReplica()) {
	    obj->PrecomputePerFrameInfo();
	}
    }
    g_Manager->Unlock(); // XXX HACK
}

// This function should resolve any "conflicts" resulting from remote updates
// to our object store (primaries or replicas).
//
// For now this just checks for collisions between remote replica clients
// and our primary clients and tries to move our client out of the way.
void DG_ResolveConflicts()
{
    int i;
    edict_t *ent;

    TimeVal now;
    gettimeofday(&now, NULL);

    for (i = 0, ent = &g_edicts[0]; i < game.maxentities; i++, ent++) {
	if (ent->inuse && ent->client && !ent->is_replica) {
	    // see if it overlaps any of our objects

	    // if we fail to "unstuck" ourselves in this many tries, then
	    // this means we are probably trapped in a corner and have no
	    // where to move to. Hopefully when the nodes that own the dudes
	    // we are stuck to resolve conflicts, they get us unstuck by
	    // moving their dude out of the way
	    static int maxbacktrack = 10;
	    bool stuck = false;
	    trace_t		tr, tr2;
	    vector<edict_t *> relink;

	    gi.unlinkentity(ent);
	    for (int i=0; i<maxbacktrack; i++) {
		tr = gi.trace (ent->s.origin, ent->mins, ent->maxs, 
			ent->s.origin, ent,
			MASK_PLAYERSOLID);
		//CONTENTS_PLAYERCLIP|CONTENTS_MONSTER);
		if (!tr.ent || tr.ent->s.number == 0) {
		    stuck = false;
		    break;
		}
		if (! tr.ent->client) {
		    gi.unlinkentity(tr.ent);
		    relink.push_back(tr.ent);
		    continue;
		}

		//INFO << "stuck together: " << ent
		//	 << " " << tr.ent << endl;

		if (!tr.ent->is_replica) {
		    DB(0) << "WTF? entity: " << ent << " is stuck inside "
			<< "a guy we own: " << tr.ent << endl;
		    break;
		}

		// else this dude is stuck inside a replica --
		// try to move my guy out of the way...

		bool same = true;
		for (int i=0; i<3; i++)
		    if (ent->s.origin[i] != tr.ent->s.origin[i])
			same = false;
		if (same) {
		    stuck = true;
		    break; // this is almost certainly a telefrag
		}

		// try moving these guys "apart" by moving our guy in
		// the direction opposite of toward the other guy
		vec3_t diff;
		VectorSubtract(ent->s.origin, tr.ent->s.origin, diff);
		float len = VectorLength(diff);
		VectorNormalize(diff);
		// we should be in 32x32x56 box -- horiz diag is ~45.3
		vec3_t move;
		VectorScale(diff, MAX(1, 32-len), move);

		for (int i=0; i<3; i++)
		    ent->velocity[i] = move[i]/FRAMETIME;

		// unlink the obstructing dude
		gi.unlinkentity(tr.ent);
		// see if we can move it there...
		bool blocked = SV_FlyMove(ent, FRAMETIME, MASK_SOLID);
		// relink the obstruting dude
		gi.linkentity(tr.ent);
		// XXX: What if there are multiple obstructing dudes?
	    }
	    gi.linkentity(ent);

	    for (int i=0; i<relink.size(); i++) {
		gi.linkentity(relink[i]);
	    }

	    if (stuck) {
		edict_t *killer = tr.ent ? tr.ent : ent;

		DB(0) << "permantently stuck to killer! " 
		    << ent
		    << " <- " << killer << endl;

		// argh! dude is still suck -- just kill it! :)
		T_Damage (ent, killer, killer, vec3_origin, killer->s.origin, 
			vec3_origin, 100000, 0, DAMAGE_NO_PROTECTION, 
			MOD_TELEFRAG);
	    }
	}
    }
}

//
// Record changes to the objects. mark each object as dirty or not. 
// call Exec which will call DG_Edict->IsDirty() and then publish them 
// accordingly.
//
// Now, Quake does not know which objects are replicas and so, it writes to 
// them unknowingly in many cases. Hence, restore the states of the objects 
// after the Exec() call is completed.
// 
void DG_RecordObjectChanges()
{
    edict_t *ent = 0, *prevent = 0;
    int  i;

    DB(10) << "In DG_RecordObjectChanges ... " << endl;

    for (i = 0, ent = &g_edicts[0], prevent = &g_shadow_edicts[0]; 
	    i < game.maxentities; i++, ent++, prevent++) {

	ent->status = DG_OBJ_UNAFFECTED;

	if (!ent->inuse && !prevent->inuse) 
	{
	    continue;
	}
	else if (!ent->inuse && prevent->inuse) 
	{
	    START(RecordObjectChanges::Delete);
	    DG_RecordDeleteObject(prevent);
	    STOP(RecordObjectChanges::Delete);
	}
	else if (ent->inuse && !prevent->inuse) 
	{
	    if (DG_is_object_distributed(ent)) {
		START(RecordObjectChanges::New);
		DG_RecordNewObject(ent);
		STOP(RecordObjectChanges::New);
	    }
	}
	// It may be the case that Quake reuses an entity slot
	// for a *new* object after we checkpoint the array. In
	// that case, Quake deleted the old object and created the new one
	// NOTE: the GUID difference depends on unique local GUIDs being
	// assigned in G_Spawn()
	else if	(ent->inuse && prevent->inuse && ent->guid != prevent->guid)
	{
	    START(RecordObjectChanges::Delete);
	    DG_RecordDeleteObject(prevent);
	    STOP(RecordObjectChanges::Delete);
	    START(RecordObjectChanges::New);
	    DG_RecordNewObject(ent);
	    STOP(RecordObjectChanges::New);
	}
	else if (ent->inuse && prevent->inuse) 
	{
	    START(RecordObjectChanges::Same);
	    DG_RecordSameObject(prevent, ent);
	    STOP(RecordObjectChanges::Same);
	}
    }
}

void DG_RecordNewObject(edict_t *ent)
{
    // should the newly created object be "distributed"? YES!
    ent->is_distributed = true;

    DB(4) << "created: " 
	<< (ent->classname ? ent->classname : "null") 
	<< " " << ent->guid << endl;
    ent->status = DG_OBJ_CREATED;
    ent->guid = g_Manager->CreateGUID();
    ent->is_replica = false;

    DG_Edict *dg_obj = new DG_Edict(ent);

    DG_MakeInitDeltaMask(dg_obj);

    ASSERT(g_ObjStore->Find(dg_obj->GetGUID()) == NULL);
    g_ObjStore->ApplicationAdd(dg_obj);

    if (g_MeasurementParams.enabled) {
	GUIDLogEntry e;
	e.status    = GUIDLogEntry::CREATE;
	e.guid      = dg_obj->GetGUID();
	e.classname = ent->classname;
	LOG(GUIDLog, e);
    }

    if (g_QuakePreferences.record_obj_deltas) {
	static bool inited = false;
	static dg_field_stack_t  init;
	static dg_field_t edict_fld;
	if (!inited) {
	    inited = true;
	    DG_InitFieldStack(&init);
	    edict_fld.type = DF_EDICT_PTR;
	}

	ObjectDeltaEntry e;
	e.frameno = SV_GetCurrentFrame();
	e.guid    = dg_obj->GetGUID();
	e.type    = (char)ObjectDeltaEntry::NEW;

	dg_fields encoded;
	DG_MakeEncodeSet(&dg_obj->GetInitDeltaMask(), &encoded);
	e.deltaSize = 
	    object_length(&init, &encoded, &edict_fld, (byte *) ent);
	e.fullSize  = e.deltaSize;

	LOG(ObjectDeltaLog, e);
    }
    // Jeff: no longer needed, auto-callback when GObject is constructed
    // created a new "primary" GUID; inform the manager 
    //g_Manager->AddGUID(dg_obj->GetGUID()); 
}

void DG_RecordDeleteObject(edict_t *prevent)
{
    // 6/30/2004 Jeff: This comment is impossible, no? What we should
    // be worried about is an entity slot that is reused by another
    // entity between check-points (ent->inuse && prevent->inuse)
    //
    // watch out for temporary entities which were spawned and 
    // deleted before we even noticed!

    // if this object is a "local" object, we don't care about its 
    // deletion. this should never happen! otherwise it should be dist
    if (DG_IsNonDistributed(prevent->guid)) {
	WARN << "deleted non-distributed object of class " 
	    << prevent->classname << endl;
	return;
    }

    DB(3) << "destroyed: " 
	<< (prevent->classname ? prevent->classname : "null") 
	<< " " << prevent->guid << endl;
    // Jeff: No reason we need to mark this (inuse is false) and it
    // will break the case where the new slot is actually used by a new ent
    //ent->status = DG_OBJ_DESTROYED;

    DG_Edict *dg_obj = (DG_Edict *) g_ObjStore->Find(prevent->guid);

    /// ASSERT(dg_obj);      XXX: changed it to do a graceful return -- Ashwin [10/05/2004]
    if (!dg_obj)
	return;

    // XXX 7/1/2004 Jeff: It looks like Quake does do this sometimes.
    // will have to handle it by sending a "remote delete" signal, but
    // for now just ignore it. Seems very uncommon.
    //ASSERT(!dg_obj->IsReplica());
    //ASSERT(!prevent->is_replica);

    DB(1) << " destroying guid: " << prevent->guid 
	<< " with classname " << prevent->classname << endl;
    g_ObjStore->ApplicationRemove(prevent->guid);
    if (!prevent->is_replica) {
	// Jeff: no longer needed, auto callback on GObject deletion
	//g_Manager->DeleteGUID(prevent->guid);

	if (g_MeasurementParams.enabled) {
	    GUIDLogEntry e;
	    e.status    = GUIDLogEntry::DESTROY;
	    e.guid      = dg_obj->GetGUID();
	    e.classname = prevent->classname;
	    LOG(GUIDLog, e);
	}

	if (g_QuakePreferences.record_obj_deltas) {
	    ObjectDeltaEntry e;
	    e.frameno = SV_GetCurrentFrame();
	    e.guid    = dg_obj->GetGUID();
	    e.type    = (char)ObjectDeltaEntry::DELETE;

	    e.deltaSize = 0;
	    e.fullSize  = 0;

	    LOG(ObjectDeltaLog, e);
	}
    }

    delete dg_obj;
}

void DG_RecordSameObject(edict_t *prevent, edict_t *ent)
{
    dg_field_t edict_fld;
    edict_fld.type = DF_EDICT_PTR;

    if (DG_IsNonDistributed(ent->guid)) {
	return;
    }

    dg_fields isdirty;
    dg_field_stack_t  init;
    DG_InitFieldStack(&init);

    if (object_is_dirty(&init, &isdirty, &edict_fld, (byte *) prevent, (byte *) ent)) {
	DB(10) << "CHANGED: " << (ent->classname ? ent->classname : 
		"null") << " " << ent->guid << endl;

	//
	// Possibly record delta info for encoding table info
	//
	if (g_QuakePreferences.deltas) {
	    DeltaEncodingEntry e(false, ent->is_replica, ent->guid,
		    ent->classname);
	    for (dg_fields::iterator it = isdirty.begin(); it != isdirty.end(); it++) {
		e.fields.push_back( dg_fieldname_map[*it] );
	    }
	    LOG(DeltaEncodingLog, e);
	}
	// -----

	ent->status = DG_OBJ_CHANGED;
	DG_Edict *dg_obj = 
	    (DG_Edict *) g_ObjStore->Find(prevent->guid);
	ASSERT(dg_obj);

	DG_MakeDeltaMask(dg_obj, &isdirty);

	// HACK: See DG_RecordNewObject() above
	dg_obj->m_InitDeltaMask.Merge(dg_obj->GetDeltaMask());
	// every primary object must have a classname set!
	ASSERT( dg_obj->IsReplica() ||
		dg_obj->m_InitDeltaMask.IsSet( 
		    _dg_field_encoding[string("classname")]->index ) ||
		dg_obj->m_InitDeltaMask.IsSet( NUM_FIELDS-1 ) );

	if (g_QuakePreferences.record_obj_deltas) {
	    static bool inited = false;
	    static dg_field_stack_t  init;
	    static dg_field_t edict_fld;
	    if (!inited) {
		inited = true;
		DG_InitFieldStack(&init);
		edict_fld.type = DF_EDICT_PTR;
	    }

	    ObjectDeltaEntry e;
	    e.frameno = SV_GetCurrentFrame();
	    e.guid    = dg_obj->GetGUID();
	    e.type    = (char)ObjectDeltaEntry::UPDATE;

	    dg_fields encoded;
	    DG_MakeEncodeSet(&dg_obj->GetDeltaMask(), &encoded);
	    e.deltaSize = object_length(&init, &encoded, 
		    &edict_fld, (byte *) ent);

	    encoded.clear();
	    DG_MakeEncodeSet(&dg_obj->GetInitDeltaMask(), &encoded);
	    e.fullSize  = object_length(&init, &encoded, 
		    &edict_fld, (byte *) ent);

	    LOG(ObjectDeltaLog, e);
	}
    }
    else {
	ent->status = DG_OBJ_UNAFFECTED;
    }
}

// Force checkpoint a single object
void DG_CheckPointObject(edict_t *ent)
{
    edict_t *prevent = g_shadow_edicts + (ent - g_edicts);
    gclient_t *prevclient, *client;

    ASSERT(ent->inuse || prevent->inuse);

    client = ent->client;
    prevclient = prevent->client;

    client = ent->client;
    prevclient = g_shadow_clients + (ent - g_edicts);

    // copy the entity structure;
    // this overwrites the CLIENT pointer also
    *prevent = *ent;
    if (client) {
	*prevclient     = *client; // copy into the shadow CLIENT array
	prevent->client = prevclient; // restore the pointer
    }
}

void DG_CheckPointObjects(bool is_replica)
{
    edict_t *ent = 0, *prevent = 0;
    gclient_t *prevclient, *client;
    int  i, length;

    for (i = 0, ent = &g_edicts[0], prevent = &g_shadow_edicts[0]; 
	    i < game.maxentities; i++, ent++, prevent++) {

	// No need to checkpoint non-distributed objects
	if (DG_IsNonDistributed(ent->guid)) {
	    continue;
	}

	// This was a free slot and was not filled. 
	if (!ent->inuse && !prevent->inuse)
	    continue;

	// No need to checkpoint objects that are not in use because
	// this is just a "free" slot in the edict array. The only 
	// information we need to checkpoint is the fact that it is free. 
	// If it is filled post-checkpoint, then the object in that spot 
	// is new. Only follow this branch though if we are checkpointing
	// the change to is_replica edicts, which we can determine from the
	// prevent if it was in use.
	if (!ent->inuse && prevent->inuse && 
		prevent->is_replica == is_replica) {
	    prevent->inuse = false;
	    continue;
	}

	if (ent->is_replica == is_replica) {
	    client = ent->client;
	    prevclient = prevent->client;

	    client = ent->client;
	    prevclient = &g_shadow_clients[i];

	    // copy the entity structure;
	    // this overwrites the CLIENT pointer also
	    *prevent = *ent;
	    if (client) {
		*prevclient     = *client; // copy into the shadow CLIENT array
		prevent->client = prevclient; // restore the pointer
	    }
	}
    }
}

// This is a hack to process multicasted events sent to clients from remote
// servers. Since some clients are remote, those events get forwarded to the
// server hosting the primary. We have to handle them here.
void DG_ProcessRemoteStatelessEvents()
{
    int i;
    edict_t *ent;

    for (i = 0; i < game.maxentities; i++) {
	ent = &g_edicts[i];

	if (ent->multicastEventList) {
	    if (ent->is_replica) {
		//INFO << "replica multicast: " << ent << endl;
		DG_MulticastEvent *next = ent->multicastEventList;
		while (next) {
		    for (uint32 j=0; j<next->data->Size(); j++)
			gi.WriteByte(next->data->data[j]);
		    gi.multicast(NULL, next->origin, next->to);
		    next = next->next;
		}
	    }

	    // clear out multicast list (for primaries, it was just sent out)
	    delete ent->multicastEventList;
	    ent->multicastEventList = NULL;
	}
    }
}

void DG_GarbageCollect()
{
    int i;
    edict_t *ent;

    for (i = 0; i < game.maxentities; i++) {
	ent = &g_edicts[i];
	// free this guy if needed and we already multicasted its events
	if (!ent->is_replica && ent->tofree) {
	    ASSERT(!ent->multicastEventList);
	    G_FreeEdict(ent);
	}
    }
}

///////////////////////////////////////////////////////////////////////////////

void DG_DisplayLookingAt()
{
    edict_t *ent;

    for (int i = 0; i < game.maxentities; i++) {
	ent = &g_edicts[i];

	if ( !ent->inuse || !ent->client || ent->is_replica ) continue;
	if ( strcmp(ent->classname, "player") ) continue;

	edict_t    *new_looking_at = NULL;
	vec3_t		forward, end;
	trace_t		tr;

	AngleVectors (ent->client->v_angle, forward, NULL, NULL);
	VectorMA (ent->s.origin, 8192, forward, end);
	tr = gi.trace (ent->s.origin, NULL, NULL, end, ent, MASK_ALL);
	if (tr.ent && tr.ent != ent) {
	    new_looking_at = tr.ent;
	}

	if (new_looking_at && 
		new_looking_at->s.number != 0 &&
		!DG_IsNonDistributed(new_looking_at->guid) &&
		ent->looking_at != new_looking_at) {
	    ent->looking_at = new_looking_at;
	    stringstream str;
	    str << new_looking_at;
	    gi.cprintf(ent, PRINT_HIGH, "%s\n", str.str().c_str());
	}
    }
}

void DG_RecordObjectInterests(bool include_items)
{
    edict_t *ent;
    ObjectInterestEntry e;

    e.frameno = SV_GetCurrentFrame();

#if 0

    // Old style of dumping everything

    for (int i = 0; i < game.maxentities; i++) {
	ent = &g_edicts[i];

	if ( !ent->inuse ) continue;

	if ( streq(ent->classname, "bot") ||
		streq(ent->classname, "player") ) {
	    e.type = ObjectInterestEntry::PLAYER;
	} else if ( streq(ent->classname, "bolt") ||
		streq(ent->classname, "grenade") ||
		streq(ent->classname, "hgrenade") ||
		streq(ent->classname, "rocket") ||
		streq(ent->classname, "bfg blast") ) {
	    e.type = ObjectInterestEntry::MISSILE;
	} else if (include_items && DG_is_object_distributed(ent)) {
	    e.type = ObjectInterestEntry::ITEM;
	} else {
	    continue;
	}

	e.guid = ent->guid;
	VectorCopy(ent->s.origin, e.orig);

	// precomputed bbox -- XXX if we record this stuff in distributed
	// mode then make sure to cache the bbox so we don't compute it
	// twice (once for the interest)
	DG_GetBoundingBox(ent->s.origin, e.min, e.max);

	LOG(ObjectInterestLog, e);
    }

#else

    // New style of dumping only primaries

    DG_Edict *obj;

    g_Manager->Lock(); // XXX HACK
    g_ObjStore->Begin();
    while ((obj = (DG_Edict *)g_ObjStore->Next()) != NULL) {
	if (obj->IsReplica())
	    continue;

	ent = obj->m_Entity;
	ASSERT(ent);

	switch(obj->GetType()) {
	case EDICT_ITEM: {
	    if (include_items) 
		e.type = ObjectInterestEntry::ITEM;
	    else
		// items are dumped once at the beginning of the game
		continue;
	    break;
	}
	case EDICT_PLAYER: {
	    e.type = ObjectInterestEntry::PLAYER;
	    break;
	}
	case EDICT_MONSTER: {
	    e.type = ObjectInterestEntry::MONSTER;
	    break;
	}
	case EDICT_MISSILE: {
	    e.type = ObjectInterestEntry::MISSILE;
	    break;
	}
	default:
	    continue;
	}

	e.guid = ent->guid;
	VectorCopy(ent->s.origin, e.orig);
	DG_GetBoundingBox(ent->s.origin, e.min, e.max);
	// this is predictive -- use actual bbox
	//VectorCopy(obj->m_CurrBBox.min, e.min);
	//VectorCopy(obj->m_CurrBBox.max, e.max);
	LOG(ObjectInterestLog, e);
    }
    g_Manager->Unlock(); // XXX HACK

#endif
}

void DG_CheckBBoxAccuracy()
{
    for (int i = 0; i < game.maxentities; i++) {
	edict_t *ent = &g_edicts[i];

	if ( ent->inuse &&
		(streq(ent->classname, "bot") ||
		 streq(ent->classname, "player") ||
		 streq(ent->classname, "bolt") ||
		 streq(ent->classname, "grenade") ||
		 streq(ent->classname, "hgrenade") ||
		 streq(ent->classname, "rocket") ||
		 streq(ent->classname, "bfg blast")) ) {

	    BBoxErrorEntry e;
	    VectorCopy(ent->s.origin, e.orig);

	    // precomputed bbox
	    vec3_t min, max;
	    e.vol = DG_GetBoundingBox(ent->s.origin, min, max);
	    VectorCopy(min, e.min);
	    VectorCopy(max, e.max);

	    // compared against direct ray tracing
	    vec3_t dmin, dmax;
	    DG_ComputeVisibilityBox(ent->s.origin, dmin, dmax, 5, 5);
	    float dvol     = BBoxVolume(dmin, dmax);
	    float dmissing = BBoxVolSubtract(dmin, dmax, min, max);
	    float dextra   = BBoxVolSubtract(min, max, dmin, dmax);
	    if (dvol == 0) {
		e.direct_frac_vol_missing = 0;
		// its all extra! but we can't normalize by 0, so just say 1
		e.direct_frac_vol_extra   = dextra == 0 ? 0 : 1;
	    } else {
		e.direct_frac_vol_missing = dmissing/dvol;
		e.direct_frac_vol_extra   = dextra/dvol;
	    }

	    // compared against pvs bbox
	    /*
	       vec3_t pmin, pmax;
	       DG_GetPVSBoundingBox(ent->s.origin, pmin, pmax);
	       float pvol = BBoxVolume(pmin, pmax);
	       float pmissing = BBoxVolSubtract(pmin, pmax, min, max);
	       float pextra   = BBoxVolSubtract(min, max, pmin, pmax);
	       if (pvol == 0) {
	       e.pvs_frac_vol_missing = 0;
	    // its all extra! but we can't normalize by 0, so just say 1
	    e.pvs_frac_vol_extra   = pextra == 0 ? 0 : 1;
	    } else {
	    e.pvs_frac_vol_missing = pmissing/pvol;
	    e.pvs_frac_vol_extra   = pextra/pvol;
	    }
	     */
	    float pmissing, pextra;
	    float pvol = DG_PVSBoundingBoxesDiff(&pmissing, &pextra,
		    ent->s.origin, min, max);
	    if (pvol == 0) {
		e.pvs_frac_vol_missing = 0;
		// its all extra! but we can't normalize by 0, so just say 1
		e.pvs_frac_vol_extra   = pextra == 0 ? 0 : 1;
	    } else {
		e.pvs_frac_vol_missing = pmissing/pvol;
		e.pvs_frac_vol_extra   = pextra/pvol;
	    }

	    LOG(BBoxErrorLog, e);
	}
    }
}

///////////////////////////////////////////////////////////////////////////////

edict_t *DG_ExistsEntity(guid_t guid)
{
    int i;
    edict_t *ent;

    // XXX
    // Jeff: Why can't we use g_ObjStore->Find() here? I have an additional
    // array storing the non-distributed objects accessible with the function
    // DG_GetNonDistEdict(guid), can we use Find() for other edicts? That
    // would be much more efficient...

    if (DG_IsNonDistributed(guid)) {
	DB(15) << "looking for a non-distributed entity... " << guid << endl;
    }

    for (i = 0; i < game.maxentities; i++) {
	ent = &g_edicts[i];
	if (ent->inuse && ent->guid == guid) {
	    if (DG_IsNonDistributed(guid)) {
		DB(15) << "found a non-distributed entity " 
		    << (ent->classname ? ent->classname : "null") 
		    << "!" << endl;
	    }
	    return ent;
	}
    }
    return NULL;
}

r_gclient_t *_DG_ExistsClient(guid_t guid) {
    int i;
    for (i = 0; i < MAX_REMOTE_CLIENTS; i++) {
	r_gclient_t *p_cl = dg_remote_clients + i;
	if (!p_cl->inuse)
	    continue;

	if (guid == p_cl->guid) {
	    return p_cl;
	}
    }
    return 0;
}

gclient_t *DG_AllocateClient(guid_t guid)
{
    int i;
    r_gclient_t *p_cl = _DG_ExistsClient(guid);
    if (p_cl)
	return &(p_cl->client);

    for (i = 0; i < MAX_REMOTE_CLIENTS; i++) {
	r_gclient_t *p_cl = dg_remote_clients + i;
	if (!p_cl->inuse) {
	    p_cl->inuse = true;
	    p_cl->guid = guid;
	    memset(&(p_cl->client), 0, sizeof(gclient_t));
	    return &(p_cl->client);
	}
    }

    // Nothing? Boy, we have run out of space. Do something sensible.
    // For now, just throw up!
    gi.error("Ran out of remote client space");
    return NULL; // will not come here
}

void DG_DeallocateClient(guid_t guid)
{
    r_gclient_t *rc = _DG_ExistsClient(guid);
    if (rc) {
	memset(rc, 0, sizeof(r_gclient_t));
	rc->inuse = false;
    }
}

/////// for the stoopid acebots
static int _dg_ace_edictnum = 1;

void DG_BeginPlayerIter() {
    _dg_ace_edictnum = 1;
}

edict_t *DG_GetNextPlayer() {
    edict_t *ent = 0;

    // check in the normal edict locations...
    while (_dg_ace_edictnum <= gcvar_maxclients->value) {
	ent = g_edicts + _dg_ace_edictnum;

	_dg_ace_edictnum++;
	if (!ent->inuse || !ent->client)
	    continue;
	return ent;
    }

    while (_dg_ace_edictnum <= gcvar_maxclients->value + MAX_REMOTE_CLIENTS) {
	int i;

	i = _dg_ace_edictnum - gcvar_maxclients->value - 1;    // index into the dg_remote_clients array
	_dg_ace_edictnum++;

	r_gclient_t *p_cl = dg_remote_clients + i;
	if (!p_cl->inuse)
	    continue;

	DG_Edict *dg_obj = (DG_Edict *) g_ObjStore->Find(p_cl->guid);
	if (!dg_obj)
	    continue;

	ent = dg_obj->GetEntity();
	if (!ent)       // i know this wont happen BUT: dont crash for stoopid acebot reasons!    - Ashwin [05/25]
	    continue;
	return ent;
    }

    return NULL;
}
int get_client_delta(gclient_t *newclient, gclient_t *oldclient)
{
    int i = 0, nbytes = 0;
    if (newclient && !oldclient) {
	return sizeof(gclient_t);
    }
    if (!newclient && oldclient) {
	return 1;
    }
    if (!newclient && !oldclient) {
	return 0;
    }

    for (i = 0; i < sizeof(gclient_t); i++) {
	if ( * ((byte *) newclient + i) != * ((byte *) oldclient + i) )
	    nbytes++;
    }
    return nbytes;
}

int get_entity_delta(edict_t *newent, edict_t *oldent)
{
    int nbytes = 0, i = 0;

    for (i = 0; i < sizeof(g_edicts[0]); i++) {
	if ( * ((byte *) newent + i) != * ((byte *) oldent + i) )
	    nbytes++;
    }

    if (newent->client && oldent->client)
	nbytes += get_client_delta(newent->client, oldent->client);

    return nbytes;
}
// the names of the methods may be highly inappropriate... 
// we'll see later...

// 
// This function is called by the DG_HandleNetworkEvents() method
// when it is able to fully deserialize an entity (including 
// resolving EDICT_PTR references)
//
edict_t *DG_HandleObjectUpdate(edict_t *ent, edict_t *newent)
{
    //edict_t *ent = DG_ExistsEntity(newent->guid);

    // 6/30/2004 Jeff: Why the hell do we touch the shadow edicts here?
    // That breaks everything... there must have been a reason because
    // this code doesn't look like it was put here by accident...
    //edict_t *shadow_ent;

    //*delta_size = 0;
    if (!ent) {

	ent = G_Spawn(); // This sets ent->s.number properly
	//shadow_ent = g_shadow_edicts + (ent - g_edicts);

	//        DB(1) << "Adding a remote object to my database at position:" << ent->s.number << endl;

	// must mean this is a "replica" object!
	*ent = *newent; // and this erases ent->s.number. nice.
	ent->s.number = ent - g_edicts;
	ent->status   = DG_OBJ_UNAFFECTED;
	ent->is_distributed = true;

	// is_replica should be set by ConstructEntity and CommitMigration. 
	// Never changes otherwise...
	//ent->is_replica = true;
	ent->is_distributed = true;

	// XXX HACK -- don't double delete this
	newent->multicastEventList = NULL;
	// XXX

	/*
	// Delta BW computation
	 *delta_size = sizeof(g_edicts[0]);
	 if (ent->client)
	 *delta_size += sizeof(gclient_t);

	 if (ent->classname && strstr(ent->classname, "item") ||
	 ent->classname && strstr(ent->classname, "missile") ||
	 ent->classname && strstr(ent->classname, "weapon"))
	 {
	 *delta_size = sizeof(entity_state_t);
	 }
	 */

	//*shadow_ent = *ent;

	if (newent->client) {
	    ent->client = DG_AllocateClient(newent->guid);
	    *(ent->client) = *(newent->client);

	    r_gclient_t *pcl = _DG_ExistsClient(newent->guid);
	    ASSERT(pcl);
	    //shadow_ent->client = dg_shadow_remote_clients + (pcl - dg_remote_clients);
	    //*(shadow_ent->client) = *(ent->client);
	}
	else {
	    ent->client = NULL;
	    //shadow_ent->client = NULL;
	}
    }
    else {
	//  Ashwin [3/18/2004]
	// I had this entity before - so this is either a remote/local update. 
	// An update to a local object should be "resolved, if there is a conflict". 
	// However, we don't know how to resolve right now. So, just overwrite.
	// So, we act as a "serializing center for the writes".
	//
	qboolean replica = ent->is_replica;
	//shadow_ent = g_shadow_edicts + (ent - g_edicts);

	gi.unlinkentity(ent);

	/*
	// Delta BW computation
	 *delta_size = get_entity_delta(newent, ent);
	 */

	*ent = *newent;
	ent->s.number = ent - g_edicts;
	ent->is_replica = replica;
	ent->status = DG_OBJ_UNAFFECTED;
	ent->is_distributed = true;
	//*shadow_ent = *ent;

	// XXX HACK -- don't double delete this
	newent->multicastEventList = NULL;
	// XXX

	if (newent->client) {
	    if (replica) {
		ent->client = DG_AllocateClient(newent->guid);
	    }
	    *(ent->client) = *(newent->client);

	    if (replica) {
		r_gclient_t *pcl = _DG_ExistsClient(newent->guid);
		ASSERT(pcl);
		//shadow_ent->client = dg_shadow_remote_clients + (pcl - dg_remote_clients);
	    }
	    //*(shadow_ent->client) = *(ent->client);
	}
	else {
	    ent->client = NULL;
	    //shadow_ent->client = NULL;
	}
    }

    gi.linkentity(ent);
    return ent;
}

void DG_HandleObjectDelete(guid_t guid)
{
    edict_t *ent = DG_ExistsEntity(guid);

    if (!ent) {
	DBG << "Destroyed object not found locally. Continuing...\n";
	return;
    }

    if (ent->client) {
	// A remote person told us that a client entity has been destroyed. 
	// Which means this must the client THERE has DISCONNECTED. 
	// In other words, this has to be a REMOTE client.
	DG_DeallocateClient(guid);
    }
    //	gi.dprintf("Destroying entity with class: %s\n", (ent->classname ? ent->classname : "null"));

    G_FreeEdict(ent);

    // 6/30/2004 Jeff: Why the F**K is everyone touching the shadow array?
    //edict_t *shadow_ent = g_shadow_edicts + (ent - g_edicts);
    //*shadow_ent = *ent;
}

bool DG_IsNonDistributed(GUID guid) {
    return guid.GetIP() == NON_DIST_IP &&
	guid.GetPort()  == NON_DIST_PORT;
}

