/* vim: set sw=4 ts=4 noet: -*- Mode:c++; c-basic-offset:4; tab-width:4; indent-tabs-mode:t -*- */

#ifndef __DG_LOGS__H
#define __DG_LOGS__H

#include <dg/dg.h>
#include <util/ExpLog.h>
#include <dg/dg_demo.h>

struct QuakeStatsLogEntry : public ExpLogEntry
{
	double frameTime;  // total time between frames
	double frameSlack; // 100 - max(timeToExecFrame, 100)
	double execTime;   // total time to exec frame (excluding idle recv())
	double dgTime;     // total time spent executing dg stuff in frame
	double dgIdleTime; // total time spent in recv() during idle period

	QuakeStatsLogEntry() : 
		frameTime(0), frameSlack(0), execTime(0), dgTime(0) {}
	virtual ~QuakeStatsLogEntry() {}

	uint32 Dump(FILE *fp) {
		return fprintf(fp, "%13.3f\t%.3f\t%.3f\t%.3f\t%.3f\t%.3f\n", 
					   (double)time.tv_sec + 
					   (double)time.tv_usec/USEC_IN_SEC, 
					   frameTime, frameSlack, 
					   execTime, dgTime, dgIdleTime);
	}
};

struct DGStoreLogEntry : public ExpLogEntry
{
    uint16 nr, np, no;
    uint16 p_pl, p_mo, p_it, p_mi, p_ot;
    uint16 r_pl, r_mo, r_it, r_mi, r_ot;
	uint16 frameno;
	
    DGStoreLogEntry() : nr(0), np(0), no(0),
						p_pl(0), p_mo(0), p_it(0), p_mi(0), p_ot(0),
						r_pl(0), r_mo(0), r_it(0), r_mi(0), r_ot(0),
						frameno(0) {};
    virtual ~DGStoreLogEntry() {};
    
    uint32 Dump(FILE *fp) {
		return fprintf(fp, "%13.3f f %hu o %hu "
					   "p %hu pl %hu mo %hu it %hu mi %hu ot %hu "
					   "r %hu pl %hu mo %hu it %hu mi %hu ot %hu\n",
					   (double)time.tv_sec + 
					   (double)time.tv_usec/USEC_IN_SEC,
					   frameno, no,
					   np, p_pl, p_mo, p_it, p_mi, p_ot,
					   nr, r_pl, r_mo, r_it, r_mi, r_ot);
    }
};

struct ObjectInterestEntry : public ExpLogEntry
{
	enum { MISSILE = 'm', PLAYER = 'p', MONSTER = 'o', ITEM = 'i' };

	int    frameno;
	// guid of the object
	GUID   guid;
	// type of object (above)
	char type;
	// the origin point, min and max of predicted bbox
	vec3_t orig;
	// the current subscription
	vec3_t min, max;

	ObjectInterestEntry() {}
	virtual ~ObjectInterestEntry() {}

	uint32 Dump(FILE *fp) {
		return fprintf(fp, "%13.3f\t%d\t%s\t%c"
					   "\t%.3f,%.3f,%.3f"
					   "\t%.3f,%.3f,%.3f"
					   "\t%.3f,%.3f,%.3f\n",
					   (double)time.tv_sec + 
					   (double)time.tv_usec/USEC_IN_SEC,
					   frameno, guid.ToString(), type,
					   orig[0], orig[1], orig[2], 
					   min[0], min[1], min[2], 
					   max[0], max[1], max[2]);
	}
};

// for projecting cost for client/server or broadcast
struct ObjectDeltaEntry : public ExpLogEntry
{
	enum { NEW = 'n', DELETE = 'd', UPDATE = 'u' };

	uint32 frameno;
	GUID   guid;
	char   type;
	uint32 deltaSize;
	uint32 fullSize;

	ObjectDeltaEntry() {}
	virtual ~ObjectDeltaEntry() {}

	uint32 Dump(FILE *fp) {
		return fprintf(fp, "%13.3f\t%d\t%s\t%c\t%d\t%d\n",
					   (double)time.tv_sec + 
					   (double)time.tv_usec/USEC_IN_SEC,
					   frameno, guid.ToString(), type, deltaSize, fullSize);
	}
};

// when each primary object is created or destroyed
struct GUIDLogEntry : public ExpLogEntry
{
	enum { CREATE = 'C', DESTROY = 'D', UPDATE = 'U' };

	char  status;
	GUID guid;
	char *classname;

	GUIDLogEntry() : status(UPDATE) {}
	virtual ~GUIDLogEntry() {}

	uint32 Dump(FILE *fp) {
		

		return fprintf(fp, "%13.3f\t%s\t%s\t%c\n",
					   (double)time.tv_sec + 
					   (double)time.tv_usec/USEC_IN_SEC,
					   guid.ToString(), classname,
					   status);
	}
};

// BBox accuracy log
struct BBoxErrorEntry : public ExpLogEntry
{
	// the origin point, min and max of predicted bbox
	vec3_t orig, min, max;
	// the volume of the predicted bbox as a fraction of the total world
	float  vol;
	// the vol missing from the predicted bbox, as a fraction of
	// the actual directly computed bbox volume
	float  direct_frac_vol_missing;
	// the vol that is extraneous in the predicted box, as a fraction of
	// the actual directly computed bbox volume
	float  direct_frac_vol_extra;
	// the vol that is missing from the predicted bbox compared to the
	// bbox calculated from the pvs
	float  pvs_frac_vol_missing;
	// the vol that is extraneous from the predicted bbox compared to the
	// bbox calculated from the pvs
	float  pvs_frac_vol_extra;

	/*
	// the number of objects missing from the predicted box in the actual box 
	int    objs_missing;
	// the number of objects in the predicted box but not in the actual box
	int    objs_extra;
	*/

	BBoxErrorEntry() {}
	virtual ~BBoxErrorEntry() {}

	uint32 Dump(FILE *fp) {
		return fprintf(fp, "%.3f,%.3f,%.3f\t%.3f,%.3f,%.3f\t%.3f,%.3f,%.3f\t%.3f\t%.3f\t%.3f\t%.3f\t%.3f\n",
					   orig[0], orig[1], orig[2], 
					   min[0], min[1], min[2], 
					   max[0], max[1], max[2], vol,
					   direct_frac_vol_missing, direct_frac_vol_extra,
					   pvs_frac_vol_missing, pvs_frac_vol_extra);
	}
};

// Delta encoding info log
struct DeltaEncodingEntry : public ExpLogEntry
{
	bool  isnew;
	bool  isreplica;
	GUID  guid;
	const char *classname;
	list<string> fields;

	DeltaEncodingEntry() {};
	DeltaEncodingEntry(bool isnew, bool isreplica, GUID& guid,
					   const char *classname) :
		isnew(isnew), isreplica(isreplica), guid(guid), classname(classname) {}
	virtual ~DeltaEncodingEntry() {}

	uint32 Dump(FILE *fp) {
		// this format for calculating what fields to delta encode normally
		fprintf(fp, "[*] %s\t%s\t%d\t", isnew?"NEW":"CHG", classname,
				isreplica?1:0);
		for (list<string>::iterator it = fields.begin() ; 
			 it != fields.end(); it++) {
			if (it != fields.begin()) fprintf(fp, ",");
			fprintf(fp, (*it).c_str());
		}
		fprintf(fp, "\n");
	}
};

// RUsage 
struct RUsageEntry : public ExpLogEntry
{
	int rss, data_rss, faults;
	
	RUsageEntry() {};
	RUsageEntry(int rss, int data_rss, int faults) : 
		rss(rss), data_rss(data_rss), faults(faults) {}
	virtual ~RUsageEntry() {}

	uint32 Dump(FILE *fp) {
		return fprintf(fp, "%13.3f\t%d\t%d\t%d\n",
					   (double)time.tv_sec + 
					   (double)time.tv_usec/USEC_IN_SEC,
					   rss, data_rss, faults);
					   
	}
};

static byte demolog_buf [MAX_MSGLEN];

struct DemoLogEntry : public ExpLogEntry {
	DemoFrame *frame;

	DemoLogEntry() {}
	DemoLogEntry(DemoFrame *df) : frame (df) {}
	
	virtual ~DemoLogEntry() {}
	uint32 Dump(FILE *fp) {
		sizebuf_t	msg;
		SZ_Init (&msg, demolog_buf, sizeof (demolog_buf));
		frame->Record (&msg);

		// strange to delete it here, but i am not very sure how 
		// the copying of log-entries works.
		delete frame;

		int len = LittleLong (msg.cursize);
		fwrite (&len, 4, 1, fp);
		fwrite (msg.data, msg.cursize, 1, fp);

		return 4 + msg.cursize;
	}
};

DECLARE_EXPLOG(QuakeStatsLog, QuakeStatsLogEntry);
DECLARE_EXPLOG(DGStoreLog, DGStoreLogEntry);
DECLARE_EXPLOG(ObjectInterestLog, ObjectInterestEntry);
DECLARE_EXPLOG(ObjectDeltaLog, ObjectDeltaEntry);
DECLARE_EXPLOG(GUIDLog, GUIDLogEntry);

DECLARE_EXPLOG(BBoxErrorLog, BBoxErrorEntry);
DECLARE_EXPLOG(DeltaEncodingLog, DeltaEncodingEntry);
DECLARE_EXPLOG(RUsageLog, RUsageEntry);

DECLARE_EXPLOG(DemoLog, DemoLogEntry);

extern void DG_InitLogs();

#endif
