#include <qcommon/qcommon.h>
#include <game/g_local.h>
#include "dg.h"
#include <util/debug.h>

#define RADIAN(x)     (M_PI * (float) (x) / (float) 180.0)
#define XY_STEP 20 // degrees
#define  Z_STEP 90 // degrees

// how many degrees to "jitter" each ray. the intuition behind jittering
// the rays is that we will be more likely to eventually produce a sub
// to match some bounding box to capture all objects in the "real" box
// (since we stay interested in objects for longer than the lifetime of
// a sub, it doesn't matter than the subs die before new ones are generated)
//
// currently the jitter is a uniform offset; we might want to sample from
// a gaussian disk perpendicular to the ray, but that may be too expensive
#define XY_JITTER 5  // degrees
#define  Z_JITTER 0 // degrees

// make sure the bounding box is at least this wide/high around the ent
#define MIN_BBOX_WIDTH  400
#define MIN_BBOX_HEIGHT 200

ostream& operator<<(ostream& os, vec3_t v)
{
    os << merc_va("<%.2f %.2f %.2f>", v[0], v[1], v[2]);
    return os;
}

//
// min, max are OUT parameters
//
void DG_ComputeVisibilityBox( vec3_t org, vec3_t dir, vec3_t min, vec3_t max )
{
    int step, n_updowns, j;
    vec3_t start, end, direction;
    trace_t tr;
    vec3_t dist;
    int   phi, theta;
    float r_phi, r_theta;

    // minimum bbox

    min[0] = org[0] - MIN_BBOX_WIDTH/2;
    min[1] = org[1] - MIN_BBOX_WIDTH/2;
    min[2] = org[2] - MIN_BBOX_HEIGHT/2;
    
    max[0] = org[0] + MIN_BBOX_WIDTH/2;
    max[1] = org[1] + MIN_BBOX_WIDTH/2;
    max[2] = org[2] + MIN_BBOX_HEIGHT/2;

    //VectorSet( min, 4000, 4000, 4000 );
    //VectorSet( max, -4000, -4000, -4000 );

    for (theta = 0; theta < 360; theta += XY_STEP) {
	r_theta = RADIAN(theta) + XY_JITTER-2*XY_JITTER*drand48();
	    
	for (phi = -90; phi <= 90; phi += Z_STEP) {
	    r_phi = RADIAN(phi) + Z_JITTER-2*Z_JITTER*drand48();

	    direction[0] = cos(r_phi) * cos(r_theta);
	    direction[1] = cos(r_phi) * sin(r_theta);
	    direction[2] = sin(r_phi);
		
	    VectorMA(org, 2, direction, start);
	    VectorMA(org, 2*4082, direction, end);
		
	    tr = gi.trace( start, vec3_origin, vec3_origin, end, 0, MASK_OPAQUE );
			
	    if ( tr.endpos[0] < min[0] ) min[0] = tr.endpos[0];
	    if ( tr.endpos[0] > max[0] ) max[0] = tr.endpos[0];
	    if ( tr.endpos[1] < min[1] ) min[1] = tr.endpos[1];
	    if ( tr.endpos[1] > max[1] ) max[1] = tr.endpos[1];
	    if ( tr.endpos[2] < min[2] ) min[2] = tr.endpos[2];
	    if ( tr.endpos[2] > max[2] ) max[2] = tr.endpos[2];

	    DB_DO(10) {
		vec3_t dist;
		float  mag_dist;
		    
		VectorSubtract( tr.endpos, org, dist );
		mag_dist = sqrt(DotProduct(dist, dist));

		DB(1) << merc_va("phi: %d, theta: %d, dist: %.3f", phi, theta, mag_dist) << endl;
	    }
		    
	}
    }

    /*

    ARGH! The BBox computing from the PVS is unsuable for anything. Its almost
    always the entire damn map! Probably because of crappy map editor 
    clustering of leaf nodes???

    vec3_t min2;
    vec3_t max2;

    CM_BoundingBox(min2, max2, org);

    vec3_t dims;
    vec3_t dims2;
    for (int i=0; i<3; i++) {
	dims[i]  = max[i] - min[i];
	dims2[i] = max2[i] - min2[i];
    }

    DB(0) << "rtrace: " << min << "\t" << max << "\t" << dims << endl;
    DB(0) << "bspbox: " << min2 << "\t" << max2 << "\t" << dims2 << endl;
    */

    // make conservative
    //    for  (j = 0; j < 3; j++) {
    //	min[j] -= 0.3 * min[j];
    //	max[j] += 0.3 * max[j];
    //    }
    DB(10) << "finally, min = " << min << "; max = " << max << endl;
}

bool _DG_is_monster(edict_t *ent)
{
    monsterinfo_t *minfo = &(ent->monsterinfo);

    // if this is a monster, one of these should be better be non-NULL
    return (minfo->stand || minfo->walk || minfo->run || minfo->attack);
}

bool _DG_is_inBox(vec3_t p, vec3_t min, vec3_t max)
{
    return (min[0] <= p[0] && p[0] <= max[0]
	    && min[1] <= p[1] && p[1] <= max[1]
	    && min[2] <= p[2] && p[2] <= max[2]);
}

void _DG_CheckVisibilityOK(edict_t *checkfor, vec3_t min, vec3_t max)
{
    edict_t *ent = &g_edicts[1];
    int invisible_in = 0, visible_in = 0, visible_out = 0;
    
    for (int i = 1; i < game.maxentities; i++, ent++) {
	if (!ent->inuse)
	    continue;
	
	if (ent == checkfor)
	    continue;

	// adds viewheight to ent->s.origin
	if (visible(checkfor, ent)) {
	    if (_DG_is_inBox(ent->s.origin, min, max)) {
		visible_in++;
	    }
	    else {
		DB(10) << "###### visible out entity: " << (ent->classname ? ent->classname : "null")
		      << " at position: " << ent->s.origin << endl;
		visible_out++;
	    }
	}
	else { // this entity is not visible;
	    if (_DG_is_inBox(ent->s.origin, min, max)) {
		invisible_in++;
	    }
	}
    }

    DB(1) << merc_va("visible_in: %d, visible_out: %d, invisible_in: %d",
		     visible_in, visible_out, invisible_in) << endl;
}

