#include <qcommon/qcommon.h>
#include <util/debug.h>
#include <util/OS.h>
#include <dg/dg.h>
#include <dirent.h>
#include <server/server.h>
#include <all_common.h>
#include <dg/dg_demo.h>
#include <dg/dg_logs.h>

// declarations
int Com_ParseEntityBits (sizebuf_t *msg, unsigned *bits);
void Com_ParseDelta (sizebuf_t *msg, entity_state_t *from, entity_state_t *to, int number, int bits); 
void Com_ParsePlayerState (sizebuf_t *msg, player_state_t *from, player_state_t *to);

entity_state_t     g_enostate;
player_state_t     g_pnostate;

#define FRAME_LIMIT 1000

/// XXX: I should really make scripts/putils.cxx into a useful sub-module in util/
static bool matches (char *pat, char *what) { 
    return strstr (what, pat) != NULL;
}

static void glob (char *pat, vector<string> *files) {
    DIR *dir = opendir (g_QuakePreferences.dg_demo_dir);
    if (!dir) {
	perror ("opendir");
	exit (1);
    }

    struct dirent *ent;
    while ((ent = readdir (dir))) {
	if (matches (pat, ent->d_name)) {
	    string s = g_QuakePreferences.dg_demo_dir;
	    s.append ("/");
	    s.append (ent->d_name);
	    files->push_back (s);
	}
    }
    closedir (dir);
}

static byte buf [MAX_MSGLEN];

static void _InitMessage (sizebuf_t *msg) {
    SZ_Init (msg, buf, sizeof (buf));
}

static int _ReadMessage (sizebuf_t *msg, FILE *fp)
{
    int msglen = 0;
    int r;

    _InitMessage (msg);
    r = fread (&msglen, 4, 1, fp);
    if (r != 1) 
	return -1;

    msglen = LittleLong (msglen);
    if (msglen == -1) 
	return -1;

    if (msglen > MAX_MSGLEN)
	Debug::die ( "SV_SendClientMessages: msglen > MAX_MSGLEN");

    r = fread (msg->data, msglen, 1, fp);
    msg->cursize = msglen;
    msg->readcount = 0;

    if (r != 1)
	return -1;

    return msglen;
}

static void _WriteMessage (sizebuf_t *msg, FILE *fp) {
    int len = LittleLong (msg->cursize);
    fwrite (&len, 4, 1, fp);
    fwrite (msg->data, msg->cursize, 1, fp);

    fflush (fp);
}

/* call this when the demo files are combined */
FILE* SVDG_DemoInit (char *name, int playernum) {
    char	buf_data[32768];
    sizebuf_t	buf;
    int		len;
    int		i;
    FILE *fp;

    FS_CreatePath (name);
    fp = fopen (name, "wb");
    if (!fp) {
	Debug::die (merc_va ("ERROR: couldn't open %s for writing.\n", name));
    }

    // setup a buffer to catch all multicasts
    SZ_Init (&svs.demo_multicast, svs.demo_multicast_buf, sizeof(svs.demo_multicast_buf));

    //
    // write a single giant fake message with all the startup info
    //
    SZ_Init (&buf, (byte *) buf_data, sizeof(buf_data));

    //
    // serverdata needs to go over for all types of servers
    // to make sure the protocol is right, and to set the gamedir
    //
    // send the serverdata
    MSG_WriteByte (&buf, svc_serverdata);
    MSG_WriteLong (&buf, PROTOCOL_VERSION);
    MSG_WriteLong (&buf, svs.spawncount);
    // 2 means server demo
    MSG_WriteByte (&buf, 2);	// demos are always attract loops
    MSG_WriteString (&buf, Cvar_VariableString ("gamedir"));
    MSG_WriteShort (&buf, playernum);
    // send full levelname
    MSG_WriteString (&buf, sv.configstrings[CS_NAME]);

    for (i=0 ; i<MAX_CONFIGSTRINGS ; i++)
	if (sv.configstrings[i][0])
	{
	    MSG_WriteByte (&buf, svc_configstring);
	    MSG_WriteShort (&buf, i);
	    MSG_WriteString (&buf, sv.configstrings[i]);
	}

    // write it to the demo file
    Com_DPrintf ("signon message length: %i\n", buf.cursize);
    len = LittleLong (buf.cursize);
    fwrite (&len, 4, 1, fp);
    fwrite (buf.data, buf.cursize, 1, fp);

    // the rest of the demo file will be individual frames

    return fp;
}

void MSG_WriteGUID (sizebuf_t *buf, guid_t guid) {
    MSG_WriteLong (buf, guid.GetIP ());
    MSG_WriteShort (buf, guid.GetPort ());
    MSG_WriteLong (buf, guid.GetLocalOID ());
}

guid_t MSG_ReadGUID (sizebuf_t *buf) {
    uint32 ip = MSG_ReadLong (buf);
    uint16 port = MSG_ReadShort (buf);
    uint32 uid = MSG_ReadLong (buf);

    guid_t g (ip, port, uid);
    return g;
}

void MSG_WriteBool (sizebuf_t *buf, bool b) { 
    byte bb = b ? 0x1 : 0x0;
    MSG_WriteByte (buf, bb);
}

bool MSG_ReadBool (sizebuf_t *buf) {
    byte b = MSG_ReadByte (buf);
    return b != 0x0;
}

ostream& operator<<(ostream& out, const vec3_t &v) { 
    out << "[" << merc_va ("%.2f", v[0]);
    out << ", " << merc_va ("%.2f", v[1]);
    out << ", " << merc_va ("%.2f", v[2]) << "]";
    return out;
}

DemoEntity::DemoEntity (edict_t *ent) { 
    is_replica = ent->is_replica;
    is_client  = ent->client != NULL;
    guid = ent->guid;
    es   = ent->s;
    if (is_client) {
	ps = ent->client->ps;
	ComputeAreaVisibility (ent);
    }
}

DemoEntity::DemoEntity (sizebuf_t *msg) {
    is_replica = MSG_ReadBool (msg);
    guid = MSG_ReadGUID (msg);

    unsigned bits;
    int number = Com_ParseEntityBits (msg, &bits);
    Com_ParseDelta (msg, &g_enostate, &es, number, bits);

    is_client = MSG_ReadBool (msg);
    // INFO << "Entity number=" << number << " guid=" << guid << " is_replica=" << is_replica << " is_client=" << is_client << endl;
    if (is_client) {
	byte cmd = MSG_ReadByte (msg);
	ASSERT (cmd == svc_playerinfo);

	Com_ParsePlayerState (msg, &g_pnostate, &ps);
	areabytes = MSG_ReadByte (msg);
	MSG_ReadData (msg, areabits, areabytes);
    }
}

void DemoEntity::SetEntityNumber (int newnum) { 
    es.number = newnum;
}

void DemoEntity::Record (sizebuf_t *msg) {
    MSG_WriteBool (msg, is_replica);
    MSG_WriteGUID (msg, guid);
    MSG_WriteDeltaEntity (&g_enostate, &es, msg, false, true);

    MSG_WriteBool (msg, is_client);
    if (is_client) {
	// this writes a svc_playerinfo byte as well
	SV_WriteDeltaPlayerstate (&g_pnostate, &ps, msg);

	MSG_WriteByte (msg, areabytes);
	SZ_Write (msg, areabits, areabytes);
    }
}

void DemoEntity::RecordPlayerState (sizebuf_t *msg) {
    ASSERT (is_client);

    MSG_WriteByte (msg, areabytes);
    SZ_Write (msg, areabits, areabytes);

    SV_WriteDeltaPlayerstate (&g_pnostate, &ps, msg);
}

void DemoEntity::RecordForClient (sizebuf_t *msg) {
    MSG_WriteDeltaEntity (&g_enostate, &es, msg, false, true);
}

ostream& DemoEntity::Print (ostream& os) const {
    return os << "guid=" << guid << " number=" << es.number << (is_replica ? " (replica) " : " (primary) ") 
	<< (is_client ? " [player] " : " [object] ") << " origin=" << es.origin << " modelindex=" << es.modelindex ;
}
int DemoEntity::ComputeAreaVisibility (edict_t *ent) {
    vec3_t org;
    int clientarea, clientcluster;
    int leafnum;

    for (int i=0 ; i<3 ; i++)
	org[i] = ent->client->ps.pmove.origin[i]*0.125 + ent->client->ps.viewoffset[i];

    leafnum = CM_PointLeafnum (org);
    clientarea = CM_LeafArea (leafnum);

    // calculate the visible areas
    areabytes = CM_WriteAreaBits (areabits, clientarea);
}

ostream& operator<<(ostream& out, const DemoEntity &de) { return out << &de; }
ostream& operator<<(ostream& out, const DemoEntity *de) { return de->Print (out); }

DemoFrame::DemoFrame (int framenum) : framenum (framenum) {
    multicast_data = new byte [MAX_MSGLEN];
    multicast_size = 0;
    frametime = -1.0;
}

DemoFrame::DemoFrame (game_export_t *ge) {
    int e;
    edict_t *ent;

    framenum = sv.framenum;

    e = 1;
    ent = EDICT_NUM(e);
    while (e < ge->num_edicts) 
    {
	// ignore ents without visible models unless they have an effect
	if (ent->inuse &&
		ent->s.number && 
		(ent->s.modelindex || ent->s.effects || ent->s.sound || ent->s.event) && 
		!(ent->svflags & SVF_NOCLIENT)) {

	    DemoEntity *de = new DemoEntity (ent);
	    dentities.push_back (de);
	}

	e++;
	ent = EDICT_NUM(e);
    }

    // now add the accumulated multicast information
    multicast_size = svs.demo_multicast.cursize;
    multicast_data = new byte [multicast_size];
    memcpy (multicast_data, svs.demo_multicast.data, multicast_size);

    SZ_Clear (&svs.demo_multicast);

    TimeVal now;
    OS::GetCurrentTime (&now);
    frametime = (double) now.tv_sec + (double) now.tv_usec / USEC_IN_SEC;
}

DemoFrame::DemoFrame (sizebuf_t *msg) {
    frametime = MSG_ReadFloat (msg);

    int nents = MSG_ReadLong (msg);
    DB (10) << "reconstructing frame; going to read " << nents << " entities" << endl;
    for (int i = 0; i < nents; i++) {
	DemoEntity *de = new DemoEntity (msg);
	dentities.push_back (de);
    }

    multicast_size = MSG_ReadLong (msg);
    multicast_data = new byte [multicast_size];
    MSG_ReadData (msg, multicast_data, multicast_size);
}

DemoFrame::~DemoFrame () { 
    for (DEVecIter it = dentities.begin (); it != dentities.end (); ++it)
	delete *it;

    if (multicast_data)
	delete[] multicast_data;
    multicast_size = -1;
}

void DemoFrame::AddEntity (DemoEntity *de) { 
    dentities.push_back (new DemoEntity (*de));
}
void DemoFrame::AddMulticastEvent (byte *data, int len) { 
    ASSERT (multicast_size + len < MAX_MSGLEN);

    memcpy (multicast_data + multicast_size, data, len);
    multicast_size += len;
}

void DemoFrame::Record (sizebuf_t *msg) {
    ASSERT (frametime > 0.0);
    MSG_WriteFloat (msg, frametime);

    MSG_WriteLong (msg, dentities.size ());	  
    for (DEVecIter it = dentities.begin (); it != dentities.end (); ++it)
	(*it)->Record (msg);

    MSG_WriteLong (msg, multicast_size);	  
    SZ_Write (msg, multicast_data, multicast_size);
}

// This has to be very carefully done!

void DemoFrame::RecordForClient (sizebuf_t *msg, guid_t player_guid) {
    MSG_WriteByte (msg, svc_frame);
    MSG_WriteLong (msg, framenum);
    MSG_WriteLong (msg, -1);	                // what we are delta'ing from
    MSG_WriteByte (msg, 0x0);                       // rate dropped packets

    bool found = false;
    for (DEVecIter it = dentities.begin (); it != dentities.end (); ++it) {
	DemoEntity *de = *it;
	if (de->IsClient () && de->GetGUID () == player_guid) {
	    de->RecordPlayerState (msg);
	    found = true;
	    break;
	}
    }
    ASSERTDO (found, { WARN << "#dents " << dentities.size () << endl; });

    MSG_WriteByte (msg, svc_packetentities);
    DB (20) << "frame=" << GetFrameNum () << endl;

    for (DEVecIter it = dentities.begin (); it != dentities.end (); ++it) {
	DemoEntity *de = *it;

	if (de->IsClient () && de->GetGUID () == player_guid)
	    continue;

	ASSERTDO (msg->cursize < MAX_MSGLEN, cerr << " msg.cursize=" << msg->cursize << endl);
	de->RecordForClient (msg);
	DB (20) << de << endl;
    }

    MSG_WriteShort (msg, 0x0);        // end of packetentities
    ASSERTDO (msg->cursize < MAX_MSGLEN, cerr << " msg.cursize=" << msg->cursize << endl);

    // FIXME: No Multicast data.. this should be sent in some other part of the packet...  
    //
    // MSG_WriteLong (msg, multicast_size);	  
    // SZ_Write (msg, multicast_data, multicast_size);
}

ostream& DemoFrame::Print (ostream& os) const {
    os << "frame " << framenum << " #entities=" << dentities.size () << endl;
    for (DEVec::const_iterator it = dentities.begin (); it != dentities.end (); ++it) {
	DemoEntity *de = (*it);
	de->Print (os);
	os << endl;
    }
    return os;	
}

ostream& operator<<(ostream& out, const DemoFrame &df) { return out << &df; }
ostream& operator<<(ostream& out, const DemoFrame *df) { return df->Print (out); }

static void _InitGlobals () {
    memset (&g_enostate, 0, sizeof (g_enostate));
    memset (&g_pnostate, 0, sizeof (g_pnostate));
}

void SVDG_BeginRecordDemo () { 
    if (!g_QuakePreferences.dg_record_demo) 
	return;

    SZ_Init (&svs.demo_multicast, svs.demo_multicast_buf, sizeof(svs.demo_multicast_buf));
    _InitGlobals ();
}

void SVDG_EndRecordDemo () { 
}

void SVDG_RecordDemoFrame () {
    DemoFrame *df = new DemoFrame (ge);
    DemoLogEntry ent (df);
    g_DemoLog->Log (ent);  // XXX: demo log will delete the frame!
}

typedef map<GUID, int, less_GUID>  GUIDNumMap;
typedef GUIDNumMap::iterator GUIDNumMapIter;

GUIDNumMap g_EntNumMap;
int g_CurEntNum;

void _InitEntMap () { 
    g_EntNumMap.clear ();
    g_CurEntNum = MAX_CLIENTS;        // The world has entity number '0'
}

void _AssignEntityNumber (DemoEntity *de) {
    int ent_num = -1;

    guid_t g = de->GetGUID ();
    GUIDNumMapIter it = g_EntNumMap.find (g);

    
    if (it == g_EntNumMap.end ()) {
	ent_num = g_CurEntNum++;

	ASSERTDO (ent_num < MAX_EDICTS, { WARN << "de=" << de << " entnum_assigned=" << ent_num << endl; });
	g_EntNumMap.insert (GUIDNumMap::value_type (g, ent_num));
    }
    else {
	ent_num = it->second;
    }

    de->SetEntityNumber (ent_num);
    DB (20) << de << endl;
}

// Also creates baselines for the primaries 
// what an ugly name :P

static DemoEntity *GetViewerGUID_and_CreateBaselines (vector<DemoFrame *>& frames) {
    vector<DemoEntity *> li;
    
    memset (sv.baselines, 0, MAX_EDICTS * sizeof (entity_state_t));

    for (vector<DemoFrame *>::iterator it = frames.begin (); it != frames.end (); ++it) {
	DEVec& dents = (*it)->GetEntities ();
	for (DEVecIter dit = dents.begin (); dit != dents.end (); ++dit) {
	    DemoEntity *de = *dit;
	    
	    if (DG_IsNonDistributed (de->GetGUID ()))
		continue;
	    if (de->IsReplica ())
		continue;
	    li.push_back (de);

	    // assign a unique entity number
	    _AssignEntityNumber (de);
	    ASSERT (de->GetENumber () < MAX_EDICTS);

	    entity_state_t es = de->GetEntityState ();
	    
	    VectorCopy (es.origin, es.old_origin);
	    sv.baselines [de->GetENumber ()] = es;
	}
    }

    ASSERT (li.size () > g_QuakePreferences.dg_demo_player);
    DemoEntity *de = li [g_QuakePreferences.dg_demo_player];
    
    INFO << "Choose guid " << de->GetGUID () << " as the guid for the viewing player " << endl;
    
    return de;
}

static void TrueDemo (DemoFrame *ff, guid_t player_guid, vector<DemoFrame *>& frames, FILE *out) {
    
    for (vector<DemoFrame *>::iterator it = frames.begin (); it != frames.end (); ++it) {
	DEVec& dents = (*it)->GetEntities ();
	
	for (DEVecIter dit = dents.begin (); dit != dents.end (); ++dit) {
	    DemoEntity *de = *dit;
	    
	    bool toadd = false;
	    
	    if (DG_IsNonDistributed (de->GetGUID ())) {
		// these objects are replicated everywhere, and dont change state
		// hence they can be picked up from any frame!
		if (it == frames.begin ()) 
		    toadd = true;
	    }
	    else { 
		// otherwise, only choose primary objects
		if (!de->IsReplica ())
		    toadd = true;
	    }

	    if (toadd) {
		DB (10) << "adding entity " << de << " to frame " << ff->GetFrameNum () << endl;
		_AssignEntityNumber (de);
		ff->AddEntity (de);
	    }
	}
    }
}

static void DistDemo (DemoFrame *ff, guid_t player_guid, vector<DemoFrame *>& frames, FILE *out) {
    // we just need to copy the frame which contains the player_guid here
    DemoFrame *needed = NULL;

    for (vector<DemoFrame *>::iterator it = frames.begin (); it != frames.end (); ++it) {
	DEVec& dents = (*it)->GetEntities ();

	for (DEVecIter dit = dents.begin (); dit != dents.end (); ++dit) {
	    DemoEntity *de = *dit;

	    if (!de->IsReplica () && de->GetGUID () == player_guid) { // got it!		
		needed = *it;
		break;
	    }
	}
	if (needed)
	    break;
    }

    ASSERT (needed != NULL);
    DEVec& dents = needed->GetEntities ();

    cerr << "frame " << ff->GetFrameNum ();
    for (DEVecIter dit = dents.begin (); dit != dents.end (); ++dit) {
	DemoEntity *de = *dit;
	_AssignEntityNumber (de);
	cerr << de << endl;
	ff->AddEntity (de);
    }
    cerr << "=================================" << endl;
}

int h_frameDelay;
list<DemoFrame *> history;

static void ServDemo (DemoFrame *ff, guid_t player_guid, vector<DemoFrame *>& frames, FILE *out) {
    // servdemo is a delayed version of truedemo. construct a true demo frame. save it off in an
    // array. return it after h_frameDelay

    DemoFrame *x = new DemoFrame (ff->GetFrameNum ());    
    TrueDemo (x, player_guid, frames, out);

    history.push_back (x);

    // HACK. The initial h_frameDelay frames are just undelayed.

    DemoFrame *ret = history.front ();
    bool popped = false;
    if (history.size () > h_frameDelay) {
	popped = true;
	history.pop_front ();
    }
	
    DEVec& dents = ret->GetEntities ();
    for (DEVecIter it = dents.begin (); it != dents.end (); ++it) 
	ff->AddEntity (*it);

    if (popped)
	delete ret;
}

static void _OpenFiles (vector<string>& demoFiles, vector<FILE *>& demoFPs) {
    glob ("Demo", &demoFiles);

    for (vector<string>::iterator it = demoFiles.begin (); it != demoFiles.end (); ++it) {
	string s = "/usr/bin/gunzip -c ";
	s.append (*it);
	FILE *fp = popen (s.c_str (), "r");
	demoFPs.push_back (fp);
    }
}

static void _CloseFiles (vector<FILE *>& demoFPs) {
    for (vector<FILE *>::iterator it = demoFPs.begin (); it != demoFPs.end (); ++it) {
	pclose (*it);
    }
}

static bool ReadAllFrames (vector<string>& demoFiles, vector<FILE *>& demoFPs, vector<DemoFrame *>& frames) {
    sizebuf_t msg;

    int i = 0;
    for (vector<FILE *>::iterator it = demoFPs.begin (); it != demoFPs.end (); ++it, ++i) {
	int msglen = _ReadMessage (&msg, *it);
	if (msglen < 0) 
	    return false;

	DB (10) << "read message of length=" << msglen << " from file " << demoFiles [i] << endl;
	DemoFrame *df = new DemoFrame (&msg);
	frames.push_back (df);
    }
}

static bool FreeAllFrames (vector<DemoFrame *>& frames) {
    for (vector<DemoFrame *>::iterator it = frames.begin (); it != frames.end (); ++it) {
	delete *it;
    }
}

typedef void (*MergingFunc) (DemoFrame *, guid_t, vector<DemoFrame *>&, FILE *);

void MergeDemo (MergingFunc merge_func) { 
    vector<string> demoFiles;
    vector<FILE *> demoFPs;
    FILE *out;
    sizebuf_t msg;

    // initialize and open files, for reading and writing
    _InitGlobals ();
    _OpenFiles (demoFiles, demoFPs);

    ASSERT (g_QuakePreferences.dg_demo_output[0] != '\0');
    _InitEntMap ();

    bool done;
    bool found_player = false;
    int framenum = 0;
    guid_t player_guid;
    
    do {
	done = false;
	vector<DemoFrame *> frames;
	
	// read one frame each from all files. if any file fails, bail.
	if (!ReadAllFrames (demoFiles, demoFPs, frames)) 
	    done = true;

	if (!done) {
	    if (!found_player) { 
		// locate the player_guid if not already located, based upon 
		// g_QuakePreferences.dg_demo_player
		
		DemoEntity *de = GetViewerGUID_and_CreateBaselines (frames);

		player_guid = de->GetGUID ();
		out = SVDG_DemoInit (g_QuakePreferences.dg_demo_output, de->GetENumber ());
		found_player = true;
	    }

	    // do the actual stuff here.
	    DemoFrame *ff = new DemoFrame (framenum);
	    (*merge_func) (ff, player_guid, frames, out);
	    
	    // serialize the frame into a message to be sent directly to the client
	    _InitMessage (&msg);
	    ff->RecordForClient (&msg, player_guid);
	    _WriteMessage (&msg, out);

	    delete ff;
	}

	FreeAllFrames (frames);
	framenum++;
	if (framenum % 50 == 0) 
	    fprintf (stderr, "%d...\n", framenum);

	if (framenum > FRAME_LIMIT) 
	    done = true;

    } while (!done);

    fclose (out);
    _CloseFiles (demoFPs);
}

void SVDG_MergeDemos () {
    char *type = g_QuakePreferences.dg_merge_demos;
    if (strstr (type, "true"))
	MergeDemo (TrueDemo);
    else if (strstr (type, "dist"))
	MergeDemo (DistDemo);
    else if (strstr (type, "serv")) {
	char *dstr = strchr (type, '-');
	ASSERT (dstr != NULL);
	dstr++;

	history.clear ();
	h_frameDelay = atoi (dstr);
	MergeDemo (ServDemo);
    }
    else {
	Com_Error (ERR_FATAL, "Invalid merge demo option given");
    }
}

// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
