/* vim: set sw=4 ts=4 noet: -*- Mode:c++; c-basic-offset:4; tab-width:4; indent-tabs-mode:t -*- */

#ifndef __DG_BBOX__H
#define __DG_BBOX__H

#include "dg.h"
#include <om/OMEvent.h>
#include <om/Manager.h>

///////////////////////////////////////////////////////////////////////////////
//
// Options for DG_ComputeVisibilityBox
//

#define XY_STEP 20 // degrees (default value)
#define  Z_STEP 90 // degrees (default value)

// how many degrees to "jitter" each ray. the intuition behind jittering
// the rays is that we will be more likely to eventually produce a sub
// to match some bounding box to capture all objects in the "real" box
// (since we stay interested in objects for longer than the lifetime of
// a sub, it doesn't matter than the subs die before new ones are generated)
//
// currently the jitter is a uniform offset; we might want to sample from
// a gaussian disk perpendicular to the ray, but that may be too expensive
#define XY_JITTER 0 // degrees
#define  Z_JITTER 0 // degrees

// make sure the bounding box is at least this wide/high around the ent
#define MIN_BBOX_WIDTH  0
#define MIN_BBOX_HEIGHT 0

///////////////////////////////////////////////////////////////////////////////
//
// Options for bounding box precomputation
//

// magic string that we begin bbox files with
#define MAGIC_STRING "q2_bbox_map"
// define if we output/input bbox file in binary
#define COMPACT_FORMAT
// allow for floating point correction
#define EPSILON 1e-6

ostream& operator<<(ostream& os, vec3_t v);

///////////////////////////////////////////////////////////////////////////////
// Use shared memory for loading the bboxes. This way, multiple virtual nodes on 
// the same machine can use the same memory -- this is only there for experiments. 
// And, is quite hacky. 
//
// Since the "flags" byte in the bbox_t structure can be mutated, 
// we will use a separate per-process array for this (only if we 
// are using shared memory).
// 
#define USE_SHARED_MEMORY

typedef struct {
#ifndef USE_SHARED_MEMORY
	byte   flags;
#endif
	vec3_t min, max;
} bbox_t;

#ifdef USE_SHARED_MEMORY

//////////////////////////////////////////////////////////////////////////////
// We manage our allocated memory here; 
// 
#define GET_OFFSET(x, y, z)     ((( x * g_ybuckets ) + y) * g_zbuckets + z)
#define GET_BBOX(x, y, z)       (* (bbox_t *) ((byte *) g_bboxes + GET_OFFSET(x, y, z) * sizeof(bbox_t)))

extern byte   ***g_bbox_flags;
void   Shmbbox_GetIndices(bbox_t *addr, int *x, int *y, int *z);
byte  &Shmbbox_GetFlags(bbox_t *addr);

#define GET_FLAGS(box)          Shmbbox_GetFlags(&box)

#else // USE_SHARED_MEMORY
#define GET_BBOX(x, y, z)       (g_bboxes[x][y][z])
#define GET_FLAGS(box)          (box.flags)
#endif // USE_SHARED_MEMORY

/////////////////////////////////////////////////////////////////////////////////
#define BBOX_VALID_BIT 0x1
#define BBOX_MARK_BIT  0x2

#define BBOX_SET_VALID(bbox, val) ( (val) ? GET_FLAGS(bbox) |= BBOX_VALID_BIT : \
									GET_FLAGS(bbox) &= ~BBOX_VALID_BIT )
#define BBOX_IS_VALID(bbox)       ( GET_FLAGS(bbox) & BBOX_VALID_BIT )
#define BBOX_SET_MARK(bbox, val)  ( (val) ? GET_FLAGS(bbox) |= BBOX_MARK_BIT : \
									GET_FLAGS(bbox) &= ~BBOX_MARK_BIT )
#define BBOX_IS_MARK(bbox)        ( GET_FLAGS(bbox) & BBOX_MARK_BIT )

#define RADIAN(x)     (M_PI * (float) (x) / (float) 180.0)

///////////////////////////////////////////////////////////////////////////////
extern int *g_CoordIndices;

extern Manager *g_Manager;
extern int g_DHTHubIndex;
extern vec3_t g_worldmin, g_worldmax;

template <class T>
void BBoxToType(T *elt, bbox_t *bbox)
{
	// assume "x" -> 0, "y" -> 1, "z" -> 2   - Ashwin [03/07/2005]
	
	for (int i = 0; i < 3; i++) { 
		int j = g_CoordIndices[i];

		Value minv = g_Manager->ConvertToValue (j, bbox->min[i], g_worldmin[i], g_worldmax[i]);

		ASSERTDO (bbox->max[i] >= -1.0e+9, WARN << "i=" << i << " max=" << bbox->max[i] << endl);
		Value maxv = g_Manager->ConvertToValue (j, bbox->max[i], g_worldmin[i], g_worldmax[i]);
		
		Constraint c (j, minv, maxv);
		elt->AddConstraint (c);
	}
}

template <class T>
void TypeToBBox(bbox_t *bbox, T *elt)
{
	for (int i = 0; i < 3; i++) {
		int j = g_CoordIndices[i];

		Constraint *c = elt->GetConstraintByAttr (j);
		if (c == NULL)
			continue;

		bbox->min[i] = g_Manager->ConvertFromValue (j, c->GetMin (), g_worldmin[i], g_worldmax[i]);
		bbox->max[i] = g_Manager->ConvertFromValue (j, c->GetMax (), g_worldmin[i], g_worldmax[i]);
	}
}

template <class T>
float TypeToDHTKey(T *elt)
{
	for (int i = 0, len = elt->GetNumConstraints (); i < len; i++) {
		const Constraint *c = elt->GetConstraint (i);
		int attr = c->GetAttrIndex ();

		if (attr == g_DHTHubIndex) 
			return g_Manager->ConvertFromValue (g_DHTHubIndex, c->GetMin (), 0.0, 1.0);
	}
	
	ASSERT(0);
	return 0;
}

template<class T>
void DHTKeyBBoxToType(T *elt, float key, bbox_t *bbox)
{
	BBoxToType<T>(elt, bbox);

	Value v = g_Manager->ConvertToValue (g_DHTHubIndex, key, 0.0, 1.0);
	Constraint c (g_DHTHubIndex, v, v);
	elt->AddConstraint (c);
}

///////////////////////////////////////////////////////////////////////////////

#define BBoxToInterest(elt, bbox) BBoxToType<OMInterest>(elt, bbox)
#define InterestToBBox(bbox, elt) TypeToBBox<OMInterest>(bbox, elt)

#define BBoxToEvent(elt, bbox) do { \
   BBoxToType<OMEvent>(elt, bbox); \
} while (false)

#define EventToBBox(bbox, elt) do { \
   TypeToBBox<OMEvent>(bbox, elt); \
} while (false)

void PointToEvent(OMEvent *ev, vec3_t pt);
void EventToPoint(vec3_t pt, OMEvent *ev);

///////////////////////////////////////////////////////////////////////////////

#define InterestToDHTKey(elt) TypeToDHTKey<OMInterest>(elt)
#define DHTKeyBBoxToInterest(elt, key, bbox) DHTKeyBBoxToType<OMInterest>(elt,key,bbox)

#define RangeEventToDHTKey(elt) TypeToDHTKey<OMEvent>(elt)
#define DHTKeyBBoxToEvent(elt, key, bbox) do { \
   DHTKeyBBoxToType<OMEvent>(elt,key,bbox); \
} while (false)

float PointEventToDHTKey(OMEvent *ev);
void DHTKeyPointToEvent(OMEvent *ev, float key, vec3_t pt);

int BBoxToNormalizedDHTPoints(float *keys, int nkeys, bbox_t *bbox);

///////////////////////////////////////////////////////////////////////////////

bool BBoxContains(bbox_t *box1, bbox_t *box2);
bool BBoxOverlaps(bbox_t *box, vec3_t min, vec3_t max);
inline bool BBoxOverlaps(bbox_t *bbox1, bbox_t *bbox2) {
	BBoxOverlaps(bbox1, bbox2->min, bbox2->max);
}
float Overlap(bbox_t *box1, bbox_t *box2, int dim);
float Overlap(bbox_t *out, bbox_t *box1, bbox_t *box2);
float BBoxVolume(bbox_t *bbox);
float BBoxVolSubtract(bbox_t *bbox1, bbox_t *bbox2);

float BBoxVolume(vec3_t min, vec3_t max);
float BBoxVolSubtract(vec3_t min1, vec3_t max1, vec3_t min2, vec3_t max2);
float Distance(vec3_t from, vec3_t to);

inline void PointToBBox(bbox_t *bbox, vec3_t pt) {
	VectorCopy(pt, bbox->min);
	VectorCopy(pt, bbox->max);
}
inline bool BBoxContains(bbox_t *bbox, vec3_t pt) {
	for (int i=0; i<3; i++) {
		if (bbox->min[i] > pt[i] || bbox->max[i] < pt[i])
			return false;
	}
	return true;
}

///////////////////////////////////////////////////////////////////////////////

float PointToStripePoint(vec3_t pt);
int BBoxToStripeRanges(float *min, float *max, int npoints, bbox_t *bbox);

///////////////////////////////////////////////////////////////////////////////

/** Load precomputed bounding boxes */
void DG_LoadBoundingBoxFile();
/** Precompute bounding boxes and export them to a file */
void DG_ExportBoundingBoxes();

///////////////////////////////////////////////////////////////////////////////

/** 
 * Compute the bounding box direction by ray tracing from the origin 
 */
void DG_ComputeVisibilityBox(vec3_t org, vec3_t min, vec3_t max,
							 float xy_step = XY_STEP, float z_step = Z_STEP);
/** 
 * Get the bounding box from the closest precomputed point
 *
 * @param orig the origin to compute the bbox for
 * @param min  output min point of the bbox
 * @param max  output max point of the bbox
 * @param diameter instead of finding the closest bbox for this point,
 *        combine all the bboxes within this diameter of that point.
 * @param zlimit limit the z-diameter to this much.
 * @param maxvol the maximum volume of the bbox allowed (only useful if
 *        diameter is specified). If specified, if the merged bbox for
 *        a particular diameter is too big, we will reduce the diameter
 *        by 2 units and try again until we find one small enough. In
 *        the worst case, if we don't find one, then we will just fall
 *        back to using the closest bbox. The maxvol is represented as
 *        a fraction of the total world volume.
 *
 * @return the vol of the bbox as a fraction of total world volume
 */
float DG_GetPrecompBoundingBox(vec3_t orig, vec3_t min, vec3_t max,
							   float diameter = 0, float zlimit = 9999,
							   float maxvol = 1);

/**
 * True if we have precomputed bounding boxes.
 */
bool DG_HasPrecompBBoxes();

/**
 * Get the bbox for the entire world if we loaded a precomputed bbox map
 */
void DG_GetPrecompWorldBBox(vec3_t min, vec3_t max);

/** 
 * Get the bounding box computed from the PVS regions 
 */
void DG_GetPVSBoundingBox( vec3_t org, vec3_t min, vec3_t max );

/** 
 * Get the bounding box using the "preferred" method from the above 3
 *
 * @return the vol of the bbox as a fraction of total world volume
 *         (only if we are using precomputed bboxes.. otherwise unknown)
 */
float DG_GetBoundingBox(vec3_t orig, vec3_t min, vec3_t max);

/**
 * Compute the difference between this bounding box and the bboxes of
 * the PVS clusters (the individual bozes, not the combined "big" one
 * which is the result of DG_GetPVSBoundingBox). Right now this computes
 * an overestimate because it doesn't take into account overlapping
 * cluster bboxes. Returns the overestimated total volume of the
 * PVS bboxes.
 */
float DG_PVSBoundingBoxesDiff( float *missing, float *extra,
							   vec3_t org, vec3_t min, vec3_t max );


///////////////////////////////////////////////////////////////////////////
// 
//

/**
 * Chop the world (as defined by DG_GetPrecompWorldBBox) into totBoxes number 
 * of boxes. First, we try to chop it up into a max of 4 z-slices. 
 * Other 2 axes are sliced equally. The number of boxes in each dimension is 
 * sent back in boxes[]. The total number of boxes (\Pi_i boxes[i]) can be
 * greater than totalBoxes if totalBoxes != 4k*k for some k.
 */
void DG_ChopWorld(int totBoxes, int boxes[3]);

/** 
 * Get the index of the box in which `point' belongs. 
 */
int  DG_GetChoppedWorldBox(vec3_t point, int boxes[3]);

/** 
 * Two special versions of the above functions for DHT chopping
 */
void DG_ChopWorldForDHT(int totBoxes);
float DG_GetNormalizedDHTBox(vec3_t point);
#endif
