/* vim: set sw=4 ts=4 noet: -*- Mode:c++; c-basic-offset:4; tab-width:4; indent-tabs-mode:t -*- */
#define BENCHMARK_REQUIRED
#include <qcommon/qcommon.h>
#include <dg/dg.h>
#include <dg/dg_bbox.h>
#include <om/Manager.h>
#include <math.h>

///////////////////////////////////////////////////////////////////////////////

typedef int ivec3_t[3];

extern Manager *g_Manager;

vec3_t g_worldmin, g_worldmax;

static vec3_t g_min;
static int g_xbuckets, g_ybuckets, g_zbuckets;
static float g_step, g_total_vol;
bbox_t ***g_bboxes = NULL;

#ifdef USE_SHARED_MEMORY
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <sys/mman.h>

/// separate space for the bbox flags 
byte   ***g_bbox_flags  = NULL;
/// shared memory is attached here
byte     *g_shmaddress  = NULL;

///////////////////////////////////////////////////////
// Get the bbox indices from the address 
// 
void Shmbbox_GetIndices(bbox_t *addr, int *x, int *y, int *z)
{
	int offset = addr - (bbox_t *) g_bboxes;
	
	*z = offset % g_zbuckets;
	offset -= *z; offset /= g_zbuckets;
	
	*y = offset % g_ybuckets;
	offset -= *y; offset /= g_ybuckets;

	*x = offset;
}

byte& Shmbbox_GetFlags(bbox_t *addr)
{
	int x, y, z;
	Shmbbox_GetIndices(addr, &x, &y, &z);
	return g_bbox_flags[x][y][z];
}

#endif

/// ASHWIN -- 
// 
// Functionality related to chopping the world into boxes
vec3_t g_worldrange;
int    g_ntotboxes;
int    g_dht_boxes[3];
uint32 g_maxUint32;

void DG_ChopWorldForDHT(int totBoxes)
{
	g_ntotboxes = totBoxes;

	DG_ChopWorld(totBoxes, g_dht_boxes);
	DG_GetPrecompWorldBBox(g_worldmin, g_worldmax);
	for (int i=0; i<3; i++) {
		g_worldrange[i] = g_worldmax[i] - g_worldmin[i];
	}

	g_maxUint32 = 0; 
	g_maxUint32 = ~g_maxUint32;
}

void DG_ChopWorld(int totBoxes, int boxes[3])
{
	boxes[2] = 1; //4; xxx: should be parameter! 
	int nXYboxes = (int) ((totBoxes / 4.0) + 0.5);     
	boxes[0] = boxes[1] = (int) (sqrt((float) nXYboxes) + 0.5);

	for (int i = 0; i < 3; i++)
		if (boxes[i] == 0)
			boxes[i] = 1;

	DG_GetPrecompWorldBBox(g_worldmin, g_worldmax);
	for (int i=0; i<3; i++) {
		g_worldrange[i] = g_worldmax[i] - g_worldmin[i];
	}

}

int  DG_GetChoppedWorldBox(vec3_t point, int boxes[3])
{
	int indices[3];
	for (int i = 0; i < 3; i++)
		indices[i] = (int) (boxes[i] * (point[i] - g_worldmin[i])/g_worldrange[i]);


	return (indices[0] * boxes[1] + indices[1]) * boxes[2] + indices[2];
}
		
inline uint32 DG_ComputeHash(uint32 key)
{
#if 0
	static uint64 a = 139990829, b = 1361165756;
	static uint64 p = 1500450271;

	return (uint32)( ((a*val + b) % p) & 0x00000000FFFFFFFFULL);
#else
    key += (key << 12);
	key ^= (key >> 22);
	key += (key << 4);
	key ^= (key >> 9);
	key += (key << 10);
	key ^= (key >> 2);
	key += (key << 7);
	key ^= (key >> 12);
	return key;	
#endif
}

float DG_GetNormalizedDHTBox(vec3_t point)
{
	int boxID = DG_GetChoppedWorldBox(point, g_dht_boxes);
	uint32 key = DG_ComputeHash(boxID);

	return (float) key / (float) g_maxUint32;
}

int BBoxToNormalizedDHTPoints(float *keys, int nkeys, bbox_t *bbox)
{
	vec3_t imin, imax;
	
	// compute the bounding box in terms of region indices
	for (int i=0; i<3; i++) {
		if (bbox->min[i] < g_worldmin[i]) return 0;
		if (bbox->max[i] > g_worldmax[i]) return 0;
		imin[i] = (int)(g_dht_boxes[i] * 
						(bbox->min[i]-g_worldmin[i])/g_worldrange[i]);
		imax[i] = (int)(g_dht_boxes[i] * 
						(bbox->max[i]-g_worldmin[i])/g_worldrange[i]);
	}
	
	int index = 0;

	// convert each discrete bounding box point into a DHT key
	for (int x = imin[0]; x <= imax[0]; x++) {
		for (int y = imin[1]; y <= imax[1]; y++) {
			for (int z = imin[2]; z <= imax[2]; z++) {
				if (index+1 > nkeys) {
					return 0;
				}

				uint32 boxID = (x * g_dht_boxes[1] + y) * g_dht_boxes[2] + z;
				uint32 k = DG_ComputeHash(boxID);
				keys[index++] = (float) k / (float) g_maxUint32;
			}
		}
	}
	
	return index;
}

float PointEventToDHTKey (OMEvent *ev)
{
	ASSERT (ev->IsPointEvent ());
	Constraint *c = ev->GetConstraint (g_DHTHubIndex);
	ASSERT (c != NULL);

	return g_Manager->ConvertFromValue (g_DHTHubIndex, c->GetMin (), 0.0, 1.0);
}

void DHTKeyPointToEvent(OMEvent *ev, float key, vec3_t pt)
{
	PointToEvent(ev, pt);

	Value v = g_Manager->ConvertToValue (g_DHTHubIndex, key, 0.0, 1.0);
	Tuple t (g_DHTHubIndex, v);
	ev->AddTuple (t);
}

///////////////////////////////////////////////////////////////////////////////

static bool DG_ExpandingRingSearch(vec3_t min, vec3_t max, 
								   vec3_t torig, int width, 
								   bool all = false, int zlimit = 9999);

ostream& operator<<(ostream& os, vec3_t v)
{
    os << merc_va("<%.2f %.2f %.2f>", v[0], v[1], v[2]);
    return os;
}

ostream& operator<<(ostream& os, ivec3_t v)
{
    os << merc_va("<%d %d %d>", v[0], v[1], v[2]);
    return os;
}

///////////////////////////////////////////////////////////////////////////////

float PointToStripePoint(vec3_t pt)
{
	float min, max;
	bbox_t bbox;

	PointToBBox(&bbox, pt);
	// a point should map to exactly 1 stripe!
	int ret = BBoxToStripeRanges(&min, &max, 1, &bbox);
	ASSERT( ret == 1 );
	ASSERT( min == max );

	return min;
}

int BBoxToStripeRanges(float *min, float *max, int npoints, bbox_t *bbox)
{
	static bool inited = false;
	static vec3_t worldmin, worldmax; // world bbox
	static vec3_t worldrange; // max - min
	static vec3_t stripes; // num stripes
	static vec3_t stripe_width; // width of each dim stripe
	static float  stripe_range; // max_stripe - min_stripe dim
	static int stripe_dim; // dimension to do stripe extension on
	static int stripemin[3];
	static int stripemax[3];

	if (!inited) {
		inited = true;
		DG_GetPrecompWorldBBox(worldmin, worldmax);
		for (int i=0; i<3; i++) {
			ASSERT(g_QuakePreferences.stripes[i]);
			worldrange[i] = worldmax[i] - worldmin[i];
			stripe_width[i] = worldrange[i]/g_QuakePreferences.stripes[i];
		}
		stripe_dim = g_QuakePreferences.stripe_dim;

		stripe_range = worldrange[stripe_dim]*
			g_QuakePreferences.stripes[0]*
			g_QuakePreferences.stripes[1]*
			g_QuakePreferences.stripes[2];

		VectorCopy(g_QuakePreferences.stripes, stripes);
	}

	vec3_t smin, smax;
	
	for (int i=0; i<3; i++) {
		ASSERT(bbox->min[i] >= worldmin[i]);
		ASSERT(bbox->max[i] <= worldmax[i]);
		stripemin[i] = (int)((bbox->min[i]-worldmin[i]-EPSILON)/stripe_width[i]);
		stripemax[i] = (int)((bbox->max[i]-worldmin[i]-EPSILON)/stripe_width[i]);

		// XXX
		smin[i] = ((bbox->min[i]-worldmin[i]-EPSILON)/stripe_width[i]);
		smax[i] = ((bbox->max[i]-worldmin[i]-EPSILON)/stripe_width[i]);
	}
	
	int index = 0;
	
	/*
	INFO << " bbox->min=" << bbox->min
		 << " bbox->max=" << bbox->max
		 << " smin=" << smin
		 << " smax=" << smax
		 << " stripemin=" << stripemin
		 << " stripemax=" << stripemax
		 << " stripe_width=" << stripe_width
		 << " stripe_range=" << stripe_range
		 << " worldmin=" << worldmin
		 << " worldmax=" << worldmax
		 << endl;
	*/
	
	// this bounding box covers the following stripes
	for (int x=stripemin[0]; x <= stripemax[0]; x++) {
		for (int y=stripemin[1]; y <= stripemax[1]; y++) {
			for (int z=stripemin[2]; z <= stripemax[2]; z++) {
				if (index >= npoints) {
					WARN << "not enough space allocated!";
					ASSERT(0);
					// not enough space!
					return -1;
				}

				int stripe_index = 
					x*(stripes[1]*stripes[2]) +
					y*(stripes[2])+
					z;

				// we need to project to the range within each of these
				// stripes, each of which is world_range[dim] in width
				min[index] =
					(-worldmin[stripe_dim] + 
					 bbox->min[stripe_dim]+
					 stripe_index*worldrange[stripe_dim])/
					stripe_range; // normalize to [0,1]
				max[index] = 
					(-worldmin[stripe_dim] + 
					 bbox->max[stripe_dim]+
					 stripe_index*worldrange[stripe_dim])/
					stripe_range; // normalize to [0,1]
				index++;
			}
		}
	}

	return index;
}

///////////////////////////////////////////////////////////////////////////////
// Bound Box code

///////////////////////////////////////////////////////////////////////////////

void PointToEvent (OMEvent *ev, vec3_t pt)
{
	for (int i = 0; i < 3; i++) {
		int j = g_CoordIndices[i];
		
		Value v = g_Manager->ConvertToValue (j, pt[i], g_worldmin[i], g_worldmax[i]);
		Tuple t (j, v);
		ev->AddTuple (t);
	}
}

void EventToPoint (vec3_t pt, OMEvent *ev)
{
	ASSERT (ev->IsPointEvent ());

	for (int i = 0; i < 3; i++) {
		int j = g_CoordIndices[i];
		
		Constraint *c = ev->GetConstraintByAttr (j);
		if (c == NULL)
			continue;
		
		pt[i] = g_Manager->ConvertFromValue (j, c->GetMin (), g_worldmin[i], g_worldmax[i]);
	}
}

bool BBoxContains(bbox_t *box1, bbox_t *box2)
{
    for (int i=0; i<3; i++) {
		if (box1->min[i] > box2->min[i] || box1->max[i] < box2->max[i]) {
			return false;
		}
    }

    return true;
}

static bool BBoxOverlapsHelper(vec3_t min1, vec3_t max1,
							   vec3_t min2, vec3_t max2)
{
	for (int i=0; i<3; i++) {
		if (!( min1[i] <= min2[i] && min2[i] <= max1[i] || 
			   min1[i] <= max2[i] && max2[i] <= max1[i] )) {
			return false;
		}
    }

	return true;
}

bool BBoxOverlaps(bbox_t *box, vec3_t min, vec3_t max)
{
    return BBoxOverlapsHelper(box->min, box->max, min, max) ||
		BBoxOverlapsHelper(min, max, box->min, box->max);
}

static float OverlapHelper(bbox_t *box1, bbox_t *box2, int i)
{
    if (box1->min[i] <= box2->min[i] && 
		box2->min[i] <= box1->max[i]) {
		return MIN(box1->max[i], box2->max[i]) - box2->min[i];
    } else if (box1->min[i] <= box2->max[i] && 
			   box2->max[i] <= box1->max[i]) {
		return box2->max[i] - MAX(box1->min[i], box2->min[i]);
    }
    return 0;
}

float Overlap(bbox_t *box1, bbox_t *box2, int dim)
{
    float ret;

    ret = OverlapHelper(box1, box2, dim);
    if (ret > 0) return ret;
    ret = OverlapHelper(box2, box1, dim);
    return ret;
}

float Overlap(bbox_t *out, bbox_t *box1, bbox_t *box2)
{
	// This function already assumes the boxes overlap!

	for (int i=0; i<3; i++) {
		// The overlapping min point is larger of the two minimums
		if (box1->min[i] > box2->min[i]) {
			out->min[i] = box1->min[i];
		} else {
			out->min[i] = box2->min[i];
		}

		// The overlapping max point is the smaller of the two maximums
		if (box1->max[i] < box2->max[i]) {
			out->max[i] = box1->max[i];
		} else {
			out->max[i] = box2->max[i];
		}
	}

}

float BBoxVolume(bbox_t *bbox)
{
	return BBoxVolume(bbox->min, bbox->max);
}

float BBoxVolSubtract(bbox_t *bbox1, bbox_t *bbox2)
{
	return BBoxVolSubtract(bbox1->min, bbox1->max, bbox2->min, bbox2->max);
}

//
// Volume of a bbox
//
float BBoxVolume(vec3_t min, vec3_t max)
{
	return (max[0] - min[0])*(max[1] - min[1])*(max[2] - min[2]);
}

//
// Get the volume of (bbox1 - bbox2). That is, the volume of that stuff that
// is in bbox1 but not in bbox2 (hence the min is 0, if bbox1 == bbox2).
//
float BBoxVolSubtract(vec3_t min1, vec3_t max1, vec3_t min2, vec3_t max2)
{
	float bbox1_vol = BBoxVolume(min1, max1);
	
	vec3_t diffmin, diffmax;

	for (int i=0; i<3; i++) {
		diffmin[i] = MAX(min1[i], min2[i]);
		diffmax[i] = MIN(max1[i], max2[i]);

		if (diffmin[i] >= diffmax[i]) {
			// no intersection of bboxes, difference is total
			return bbox1_vol;
		}
	}

	return bbox1_vol - BBoxVolume(diffmin, diffmax);
}

//
// Distance between two points
//
float Distance(vec3_t from, vec3_t to)
{
	float dx = from[0]-to[0];
	float dy = from[1]-to[1];
	float dz = from[2]-to[2];

	return sqrt(dx*dx+dy*dy+dz*dz);
}

void IndexToPoint(vec3_t out, ivec3_t index)
{
	for (int i=0; i<3; i++)
		out[i] = g_min[i] + g_step*index[i];
}

///////////////////////////////////////////////////////////////////////////////

FILE * DG_LoadBBoxFromNormalFile()
{
	FILE *fp = fopen(g_QuakePreferences.bbox_file, "r");
    if (!fp) {
		WARN << "can't open bbox file: " << g_QuakePreferences.bbox_file;
		exit(1);
    }

    char *junk = (char *)calloc(strlen(MAGIC_STRING)+1, 1);
    fread((void *)junk, strlen(MAGIC_STRING), 1, fp); // magic

	if (strcmp(junk, MAGIC_STRING)) {
		fprintf(stderr, "bbox file doesn't begin with magic string: " 
				MAGIC_STRING "\n");
		exit(1);
    }
	free(junk);

    fread((void *)g_min, sizeof(vec3_t), 1, fp);    // starting point
    fread((void *)&g_xbuckets, sizeof(int), 1, fp); // number of x points
    fread((void *)&g_ybuckets, sizeof(int), 1, fp); // number of y points
    fread((void *)&g_zbuckets, sizeof(int), 1, fp); // number of z points
    fread((void *)&g_step, sizeof(float), 1, fp);   // size of step between pts

	return fp;
}

void DG_ReadBBoxesFromNormalFile(FILE *fp)
{
	g_bboxes = new bbox_t**[g_xbuckets];
	
	for (int x_index = 0; x_index < g_xbuckets; x_index++) {
		g_bboxes[x_index] = new bbox_t*[g_ybuckets];
		for (int y_index = 0; y_index < g_ybuckets; y_index++) {
			g_bboxes[x_index][y_index] = new bbox_t[g_zbuckets];
			for (int z_index = 0; z_index < g_zbuckets; z_index++) {
				int ret;

				ret = fread((void *) GET_BBOX(x_index, y_index, z_index).min, 
						sizeof(vec3_t), 1, fp); // bbox min
				if (ret == 0) {
					WARN << "EOF encountered reading bbox file!" << endl;
					ASSERT(0);
				}
				ret = fread((void *) GET_BBOX(x_index, y_index, z_index).max, 
						sizeof(vec3_t), 1, fp); // bbox max
				if (ret == 0) {
					WARN << "EOF encountered reading bbox file!" << endl;
					ASSERT(0);
				}

			}
		}
	}
}

#ifdef USE_SHARED_MEMORY
#include <errno.h>
void DG_LoadBBoxFromMmapedFile()
{
	int fd = open(g_QuakePreferences.bbox_file, O_RDONLY);
	if (fd < 0) 
	{
		perror("open");
		WARN << "can't open bbox file: " << g_QuakePreferences.bbox_file;
		exit(1);
	}
	
	fprintf(stderr, "opened file successfully...\n");
	struct stat buf;
	if (fstat(fd, &buf) < 0) { 
		perror("fstat"); 
		Debug::die("fstat failed!"); 
	}
	
	fprintf(stderr, "stat happenned successfully...file length = %d\n", buf.st_size);
	g_shmaddress = (byte *) mmap(NULL, buf.st_size, PROT_READ, MAP_SHARED, fd, 0);
	if (g_shmaddress == NULL)
	{
		fprintf(stderr, "stat error...\n");
		perror("mmap");
		WARN << "error mmaping bbox file: " << g_QuakePreferences.bbox_file;
		exit(1);
	}
	fprintf(stderr, "mmap happened successfully...\n");
	if (strncmp((char *) g_shmaddress, MAGIC_STRING, strlen(MAGIC_STRING))) {
		fprintf(stderr, "bbox file doesn't begin with magic string: " 
				MAGIC_STRING "\n");
		exit(1);
    }	
	g_shmaddress += strlen(MAGIC_STRING);
	
	memcpy(g_min, g_shmaddress, sizeof(vec3_t));  g_shmaddress += sizeof(vec3_t);
	memcpy(&g_xbuckets, g_shmaddress, sizeof(int)); g_shmaddress += sizeof(int);
	memcpy(&g_ybuckets, g_shmaddress, sizeof(int)); g_shmaddress += sizeof(int);
	memcpy(&g_zbuckets, g_shmaddress, sizeof(int)); g_shmaddress += sizeof(int);
	memcpy(&g_step, g_shmaddress, sizeof(float)); g_shmaddress += sizeof(float);
}

void DG_ReadBBoxesFromMmapedFile()
{
	g_bboxes = (bbox_t ***) g_shmaddress;

	g_bbox_flags = new byte**[g_xbuckets];
	for (int x_index = 0; x_index < g_xbuckets; x_index++)
	{
		g_bbox_flags[x_index] = new byte*[g_ybuckets];
		for (int y_index = 0; y_index < g_ybuckets; y_index++) {
			g_bbox_flags[x_index][y_index] = new byte[g_zbuckets];

			for (int z_index = 0; z_index < g_zbuckets; z_index++) {
				g_bbox_flags[x_index][y_index][z_index] = 0;
			}
		}
	}
}
#endif

//
// Load the bounding box file for use in sub computation
//
void DG_LoadBoundingBoxFile() {
#ifdef USE_SHARED_MEMORY
	DG_LoadBBoxFromMmapedFile();
#else
	FILE *fp = DG_LoadBBoxFromNormalFile();
#endif
	
	// ok to create!
	INFO << "g_xbuckets = " << g_xbuckets << 
		" g_ybuckets = " << g_ybuckets <<
		" g_zbuckets = " << g_zbuckets << 
		" sizeof(bbox_t) = " << sizeof(bbox_t) << endl;
			

	g_total_vol = g_xbuckets*g_ybuckets*g_zbuckets*g_step*g_step*g_step;

#ifdef USE_SHARED_MEMORY
	DG_ReadBBoxesFromMmapedFile();
#else
	DG_ReadBBoxesFromNormalFile(fp);
#endif

	for (int i = 0 ; i < g_xbuckets * g_ybuckets * g_zbuckets * sizeof(bbox_t); i++)
	{
		byte *mem = (byte *) g_bboxes;
		mem = mem + i;
		
		byte access = *mem;
	}
    for (int x_index = 0; x_index < g_xbuckets; x_index++) {
		for (int y_index = 0; y_index < g_ybuckets; y_index++) {
			for (int z_index = 0; z_index < g_zbuckets; z_index++) {
				bbox_t *t = &(GET_BBOX(x_index, y_index, z_index));
				GET_FLAGS(*t) = 0;

				if (BBoxVolume(GET_BBOX(x_index, y_index, z_index).min,
							GET_BBOX(x_index, y_index, z_index).max) <= 0) {
					BBOX_SET_VALID(GET_BBOX(x_index, y_index, z_index), 0);
				} else {
					BBOX_SET_VALID(GET_BBOX(x_index, y_index, z_index), 1);
				}
			}
		}
    }

#ifndef USE_SHARED_MEMORY
	fclose(fp);
#endif

	DG_GetPrecompWorldBBox(g_worldmin, g_worldmax);
}

///////////////////////////////////////////////////////////////////////////////

bool DG_HasPrecompBBoxes()
{
	return g_bboxes != NULL;
}

float DG_GetBoundingBox(vec3_t orig, vec3_t min, vec3_t max)
{
	if (g_bboxes != NULL) {
		// if we precomputed bboxes, use that
		return DG_GetPrecompBoundingBox(orig, min, max);
	} else {
		// otherwise we have to compute it on the fly
		DG_ComputeVisibilityBox(orig, min, max);
		return 0;
	}
}

///////////////////////////////////////////////////////////////////////////////

static void ClearMarks(ivec3_t trunc, int width)
{
	for (int x = MAX(trunc[0], 0);
		 x <= MIN(trunc[0] + width, g_xbuckets-1); x++) {
	 	for (int y = MAX(trunc[1], 0);
			 y <= MIN(trunc[1] + width, g_ybuckets-1); y++) {
		  	for (int z = MAX(trunc[2], 0);
               	 z <= MIN(trunc[2] + width, g_zbuckets-1); z++) {
                 	   BBOX_SET_MARK( GET_BBOX(x, y, z), 0 );
		  	}
	 	}
	}
}
//
// Perform an "expanding-ring" search around torig (which is assumed to be
// in the coordinate-space after translation by g_min) to find the closest
// valid point with a bbox.
//
// bbox_min - the resulting bbox_min point
// bbox_max - the resulting bbox_max point
// torig    - the origin point (in g_min translated space)
// width    - the number of g_step units for the diameter of the ring
// all      - combine the bboxes from all the points within width diameter
//            instead of finding the closest point (this might return 
//            the null box if nothing is in range)
// zlimit   - a HACK that allows us to limit the range search in the
//            z-direction to a given diameter. this is useful since most
//            rooms have ceilings and it makes no sense to have the search
//            space include the areas "above" the ceiling or "below" the
//            floors. (9999 ~ no limit)
//
// returns true if we found a bbox.
//
bool DG_ExpandingRingSearch(vec3_t bbox_min, vec3_t bbox_max,
							vec3_t torig, int width, bool all, int zlimit)
{
	bool found = false;
	ivec3_t res;
	vec3_t test;
	float mindist;

	// Truncate the point to the closest "lower-left" corner bbox point
	ivec3_t trunc;
	trunc[0] = (int)(torig[0]/g_step) - width/2;
	trunc[1] = (int)(torig[1]/g_step) - width/2;
	trunc[2] = (int)(torig[2]/g_step) - MIN(width, zlimit)/2;

	/*
	INFO << "torig=" << torig << endl;
	INFO << "width=" << width << endl;
	INFO << "trunc=" << trunc << endl;
	*/

	// Now sweep the box looking for valid points we haven't marked yet
	for (int x = MAX(trunc[0], 0); 
		 x <= MIN(trunc[0] + width, g_xbuckets-1); x++) {

		for (int y = MAX(trunc[1], 0); 
			 y <= MIN(trunc[1] + width, g_ybuckets-1); y++) {

			for (int z = MAX(trunc[2], 0); 
				 z <= MIN(trunc[2] + MIN(width, zlimit), g_zbuckets-1); z++) {

				if (BBOX_IS_MARK( GET_BBOX(x, y, z) ) ||
					! BBOX_IS_VALID( GET_BBOX(x, y, z) ))
					continue;
				
				test[0] = x*g_step;
				test[1] = y*g_step;
				test[2] = z*g_step;
				
				float dist = Distance(torig, test);
				if (dist <= (g_step*width)/2) {

					// XXX We should also make sure the bbox point is
					// visible from the origin point! E.G., we may be
					// blocked by ceilings or go through a wall! But
					// this requires doing a ray-trace, which is
					// potentially expensive, so just hope we will find
					// a close point fast for now so that is unnecessary
					if (!all) {
						// mark it so we don't look at it again
						BBOX_SET_MARK( GET_BBOX(x, y, z), 1 );

						// we are just looking for the min point
						if (!found) {
							found   = true;
							mindist = dist;
							res[0] = x; res[1] = y; res[2] = z;
						} else {
							if (dist < mindist) {
								mindist = dist;
								res[0] = x; res[1] = y; res[2] = z;
							}
						}
					} else {
						// we are combining all the bboxes for all points
						if (!found) {
							found = true;
							VectorCopy(GET_BBOX(x, y, z).min, bbox_min);
							VectorCopy(GET_BBOX(x, y, z).max, bbox_max);
						} else {
							for (int i=0; i<3; i++) {
								bbox_min[i] = MIN(bbox_min[i], 
												  GET_BBOX(x, y, z).min[i]);
								bbox_max[i] = MAX(bbox_max[i], 
												  GET_BBOX(x, y, z).max[i]);
							}
						}
					}
				}
				
			}
		}
	}

	// if we were getting the large combined bbox we are now done
	if (all) {
		return found;
	}

	// otherwise, we need to unclear the bits or possibly recurse if we
	// didn't find a valid point.
	if (found) {
		/*
		if (all) {
			INFO << "width=" << width << endl;
			INFO << "found=" << found << endl;
			INFO << bbox_min << " " << bbox_max << endl;
		}
		*/

		// sweep across the search box and unclear the marked bits
		ClearMarks(trunc, width);
		
		VectorCopy(GET_BBOX(res[0], res[1], res[2]).min, bbox_min);
		VectorCopy(GET_BBOX(res[0], res[1], res[2]).max, bbox_max);

		return true;
	} else {
		// sanity check: if this happens then it implies that the bbox map
		// was malformed because there are no valid points in the search space
		if (width > g_xbuckets && width > g_ybuckets && width > g_zbuckets) {
			// This seems to sometimes happen for missiles that fly out
			// into space? Not sure, but don't crash because of it...
			DB(1) << "ring search exhausted search space without a result!"
				  << endl;
			ClearMarks(trunc, width);
			bzero((byte *)bbox_min, sizeof(vec3_t));
			bzero((byte *)bbox_min, sizeof(vec3_t));
			return false;
		}

		// didn't find a valid box!, try again with larger diameter
		return DG_ExpandingRingSearch(bbox_min, bbox_max, torig, 
									  width+1, all, zlimit);
	}
}

//
// Get the closest precomputed bounding box for a point in space.
//
float DG_GetPrecompBoundingBox(vec3_t orig, vec3_t bbox_min, vec3_t bbox_max,
							   float diameter, float zlimit, float maxvol)
{
	ASSERT(g_bboxes != NULL);

	// check if quake put this object in some wierd place...
	for (int i=0; i<3; i++) {
		if (orig[i] < g_worldmin[i] ||
			orig[i] > g_worldmax[i]) {
			/*
			INFO << " orig=" << orig << " min=" << g_worldmin
				 << " max=" << g_worldmax << endl;
			*/
			return 0;
		}
	}

	// translate the origin to the coordinate space starting from the
	// "edge" of the spots for which we have bboxes for
	vec3_t torig;
	torig[0] = orig[0] - g_min[0];
	torig[1] = orig[1] - g_min[1];
	torig[2] = orig[2] - g_min[2];

	bool found = false;
	int zlim = (int)ceil( zlimit/g_step );

	if (diameter > 0) {
		// get the box of all potential points within this diameter!
		int width  = (int)ceil( diameter/g_step );

		START(DG_ExpandingRingSearch);
		
		while (width > 0) {
			found = DG_ExpandingRingSearch(bbox_min, bbox_max, torig, 
										   width, true, zlim);
			if (!found)
				break;
			float vol = BBoxVolume(bbox_min, bbox_max)/g_total_vol;
			if (vol <= maxvol)
				break;
			else {
				DB(1) << "bbox with width=" << width << " too big!" << endl;
				// too big! try again with smaller diameter
				width -= 2;
			}
		}
			
		STOP(DG_ExpandingRingSearch);

		if (!found) {
			START(DG_ExpandingRingSearch);
			// no points found in diameter! just find the closest then
			found = DG_ExpandingRingSearch(bbox_min, bbox_max, torig, 
					1, false, zlim);
			STOP(DG_ExpandingRingSearch);
		}
		/* else {
			ASSERT(bbox_max[0] - bbox_min[0] > 0 &&
				   bbox_max[1] - bbox_min[1] > 0 &&
				   bbox_max[2] - bbox_min[2] > 0);
				   }*/
	} else {
		// Now perform an "expanding-ring" search for the closest valid 
		// bbox point
		START(DG_ExpandingRingSearch);
		found = DG_ExpandingRingSearch(bbox_min, bbox_max, torig, 
				1, false, zlim);
		STOP(DG_ExpandingRingSearch);
	}

	/*
	// XXX
	vec3_t closest;
	IndexToPoint(closest, res);
	
	INFO << "origin =" << orig << endl;
	INFO << "closest=" << closest << endl;
	INFO << "dist   =" << Distance(orig, closest) << endl;
	*/

	if (found)
		return BBoxVolume(bbox_min, bbox_max)/g_total_vol;
	else
		return 0;
}

void DG_GetPrecompWorldBBox(vec3_t min, vec3_t max)
{
	ASSERT(g_bboxes != NULL);

	for (int i=0; i<3; i++) {
		min[i] = g_min[i];
	}

	max[0] = g_min[0] + g_xbuckets*g_step;
	max[1] = g_min[1] + g_ybuckets*g_step;
	max[2] = g_min[2] + g_zbuckets*g_step;

	INFO << "* computed world bbox as: " << min << " -> " << max << endl;
	INFO << " (params were: g_min=" << g_min << " g_step=" << g_step 
		<< " xbkt=" << g_xbuckets << " ybkt=" << g_ybuckets 
		<< " zbkt=" << g_zbuckets << endl;
}

///////////////////////////////////////////////////////////////////////////////

//
// Compute the bounding boxes for the map and write them to the file
//
void DG_ExportBoundingBoxes() {
    DBG_INIT(&g_LocalSID);

	if (!strcmp(g_QuakePreferences.bbox_file, "")) {
		WARN << "no bbox output file specified!" << endl;
		Debug::die("");
	}

    // open the bbox file to write to
    FILE *fp = fopen(g_QuakePreferences.bbox_file, "w");

    if (!fp) {
		WARN << "unable to open " 
			 << g_QuakePreferences.bbox_file << " for writing: "
			 << strerror(errno) << endl;
		Debug::die("");
    }

    // parse the x,y,z min and max for the space we are interested in
    // computing bboxes for...
    vec3_t min, max;
    float step = g_QuakePreferences.bbox_unit;
    char *str;

    str = strtok(g_QuakePreferences.bbox_min, ",");
    ASSERT(str);
    min[0] = atof(str) + step/2;
    str = strtok(NULL, ",");
    ASSERT(str);
    min[1] = atof(str) + step/2;
    str = strtok(NULL, ",");
    ASSERT(str);
    min[2] = atof(str) + step/2;

    str = strtok(g_QuakePreferences.bbox_max, ",");
    ASSERT(str);
    max[0] = atof(str) - step/2;
    str = strtok(NULL, ",");
    ASSERT(str);
    max[1] = atof(str) - step/2;
    str = strtok(NULL, ",");
    ASSERT(str);
    max[2] = atof(str) - step/2;

    // calculate the number of buckets to calculate (int's cheaper than floats)
    int xbuckets = (int)( (max[0]-min[0])/step + 1 );
    int ybuckets = (int)( (max[1]-min[1])/step + 1 );
    int zbuckets = (int)( (max[2]-min[2])/step + 1 );
    int total    = xbuckets*ybuckets*zbuckets;
	float total_vol = BBoxVolume(min, max);

#ifdef COMPACT_FORMAT
	fwrite((void *)MAGIC_STRING, strlen(MAGIC_STRING), 1, fp); // magic
	fwrite((void *)min, sizeof(vec3_t), 1, fp);    // starting point
	fwrite((void *)&xbuckets, sizeof(int), 1, fp); // number of x points
	fwrite((void *)&ybuckets, sizeof(int), 1, fp); // number of y points
	fwrite((void *)&zbuckets, sizeof(int), 1, fp); // number of z points
	fwrite((void *)&step, sizeof(float), 1, fp);   // size of step between pts
#endif

    INFO << "bbox buckets = (" << xbuckets << "," 
		 << ybuckets << "," << zbuckets << ") = "
		 << total << " total, vol = " << total_vol << endl;

    // for each point in the space we are interested in, calculate and
    // write the bbox for that point.
    vec3_t orig, bmin, bmax;
    int x_index, y_index, z_index;
    int todo = total;

    for (x_index = 0; x_index < xbuckets; x_index++) {
		orig[0] = min[0] + step*x_index;
		
		for (y_index = 0; y_index < ybuckets; y_index++) {
			orig[1] = min[1] + step*y_index;
			
			for (z_index = 0; z_index < zbuckets; z_index++) {
				orig[2] = min[2] + step*z_index;
				START(BBOX_TIMER);
				DG_ComputeVisibilityBox(orig, bmin, bmax,
										g_QuakePreferences.bbox_xystep,
										g_QuakePreferences.bbox_zstep);
				
#ifndef COMPACT_FORMAT
				float vol = BBoxVolume(bmin, bmax);
				fprintf(fp, "%8.3f,%8.3f,%8.3f\t%8.3f,%8.3f,%8.3f\t%8.3f,%8.3f,%8.3f\t%7.5f\n",
						orig[0], orig[1], orig[2],
						bmin[0], bmin[1], bmin[2],
						bmax[0], bmax[1], bmax[2], vol/total_vol);
#else
				fwrite((void *)bmin, sizeof(vec3_t), 1, fp); // bbox min
				fwrite((void *)bmax, sizeof(vec3_t), 1, fp); // bbox max
#endif
				
				todo--;
				
				TimeVal now;
				gettimeofday(&now, NULL);
				STOP(BBOX_TIMER);

				// XXX This isn't very accurate since we aren't sampling points
				// from the map space randomly...
				//	PERIODIC(30000, now, 
						
				{
					PERIODIC (30000, now, {
						float time;
						AVERAGE_ASSIGN(time, BBOX_TIMER);
						fprintf(stderr, "Time Left: %7.1f min     Todo: %10d\n", 
								((float)todo/(MSEC_IN_SEC*60))*time, 
								todo);
						// Restart the timer average after the cache has been
						// primed with a few runs to get an accurate ETA
						static bool reset = 0;
						if (!reset && todo < total - 100) {
							RESTART(BBOX_TIMER);
							reset = 1;
						}
					});
				}
			}
		}
    }
    
    fclose(fp);
}

///////////////////////////////////////////////////////////////////////////////

//
// Compute the bounding box for a point on the fly using ray-tracing
//
// org     - the origin point
// min     - the output min-corner of the bbox
// max     - the output max-corner of the bbox
// xy_step - the optional step to use when sweeping in the xy-plane (degrees)
// z_step  - the optional step to use when sweeping on the z-axis (degrees)
//
void DG_ComputeVisibilityBox( vec3_t org, vec3_t min, vec3_t max,
							  float xy_step, float z_step )
{
	static vec3_t zero_vec = { 0, 0, 0 };

    int step, n_updowns, j;
    vec3_t start, end, direction;
    trace_t tr;
    vec3_t dist;
    int   phi, theta;
    float r_phi, r_theta;

    // minimum bbox

    min[0] = org[0] - MIN_BBOX_WIDTH/2;
    min[1] = org[1] - MIN_BBOX_WIDTH/2;
    min[2] = org[2] - MIN_BBOX_HEIGHT/2;
    
    max[0] = org[0] + MIN_BBOX_WIDTH/2;
    max[1] = org[1] + MIN_BBOX_WIDTH/2;
    max[2] = org[2] + MIN_BBOX_HEIGHT/2;

    for (theta = 0; theta < 360; theta += xy_step) {
		r_theta = RADIAN(theta) + XY_JITTER-2*XY_JITTER*drand48();
	    
		for (phi = -90; phi <= 90; phi += xy_step) {
			r_phi = RADIAN(phi) + Z_JITTER-2*Z_JITTER*drand48();

			direction[0] = cos(r_phi) * cos(r_theta);
			direction[1] = cos(r_phi) * sin(r_theta);
			direction[2] = sin(r_phi);
		
			VectorMA(org, 2, direction, start);
			VectorMA(org, 2*4082, direction, end);
		
			tr = gi.trace( start, vec3_origin, vec3_origin, 
						   end, 0, MASK_OPAQUE );

			if ( tr.startsolid || tr.fraction >= 1.0 ) {
				// the origin was in a solid! that means bbox is null
				// so we can just return with the default bbox
				// OR
				// didn't hit anything! that means we are outside the
				// "game" area and can't possibly be here!
				//INFO << "here: min=" << min << " max=" << max << endl;
				VectorCopy(zero_vec, min);
				VectorCopy(zero_vec, max);
				return;
			}
			
			if ( tr.endpos[0] < min[0] ) min[0] = tr.endpos[0];
			if ( tr.endpos[0] > max[0] ) max[0] = tr.endpos[0];
			if ( tr.endpos[1] < min[1] ) min[1] = tr.endpos[1];
			if ( tr.endpos[1] > max[1] ) max[1] = tr.endpos[1];
			if ( tr.endpos[2] < min[2] ) min[2] = tr.endpos[2];
			if ( tr.endpos[2] > max[2] ) max[2] = tr.endpos[2];

			/*
			DB_DO(10) {
				vec3_t dist;
				float  mag_dist;
		    
				VectorSubtract( tr.endpos, org, dist );
				mag_dist = sqrt(DotProduct(dist, dist));

				DB(1) << merc_va("phi: %d, theta: %d, dist: %.3f", phi, theta, mag_dist) << endl;
			}
			*/
		    
		}
    }

}

///////////////////////////////////////////////////////////////////////////////

void DG_GetPVSBoundingBox( vec3_t org, vec3_t min, vec3_t max )
{
    CM_BoundingBox(min, max, org);
}

float DG_PVSBoundingBoxesDiff( float *missing, float *extra,
							   vec3_t org, vec3_t min, vec3_t max )
{
	static vec3_t mins[1024], maxs[1024];

	int num = CM_BoundingBoxes(mins, maxs, 1024, org);
	if (num < 0) {
		return -1;
	}

	float total = 0;
	*missing = 0;
	*extra   = 0;

	// XXX This is an OVER-estimate, since we are subtracting a union
	// of boxes which may overlap (but hopefully they don't too much
	// since they are different clusters). To do this correctly would
	// require num factorial iterations for inclusion-exclusion...
	for (int i=0; i<num; i++) {
		*missing += BBoxVolSubtract(mins[i], maxs[i], min, max);
		// extra number is meaningless without inclusion-exclusion...
		*extra   += BBoxVolSubtract(min, max, mins[i], maxs[i]);
		total    += BBoxVolume(mins[i], maxs[i]);
	}

	return total;
}
