/* vim: set sw=4 ts=4 noet: -*- Mode:c++; c-basic-offset:4; tab-width:4; indent-tabs-mode:t -*- */

#ifndef __DG__H
#define __DG__H

#include "dg_types.h"
#include "dg_delta_tables.h"
#include "dg_delta_encoding.h"
#include <dg/DG_Edict.h>
#include <dg/DG_ObjStore.h>
#include <dg/QuakeAdaptor.h>

// Commands passed to DG_GameFunc(...). The order of events should be:
// 
// (1) DG_CMD_DO_MAINTENANCE
//
// Foreach frame_interval:
// {
//   (2) DG_CMD_CHECKPOINT_REPLICAS
//   (3) G_RunFrame()
//   (4) DG_CMD_SEND_UPDATES
//   (5) DG_CMD_PRIM_CHECKPOINT
//   (6) send_to_clients
// }
enum { 
	// this is called once per frame. it handles timers and handlers
	// etc. in the manager. 
	DG_CMD_DO_MAINTENANCE,
    // This should be called as often as possible. It just processes
    // incoming remote packets, and gives mercury processing time
    DG_CMD_DO_RECV,
	// This should be called just before Quake does its thing with the
	// object state; it resolves any "conflicts" that arise because of
	// remote updates so that the generic quake code only "sees" a valid
	// state of objects
	DG_RESOLVE_CONFLICTS,
    // This should be called immediately after SEND_UPDATES to checkpoint
    // the latest version of the objects that have been synchronized with
    // other servers. We can not just call this before G_RunFrame() because
    // we need to take into account remote updates as well.
    DG_CMD_CHECKPOINT_PRIMARIES,
    // This should be called immediately before G_RunFrame() to checkpoint
    // the latest version of replica objects just before we make any changes
    // to them.
    DG_CMD_CHECKPOINT_REPLICAS,
	// Precompute some info just after the game changes the objects
	DG_PRECOMPUTE_INFO,
    // This should be called immediately after G_RunFrame() which will
    // send deltas to other servers (the delta being the current state of
    // g_edicts vs. the state since the last CHECKPOINT)
    DG_CMD_SEND_UPDATES,
	// This should be called at the end of the frame after we checkpoint
	// the primaries... some objects we need to free so they will be
	// properly deleted next time
	DG_GARBAGE_COLLECT,
};

extern DG_ObjStore *g_ObjStore;

void DG_Init();
void DG_PrecomputePerFrameInfo();
void DG_ResolveConflicts();
void DG_RecordObjectChanges();
void DG_RecordNewObject(edict_t *ent);
void DG_RecordDeleteObject(edict_t *prevent);
void DG_RecordSameObject(edict_t *prevent, edict_t *ent);
void DG_CheckPointObject(edict_t *ent); // only works for g_edict edicts!
void DG_CheckPointObjects(bool is_replica);
void DG_ProcessRemoteStatelessEvents();
void DG_GarbageCollect();
void DG_GameFunc(int command);
edict_t *DG_ExistsEntity(guid_t guid);
void DG_SendVisualizerUpdates();

enum { DG_OBJ_CREATED, DG_OBJ_DESTROYED, DG_OBJ_CHANGED, DG_OBJ_UNAFFECTED };

// XXX HACK: in remote updates, if we only sent multicasted events, don't
// deserialize the remainder of the object...
#define DUMMY_EVENTS_ONLY_MASK 0xFFFF0000

//////////////////////////////////////////////////////////////////////////
void DG_GetClientGUID(byte *pp_client, guid_t *guid);
bool DG_IsClient(int edict_num);
gclient_t *DG_AllocateClient(guid_t guid);
void DG_DeallocateClient(guid_t guid);
edict_t *DG_HandleObjectUpdate(edict_t *, edict_t *);
void DG_HandleObjectDelete(guid_t guid);
bool DG_is_object_distributed(edict_t *ent);

// call when we exit please!
void DG_Halt();

//////////////////////////////////////////////////////////////////////////

// XXX TODO: change refs from zero_* to *_NONE
extern guid_t GUID_NONE;
extern sid_t  SID_NONE;
#define zero_guid GUID_NONE
#define zero_sid  SID_NONE

//////////////////////////////////////////////////////////////////////////
// Special Hacks for non-distributed objects

// This special IP:Port pair is assigned to GUIDs of objects that are not
// distributed upon serialization so that we can deserialize their references
// locally (since they are guaranteed to be shared among all nodes)
//
// We assume that all these entities are started up with the map and are
// created in the same order on all nodes (which I think is reasonable)
// See G_InitEdict in game/g_utils.c for the assignment code.
const  uint32    NON_DIST_IP   = 0xFFFFFFFF;
const  uint16    NON_DIST_PORT = 0xFFFF;
extern unsigned long g_NonDistLocalOIDCounter; // 0 is for worldspawn

bool DG_IsNonDistributed(GUID guid);

//////////////////////////////////////////////////////////////////////////
// Iterating through player lists now require looking at remote players too

extern void DG_BeginPlayerIter();
extern edict_t * DG_GetNextPlayer();

//////////////////////////////////////////////////////////////////////////

//------- change to new/delete if need be
//#define ALLOC_MEM(amt)      gi.TagMalloc(amt, TAG_GAME)
//#define FREE_MEM(ptr)       gi.TagFree(ptr)
#define ALLOC_MEM(amt)        ((void *)new byte[amt])
#define FREE_MEM(ptr)         (delete[] ((byte *)ptr))
//-----------------------------------------

int search_pointer(void **ptr_array, int len, void *ptr);
int search_char_pointer(void **ptr_array, int len, void *ptr);

#define streq(a,b) (!strcmp(a,b))

/////////////////////////////////////////////////////////////////////////
// Visualization

#include <gameapi/GameVisualizer.h>

extern struct sockaddr_in g_VisAddr;
extern int g_DHTHubIndex;
extern int g_StripeHubIndex; 

#endif // __DG__H
