/* vim: set sw=4 ts=4 noet: -*- Mode:c++; c-basic-offset:4; tab-width:4; indent-tabs-mode:t -*- */

#include <qcommon/qcommon.h>
#include <dg/dg.h>
#include <dg/dg_bbox.h>
#include <dg/QuakeAdaptor.h>
#include <dg/DG_Edict.h>
#include <dg/DG_ObjStore.h>
#include <om/Manager.h>

extern Manager *g_Manager;

//////////////////////////////////////////////////////////////////////////
// QuakeAdaptor
//////////////////////////////////////////////////////////////////////////

QuakeAdaptor::QuakeAdaptor()
{
    m_DG_Edicts = new DG_ObjStore(false);
    m_Half_Edicts = new DG_ObjStore(true);
}


void QuakeAdaptor::FillInterestsDHT(InterestList *reg,
									InterestList *unreg,
									GUIDMap *reg_assoc,
									InterestMap *curr)
{
#define MAX_KEYS 128

	uint32 ttl;
	float pred;
	ObjectStore *store = GetObjectStore();

	GUIDSet inuse;

	// what keys are we still subscribed to in the system
	map<float, OMInterest *> have_keys;
	for (InterestMapIter it = curr->begin(); it != curr->end(); it++) {
		float key = InterestToDHTKey(it->second);
		have_keys[key] = it->second;
	}
	
	bbox_t bbox;
	DG_Edict *obj;
	store->Begin();
    while ( (obj = (DG_Edict *)store->Next()) != NULL ) {
		if ( ! obj->IsReplica() ) {

			// items don't subscribe
			if (obj->GetType() == EDICT_ITEM)
				continue;

			if (obj->GetType() == EDICT_PLAYER) {
				ttl  = g_QuakePreferences.subttl_player;
				pred = g_QuakePreferences.subpred_player;
			} else if (obj->GetType() == EDICT_MONSTER) {
				ttl  = g_QuakePreferences.subttl_monster;
				pred = g_QuakePreferences.subpred_monster;
			} else if (obj->GetType() == EDICT_MISSILE) {
				ttl  = g_QuakePreferences.subttl_missile;
				pred = g_QuakePreferences.subpred_missile;
			} else {
				Debug::die("Unknown EDICT TYPE: %d", obj->GetType());
			}

			// add 0-5% of time to the TTL to prevent synchronization
			ttl = (uint32)(ttl*(1.0 + drand48()/20.0));

			GUIDSet *ids = &obj->m_InterestIDs;
			if (! obj->GetVisibleBBox(&bbox, 0) ) {
				// object said something funny happened and no bbox!
				obj->m_InterestIDs.clear();
				continue;
			}

			static float np_keys[MAX_KEYS];
			int num_np_keys = BBoxToNormalizedDHTPoints(np_keys, MAX_KEYS,
														&bbox);
			bool suffices = true;
			for (int i=0; i<num_np_keys; i++) {
				map<float, OMInterest *>::iterator p = have_keys.find(np_keys[i]);
				if (p != have_keys.end()) {
					OMInterest *in = p->second;
					bbox_t old_bbox;
					InterestToBBox(&old_bbox, in);
					if (BBoxContains(&old_bbox, &bbox)) { 
						obj->m_InterestIDs.insert(p->second->GetGUID());
						inuse.insert(p->second->GetGUID());
						// duplicate are ok, they will be removed later
						reg_assoc->insert(pair<GUID,GUID>(obj->GetGUID(), 
														  p->second->GetGUID()));
					} else {
						suffices = false;
						break;
					}
				} else {
					// oops! missing a key we need, need to resub
					suffices = false;
					break;
				}
			}
			if (suffices)
				return;
			
			// otherwise we need a new bbox!
			if (! obj->GetVisibleBBox(&bbox, pred) ) {
				// object said couldn't make bounding box!
				obj->m_InterestIDs.clear();
				continue;
			}

			static float new_keys[MAX_KEYS];
			int num_new_keys = BBoxToNormalizedDHTPoints(new_keys, MAX_KEYS,
														 &bbox);

			int num_ok = 0;

			for (int i=0; i<num_new_keys; i++) {
				map<float, OMInterest *>::iterator p = 
					have_keys.find(new_keys[i]);
				if (p != have_keys.end()) {
					OMInterest *in = p->second;
					bbox_t old_bbox;
					InterestToBBox(&old_bbox, in);
					if (BBoxContains(&old_bbox, &bbox)) { 

						// had a sub to the key already -- reuse it
						obj->m_InterestIDs.insert(p->second->GetGUID());
						inuse.insert(p->second->GetGUID());
						// duplicate are ok, they will be removed later
						reg_assoc->insert(pair<GUID,GUID>(obj->GetGUID(), 
														  p->second->GetGUID()));
						
						num_ok++;
						continue;
					}
				}

				// oops! missing a key we need, need a new sub
				OMInterest *in = g_Manager->CreateInterest();
				DHTKeyBBoxToInterest(in, new_keys[i], &bbox);
				obj->m_InterestIDs.insert( in->GetGUID() );
				
				ASSERT(ttl > 0);
				in->SetLifeTime(ttl);
				
				reg->push_back(in);
				
				inuse.insert(in->GetGUID());
				reg_assoc->insert(pair<GUID,GUID>(obj->GetGUID(), 
												  in->GetGUID()));
				have_keys[new_keys[i]] = in;
			}

		}
	}
	
	// unregister all the interests that are no longer in use
	for (InterestMapIter it = curr->begin(); it != curr->end(); it++) {
		if ( inuse.find(it->first) == inuse.end() ) {
			unreg->push_back(it->second);
		}
	}
}

void QuakeAdaptor::FillInterests(InterestList *reg,
								 InterestList *unreg,
								 GUIDMap *reg_assoc,
								 InterestMap *curr)
{
	if (g_QuakePreferences.dht) {
		FillInterestsDHT(reg, unreg, reg_assoc, curr);
		return;
	}

	static bool inited = false;
	static vec3_t worldmin, worldmax;
	if (!inited) {
		inited = true;
		DG_GetPrecompWorldBBox(worldmin, worldmax);
	}

	DG_Edict *obj;
	bbox_t bbox, oldbbox;
	uint32 ttl;
	float pred;
	ObjectStore *store = GetObjectStore();

	GUIDSet inuse;

	store->Begin();
    while ( (obj = (DG_Edict *)store->Next()) != NULL ) {
		if ( ! obj->IsReplica() ) {

			// items don't subscribe
			if (obj->GetType() == EDICT_ITEM)
				continue;

			if (obj->GetType() == EDICT_PLAYER) {
				ttl  = g_QuakePreferences.subttl_player;
				pred = g_QuakePreferences.subpred_player;
			} else if (obj->GetType() == EDICT_MONSTER) {
				ttl  = g_QuakePreferences.subttl_monster;
				pred = g_QuakePreferences.subpred_monster;
			} else if (obj->GetType() == EDICT_MISSILE) {
				ttl  = g_QuakePreferences.subttl_missile;
				pred = g_QuakePreferences.subpred_missile;
			} else {
				Debug::die("Unknown EDICT TYPE: %d", obj->GetType());
			}

			// add 0-5% of time to the TTL to prevent synchronization
			ttl = (uint32)(ttl*(1.0 + drand48()/20.0));

			GUIDSet *ids = &obj->m_InterestIDs;
			if (! obj->GetVisibleBBox(&bbox, 0) ) {
				// object said something funny happened and no bbox!
				obj->m_InterestIDs.clear();
				continue;
			}

			// XXX: This currently assumes that the old bbox is sent to
			// all the correct stripes and all have the same TTL so that
			// if one of them suffices then we know that the remainder in
			// the other stripes also suffice... man this is hacky :P
			bool suffices = false;
			for (GUIDSetIter it = ids->begin(); it != ids->end(); it++) {
				GUID id = *it;
				InterestMapIter p = curr->find( id );
				OMInterest *old = NULL;
				if (p != curr->end())
					old = p->second;

				// see if the old bbox suffices
				if (old) {
					InterestToBBox(&oldbbox, old); 
					if (BBoxContains(&oldbbox, &bbox)) {
						NOTE(OLD_BBOX_OK, 1);
						// old bbox we subscribed to suffices for our purposes!
						inuse.insert(old->GetGUID());
						suffices = true;
					} else 
						NOTE(OLD_BBOX_NOT_OK, 1);
				}
			}
			if (suffices)
				continue;

			obj->m_InterestIDs.clear();

			// see if my owner's bbox suffices
			if (obj->m_Entity->owner) {
				DG_Edict *owner = 
					(DG_Edict *)store->Find(obj->m_Entity->owner->guid);
				if (owner) {
					GUIDSet *ids = &owner->m_InterestIDs;
					for (GUIDSetIter it = ids->begin(); 
						 it != ids->end(); it++) {
						GUID id = *it;

						InterestMapIter op = curr->find( id );
						if (op != curr->end()) {
							OMInterest *owner_in = op->second;
							bbox_t owner_bbox;
							InterestToBBox(&owner_bbox, owner_in);
							if (BBoxContains(&owner_bbox, &bbox)) {
								NOTE(OWNER_BBOX_OK, 1);
								obj->m_InterestIDs.insert(id);
								inuse.insert(id);
								
								reg_assoc->insert(pair<GUID,GUID>
												  (obj->GetGUID(), 
												   id));
								
								suffices = true;
							} else {
								NOTE(OWNER_BBOX_NOT_OK, 1);
							}
						}
					}
				}
			}
			if (suffices)
				continue;

			ASSERT(obj->m_InterestIDs.size() == 0);

			// otherwise we need a new bbox!
			if (! obj->GetVisibleBBox(&bbox, pred) ) {
				// object said couldn't make bounding box!
				obj->m_InterestIDs.clear();
				continue;
			}

			obj->m_LastPredBBox = bbox;

			if (g_QuakePreferences.stripe_dim >= 0) {

#define MAX_RANGES 64
				static float min[MAX_RANGES], max[MAX_RANGES];
				int num = BBoxToStripeRanges(min, max, MAX_RANGES, &bbox);
				ASSERT(num > 0);

				//DB(0) << "num stripes: " << num << endl;

				for (int i=0; i<num; i++) {
					// need 1 sub per stripe published to
					OMInterest *in = g_Manager->CreateInterest();
					BBoxToInterest (in, &bbox);
					obj->m_InterestIDs.insert( in->GetGUID() );

					ASSERT(ttl > 0);
					in->SetLifeTime(ttl);

					//INFO << "bbox=" << bbox.min << " " << bbox.max << endl;
					//INFO << "min=" << min[i] << " max=" << max[i] << endl; 
					
					Value minv = g_Manager->ConvertToValue (g_StripeHubIndex, min[i], 0, 1);
					Value maxv = g_Manager->ConvertToValue (g_StripeHubIndex, max[i], 0, 1);
					Constraint c (g_StripeHubIndex, minv, maxv);

					in->AddConstraint (c);
					reg->push_back(in);

					inuse.insert(in->GetGUID());
					reg_assoc->insert(pair<GUID,GUID>(obj->GetGUID(), 
													  in->GetGUID()));
				}

			} else {

				OMInterest *in = g_Manager->CreateInterest();
				BBoxToInterest(in, &bbox);
				obj->m_InterestIDs.insert( in->GetGUID() );
			
				ASSERT(ttl > 0);				
				in->SetLifeTime( ttl );
				
				DB(1) << "obj " << obj->GetGUID()
					  << " filling sub: " << in
					  << endl;
				
				reg->push_back(in);
				inuse.insert(in->GetGUID());
				reg_assoc->insert(pair<GUID,GUID>(obj->GetGUID(), 
												  in->GetGUID()));
			}
		}
	}

	// unregister all the interests that are no longer in use
	for (InterestMapIter it = curr->begin(); it != curr->end(); it++) {
		if ( inuse.find(it->first) == inuse.end() ) {
			unreg->push_back(it->second);
		}
	}
}

GObject *QuakeAdaptor::Construct(GObjectInfoIface *info,
								 Packet *pkt, const DeltaMask& mask,
								 SIDMap *unresolved)
{
    DG_Edict *dg_obj = new DG_Edict(info);

    dg_obj->ConstructEntity(pkt, mask, unresolved);

	/*
	INFO << "constructing: scratch? "
		 << (dg_obj->m_ScratchEnt == NULL) << "  " << 
		(dg_obj->m_ScratchEnt ? dg_obj->m_ScratchEnt : dg_obj->m_Entity)
		 << endl;
	*/

    return dg_obj;
}

void QuakeAdaptor::Destroy(guid_t guid)
{
    bool found = false;

    if (m_DG_Edicts->Find(guid) != NULL) {
		m_DG_Edicts->Destroy(guid);
        found = true;
    }
    if (m_Half_Edicts->Find(guid) != NULL) {
		m_Half_Edicts->Destroy(guid);
		found = true;
    }

    if (!found) {
		WARN << "Object being removed not present in the "
			 << "normal or pending store (guid:" << guid << ")" << endl;
    }
}

void QuakeAdaptor::CheckInvariants() {
    DG_ObjStore *normal = (DG_ObjStore *) GetObjectStore();
    DG_ObjStore *pending = (DG_ObjStore *) GetPendingStore();

    normal->Begin();
    GObject *obj;
    while ((obj = normal->Next())) {
		DG_Edict *dg_obj = (DG_Edict *) obj;
		ASSERT(dg_obj->m_Entity);
		//ASSERT(!dg_obj->m_ScratchEnt);

		ASSERT(dg_obj->m_Entity->inuse);
		ASSERT(dg_obj->m_Entity->classname);
    }
    pending->Begin();
    while ((obj = pending->Next())) {
		DG_Edict *dg_obj = (DG_Edict *) obj;
		ASSERT(!dg_obj->m_Entity);
		ASSERT(dg_obj->m_ScratchEnt);

		// non-players should not have client pointers
		if (strcmp(dg_obj->m_ScratchEnt->classname, "player") &&
			strcmp(dg_obj->m_ScratchEnt->classname, "bot") &&
			strcmp(dg_obj->m_ScratchEnt->classname, "disconnected")) {
			ASSERT(!dg_obj->m_ScratchEnt->client);
		}
    }

	edict_t *ent;
	for (ent = g_edicts; ent < g_edicts + game.maxentities; ent++) {
		// non-players should not have client pointers
		if (ent->inuse && 
			strcmp(ent->classname, "player") &&
			strcmp(ent->classname, "bot") &&
			strcmp(ent->classname, "disconnected")) {
			ASSERT(!ent->client);
		}

		if (ent->s.number > 0 &&
			ent->s.number < game.maxclients + 1 &&
			ent->classname &&
			(!strcmp(ent->classname, "player") ||
			 !strcmp(ent->classname, "bot"))) {
			ASSERT(ent->inuse);
		}

		if (ent->s.number > 0 &&
			ent->s.number < game.maxclients + 1 && ent->inuse) {
			ASSERT(!strcmp(ent->classname, "bot") ||
				   !strcmp(ent->classname, "bot"));
		}
			
	}
}

///////////////////////////////////////////////////////////////////////////////

static void handleVersion(ostream& out, char **args);
static void handleGEdicts(ostream& out, char **args);
static void handleVisIP(ostream& out, char **args);
static void handleQuakeCommand(ostream& out, char **args);

static InputHandler g_QuakeHandlers[] = {
	{ "QVER", "QV", handleVersion, "",
	  "Show the version of quake2" },
	{ "GEDICTS", "GE",  handleGEdicts,   "[{P|R|A|N} [{P|M|I}]]",
	  "Show distributed entities in g_edicts" },
	{ "VISIP", "VI", handleVisIP, "ipaddress",
	  "Set the IP the visualizer should send to (use 'none' to disable)" },
	{ "QCMD", "QC", handleQuakeCommand, "quake_console_command...",
	  "Execute a quake console command on this server" },
	{ NULL, NULL, NULL, NULL }
};

void QuakeAdaptor::InstallTerminalHandlers(InputHandlers *tofill)
{
	InputHandler *ih = g_QuakeHandlers;
	while (ih->cmd) {
		tofill->push_back(ih);
		++ih;
	}
}

void handleVersion(ostream& out, char **args) {
    out << "quake2 " << g_QuakeVersion 
		<< " (" << g_QuakeBuildTime << ")" << "\n";
    return;
}

void handleGEdicts(ostream& out, char **args)
{
	bool distributed = true;
	bool showReplicas = true;
	bool showPrimaries = true;
	bool showPlayers = true;
	bool showMissiles = true;
	bool showItems   = true;
	int i;
	edict_t *ent;

	if (args[1]) {
		if (!strcasecmp(args[1], "P")) {
			showReplicas = false;
		} else if (!strcasecmp(args[1], "R")) {
			showPrimaries = false;
		} else if (!strcasecmp(args[1], "A")) {
			// already true
		} else if (!strcasecmp(args[1], "N")) {
			distributed = false;
		} else {
			out << "ERROR: unknown filter argument: " << args[1] << endl;
			return;
		}

		if (args[2]) {
			if (!strcasecmp(args[2], "P")) {
				showMissiles = false;
				showItems = false;
			} else if (!strcasecmp(args[2], "M")) {
				showPlayers = false;
				showItems = false;
			} else if (!strcasecmp(args[2], "I")) {
				showPlayers = false;
				showMissiles = false;
			} else {
				out << "ERROR: unknown filter argument: " << args[2] << endl;
				return;
			}
		}
	}

	ent = &g_edicts[0];
	for (i = 0; i < game.maxentities; i++, ent++) {
		if (!ent->inuse && !ent->s.modelindex ||
			(distributed && DG_IsNonDistributed(ent->guid)) ||
			(!distributed && !DG_IsNonDistributed(ent->guid))) {
			continue;
		}
		if (!showReplicas && ent->is_replica ||
			!showPrimaries && !ent->is_replica) {
			continue;
		}

		if ( !showPlayers &&
			 (streq(ent->classname, "bot") ||
			  streq(ent->classname, "player")) ) {
			continue;
		} else if ( !showMissiles &&
					(streq(ent->classname, "bolt") ||
					 streq(ent->classname, "grenade") ||
					 streq(ent->classname, "hgrenade") ||
					 streq(ent->classname, "rocket") ||
					 streq(ent->classname, "bfg blast")) ) {
			continue;
		} else if ( !showItems &&
					!(streq(ent->classname, "bot") ||
					  streq(ent->classname, "player") ||
					  streq(ent->classname, "bolt") ||
					  streq(ent->classname, "grenade") ||
					  streq(ent->classname, "hgrenade") ||
					  streq(ent->classname, "rocket") ||
					  streq(ent->classname, "bfg blast")) ) {
			continue;
		}
		
		out << ent << endl;
	}
}

void handleVisIP(ostream& out, char **args)
{
	if (!args[1]) {
		out << "ERROR: expected ipaddress or 'none'" << endl;
		return;
	}

	if (!strcasecmp(args[1], "NONE")) {
		strcpy(g_QuakePreferences.visualizer_ip, "");
	} else {
		strncpy(g_QuakePreferences.visualizer_ip, args[1], 255);

		IPEndPoint visIP(g_QuakePreferences.visualizer_ip, VISUALIZER_PORT);
		if (visIP == SID_NONE) {
			out << "ERROR: invalid address" << endl;
			return;
		}

		memset((char *) &g_VisAddr, 0, sizeof(g_VisAddr));
		g_VisAddr.sin_family = AF_INET;
		g_VisAddr.sin_addr.s_addr = visIP.GetIP();
		g_VisAddr.sin_port = htons(visIP.GetPort());
	}

	out << "OK" << endl;
}

void handleQuakeCommand(ostream& out, char **args)
{
	if (!args[1]) {
		out << "ERROR: expected quake command" << endl;
		return;
	}

	args++;

	char cmd[512];
	char *ptr = cmd;
	int len = 0;
	while (args[0] != NULL) {
		int thislen = strlen(args[0]);
		if (thislen >= 512-len-1) {
			out << "ERROR: command too long" << endl;
			return;
		}

		strcpy(ptr, args[0]);
		ptr += thislen;
		len += thislen;
		strcpy(ptr, " ");
		ptr += 1;
		len += 1;

		args++;
	}

	Cbuf_Init();
	Cbuf_InsertText(cmd);
	Cbuf_Execute();

	out << "OK: " << cmd << endl;
}
