/* vim: set sw=4 ts=4 noet: -*- Mode:c++; c-basic-offset:4; tab-width:4; indent-tabs-mode:t -*- */

#include <dg/dg.h>
#include <om/Manager.h>
#include <dg/DG_ObjStore.h>
#include <dg/DG_Edict.h>

extern Manager *g_Manager;

//////////////////////////////////////////////////////////////////////////
// DG_ObjStore
//////////////////////////////////////////////////////////////////////////

DG_ObjStore::DG_ObjStore(qboolean pending)
{
    m_IsPending = pending;
    m_Iter = m_Objmap.begin();
}

void DG_ObjStore::Begin()
{
    m_Iter = m_Objmap.begin();
}

GObject *DG_ObjStore::Next()
{
    if (m_Iter == m_Objmap.end()) 
        return NULL;
    GObject *o = m_Iter->second;
    m_Iter++;
    return o;
}

GObject *DG_ObjStore::Find(guid_t guid)
{
    GObjectMapIter it = m_Objmap.find(guid);
    if (it == m_Objmap.end()) 
		return NULL;
    else
		return it->second;
}
    
void DG_ObjStore::Dump()
{
    
    DB_DO(5) {
		DB(1) << "Object store status" << endl;
	
		for (GObjectMapIter it = m_Objmap.begin(); it != m_Objmap.end(); it++) {
			DG_Edict *dg_obj = (DG_Edict *) it->second;
			edict_t *ent = dg_obj->GetEntity();
			if (!ent) { ent = dg_obj->GetScratchEntity(); }
			ASSERT(ent);

			ASSERT(ent->s.number == (ent - g_edicts));
			//if (ent->is_replica) {
			DB(1) <<  " ent->s:" << ent->s.number << ", guid: " << ent->guid <<
				" replica: " << ent->is_replica << " classname: " << (ent->classname ? ent->classname : "null") << endl;
			//		    " coords = " << ent->s.origin[0] << ":" << ent->s.origin[1] << ":" << ent->s.origin[2] << endl;
			//}
		}

		DB(1) << "-------------" << endl;
    }
}

void DG_ObjStore::_ManagerRemove(guid_t guid)
{
    // Only called from Manager, no need to lock

    DB(1) << "Removing " << guid << " from "
		  << (m_IsPending?"pending":"normal") << " store" << endl;
 
    GObjectMapIter it = m_Objmap.find(guid);
    if (it == m_Objmap.end()) {
        WARN << "Object being removed not present in the "
			 << (m_IsPending?"pending":"normal") << " store" << endl;
        return;
    }

    DG_Edict *dg_obj = (DG_Edict *) it->second;
    if (m_IsPending) {
		//
        // is it possible for the pending store to be invoked a DELETE operation?
		// Jeff: Yes -- this is called on the PendingStore when an object is
		// moved from it to the normal ObjectStore. Thus it should not be
		// deleted. (In fact, I don't think it should be deleted in the
		// normal case either... but since the Manager currently will never
		// call it on the object store, it doesn't really matter)
		//
        m_Objmap.erase(it);
    } else {
		// being removed from the NORMAL object store?  Must be a delete operation
		DG_HandleObjectDelete(guid);
		delete dg_obj;
		m_Objmap.erase(it);
    }
}

//
// Ashwin [05/20]; this is basically a copy of the Remove() routine except
// that it "deletes" the object irrespective of whether it is invoked on the
// pending store or the normal store
//
void DG_ObjStore::Destroy(guid_t guid)
{
    // Only called from Manager, no need to lock

    GObjectMapIter it = m_Objmap.find(guid);
    DB(1) << " got DESTROY for " << guid << " in the " <<
		(m_IsPending ? " pending " : " normal ") << "store" << endl;
    
    ASSERT(it != m_Objmap.end());

    DG_Edict *dg_obj = (DG_Edict *) it->second;
    ASSERT(dg_obj->IsReplica());

	/*
	if (dg_obj->m_Entity) {
		INFO << "Destroying: " << dg_obj->m_Entity << endl;
	}
	*/
  
    if (!m_IsPending) {
		DG_HandleObjectDelete(guid);
    }
    delete dg_obj;
    m_Objmap.erase(it);

}

// This is a remote Add from the lower OM layer
void DG_ObjStore::_ManagerAdd(GObject *obj)
{
    // Only called from Manager, no need to lock

    GObject *old_obj = Find(obj->GetGUID());
    if (old_obj) {
        if (!m_IsPending) {
			// 4/29/2004 Jeff:
			// This can happen in the following circumstance:
			// (1) Get an object A that has a unresolved ref to object B
			// (2) A goes in the pending store, and we request B
			// (3) A publication for B arrives, A is resolved, B is resolved,
			//     both go in object store.
			// (4) FetchResp for B finally returns, it creates a new copy of
			//     B, B gets added to the object store again.
			//
			// This is only one of the situations it can occur; in general,
			// If we get *any* asynchronous FetchResp, we will attempt to
			// add the object to the object store because a FetchResp's
			// object is almost certainly more up-to-date than our current
			// copy. Do we want these semantics? I allowed multiple adds
			// in case the application wanted to be fancy about keeping
			// versions about its objects and hence this function could
			// decide whether to keep the old one or the new one. In any case,
			// if we don't keep the new one, we must delete it (otherwise
			// it is a memory leak)
			//
			// XXX TODO: we should use the new copy, not the old one
            DBG << "Duplicate 'Add' for the ObjectStore Ignoring... " 
				<< obj->GetGUID() << endl;
            delete obj;
			return;
        }

        delete old_obj; // should invoke the right destructor... 
    }

    DB(1) << "[34madding an object with guid : [m" << obj->GetGUID() 
		  << (m_IsPending ? " to pending store " : " to normal object store ") << endl;


    ASSERT(obj);
    ASSERT(obj->IsReplica()); // om layer shouldn't add primaries
    m_Objmap[obj->GetGUID()] = obj;

    if (!m_IsPending) {
        // a new object has been added. should inform the upper layer.
		// However, the best way is for the DG_Edict to do the job.
		((DG_Edict *) obj)->OnObjectAdd();
    }
}

void DG_ObjStore::_ApplicationAdd(GObject *obj)
{
    // XXX HACK
    g_Manager->Lock();
    // XXX

    DB(1) << "Application Adding " << obj->GetGUID() << " to "
		  << (m_IsPending?"pending":"normal") << " store" << endl;

    ASSERT(!m_IsPending);
    GObject *old_obj = Find(obj->GetGUID());
    if (old_obj) {
		// Jeff: This is a bug, not a warning
		ASSERT(0);
        //WARN << "Conflicting object addition from the application? Overwriting the earlier object" << endl;
        // XXX A memory leak here, potentially - Ashwin [4/7/2004]
    }

	//    DB(2) << "Adding object :" << obj->GetGUID() << ":" << endl;
    ASSERT(obj != NULL);
    m_Objmap[obj->GetGUID()] = obj;

    // XXX HACK
    g_Manager->Unlock();
    // XXX
}

void DG_ObjStore::_ApplicationRemove(guid_t guid)
{
    // XXX HACK
    g_Manager->Lock();
    // XXX

    DB(2) << "Application Deleting " << guid << " from "
		  << (m_IsPending?"pending":"normal") << " store" << endl;

    // The application wants to get rid of an object.
    // before doing this, DeleteGUID has been called
    // which informs remote interested parties accordingly.

    GObjectMapIter it = m_Objmap.find(guid);
    ASSERT(it != m_Objmap.end());

	// the application frees the object - Ashwin [05/20]
    m_Objmap.erase(it);

    // XXX HACK
    g_Manager->Unlock();
    // XXX
}

///////////////////////////////////////////////////////////////////////////////

edict_t *debug_get_entity(GObject *obj)
{
    return ((DG_Edict *) obj)->m_Entity;
}

edict_t *debug_get_sentity(GObject *obj)
{
    return ((DG_Edict *) obj)->m_ScratchEnt;
}
