/* vim: set sw=4 ts=4 noet: -*- Mode:c++; c-basic-offset:4; tab-width:4; indent-tabs-mode:t -*- */

#ifndef DG_EDICT_H
#define DG_EDICT_H

#include <hash_map.h>
#include <game/g_local.h>
#include <om/GObject.h>
#include <dg/dg_delta_encoding.h>
#include <dg/dg_bbox.h>

///////////////////////////////////////////////////////////////////////////////

struct eqstr {
	bool operator()(const char* s1, const char* s2) const {
		return strcmp(s1, s2) == 0;
	}
};

typedef hash_map<const char*, float, hash<const char*>, eqstr> CostMap;
typedef CostMap::iterator CostMapIter;
extern CostMap g_CostMap;

///////////////////////////////////////////////////////////////////////////////

typedef list<byte *> ObjLocList;
typedef ObjLocList::iterator ObjLocListIter;

///////////////////////////////////////////////////////////////////////////////

class EdictRef : public GObjectRef {
public:
	bool client; // true if in gclient_t rather than edict_t
	int  offset; // offset into gclient_t or edict_t of the pointer
	GUID guid;   // unresolved GUID ref
	SID  owner;  // last known owner

    EdictRef(GUID guid, SID owner, bool client, int offset) :
		guid(guid), owner(owner), client(client), offset(offset) {
	}

	virtual ~EdictRef() {}

	GUID GetTarget() { return guid; }
};

struct less_EdictRef {
	bool operator() ( const EdictRef *a, const EdictRef *b) const {
		return (!a->client && b->client) || (a->offset < b->offset);
    }
};

typedef set<EdictRef *, less_EdictRef> EdictRefs;
typedef EdictRefs::iterator EdictRefsIter;

struct EdictRefToFix {
	DG_Edict *obj;
	EdictRef ref;

	EdictRefToFix(DG_Edict *obj, EdictRef& ref) :
		obj(obj), ref(ref) {}
};

///////////////////////////////////////////////////////////////////////////////

// types of quake objects
typedef enum { EDICT_INVALID, EDICT_PLAYER, EDICT_MONSTER, 
			   EDICT_MISSILE, EDICT_ITEM } EdictType;

// encapsulates a edict_t
class DG_Edict : public GObject 
{
    friend ostream& operator<<(ostream& out, DG_Edict *obj);

	void MakeFatPVS (vec3_t org);
	void MakePredictedOrigin(vec3_t org, edict_t *ent);
	bool PredictedCheckIsVisible(edict_t *ent);

	void FixPointer(EdictRef *eref, edict_t *me, edict_t *other);

protected:

	static ClassChunker<edict_t>   *m_ScratchEnts;
	static ClassChunker<gclient_t> *m_ScratchClients;

	static void Init();

	// pointers in our edict that are unresolved
	EdictRefs m_UnresolvedPtrs;
	// pointers in other objects that point to our edict that should
	// be fixed when we are added to the object store... why do we need
	// this? because when the pointer is originally resolved, we might
	// still be in the pending store, so we don't have an m_Entity yet
	// (this could happen because both objects may be added in the same
	// round of resolution, and one of them has to be resolved and added
	// first, so the other will still be at the pending store at the time)
	list<EdictRefToFix> m_PtrsToFixOnAdd;

public: // HACK for testing: 5/16 Jeff
	// This one is a pointer into g_edicts array
    edict_t   *m_Entity;
	// These are scratch, used for construction
    edict_t   *m_ScratchEnt;
    gclient_t *m_ScratchClient;
	GUIDSet    m_InterestIDs; // ID of current interest
	EdictType  m_Type;

	// Delta mask required for constructing a "new" copy of this object
	DeltaMask  m_InitDeltaMask;

	///////////////////////////////////////////////////////////////////////////

	// current (predicted) origin
	vec3_t  m_CurrOrg;
	// current bounding box (for non-items)
	bbox_t  m_CurrBBox;
	// last predicted box
	bbox_t  m_LastPredBBox;
	// for prediction of clients pvs
	byte	m_CurrFatPVS[65536/8]; // 32767 is MAX_MAP_LEAFS

	///////////////////////////////////////////////////////////////////////////

	void AllocScratchEnt();
    void FreeScratchEnt();
//public:
    DG_Edict(edict_t *ent);
    DG_Edict(GObjectInfoIface *info);
    ~DG_Edict();

	void PrecomputePerFrameInfo();

	EdictType    GetType();
    edict_t     *GetEntity() { return m_Entity; }
    edict_t     *GetScratchEntity() { return m_ScratchEnt; }

	void         AttachedTo(GUIDList *others);

	uint32       InterestTime(OMEvent *ev, InterestMap *curr);
	uint32       InterestTime(GObject *obj, InterestMap *curr);

	void         FillEventsDHT(EventList *reg, EventList *unreg, EventSet *curr);
	void         FillEvents(EventList *reg, EventList *unreg, EventSet *curr);
	bool         GetVisibleBBox(bbox_t *bbox, float predict);

	const DeltaMask& GetInitDeltaMask() { return m_InitDeltaMask; }
    void         PackUpdate(Packet *pkt, const DeltaMask& mask);
    void         UnpackUpdate(Packet *pkt, const DeltaMask& mask,
				 SIDMap *unresolved);

	void         GetRefs(GObjectRefList *tofill);
	void         ResolveRef(GObjectRef *ref, GObject *obj);
    void         RegisterRef(guid_t guid, sid_t sid, 
							 bool client, int offset, bool missing);
    
    bool         IsMigratable();
    void         CommitMigration(sid_t target);
    bool         IsNew();

    CostMetric   GetFixedCost();
    CostMetric   GetDeltaCost();

    void         ConstructEntity(Packet *pkt, const DeltaMask& mask,
				 SIDMap *unresolved);
    void         OnObjectAdd();

    void         PrintFields(ostream& out);
};

///////////////////////////////////////////////////////////////////////////////

ostream& operator<<(ostream& out, edict_t *edict);
ostream& operator<<(ostream& out, DG_Edict *obj);

#endif
