/* vim: set sw=4 ts=4 noet: -*- Mode:c++; c-basic-offset:4; tab-width:4; indent-tabs-mode:t -*- */

#include <qcommon/qcommon.h>
#include <dg/dg.h>
#include <dg/dg_logs.h>
#include <dg/dg_bbox.h>
#include <dg/DG_Edict.h>
#include <util/Benchmark.h>
#include <om/Manager.h>

extern Manager *g_Manager;

//////////////////////////////////////////////////////////////////////////
// DG_Edict
//////////////////////////////////////////////////////////////////////////

ClassChunker<edict_t>  *DG_Edict::m_ScratchEnts     = NULL;
ClassChunker<gclient_t> *DG_Edict::m_ScratchClients  = NULL;

void DG_Edict::Init()
{
	m_ScratchEnts    = new ClassChunker<edict_t>();
	m_ScratchClients = new ClassChunker<gclient_t>();
}

void DG_Edict::AllocScratchEnt()
{
	m_ScratchEnt      = m_ScratchEnts->alloc();
	m_ScratchClient   = m_ScratchClients->alloc();

	memset((void *)m_ScratchEnt, 0, sizeof(edict_t));
	memset((void *)m_ScratchClient, 0, sizeof(gclient_t));
	// Don't set this here -- only set it when we need to deserialize it
	//m_ScratchEnt->client = m_ScratchClient;
}

void DG_Edict::FreeScratchEnt()
{
	ASSERT(m_ScratchEnt);
	ASSERT(m_ScratchClient);

	if (m_ScratchEnt->multicastEventList) {
		delete m_ScratchEnt->multicastEventList;
	}
	m_ScratchEnts->free(m_ScratchEnt);
	m_ScratchClients->free(m_ScratchClient);

    m_ScratchEnt    = NULL;
    m_ScratchClient = NULL;
}

void DG_Edict::OnObjectAdd()
{
    m_Entity = DG_HandleObjectUpdate(m_Entity, m_ScratchEnt);
	ASSERT(m_Entity->inuse);

	// see if there are any pointers in other objects that are waiting
	// to be fixed when we are added to the object store...
	for (list<EdictRefToFix>::iterator it = m_PtrsToFixOnAdd.begin();
		 it != m_PtrsToFixOnAdd.end(); it++) {
		// the other guy should be in object store already...
		ASSERT(it->obj->m_Entity);
		FixPointer(&it->ref, it->obj->m_Entity, m_Entity);
	}
	m_PtrsToFixOnAdd.clear();
	
	PrecomputePerFrameInfo();
}

// Constructor would get invoked when:
//   a) GameAdaptor::Construct is called 
//   b) The application (Quake) registers a new object 
DG_Edict::DG_Edict(edict_t *ent) : 
	GObject(ent->guid), m_Type(EDICT_INVALID)
{
	if (!m_ScratchEnts || !m_ScratchClients) {
		Init();
	}

    m_Entity = ent;
	ASSERT(m_Entity->inuse);
    AllocScratchEnt();

	bzero(m_CurrOrg, sizeof(vec3_t));
	bzero(&m_CurrBBox, sizeof(bbox_t));
	bzero(&m_CurrFatPVS, sizeof(m_CurrFatPVS));
}

DG_Edict::DG_Edict(GObjectInfoIface *info) : 
	m_Type(EDICT_INVALID), GObject(info) { 

	if (!m_ScratchEnts || !m_ScratchClients) {
		Init();
	}
	
	m_Entity = NULL;
	AllocScratchEnt();
}

DG_Edict::~DG_Edict()
{
    FreeScratchEnt();

	// Do not G_FreeEdict m_Entity.
	//
	// (1) if it was a primary, then it is already freed by the game
	//     (and possibly replaced by something else) in DG_RecordChanges
	// (2) if it was a replica, then it will be freed by DG_HandleObjectDelete

    for (EdictRefsIter it = m_UnresolvedPtrs.begin(); 
		 it != m_UnresolvedPtrs.end(); it++) {
        EdictRef *ref = (EdictRef *) *it;
        delete ref;
    }
}

void DG_Edict::GetRefs(GObjectRefList *tofill)
{
	for (EdictRefsIter it = m_UnresolvedPtrs.begin();
		 it != m_UnresolvedPtrs.end(); it++) {
		tofill->push_back(*it);
	}
}

void DG_Edict::ResolveRef(GObjectRef *ref, GObject *obj)
{
	EdictRef *eref = (EdictRef *)ref;

	EdictRefsIter p = m_UnresolvedPtrs.find(eref);
	if (p == m_UnresolvedPtrs.end()) {
		DB(0) << "Trying to resolve ref that is not unresolved! " 
			  << "at: " << (eref->client?"client":"edict")
			  << ":" << eref->offset << " to: "
			  << obj->GetGUID() << endl;
	} else {
		m_UnresolvedPtrs.erase(p);
	}

	// INVARIANT: edicts should only point to the ObjectStore, not
	// the PendingStore (i.e., edicts in g_edicts). To ensure this,
	// we have to delay resolution of pointers to objects currently
	// in the PendingStore. The Manager gives us the invariant that
	// Objects will only be added to the ObjectStore when all their
	// pointers have been resolved, but since it must resolve and
	// add one object at a time, when this function is called, the
	// other object may not yet be in the ObjectStore. However, we
	// are safe in delaying resolution until the other object *is*
	// added to the ObjectStore (OnObjectAdd()), since the Manager
	// guarantees that it will be added in the same "resolution round"
	// as this one.

	DG_Edict *eobj = (DG_Edict *)obj;
	if (!eobj || eobj->m_Entity) {
		// if the remote object was deleted, hopefully setting this
		// to NULL will do the "right thing" ... :|
		edict_t *other = eobj == NULL ? NULL : eobj->m_Entity;
		edict_t **ptr;

		if (other) {
			ASSERT((uint32)other >= (uint32)g_edicts);
			ASSERT((uint32)other < (uint32)(g_edicts + game.maxentities));
		}

		ASSERT(!m_Entity);    // XXX: can we assume always in pending store?
		ASSERT(m_ScratchEnt);
		FixPointer(eref, m_ScratchEnt, other);
	} else {
		// Not added to object store yet! Presumably will be added immediately
		// following this one in resolution order, so resolve the ref after it
		// is added... (See OnObjectAdd())
		eobj->m_PtrsToFixOnAdd.push_back(EdictRefToFix(this, *eref));
	}

	delete eref;
}

void DG_Edict::FixPointer(EdictRef *eref, edict_t *me, edict_t *other)
{
	edict_t **ptr;
	
	if (eref->client) {
		ptr = (edict_t **)(((byte *)me->client) + eref->offset);
	} else {
		ptr = (edict_t **)(((byte *)me) + eref->offset);
	}
	*ptr = other;
}

extern typeinfo_t typeinfo[];

// Register a missing reference.
void DG_Edict::RegisterRef(guid_t guid, sid_t sid, 
						   bool client, int offset, bool missing)
{
	if (missing) {
		// new missing pointer or changed missing pointer
		EdictRef *ref = new EdictRef(guid, sid, client, offset);
	
		EdictRefsIter p = m_UnresolvedPtrs.find(ref);
		if (p != m_UnresolvedPtrs.end()) {
			EdictRef *old = *p;
			m_UnresolvedPtrs.erase(p);
			delete old;
		}

		m_UnresolvedPtrs.insert(ref);
	} else {
		// missing pointer no longer missing
		EdictRef ref(guid, sid, client, offset);
		EdictRefsIter p = m_UnresolvedPtrs.find(&ref);
		if (p != m_UnresolvedPtrs.end()) {
			EdictRef *old = *p;
			m_UnresolvedPtrs.erase(p);
			delete old;
		}
	}
}

EdictType DG_Edict::GetType()
{
	if (m_Type == EDICT_INVALID) {
		edict_t *ent;
		if (m_Entity)
			ent = m_Entity;
		else
			ent = m_ScratchEnt;

		if ( ent->classname == NULL ) {
			m_Type = EDICT_INVALID;
		} else if ( streq(ent->classname, "bot") ||
					streq(ent->classname, "player") ) {
			if (m_Entity->client->is_monster)
				m_Type = EDICT_MONSTER;
			else
				m_Type = EDICT_PLAYER;
		} else if ( streq(ent->classname, "bolt") ||
					streq(ent->classname, "grenade") ||
					streq(ent->classname, "hgrenade") ||
					streq(ent->classname, "rocket") ||
					streq(ent->classname, "bfg blast") ) {
			m_Type = EDICT_MISSILE;
		} else {
			m_Type = EDICT_ITEM;
		}
	}
	return m_Type;
}

namespace Tmp {
	static bool inited = false;
	static vec3_t worldmin, worldmax;
};

void DG_Edict::AttachedTo(GUIDList *others)
{
	// attach missiles to their owner, so they will be proactively replicated
	// to where their owner is
	if ( GetType() == EDICT_MISSILE && !IsReplica() ) {
		if ( m_Entity->owner ) {
			ASSERT( m_Entity->owner->guid != GUID_NONE );
			others->push_back( m_Entity->owner->guid );
		}
	}
}

///////////////////////////////////////////////////////////////////////////////

// precompute some information that is used often per frame!
void DG_Edict::PrecomputePerFrameInfo()
{
	if (!Tmp::inited) {
		Tmp::inited = true;
		DG_GetPrecompWorldBBox(Tmp::worldmin, Tmp::worldmax);
	}

	if (GetType() == EDICT_PLAYER ||
		GetType() == EDICT_MONSTER) {
		MakePredictedOrigin(m_CurrOrg, m_Entity);
	} else {
		// XXX TODO: we should do MORE prediction for missiles
		// since they are easy to predict!
		VectorCopy(m_Entity->s.origin, m_CurrOrg);
	}

	if (GetType() == EDICT_ITEM)
		return;

	for (int i=0; i<3; i++) {
		if (m_CurrOrg[i] < Tmp::worldmin[i] ||
			m_CurrOrg[i] > Tmp::worldmax[i]) {
			// stupid damn bots and missiles!
			/*
			DB(1) << "stupid object outside of world! : "
				  << GetGUID() << " " << m_Entity->classname 
				  << " origin=" << m_CurrOrg
				  << " worldmin=" << Tmp::worldmin
				  << " worldmax=" << Tmp::worldmax << endl;
			*/
			return;
		}
	}

	MakeFatPVS(m_CurrOrg);
	DG_GetPrecompBoundingBox(m_CurrOrg, m_CurrBBox.min, m_CurrBBox.max, 0);

	/*
	INFO << "called: type=" 
		 << GetType() << " org="
		 << m_Entity->s.origin
		 << " predorg=" << m_CurrOrg
		 << " bbox=" 
		 << "[min=" << m_CurrBBox.min
		 << ",max=" << m_CurrBBox.max << "]" << endl;
	*/
}

// From sv_ents -- makes a predicted pvs around origin
void DG_Edict::MakeFatPVS (vec3_t org)
{
	int		leafs[64];
	int		i, j, count;
	int		longs;
	byte	*src;
	vec3_t	mins, maxs;

	for (i=0 ; i<3 ; i++)
	{
		mins[i] = org[i] - 8;
		maxs[i] = org[i] + 8;
	}

	count = CM_BoxLeafnums (mins, maxs, leafs, 64, NULL);
	if (count < 1) {
		WARN << "SV_FatPVS: count < 1" << endl;
		ASSERT(0);
	}
	longs = (CM_NumClusters()+31)>>5;

	// convert leafs to clusters
	for (i=0 ; i<count ; i++)
		leafs[i] = CM_LeafCluster(leafs[i]);

	memcpy (m_CurrFatPVS, CM_ClusterPVS(leafs[0]), longs<<2);
	// or in all the other leaf bits
	for (i=1 ; i<count ; i++)
	{
		for (j=0 ; j<i ; j++)
			if (leafs[i] == leafs[j])
				break;
		if (j != i)
			continue;		// already have the cluster we want
		src = CM_ClusterPVS(leafs[i]);
		for (j=0 ; j<longs ; j++)
			((long *)m_CurrFatPVS)[j] |= ((long *)src)[j];
	}
}

// This generates the predicted origin of the client
void DG_Edict::MakePredictedOrigin(vec3_t org, edict_t *ent)
{
	ASSERT(ent->client);

	for (int i=0 ; i<3 ; i++)
		org[i] = ent->client->ps.pmove.origin[i]*0.125 + 
			ent->client->ps.viewoffset[i];
}

bool DG_Edict::PredictedCheckIsVisible(edict_t *ent)
{
	int		e, i;
	vec3_t	org;
	int		l;
	int		clientarea, clientcluster;
	int		leafnum;
	byte	*bitvector;

	leafnum = CM_PointLeafnum (m_CurrOrg);
	clientarea = CM_LeafArea (leafnum);
	clientcluster = CM_LeafCluster (leafnum);

	// check area
	if (!CM_AreasConnected (clientarea, ent->areanum))
	{
		// doors can legally straddle two areas, so
		// we may need to check another one
		if (!ent->areanum2
			|| !CM_AreasConnected (clientarea, ent->areanum2))
			return false;		// blocked by a door
	}

	bitvector = m_CurrFatPVS;

	if (ent->num_clusters == -1)
	{
		if (!CM_HeadnodeVisible (ent->headnode, bitvector))
			return false;
	}
	else
	{	// check individual leafs
		for (i=0 ; i < ent->num_clusters ; i++)
		{
			l = ent->clusternums[i];
			if (bitvector[l >> 3] & (1 << (l&7) ))
				break;
		}
		if (i == ent->num_clusters)
			return false;		// not visible
	}

	return true;
}

///////////////////////////////////////////////////////////////////////////////

uint32 DG_Edict::InterestTime(OMEvent *ev, InterestMap *curr)
{
	uint32 interestTime = ev->GetLifeTime() + ManagerParams::PUBSUB_TTL_EXTRA;

	if (ev->IsPointEvent ()) {
		vec3_t pt;
		EventToPoint(pt, ev);

		// Assume that the PVS is a better filter than our predictive bbox
		// which is probably true... --- must use this func instead of
		// fatpvs since we don't have the edict yet

#if 1
		// use bboxes
		if (BBoxContains(&m_CurrBBox, pt))
#else
		if ( gi.inPVS (m_CurrOrg, pt) ) 
#endif
		{
			return interestTime;
		} else {
			return 0;
		}
		
		for (GUIDSetIter it = m_InterestIDs.begin(); 
			 it != m_InterestIDs.end(); it++) {
			GUID id = *it;
			InterestMapIter p = curr->find( id );
			if (p != curr->end()) {
				OMInterest *in = p->second;
				if (in->Overlaps (ev)) {
					return interestTime;
				}
			}
		}

		// It may still be the case that we are just missing
		// our latest sub in the system. But in this case just
		// wait until the re-sub since items aren't that important...
		return 0;

	} 
	else {
		// If it is a ranged pub, that means it is predictive, so we
		// have no choice but to just see if it overlaps our BBOX
		bbox_t other_bbox;
		EventToBBox(&other_bbox, ev);

		bool intOverlaps = false;

		//DG_GetPrecompBoundingBox(m_Entity->s.origin, 
		//						 bbox.min, bbox.max, 0);
		if (BBoxOverlaps(&m_CurrBBox, &other_bbox)) {
			return interestTime;
		} else {
			return 0;
		}

	} 
}

uint32 DG_Edict::InterestTime(GObject *obj, InterestMap *curr)
{
	if (GetType() == EDICT_ITEM) {
		WARN << "How did an item get here?" << endl;
		return 0;
	}

	DG_Edict *dg_obj = (DG_Edict *)obj;
	edict_t *other;
	if (dg_obj->m_Entity)
		other = dg_obj->m_Entity;
	else
		other = dg_obj->m_ScratchEnt;
	ASSERT(other);

	
	int divisor;
	/*
	if ( visible(m_Entity, other) )
		// we are interested in things that are visible for a while
		divisor = 1;
	else if ( PredictedCheckIsVisible(other) )
		// if in FatPVS, maybe interested for a little while, maybe not
		// ask us again soon...
		divisor = 4;
	else
		// otherwise definitely not interested
		return 0;
	*/
	// Since we define consistency in terms of bbox only, use that for interest

	if (BBoxContains(&m_CurrBBox, other->s.origin))
		divisor = 1;
	else
		return 0;
		
	switch (GetType()) {
	case EDICT_PLAYER:
		switch(dg_obj->GetType()) {
		case EDICT_PLAYER:
			return 6000/divisor;
		case EDICT_MONSTER:
			return 6000/divisor;
		case EDICT_MISSILE:

			// missiles don't last long, so we will probably be
			// interested in it all our life
			return 1000;
		case EDICT_ITEM:
			//if (dg_obj->GetSID() != GetSID())
			//	INFO << "interested in item: "
			//		 << (GObject *)dg_obj
			//		 << " m_Entity=" << dg_obj->m_Entity
			//		 << " divisor=" << divisor
			//		 << endl;

			// if we are going to fetch an item, might as well keep
			// it for a while since it is unlikely to be updated
			return 10000;
		default:
			return 0;
		}
	case EDICT_MONSTER:
		switch(dg_obj->GetType()) {
		case EDICT_PLAYER:
			return 6000/divisor;
		case EDICT_MONSTER:
			return 6000/divisor;
		case EDICT_MISSILE:
			return 1000;
		case EDICT_ITEM:
			return 10000; 
		default:
			return 0;
		}
	case EDICT_MISSILE:
		switch(dg_obj->GetType()) {
		case EDICT_PLAYER:
			return 1000;
		case EDICT_MONSTER:
			return 1000;
		case EDICT_MISSILE:
			return 0;
		case EDICT_ITEM:
			return 0; 
		default:
			return 0;
		}
	default:
			return 0;
	}
}

void DG_Edict::FillEventsDHT(EventList *reg, EventList *unreg, EventSet *curr)
{
#define MAX_KEYS 256

	uint32 ttl;
	float pred;

	OMEvent *old = NULL;
	if (curr) {
		ASSERT(curr->size() > 0);
		old = *curr->begin();

		// Quick Optimization: if we are an item, and have an old pub, no
		// need to go further since we can't move
		if (GetType() == EDICT_ITEM)
			return;
	}

	// check if quake put this object in some wierd place...
	for (int i=0; i<3; i++) {
		if (m_CurrOrg[i] < Tmp::worldmin[i] ||
			m_CurrOrg[i] > Tmp::worldmax[i]) {
			// stupid damn bots
			DB(1) << "stupid object outside of world! : "
				  << GetGUID() << " " << m_Entity->classname 
				  << " origin=" << m_CurrOrg
				  << " worldmin=" << Tmp::worldmin
				  << " worldmax=" << Tmp::worldmax << endl;
			return;
		}
	}

	float pt_key = DG_GetNormalizedDHTBox(m_CurrOrg);

	// Old pub didn't suffice, but some of the keys might still be valid...
	map<float, OMEvent *> have_keys;
	if (curr) {
		for (EventSetIter it = curr->begin(); it != curr->end(); it++) {
			float key;
			if ((*it)->IsPointEvent ()) {
				key = PointEventToDHTKey(*it);
				
				vec3_t old_pt;
				EventToPoint(old_pt, *it);
				
				// I didn't move! old pub is ok! -- don't have to check key
				// since it must be the same (point)
				if (VectorCompare(old_pt, m_CurrOrg) ) {
					return;
				}
			} else {
				key = RangeEventToDHTKey(*it);
				
				bbox_t old_bbox;
				EventToBBox(&old_bbox, old);
				
				if (pt_key == key && BBoxContains(&old_bbox, m_CurrOrg)) {
					// still pub'd to the point I'm at! ok!
					return;
				}
			}
			have_keys[key] = *it;
		}
	}
 
	// If we get here then our old pubs aren't sufficient!

	if (GetType() == EDICT_PLAYER) {
		ttl  = g_QuakePreferences.pubttl_player;
		pred = g_QuakePreferences.pubpred_player;
	} else if (GetType() == EDICT_MONSTER) {
		ttl  = g_QuakePreferences.pubttl_monster;
		pred = g_QuakePreferences.pubpred_monster;
	} else if (GetType() == EDICT_MISSILE) {
		ttl  = g_QuakePreferences.pubttl_missile;
		pred = g_QuakePreferences.pubpred_missile;
	} else if (GetType() == EDICT_ITEM) {
		ttl  = g_QuakePreferences.pubttl_item;
		pred = 0;
	} else {
		Debug::die("Unknown EDICT TYPE: %d", GetType());
	}

	EventSet inuse;

	if (pred == 0) {
		OMEvent *ev = g_Manager->CreateEvent(GetGUID());
		DHTKeyPointToEvent(ev, DG_GetNormalizedDHTBox(m_CurrOrg), m_CurrOrg);
		ev->SetLifeTime(ttl);
		reg->push_back(ev);
	} else {
		bbox_t bbox;

		// find the closest bbox for this dude, use the z-values
		// so that we remain published to the area even if we jump
		if ( DG_GetPrecompBoundingBox(m_CurrOrg, bbox.min, bbox.max, 
									  0, 1.0, 0) == 0 ) {
			return; // no bbox -- ignore
		}
		// for the other coordinates, predict only a limited amount
		// -- either the viewable bbox-limit, or some number of units
		// radius around the dude
		for (int i=0; i<2; i++) {
			bbox.min[i] = MAX(MAX(m_CurrOrg[i] - pred, bbox.min[i]), 
							  Tmp::worldmin[i]);
			bbox.max[i] = MIN(MIN(m_CurrOrg[i] + pred, bbox.max[i]), 
							  Tmp::worldmax[i]);
		}
		
		//bbox.min[2] = MAX(m_CurrOrg[2] - MIN(pred, 1), Tmp::worldmin[2]);
		//bbox.max[2] = MIN(m_CurrOrg[2] + MIN(pred, 1), Tmp::worldmax[2]);

		static float new_keys[MAX_KEYS];
		int num_new_keys = BBoxToNormalizedDHTPoints(new_keys, MAX_KEYS,
													 &bbox);

		int num_old_keys_valid = 0;
		
		for (int i=0; i<num_new_keys; i++) {
			map<float, OMEvent *>::iterator p = have_keys.find(new_keys[i]);
			if (p != have_keys.end() &&
				!p->second->IsPointEvent ()) {
				bbox_t old_bbox;
				EventToBBox(&old_bbox, p->second);
				if (BBoxContains(&old_bbox, &bbox)) {
					// had a pub to the key already -- reuse it
					inuse.insert(p->second);
					num_old_keys_valid++;
					continue;
				}
			}

			// need a new pub
			OMEvent *ev = g_Manager->CreateEvent(GetGUID());
			ASSERT(new_keys[i] != 0);
			DHTKeyBBoxToEvent(ev, new_keys[i], &bbox);
			ev->SetLifeTime(ttl);
			reg->push_back(ev);
		}

	}

	if (curr) {
		// unregister all the interests that are no longer in use
		for (EventSetIter it = curr->begin(); it != curr->end(); it++) {
			if ( inuse.find(*it) == inuse.end() ) {
				unreg->push_back(*it);
			}
		}
	}
}

void DG_Edict::FillEvents(EventList *reg, EventList *unreg, EventSet *curr)
{
	if (g_QuakePreferences.dht) {
		FillEventsDHT(reg, unreg, curr);
		return;
	}

	if (!Tmp::inited) {
		Tmp::inited = true;
		DG_GetPrecompWorldBBox(Tmp::worldmin, Tmp::worldmax);
	}

	uint32 ttl;
	float pred;

	OMEvent *old = NULL;
	if (curr) {
		ASSERT(curr->size() > 0);
		old = *curr->begin();

		// Quick Optimization: if we are an item, and have an old pub, no
		// need to go further since we can't move
		if (GetType() == EDICT_ITEM)
			return;
	}

	// check if quake put this object in some wierd place...
	for (int i=0; i<3; i++) {
		if (m_CurrOrg[i] < Tmp::worldmin[i] ||
			m_CurrOrg[i] > Tmp::worldmax[i]) {
			// stupid damn bots
			DB(1) << "stupid object outside of world! : "
				  << GetGUID() << " " << m_Entity->classname 
				  << " origin=" << m_CurrOrg
				  << " worldmin=" << Tmp::worldmin
				  << " worldmax=" << Tmp::worldmax << endl;
			return;
		}
	}

	if (old) {
		if (old->IsPointEvent ()) {
			vec3_t old_pt;
			EventToPoint(old_pt, old);

			// I didn't move! old pub is ok!
			if( VectorCompare(old_pt, m_CurrOrg) ) {
				//NOTE(OLD_PUB_OK, 1);
				return;
			}
		}
		else {
			bbox_t pt_bbox, old_bbox;
			EventToBBox(&old_bbox, old);

			if ( BBoxContains(&old_bbox, m_CurrOrg) ) {
				// old bbox we published to suffices for our purposes!
				NOTE(OLD_PUB_OK, 1);
				return;
			} else {
				NOTE(OLD_PUB_NOT_OK, 1);

				if (GetType() == EDICT_PLAYER) {
					NOTE(OLD_PUB_NOT_OK_PLAYER, 1);
				} else if (GetType() == EDICT_MONSTER) {
					NOTE(OLD_PUB_NOT_OK_MONSTER, 1);
				} else if (GetType() == EDICT_MISSILE) {
					NOTE(OLD_PUB_NOT_OK_MISSILE, 1);
				} else {
					NOTE(OLD_PUB_NOT_OK_OTHER, 1);
				}
			}
		}
	} else {
		NOTE(OLD_PUB_EXPIRED, 1);
	}

	if (GetType() == EDICT_PLAYER) {
		ttl  = g_QuakePreferences.pubttl_player;
		pred = g_QuakePreferences.pubpred_player;
	} else if (GetType() == EDICT_MONSTER) {
		ttl  = g_QuakePreferences.pubttl_monster;
		pred = g_QuakePreferences.pubpred_monster;
	} else if (GetType() == EDICT_MISSILE) {
		ttl  = g_QuakePreferences.pubttl_missile;
		pred = g_QuakePreferences.pubpred_missile;
	} else if (GetType() == EDICT_ITEM) {
		ttl  = g_QuakePreferences.pubttl_item;
		pred = 0;
	} else {
		Debug::die("Unknown EDICT TYPE: %d", GetType());
	}

	// add 0-5% of time to the TTL to prevent synchronization
	ttl = (uint32)(ttl*(1.0 + drand48()/20.0));
	
	if (pred == 0) {
		OMEvent *ev = g_Manager->CreateEvent(GetGUID());

		PointToEvent(ev, m_CurrOrg);
		// INFO << " org=" << m_CurrOrg << " ev=" << ev;
		
		// if we are using striping then we need to insert the correct
		// number to use on the single dimensional hub
		if (g_QuakePreferences.stripe_dim >= 0) {
			float val = PointToStripePoint(m_CurrOrg);
			
			Value v = g_Manager->ConvertToValue (g_StripeHubIndex, val, 0, 1);
			Tuple t (g_StripeHubIndex, v);
			ev->AddTuple (t);
		}

		// cerr << "here: " << ev << endl << endl;

		ev->SetLifeTime(ttl);
		reg->push_back(ev);
	} else {
		bbox_t bbox;

		// find the closest bbox for this dude, use the z-values
		// so that we remain published to the area even if we jump
		if ( DG_GetPrecompBoundingBox(m_CurrOrg, bbox.min, bbox.max, 
									  0, 1.0, 0) == 0 ) {
			return; // no relevant bbox -- don't publish
		}
		// for the other coordinates, predict only a limited amount
		// -- either the viewable bbox-limit, or some number of units
		// radius around the dude
		for (int i=0; i<2; i++) {
			ASSERTDO (bbox.max[i] >= -1.0e+9, { 
					WARN << "i=" << i << " max[i]=" << bbox.max[i] << " currorig[i]=" << m_CurrOrg[i] << endl;
					WARN << " pred=" << pred << " worldmin[i]=" << Tmp::worldmin[i] << " worldmax[i]=" << Tmp::worldmax[i] 
					<< " org+pred=" << m_CurrOrg[i] + pred << endl;
					});
			bbox.min[i] = MAX(MAX(m_CurrOrg[i] - pred, bbox.min[i]), 
							  Tmp::worldmin[i]);
			bbox.max[i] = MIN(MIN(m_CurrOrg[i] + pred, bbox.max[i]), 
							  Tmp::worldmax[i]);
		}

		bbox.min[2] = MAX(bbox.min[2], Tmp::worldmin[2]);
		bbox.max[2] = MIN(bbox.max[2], Tmp::worldmax[2]);
		
		if (g_QuakePreferences.stripe_dim >= 0) {
#define MAX_RANGES 64
			static float min[MAX_RANGES], max[MAX_RANGES];
			int num = BBoxToStripeRanges(min, max, MAX_RANGES, &bbox);
			ASSERT(num > 0);

			//DB(0) << "num stripes: " << num << endl;

			/*
			INFO << "origin=" << m_Entity->s.origin
				 << " bbox=" << bbox.min << " -> " << bbox.max
				 << ": stripe ranges: " << endl;
			INFO << "normmin=" << normmin << " normmax=" << normmax << endl;
			*/

			for (int i=0; i<num; i++) {
				// need 1 event per stripe published to
				OMEvent *ev = g_Manager->CreateEvent(GetGUID());
				BBoxToEvent(ev, &bbox);
				ev->SetLifeTime(ttl);

				Value minv = g_Manager->ConvertToValue (g_StripeHubIndex, min[i], 0, 1);
				Value maxv = g_Manager->ConvertToValue (g_StripeHubIndex, max[i], 0, 1);
				Constraint c (g_StripeHubIndex, minv, maxv);

				ev->AddConstraint (c);
				reg->push_back(ev);

				//INFO << min[i] << " -> " << max[i] << endl;
			}
		} else {
			OMEvent *ev = g_Manager->CreateEvent(GetGUID());
			BBoxToEvent(ev, &bbox);
			ev->SetLifeTime(ttl);
			reg->push_back(ev);
		}

	}

	if (curr) {
		for (EventSetIter it = curr->begin(); it != curr->end(); it++)
			unreg->push_back(*it);
	}
}

bool DG_Edict::GetVisibleBBox(bbox_t *bbox, float predict /* radius */)
{
	if (!Tmp::inited) {
		Tmp::inited = true;
		DG_GetPrecompWorldBBox(Tmp::worldmin, Tmp::worldmax);
	}

    vec3_t min, max;
    ASSERT (m_Entity->s.number != 0);

	ASSERT(m_Entity->inuse);
	ASSERT(GetType() != EDICT_ITEM); // should not be calling this!

	if (DG_HasPrecompBBoxes()) {
		if (m_CurrOrg[0] < Tmp::worldmin[0] || 
			m_CurrOrg[0] > Tmp::worldmax[0]
		 || m_CurrOrg[1] < Tmp::worldmin[1] || 
			m_CurrOrg[1] > Tmp::worldmax[1]
		 || m_CurrOrg[2] < Tmp::worldmin[2] || 
			m_CurrOrg[2] > Tmp::worldmax[2]
		   )
		{
			// seems to happen for stupid dudes when they die and sink
			DB(1) << "entity position outside the world - ignoring" << endl;
			return false;
		}
		else {
			
			if (DG_GetPrecompBoundingBox(m_CurrOrg, bbox->min, bbox->max, 
										 // grow the predicted sub location by 
										 // this much (x2 because its a 
										 // diameter)
										 2*predict, 
										 // limit the bbox growth in the 
										 // z-dimension so we don't go through 
										 // ceilings or floors
										 1.0,
										 // hysterisies check to ensure the bbox
										 // isn't ridicuously large -- max sub
										 // to 10% of the volume of the world.
										 0.10) == 0) {
				return false; // no relevant bounding box -- don't sub
			}
			for (int i = 0; i < 3; i++) {
				bbox->min[i] =
					MAX(MIN(bbox->min[i], Tmp::worldmax[i]), 
						Tmp::worldmin[i]);
				bbox->max[i] =
					MAX(MIN(bbox->max[i], Tmp::worldmax[i]), 
						Tmp::worldmin[i]);
			}
			return true;
		}
	} else {
		WARN << "No precomputed bboxes! falling back to old mode!" << endl;
		DG_GetBoundingBox(m_CurrOrg, bbox->min, bbox->max);
		return true;
	}
}

void DG_Edict::PackUpdate(Packet *pkt, const DeltaMask& mask)
{
	ASSERT(m_Entity->inuse);

    dg_field_t edict_fld;
    edict_fld.type = DF_EDICT_PTR;

	//DB(0) << "encoding obj of type=" << m_Entity->classname
	//	  << " with mask=" << mask << endl;

	dg_fields encoded;
	DG_MakeEncodeSet(&mask, &encoded);

	DB_DO(5) {
		if (strstr(m_Entity->classname, "item")) {
			DB(0) << "class=" << m_Entity->classname
				  << " replica=" << IsReplica()
				  << " guid=" << GetGUID()
				  << ": encoded(" << encoded.size() << ")" << endl;
		}
	}
	

	dg_field_stack_t  init;
	DG_InitFieldStack(&init);

	// no need to get length, packet is MAX_SIZE
	//START(DG_Edict::object_length);
	//int len = object_length(&init, &encoded, &edict_fld, (byte *) m_Entity);
	//STOP(DG_Edict::object_length);

    ASSERT(pkt);

	// XXX HACK: For players, the skinnum is actually an index into a client
	// array (on the server as well as the clients on that server). This
	// array contains the actual info that is used for that entity's
	// modelindex (model), modelindex2 (weapon model), and skinnum (skin).
	// Since the replica "players" don't have player info attached to them
	// the skinnum on those guys will be pretty much meaningless (it will
	// be an index into the clients array at *that* server) and this will
	// result in strange graphics artifacts because of wrong models and
	// the like. To prevent this, we insist that all replica players use
	// skin number 0 -- this means they will use the same model info as
	// the first player on that server; this player will always exist when
	// there is a client to observe the effects of the entity.
	//
	// Nonetheless, this means all replica players will look the same, etc...
	
	int old_skinnum = m_Entity->s.skinnum;
	if (!IsReplica() && m_Entity->client) {
		m_Entity->s.skinnum &= 0xFFFFFF00; // preserve weapon index
		m_Entity->s.skinnum |= 11;         // manually set model index
	}
	
	// XXX
	
	//START(DG_Edict::object_serialize);
    object_serialize(&init, &encoded, this, &edict_fld, (byte *) m_Entity, pkt);
	//STOP(DG_Edict::object_serialize);

	// XXX
	m_Entity->s.skinnum = old_skinnum;
	// XXX
}

void DG_Edict::ConstructEntity(Packet *pkt, 
							   const DeltaMask& mask,
							   SIDMap *unresolved) 
{
	ASSERT(m_ScratchEnt);
	ASSERT(m_ScratchClient);

	if (m_Entity) {

		// XXX: a bit of these semantics are wrong... if we didn't apply the
		// last update, the "latest" version is actually the scratch ent,
		// not the last version...
		//
		// has old entity to copy from
		*m_ScratchEnt = *m_Entity;

		if (m_Entity->client) {
			m_ScratchEnt->client = m_ScratchClient;
			*m_ScratchClient = *m_Entity->client;
		}
	}

	m_ScratchEnt->guid           = GetGUID();
	m_ScratchEnt->inuse          = true;
    m_ScratchEnt->is_replica     = IsReplica();
    m_ScratchEnt->is_distributed = true;
    m_ScratchEnt->linkcount      = 1;
	
	static int clientIndex = _dg_field_encoding["client"]->index;
	if (mask.IsSet(clientIndex)) {
	    m_ScratchEnt->client = m_ScratchClient;
	}

    dg_field_t edict_fld;
    edict_fld.type = DF_EDICT_PTR;

	dg_fields encoded;
	DG_MakeEncodeSet(&mask, &encoded);
	dg_field_stack_t  init;
	DG_InitFieldStack(&init);

    object_deserialize(&init, &encoded, this, &edict_fld, pkt, 
					   (byte *) m_ScratchEnt, DF_EDICT_PTR, 0);

	DB_DO(5) {
		if (m_ScratchEnt->classname &&
			strstr(m_ScratchEnt->classname, "item")) {
			DB(0) << "class=" << m_ScratchEnt->classname
				  << " replica=" << IsReplica()
				  << " guid=" << GetGUID()
				  << ": encoded(" << encoded.size() << "): " << endl;
		}
	}

    for (EdictRefsIter it = m_UnresolvedPtrs.begin(); 
		 it != m_UnresolvedPtrs.end(); it++) {
        EdictRef *ref = *it;
        unresolved->insert(SIDMap::value_type(ref->guid, ref->owner));
    }

	ASSERT(m_ScratchEnt->classname);

    DB(5) << " constructing object [" 
		  << m_ScratchEnt->classname 
		  << "] with guid " << m_ScratchEnt->guid << endl;
}

//
// Acts like the HandleObjectUpdate
// XXX TODO: too many copies? 
//
void DG_Edict::UnpackUpdate(Packet *pkt, const DeltaMask& mask,
							SIDMap *unresolved)
{
    ConstructEntity(pkt, mask, unresolved);

    qboolean all_clear = m_UnresolvedPtrs.size() == 0;

    if (!m_Entity) {  // this guy belongs to the pending store!
		// we got the updated version of scratch ent; so go back!
		return;
    }
	
    if (!all_clear) {
        // somebody should tell us when we will get resolved.
        DB(1) << "update resulting in unresolved reference: "
			  << (GObject *)this << endl;
        // BIG XXX TODO: if the lower layer really doesn't inform us later, 
        // why should it bother with the unresolved references then?

		// we save the scratch ent so that when the pointer is resolved, it
		// will be resolved in the appropriate edict structure.
		// However, delete the multicast events, otherwise we may memory
		// leak it since it may be overwritten. They are "unreliable" anyway
		// so we just miss some special fx
		if (m_ScratchEnt->multicastEventList) {
			delete m_ScratchEnt->multicastEventList;
			m_ScratchEnt->multicastEventList = 0;
		}

		return;
    } else {

		// if we had some multicast events, and then we got some more
		// disregard the old ones (really should merge them, but let's
		// just do the simple thing :P)
		if (m_Entity && m_Entity->multicastEventList &&
			m_ScratchEnt->multicastEventList &&
			m_Entity->multicastEventList != m_ScratchEnt->multicastEventList) {
			delete m_Entity->multicastEventList;
			m_Entity->multicastEventList = NULL;
		}

		// and the next delta will not correspond correctly to our version.
		m_Entity = DG_HandleObjectUpdate(m_Entity, m_ScratchEnt);
		ASSERT(m_Entity->inuse);

		return;
	}
}

bool DG_Edict::IsMigratable()
{
    // don't migrate players...
    if ( !m_Entity || !strcmp(m_Entity->classname, "player") )
		return false;

    return true;
}

void DG_Edict::CommitMigration(sid_t target)
{
    ASSERT(m_Entity);
	ASSERT(m_Entity->inuse);
    // This is a necessary hack because this flag shadows the "actual"
    // flag that is in GObject
    m_Entity->is_replica = !m_Entity->is_replica;

	/*
	INFO << "COMMITTED:" << endl;
	cout << "guid=" << m_Entity->guid << " linkcount=" 
		 << m_Entity->linkcount << endl;
	cout << m_Entity << endl;
	*/

	/*
	if (!m_Entity->is_replica) {
		DB(-2) << "committing migration: " << (GObject *)this << endl;
	}
	*/

	// get rid of any pending multicast events --
	// (1) if i was a replica, then these events are now mine and useless
	// (2) if i was a primary, then these events are no longer mine
	// XXX FIXME: should really fix multicast event semantics
	// this is very hacky
	if (m_Entity->multicastEventList) {
		delete m_Entity->multicastEventList;
		m_Entity->multicastEventList = NULL;
		// XXX Is there a memory leak for events somewhere dealing with
		// migration? I think migration breaks our invariants for multicast
		// event lists!
	}

	// If this is now a primary object, we need to make a initial delta
	// mask for it.
	if (!m_Entity->is_replica) {
		DG_MakeInitDeltaMask(this);

		// we also need to move it to the "primary" edict/client area
		if (GetType() == EDICT_PLAYER) {
			edict_t *free = ACESP_FindFreeClient();
			if (!free) {
				WARN << "couldn't find free client spot for player "
					 << "we migrated!" << endl;
				ASSERT(0);
			}
			memcpy(free, m_Entity, sizeof(edict_t));
			free->s.number = free - g_edicts;
			free->client = game.clients + (free - g_edicts - 1);
			memcpy(free->client, m_Entity->client, sizeof(gclient_t));

			gi.unlinkentity (m_Entity);

			DG_DeallocateClient(GetGUID()); // this deallocates remote clients
			G_FreeEdict(m_Entity);

			m_Entity = free;

			gi.linkentity (m_Entity);
		}
	} else {

		// need to move it to the "replica" edict/client area
		if (GetType() == EDICT_PLAYER) {
			edict_t *free = G_Spawn();
			if (!free) {
				WARN << "couldn't find free client spot for player "
					 << "migrating to us!" << endl;
				ASSERT(0);
			}
			memcpy(free, m_Entity, sizeof(edict_t));
			free->s.number = free - g_edicts;
			free->client = DG_AllocateClient(GetGUID());
			memcpy(free->client, m_Entity->client, sizeof(gclient_t));
			
			gi.unlinkentity (m_Entity);

			m_Entity->s.modelindex = 0;
			m_Entity->solid = SOLID_NOT;
			m_Entity->inuse = false;
			m_Entity->classname = "disconnected";
			m_Entity->client->pers.connected = false;
			
			m_Entity = free;
			
			gi.linkentity (m_Entity);
		}
	}

	// Force Checkpoint the object so we don't think it is new
	DG_CheckPointObject(m_Entity);
}

bool DG_Edict::IsNew()
{
    ASSERT(m_Entity);
    return (m_Entity->status == DG_OBJ_CREATED);
}

CostMetric DG_Edict::GetFixedCost() {
    if (m_Entity->is_bot || 
		(m_Entity->s.number > 0 && m_Entity->s.number <= 
		 gcvar_maxclients->value)) {
		return 20;
    }
    else {
		return 0;
    }
}

CostMetric DG_Edict::GetDeltaCost() {
    return g_CostMap[m_Entity->classname];
}

void DG_Edict::PrintFields(ostream& out) {
    if (m_Entity) {
		out << " predicted_org=" << m_CurrOrg << " bbox="
			<< "[min=" << m_CurrBBox.min
			<< ",max=" << m_CurrBBox.max << "] ";
		out << "entity=" << m_Entity;
    } else {
		out << "scratch=" << m_ScratchEnt;
    }
}

ostream& operator<<(ostream& out, edict_t *edict)
{
#if 1
    out << "[edict:"
		<< " guid=" << edict->guid
		<< " class=" << edict->classname
		<< " origin=(" 
		<< edict->s.origin[0] << "," 
		<< edict->s.origin[1] << "," 
		<< edict->s.origin[2] << ")";

	if (edict->client != NULL) {
		char *dead_str;
		switch(edict->deadflag) {
		case DEAD_NO: dead_str = "NO"; break;
		case DEAD_DYING: dead_str = "DYING"; break;
		case DEAD_DEAD: dead_str = "DEAD"; break;
		case DEAD_RESPAWNABLE: dead_str = "RESPAWNABLE"; break;
		default: dead_str = "???";
		}

		out << " name=" << edict->client->pers.netname
			<< " score=" << edict->client->resp.score
			<< " health=" << edict->health
			<< " dead=" << dead_str
			<< " respawn_delay=" << edict->client->resp.respawn_delay
			<< " respawn_time=" << MAX(0, edict->client->respawn_time-level.time);
	}
	
	out << "]";
#else
	dg_field_stack_t  init;
	DG_InitFieldStack(&init);
	dg_field_t edict_fld;
	edict_fld.type = DF_EDICT_PTR;
	object_print(0, out, &init, NULL, &edict_fld, (byte *)edict);
#endif
    return out;
}
