////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////
#include <Mercury.h>
#include <util/debug.h>
#include <wan-env/RealNet.h>
#include <wan-env/UDPTransport.h>
#include <wan-env/TCPTransport.h>
#include <wan-env/WANScheduler.h>
#include <mercury/Timer.h>
#include <vector>
#include <algorithm>

#define USE_THREADS

bool finished = false;
uint32 period = 5;
uint32 bsize;
uint32 timelimit;
uint32 rcvd = 0, sent = 0;

uint32 send_period_millis = 1;     // time between sends
u_long realnet_timeslice = 10000;  // usecs
u_long procmsg_timeslice = 10000;  // usecs

TransportType proto;
char  myaddr[255];
char  servers[100*80];
char  protostr[255];
vector<IPEndPoint> serveraddrs;

OptionType g_Options[] =
{
    { '#', "block-size", OPT_INT,
	"packet block size",
	&bsize, "1024", NULL },
	{ '#', "myaddr", OPT_STR,
	    "my listen address",
	    myaddr, "127.0.0.1:5000", NULL },
	    { '#', "servers", OPT_STR,
		"receivers of packets",
		servers, "127.0.0.1:5000", NULL },
		{ '#', "timelimit", OPT_INT,
		    "how long to run before exiting",
		    &timelimit, "0", NULL },
		    { '#', "proto", OPT_STR,
			"protocol",
			protostr, "UDP", NULL },
			{ '#', "net-slice", OPT_INT,
			    "usecs of slice given to RealNet::DoWork()",
			    &realnet_timeslice, "10000", NULL },
			    { '#', "proc-slice", OPT_INT,
				"usecs of slice given to processing dequeued messages",
				&procmsg_timeslice, "10000", NULL },
				{ 0, 0, 0, 0, 0, 0, 0 }
};

IPEndPoint me;
RealNet *net;
DummyNode *dnode;
WANScheduler *scheduler;

TimeVal istart = TIME_NONE;

sint64 diff(struct timeval& a, struct timeval& b) {
    return ((sint64)a.tv_sec - (sint64)b.tv_sec)*USEC_IN_SEC + 
	((sint64)a.tv_usec - (sint64)b.tv_usec);
}

class InfoTimer : public Timer { 
public:
    InfoTimer () : Timer (0) {}
    void OnTimeout () {
	TimeVal now = scheduler->TimeNow ();

	double tput_pps = (double)(rcvd) / ((double)(diff(now, istart))/1000000.0);
	INFO << " recv tput: " << tput_pps << " pps" << endl;
	// tput_pps = (double)(sent) / ((double)(diff(now, istart))/1000000.0);
	// INFO << " send tput: " << tput_pps << " pps" << endl;

	rcvd = 0;
	OS::GetCurrentTime (&istart);
	scheduler->RaiseEvent (mkref(this), me, period * 1000);
    }
};

#ifndef USE_THREADS
class SendTimer : public Timer {
public:
    SendTimer () : Timer (0) {}
    void OnTimeout () {
	static MsgBlob msg (bsize, me);

	for (vector<IPEndPoint>::iterator it = serveraddrs.begin(); it != serveraddrs.end(); ++it) {
	    net->SendMessage (&msg, &(*it), proto);
	    sent++;
	}
	scheduler->RaiseEvent (mkref(this), me, send_period_millis);
    }
};
#else
#include <pthread.h>

void *start_thread (void *args) {
    static MsgBlob msg (bsize, me);
    while (!finished) { 
	for (vector<IPEndPoint>::iterator it = serveraddrs.begin(); it != serveraddrs.end(); ++it) {
	    net->SendMessage (&msg, &(*it), proto);
	    sent++;
	}
    }

    return NULL;
}

#endif

class FinishEvent : public SchedulerEvent {
public:
    void Execute (Node& node, TimeVal& now) {
	finished = true;
    }
};

ConnStatusType ProcessMessage ()
{
    IPEndPoint from = SID_NONE;
    Message *msg = NULL;
    ConnStatusType status = net->GetNextMessage (&from, &msg);

    switch (status) {
    case CONN_NEWINCOMING:
    case CONN_OK:
	rcvd++;
	net->FreeMessage (msg);
	break;
    case CONN_CLOSED:
    case CONN_NOMSG:
	break;
    case CONN_ERROR:
    default:
	WARN << "error" << endl;
	break;
    }
    return status;
}

void DoWork (u_long slice)
{
    unsigned long long stoptime = CurrentTimeUsec () + (unsigned long long) slice;
    scheduler->ProcessTill (scheduler->TimeNow ());    

    while (CurrentTimeUsec () < stoptime) {
	if (ProcessMessage () == CONN_NOMSG)
	    break;
    }
    scheduler->ProcessTill (scheduler->TimeNow ());
}


int main(int argc, char **argv) {
    InitializeColyseus (&argc, argv, g_Options, false);
    DBG_INIT (NULL);

    me = IPEndPoint (myaddr);
    vector<string> result;
    tokenizer<comma_sep>::tokenize (result, servers);

    for (vector<string>::iterator it = result.begin(); it != result.end(); ++it) {
	string s = *it;
	serveraddrs.push_back (IPEndPoint ((char *) s.c_str()));
    }

    scheduler = new WANScheduler ();
    dnode = new DummyNode (NULL, scheduler, me);
    scheduler->SetNode (dnode);
    net = new RealNet (scheduler, me);

    proto = Transport::GetProtoByName(protostr);
    net->StartListening(proto);

    scheduler->RaiseEvent (new refcounted<FinishEvent> (), me, timelimit * 1000);
#ifndef USE_THREADS
    scheduler->RaiseEvent (new refcounted<SendTimer> (), me, 100);
#else
    pthread_t tid;
    pthread_create (&tid, NULL, start_thread, NULL);
#endif

    scheduler->RaiseEvent (new refcounted<InfoTimer> (), me, period * 1000);	    

    OS::GetCurrentTime(&istart);
    cerr << " starting " << endl;
    while (!finished) {
	RealNet::DoWorkUsec (realnet_timeslice);
	DoWork (procmsg_timeslice);
    }

#ifdef USE_THREADS
    INFO << " waiting for thread" << endl;
    pthread_join (tid, NULL);
#endif
    INFO << " ending " << endl;
    exit (0);
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
