////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

#include "Options.h"
#include <util/debug.h>
#include <vector>
#include <string>
using namespace std;

#define MAX_OPT_LENGTH 128

static string _GetValueString (OptionType *ptr, void *value, bool is_default_val = false) { 
    char sbuffer [128];

    if (ptr->flags == OPT_SEP)
	return "";
    if (ptr->flags & OPT_STR)
	return (char *) value;

    if (ptr->flags & OPT_CHR) 
	sprintf (sbuffer, "%c", * (char *) value);
    else if (ptr->flags & OPT_INT)
	sprintf (sbuffer, "%d", (is_default_val ? atoi ((char *) value) : * (int *) value));
    else if (ptr->flags & OPT_FLT)
	sprintf (sbuffer, "%f", (is_default_val ? atof ((char *) value) : * (float *) value));
    else if (ptr->flags & OPT_DBL)
	sprintf (sbuffer, "%lf", (is_default_val ? atof ((char *) value) : * (double *) value));
    else if (ptr->flags & OPT_BOOL) {
	bool val = false;
	if (is_default_val) {
	    if (strcmp ((char *) value, "1") == 0)
		val = true;
	}
	else 
	    val = * (bool *) value;
	sprintf (sbuffer, "%s", (val ? "true" : "false"));
    }

    return sbuffer;
}


void PrintUsage(OptionType * options_def)
{
    OptionType *ptr;

    fprintf(stderr, "Usage: ProgramName [options]:\n");
    unsigned int maxlen = 0;
    for (ptr = options_def; ptr->longword; ptr++) {
	if (ptr->flags == OPT_SEP)
	    continue;

	if (strlen(ptr->longword) > maxlen)
	    maxlen = strlen(ptr->longword);
    }

    for (ptr = options_def; ptr->longword; ptr++) {
	if (ptr->flags == OPT_SEP) {
	    fprintf (stderr, "\n[31mOptions related to:[m [32m%s[m\n", ptr->longword); 
	    /*
	    for (int i = 0; i < 80; i++) 
		fprintf (stderr, "-");
	    fprintf (stderr, "\n");
	    */
	}
	else {
	    fprintf(stderr, "\t-%c,--%s:", ptr->shortletter,
		    ptr->longword);
	    for (unsigned int i = 0; i <= maxlen - strlen (ptr->longword); i++)
		fprintf (stderr, " ");

	    string def = _GetValueString (ptr, (void *) ptr->default_val, true /* is_default_val */);
	    fprintf (stderr, "%s [%s]\n", ptr->comment, def.c_str ());
	}
    }
    fprintf(stderr, "\n");
}

static void _AssignValue(OptionType * ptr, char *value)
{
    if (ptr->flags == OPT_SEP)
	return;

    if (ptr->flags & OPT_CHR) {
	*(char *) ptr->varptr = *value;
    } else if (ptr->flags & OPT_INT) {
	*(int *) ptr->varptr = atoi(value);
    } else if (ptr->flags & OPT_FLT) {
	*(float *) ptr->varptr = (float) atof(value);
    } else if (ptr->flags & OPT_DBL) {
	*(double *) ptr->varptr = atof(value);		
    } else if (ptr->flags & OPT_STR) {
	// XXX security risk!! change to strncpy, but then OPT_STR needs 
	// an additional parameter for max-size.. perhaps, ptr->val_to_set 
	// can be used hackily
	strcpy((char *) ptr->varptr, value);
    } else if (ptr->flags & OPT_BOOL) {
	bool toSet = (strcmp(value, "1") == 0 ? true : false);
	*((bool *) ptr->varptr) = toSet;
    }
}

void PrintOptionValues(OptionType * options_def)
{
    OptionType *ptr;

    unsigned int maxlen = 0;
    for (ptr = options_def; ptr->longword; ptr++) 
	if (strlen(ptr->longword) > maxlen)
	    maxlen = strlen(ptr->longword);

    fprintf(stderr, "Current option values...\n");
    for (ptr = options_def; ptr->longword; ptr++) {
	if (ptr->flags == OPT_SEP)
	    fprintf (stderr, "[%s]\n", ptr->longword); 
	else {
	    fprintf(stderr, "\t-%c,--%s:", ptr->shortletter, ptr->longword);
	    for (unsigned int i = 0; i <= maxlen - strlen (ptr->longword); i++)
		fprintf (stderr, " ");

	    string str = _GetValueString (ptr, ptr->varptr);
	    fprintf (stderr, "%s\n", str.c_str ());
	}
    }
    fprintf(stderr, "\n");
}

static void _AssignDefaults(OptionType * options_def)
{
    OptionType *ptr;

    for (ptr = options_def; ptr->longword; ptr++) {
	if (ptr->flags == OPT_SEP)
	    continue;
	_AssignValue(ptr, ptr->default_val);
    }
}

//
// Process the options defined by 'options_def'
//   The function changes the argv[] array by removing all
//                 the options that were processed.
//
int ProcessOptions(OptionType * options_def, int argc, char *argv[],
	bool complain)
{
    int i = 1;
    char sbuf[3], lbuf[MAX_OPT_LENGTH + 2];
    OptionType *ptr;
    vector < int >notprocessed;

    sbuf[0] = '-';
    sbuf[2] = 0;
    lbuf[0] = lbuf[1] = '-';

    DB (1) << "assigning defaults..." << endl;
    _AssignDefaults(options_def);
    DB (1) << "processing arguments... " << endl;

    while (i < argc) {
	if (!strcmp(argv[i], "-h") || !strcmp(argv[i], "--help") ||
		!strcmp(argv[i], "--h")) {
	    PrintUsage(options_def);
	    exit(0);
	}

	bool proc = false;
	if (argv [i][0] == '-') {
	    ptr = options_def;

	    while (ptr->longword) {
		if (ptr->flags != OPT_SEP) {
		    sbuf[1] = ptr->shortletter;
		    strncpy(lbuf + 2, ptr->longword, MAX_OPT_LENGTH);

		    if (!strcmp(argv[i], sbuf) || !strcmp(argv[i], lbuf)) {
			//
			// Process this option now.
			//
			if (ptr->flags & OPT_NOARG) {
			    _AssignValue(ptr,
				    (ptr->val_to_set ? (char *) ptr->
				     val_to_set : (char *) "1"));
			    i += 1;
			} else {
			    int index = i + 1;
			    ASSERT(index < argc);
			    //fprintf(stderr, "%s\n", argv[index]);
			    _AssignValue(ptr, argv[index]);
			    i += 2;
			}
			proc = true;
			break;
		    }
		}

		ptr++;
	    }
	    if (!ptr->longword) {
		if (complain) 
		    cerr << " *** unknown option: " << argv[i] << endl;
	    }
	}

	if (!proc) {
	    notprocessed.push_back(i);
	    i++;
	    continue;
	}
    }


    /* Now, remove all the processed elements from argv and update argc accordingly. */
    for (int j = 0; j < (int) notprocessed.size(); j++) {
	argv[j + 1] = argv[notprocessed[j]];	// in memory copying. but works. (small := large)
    }
    return notprocessed.size() + 1;
}

static int GetLength (OptionType *ptr)
{
    int ret = 0;
    while (ptr->longword) {
	ret++;
	ptr++;
    }		
    return ret;
}

OptionType *MergeOptions (OptionType *optsarray[], int alen)
{
    int total = 0;
    for (int i = 0; i < alen; i++) {
	total += GetLength (optsarray[i]);
    }
    total += 1;

    OptionType *ret = new OptionType[total];
    int cur = 0;
    for (int i = 0; i < alen; i++) {
	OptionType *ptr = optsarray[i];
	while (ptr->longword) {
	    ret[cur++] = *ptr;
	    ptr++;
	}
    }
    memset (&ret[total - 1], 0, sizeof (OptionType));

    return ret;
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
