/*
   ===========================================================================
   Copyright (C) 1999-2005 Id Software, Inc.

   This file is part of Quake III Arena source code.

   Quake III Arena source code is free software; you can redistribute it
   and/or modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the License,
   or (at your option) any later version.

   Quake III Arena source code is distributed in the hope that it will be
   useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Foobar; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
   ===========================================================================
 */

// derived from quake3/qcommon.msg.c

#include "HuffmanPacket.h"
#include <string.h>

static short   ShortSwap (short l)
{
    byte    b1,b2;

    b1 = l&255;
    b2 = (l>>8)&255;

    return (b1<<8) + b2;
}

static short	ShortNoSwap (short l)
{
    return l;
}

static int    LongSwap (int l)
{
    byte    b1,b2,b3,b4;

    b1 = l&255;
    b2 = (l>>8)&255;
    b3 = (l>>16)&255;
    b4 = (l>>24)&255;

    return ((int)b1<<24) + ((int)b2<<16) + ((int)b3<<8) + b4;
}

static int	LongNoSwap (int l)
{
    return l;
}

typedef union {
    float	f;
    unsigned int i;
} _FloatByteUnion;

float FloatSwap (const float *f) {
    const _FloatByteUnion *in;
    _FloatByteUnion out;

    in = (_FloatByteUnion *)f;
    out.i = LongSwap(in->i);

    return out.f;
}

float FloatNoSwap (const float *f)
{
    return *f;
}

#define ID_INLINE __inline 

static ID_INLINE short BigShort( short l) { return ShortSwap(l); }
#define LittleShort
static ID_INLINE int BigLong(int l) { return LongSwap(l); }
#define LittleLong
static ID_INLINE float BigFloat(const float *l) { return FloatSwap(l); }
#define LittleFloat

/*
   ==============================================================================

   MESSAGE IO FUNCTIONS

   Handles byte ordering and avoids alignment errors
   ==============================================================================
 */

bool HuffmanPacket::msgInit = false;
huffman_t HuffmanPacket::msgHuff;

void HuffmanPacket::MSG_Init( msg_t *buf, byte *data, int length ) {
    if (!msgInit) {
	MSG_initHuffman();
    }
    memset (buf, 0, sizeof(*buf));
    buf->data = data;
    buf->maxsize = length;
}

void HuffmanPacket::MSG_InitOOB( msg_t *buf, byte *data, int length ) {
    if (!msgInit) {
	MSG_initHuffman();
    }
    memset (buf, 0, sizeof(*buf));
    buf->data = data;
    buf->maxsize = length;
    buf->oob = true;
}

void HuffmanPacket::MSG_Clear( msg_t *buf ) {
    buf->cursize = 0;
    buf->overflowed = false;
    buf->bit = 0;					//<- in bits
}


void HuffmanPacket::MSG_Bitstream( msg_t *buf ) {
    buf->oob = false;
}

void HuffmanPacket::MSG_BeginReading( msg_t *msg ) {
    msg->readcount = 0;
    msg->bit = 0;
    msg->oob = false;
}

void HuffmanPacket::MSG_BeginReadingOOB( msg_t *msg ) {
    msg->readcount = 0;
    msg->bit = 0;
    msg->oob = true;
}

/*
   =============================================================================

   bit functions

   =============================================================================
 */

// negative bit values include signs
void HuffmanPacket::MSG_WriteBits( msg_t *msg, int value, int bits ) {
    int	i;
    //	FILE*	fp;

    oldsize += bits;

    // this isn't an exact overflow check, but close enough
    if ( msg->maxsize - msg->cursize < 4 ) {
	msg->overflowed = true;
	return;
    }

    if ( bits == 0 || bits < -31 || bits > 32 ) {
	Debug::die( "MSG_WriteBits: bad bits %i", bits );
    }

    // check for overflows
    if ( bits != 32 ) {
	if ( bits > 0 ) {
	    if ( value > ( ( 1 << bits ) - 1 ) || value < 0 ) {
		overflows++;
	    }
	} else {
	    int	r;

	    r = 1 << (bits-1);

	    if ( value >  r - 1 || value < -r ) {
		overflows++;
	    }
	}
    }
    if ( bits < 0 ) {
	bits = -bits;
    }
    if (msg->oob) {
	if (bits==8) {
	    msg->data[msg->cursize] = value;
	    msg->cursize += 1;
	    msg->bit += 8;
	} else if (bits==16) {
	    unsigned short *sp = (unsigned short *)&msg->data[msg->cursize];
	    *sp = LittleShort(value);
	    msg->cursize += 2;
	    msg->bit += 16;
	} else if (bits==32) {
	    unsigned int *ip = (unsigned int *)&msg->data[msg->cursize];
	    *ip = LittleLong(value);
	    msg->cursize += 4;
	    msg->bit += 8;
	} else {
	    Debug::die( "can't read %d bits\n", bits);
	}
    } else {
	//		fp = fopen("c:\\netchan.bin", "a");
	value &= (0xffffffff>>(32-bits));
	if (bits&7) {
	    int nbits;
	    nbits = bits&7;
	    for(i=0;i<nbits;i++) {
		Huff_putBit((value&1), msg->data, &msg->bit);
		value = (value>>1);
	    }
	    bits = bits - nbits;
	}
	if (bits) {
	    for(i=0;i<bits;i+=8) {
		//				fwrite(bp, 1, 1, fp);
		Huff_offsetTransmit (&msgHuff.compressor, (value&0xff), msg->data, &msg->bit);
		value = (value>>8);
	    }
	}
	msg->cursize = (msg->bit>>3)+1;
	//		fclose(fp);
    }
}

int HuffmanPacket::MSG_ReadBits( msg_t *msg, int bits ) {
    int			value;
    int			get;
    bool	sgn;
    int			i, nbits;
    //	FILE*	fp;

    value = 0;

    if ( bits < 0 ) {
	bits = -bits;
	sgn = true;
    } else {
	sgn = false;
    }

    if (msg->oob) {
	if (bits==8) {
	    value = msg->data[msg->readcount];
	    msg->readcount += 1;
	    msg->bit += 8;
	} else if (bits==16) {
	    unsigned short *sp = (unsigned short *)&msg->data[msg->readcount];
	    value = LittleShort(*sp);
	    msg->readcount += 2;
	    msg->bit += 16;
	} else if (bits==32) {
	    unsigned int *ip = (unsigned int *)&msg->data[msg->readcount];
	    value = LittleLong(*ip);
	    msg->readcount += 4;
	    msg->bit += 32;
	} else {
	    Debug::die( "can't read %d bits\n", bits);
	}
    } else {
	nbits = 0;
	if (bits&7) {
	    nbits = bits&7;
	    for(i=0;i<nbits;i++) {
		value |= (Huff_getBit(msg->data, &msg->bit)<<i);
	    }
	    bits = bits - nbits;
	}
	if (bits) {
	    //			fp = fopen("c:\\netchan.bin", "a");
	    for(i=0;i<bits;i+=8) {
		Huff_offsetReceive (msgHuff.decompressor.tree, &get, msg->data, &msg->bit);
		//				fwrite(&get, 1, 1, fp);
		value |= (get<<(i+nbits));
	    }
	    //			fclose(fp);
	}
	msg->readcount = (msg->bit>>3)+1;
    }
    if ( sgn ) {
	if ( value & ( 1 << ( bits - 1 ) ) ) {
	    value |= -1 ^ ( ( 1 << bits ) - 1 );
	}
    }

    return value;
}



//================================================================================

//
// writing functions
//

void HuffmanPacket::MSG_WriteChar( msg_t *sb, int c ) {
#ifdef PARANOID
    if (c < -128 || c > 127)
	Debug::die ( "MSG_WriteChar: range error");
#endif

    MSG_WriteBits( sb, c, 8 );
}

void HuffmanPacket::MSG_WriteByte( msg_t *sb, int c ) {
#ifdef PARANOID
    if (c < 0 || c > 255)
	Debug::die ( "MSG_WriteByte: range error");
#endif

    MSG_WriteBits( sb, c, 8 );
}

void HuffmanPacket::MSG_WriteData( msg_t *buf, const void *data, int length ) {
    int i;
    for(i=0;i<length;i++) {
	MSG_WriteByte(buf, ((byte *)data)[i]);
    }
}

void HuffmanPacket::MSG_WriteShort( msg_t *sb, int c ) {
#ifdef PARANOID
    if (c < ((short)0x8000) || c > (short)0x7fff)
	Debug::die ( "MSG_WriteShort: range error");
#endif

    MSG_WriteBits( sb, c, 16 );
}

void HuffmanPacket::MSG_WriteLong( msg_t *sb, int c ) {
    MSG_WriteBits( sb, c, 32 );
}

void HuffmanPacket::MSG_WriteFloat( msg_t *sb, float f ) {
    union {
	float	f;
	int	l;
    } dat;

    dat.f = f;
    MSG_WriteBits( sb, dat.l, 32 );
}

//============================================================

//
// reading functions
//

// returns -1 if no more characters are available
int HuffmanPacket::MSG_ReadChar (msg_t *msg ) {
    int	c;

    c = (signed char)MSG_ReadBits( msg, 8 );
    if ( msg->readcount > msg->cursize ) {
	c = -1;
    }	

    return c;
}

int HuffmanPacket::MSG_ReadByte( msg_t *msg ) {
    int	c;

    c = (unsigned char)MSG_ReadBits( msg, 8 );
    if ( msg->readcount > msg->cursize ) {
	c = -1;
    }	
    return c;
}

int HuffmanPacket::MSG_ReadShort( msg_t *msg ) {
    int	c;

    c = (short)MSG_ReadBits( msg, 16 );
    if ( msg->readcount > msg->cursize ) {
	c = -1;
    }	

    return c;
}

int HuffmanPacket::MSG_ReadLong( msg_t *msg ) {
    int	c;

    c = MSG_ReadBits( msg, 32 );
    if ( msg->readcount > msg->cursize ) {
	c = -1;
    }	

    return c;
}

float HuffmanPacket::MSG_ReadFloat( msg_t *msg ) {
    union {
	byte	b[4];
	float	f;
	int	l;
    } dat;

    dat.l = MSG_ReadBits( msg, 32 );
    if ( msg->readcount > msg->cursize ) {
	dat.f = -1;
    }	

    return dat.f;	
}

void HuffmanPacket::MSG_ReadData( msg_t *msg, void *data, int len ) {
    int		i;

    for (i=0 ; i<len ; i++) {
	((byte *)data)[i] = MSG_ReadByte (msg);
    }
}

///////////////////////////////////////////////////////////////////////////////

/*
static int msg_hData[256] = {
    250315,			// 0
    41193,			// 1
    6292,			// 2
    7106,			// 3
    3730,			// 4
    3750,			// 5
    6110,			// 6
    23283,			// 7
    33317,			// 8
    6950,			// 9
    7838,			// 10
    9714,			// 11
    9257,			// 12
    17259,			// 13
    3949,			// 14
    1778,			// 15
    8288,			// 16
    1604,			// 17
    1590,			// 18
    1663,			// 19
    1100,			// 20
    1213,			// 21
    1238,			// 22
    1134,			// 23
    1749,			// 24
    1059,			// 25
    1246,			// 26
    1149,			// 27
    1273,			// 28
    4486,			// 29
    2805,			// 30
    3472,			// 31
    21819,			// 32
    1159,			// 33
    1670,			// 34
    1066,			// 35
    1043,			// 36
    1012,			// 37
    1053,			// 38
    1070,			// 39
    1726,			// 40
    888,			// 41
    1180,			// 42
    850,			// 43
    960,			// 44
    780,			// 45
    1752,			// 46
    3296,			// 47
    10630,			// 48
    4514,			// 49
    5881,			// 50
    2685,			// 51
    4650,			// 52
    3837,			// 53
    2093,			// 54
    1867,			// 55
    2584,			// 56
    1949,			// 57
    1972,			// 58
    940,			// 59
    1134,			// 60
    1788,			// 61
    1670,			// 62
    1206,			// 63
    5719,			// 64
    6128,			// 65
    7222,			// 66
    6654,			// 67
    3710,			// 68
    3795,			// 69
    1492,			// 70
    1524,			// 71
    2215,			// 72
    1140,			// 73
    1355,			// 74
    971,			// 75
    2180,			// 76
    1248,			// 77
    1328,			// 78
    1195,			// 79
    1770,			// 80
    1078,			// 81
    1264,			// 82
    1266,			// 83
    1168,			// 84
    965,			// 85
    1155,			// 86
    1186,			// 87
    1347,			// 88
    1228,			// 89
    1529,			// 90
    1600,			// 91
    2617,			// 92
    2048,			// 93
    2546,			// 94
    3275,			// 95
    2410,			// 96
    3585,			// 97
    2504,			// 98
    2800,			// 99
    2675,			// 100
    6146,			// 101
    3663,			// 102
    2840,			// 103
    14253,			// 104
    3164,			// 105
    2221,			// 106
    1687,			// 107
    3208,			// 108
    2739,			// 109
    3512,			// 110
    4796,			// 111
    4091,			// 112
    3515,			// 113
    5288,			// 114
    4016,			// 115
    7937,			// 116
    6031,			// 117
    5360,			// 118
    3924,			// 119
    4892,			// 120
    3743,			// 121
    4566,			// 122
    4807,			// 123
    5852,			// 124
    6400,			// 125
    6225,			// 126
    8291,			// 127
    23243,			// 128
    7838,			// 129
    7073,			// 130
    8935,			// 131
    5437,			// 132
    4483,			// 133
    3641,			// 134
    5256,			// 135
    5312,			// 136
    5328,			// 137
    5370,			// 138
    3492,			// 139
    2458,			// 140
    1694,			// 141
    1821,			// 142
    2121,			// 143
    1916,			// 144
    1149,			// 145
    1516,			// 146
    1367,			// 147
    1236,			// 148
    1029,			// 149
    1258,			// 150
    1104,			// 151
    1245,			// 152
    1006,			// 153
    1149,			// 154
    1025,			// 155
    1241,			// 156
    952,			// 157
    1287,			// 158
    997,			// 159
    1713,			// 160
    1009,			// 161
    1187,			// 162
    879,			// 163
    1099,			// 164
    929,			// 165
    1078,			// 166
    951,			// 167
    1656,			// 168
    930,			// 169
    1153,			// 170
    1030,			// 171
    1262,			// 172
    1062,			// 173
    1214,			// 174
    1060,			// 175
    1621,			// 176
    930,			// 177
    1106,			// 178
    912,			// 179
    1034,			// 180
    892,			// 181
    1158,			// 182
    990,			// 183
    1175,			// 184
    850,			// 185
    1121,			// 186
    903,			// 187
    1087,			// 188
    920,			// 189
    1144,			// 190
    1056,			// 191
    3462,			// 192
    2240,			// 193
    4397,			// 194
    12136,			// 195
    7758,			// 196
    1345,			// 197
    1307,			// 198
    3278,			// 199
    1950,			// 200
    886,			// 201
    1023,			// 202
    1112,			// 203
    1077,			// 204
    1042,			// 205
    1061,			// 206
    1071,			// 207
    1484,			// 208
    1001,			// 209
    1096,			// 210
    915,			// 211
    1052,			// 212
    995,			// 213
    1070,			// 214
    876,			// 215
    1111,			// 216
    851,			// 217
    1059,			// 218
    805,			// 219
    1112,			// 220
    923,			// 221
    1103,			// 222
    817,			// 223
    1899,			// 224
    1872,			// 225
    976,			// 226
    841,			// 227
    1127,			// 228
    956,			// 229
    1159,			// 230
    950,			// 231
    7791,			// 232
    954,			// 233
    1289,			// 234
    933,			// 235
    1127,			// 236
    3207,			// 237
    1020,			// 238
    927,			// 239
    1355,			// 240
    768,			// 241
    1040,			// 242
    745,			// 243
    952,			// 244
    805,			// 245
    1073,			// 246
    740,			// 247
    1013,			// 248
    805,			// 249
    1008,			// 250
    796,			// 251
    996,			// 252
    1057,			// 253
    11457,			// 254
    13504,			// 255
};

*/

// Frequency counts generated using MsgAppUpdate dumps...
//
static int msg_hData [256] = {
	         250315,     //   0
	           8638,     //   1
	          10509,     //   2
	          10652,     //   3
	           4284,     //   4
	           1065,     //   5
	           2470,     //   6
	           7120,     //   7
	            934,     //   8
	            589,     //   9
	            657,     //  10
	            764,     //  11
	            644,     //  12
	            560,     //  13
	            811,     //  14
	            800,     //  15
	           1261,     //  16
	            692,     //  17
	            789,     //  18
	            766,     //  19
	            885,     //  20
	            639,     //  21
	            722,     //  22
	            852,     //  23
	           5767,     //  24
	            540,     //  25
	            417,     //  26
	            443,     //  27
	            621,     //  28
	            302,     //  29
	            405,     //  30
	            326,     //  31
	           5004,     //  32
	            524,     //  33
	            415,     //  34
	            372,     //  35
	            544,     //  36
	            395,     //  37
	            581,     //  38
	            377,     //  39
	            625,     //  40
	            371,     //  41
	            437,     //  42
	            356,     //  43
	            584,     //  44
	            361,     //  45
	            620,     //  46
	           1240,     //  47
	           1144,     //  48
	            574,     //  49
	           1173,     //  50
	            524,     //  51
	            562,     //  52
	            529,     //  53
	            480,     //  54
	            418,     //  55
	            673,     //  56
	            364,     //  57
	            444,     //  58
	            359,     //  59
	            486,     //  60
	            354,     //  61
	            533,     //  62
	            986,     //  63
	           2464,     //  64
	           1724,     //  65
	           1889,     //  66
	           8542,     //  67
	           2375,     //  68
	           3208,     //  69
	            361,     //  70
	            279,     //  71
	            428,     //  72
	            296,     //  73
	            363,     //  74
	            299,     //  75
	            357,     //  76
	            246,     //  77
	            337,     //  78
	            310,     //  79
	            618,     //  80
	            277,     //  81
	            391,     //  82
	            250,     //  83
	            460,     //  84
	            270,     //  85
	            329,     //  86
	            265,     //  87
	            593,     //  88
	            283,     //  89
	            312,     //  90
	            319,     //  91
	            829,     //  92
	            305,     //  93
	            405,     //  94
	            282,     //  95
	            856,     //  96
	           4733,     //  97
	            363,     //  98
	            381,     //  99
	            739,     // 100
	            406,     // 101
	            453,     // 102
	            368,     // 103
	            446,     // 104
	            271,     // 105
	            316,     // 106
	            272,     // 107
	            470,     // 108
	            279,     // 109
	            323,     // 110
	            361,     // 111
	            859,     // 112
	            311,     // 113
	            423,     // 114
	            352,     // 115
	            528,     // 116
	            276,     // 117
	            348,     // 118
	            325,     // 119
	            543,     // 120
	            276,     // 121
	            321,     // 122
	            264,     // 123
	            357,     // 124
	            260,     // 125
	            381,     // 126
	            551,     // 127
	           7358,     // 128
	            237,     // 129
	            396,     // 130
	            350,     // 131
	            498,     // 132
	            282,     // 133
	            375,     // 134
	            707,     // 135
	            423,     // 136
	            255,     // 137
	            424,     // 138
	            397,     // 139
	            380,     // 140
	            248,     // 141
	            374,     // 142
	            374,     // 143
	            801,     // 144
	            247,     // 145
	            344,     // 146
	            319,     // 147
	            378,     // 148
	            256,     // 149
	            452,     // 150
	            342,     // 151
	            443,     // 152
	            298,     // 153
	            412,     // 154
	            304,     // 155
	            442,     // 156
	            340,     // 157
	            541,     // 158
	            290,     // 159
	           1087,     // 160
	            216,     // 161
	            290,     // 162
	            221,     // 163
	            312,     // 164
	            209,     // 165
	            296,     // 166
	            202,     // 167
	           4775,     // 168
	            208,     // 169
	            287,     // 170
	            207,     // 171
	            309,     // 172
	            206,     // 173
	            292,     // 174
	            214,     // 175
	            516,     // 176
	            207,     // 177
	            284,     // 178
	            207,     // 179
	            348,     // 180
	            449,     // 181
	            349,     // 182
	            194,     // 183
	            362,     // 184
	            207,     // 185
	            286,     // 186
	            222,     // 187
	            418,     // 188
	            228,     // 189
	            376,     // 190
	            510,     // 191
	           1669,     // 192
	           1197,     // 193
	           1597,     // 194
	           2335,     // 195
	           2310,     // 196
	           3845,     // 197
	            296,     // 198
	            213,     // 199
	            456,     // 200
	            213,     // 201
	            300,     // 202
	            196,     // 203
	            304,     // 204
	            257,     // 205
	            283,     // 206
	            192,     // 207
	            487,     // 208
	            182,     // 209
	            284,     // 210
	            189,     // 211
	            284,     // 212
	           4602,     // 213
	            299,     // 214
	            286,     // 215
	            338,     // 216
	            185,     // 217
	            269,     // 218
	            190,     // 219
	            301,     // 220
	            187,     // 221
	            278,     // 222
	            220,     // 223
	           2177,     // 224
	            200,     // 225
	            324,     // 226
	            208,     // 227
	            298,     // 228
	            195,     // 229
	            279,     // 230
	            193,     // 231
	            359,     // 232
	            196,     // 233
	            271,     // 234
	            193,     // 235
	            319,     // 236
	            195,     // 237
	            357,     // 238
	            214,     // 239
	            591,     // 240
	            203,     // 241
	            315,     // 242
	            221,     // 243
	            651,     // 244
	            217,     // 245
	            322,     // 246
	            205,     // 247
	           1049,     // 248
	            872,     // 249
	           1129,     // 250
	            981,     // 251
	           1066,     // 252
	           1041,     // 253
	            539,     // 254
	          29487,     // 255
};
void HuffmanPacket::MSG_initHuffman() {
    int i,j;

    msgInit = true;
    Huff_Init(&msgHuff);
    for(i=0;i<256;i++) {
	for (j=0;j<msg_hData[i];j++) {
	    Huff_addRef(&msgHuff.compressor,	(byte)i);			// Do update
	    Huff_addRef(&msgHuff.decompressor,	(byte)i);			// Do update
	}
    }
}

///////////////////////////////////////////////////////////////////////////////

HuffmanPacket::HuffmanPacket() : 
oldsize(0), overflows(0), ispeek(false), Packet(NULL, true)
{
    MSG_Init(&sb, new byte[DEFAULT_PACKET_SIZE], DEFAULT_PACKET_SIZE);
    // assume we are just writing
}

HuffmanPacket::HuffmanPacket(int size) : 
oldsize(0), overflows(0), ispeek(false), Packet(NULL, true)
{
    MSG_Init(&sb, new byte[size], size);
    // assume we are just writing
}

HuffmanPacket::HuffmanPacket(Packet *pkt) : 
oldsize(0), overflows(0), ispeek(false), Packet(NULL, true)
{
    uint16 size = pkt->ReadShort();
    MSG_Init(&sb, new byte[size], size);
    pkt->ReadBuffer(sb.data, size);
    sb.cursize = size;
    sb.maxsize = size;
    //MSG_BeginReading(&sb);
    //MSG_Bitstream(&sb);
    // assume we are just reading
}

HuffmanPacket::HuffmanPacket(const HuffmanPacket& other) : Packet(NULL, true)
{
    sb = other.sb;
    sb.data = new byte[other.sb.maxsize];
    memcpy(sb.data, other.sb.data, other.sb.maxsize);
    oldsize = other.oldsize;
    overflows = other.overflows;
    peek = other.peek;
    ispeek = other.ispeek;
}

HuffmanPacket::~HuffmanPacket()
{
    delete[] sb.data;
}

void HuffmanPacket::Serialize(Packet *pkt)
{
    ASSERT(sb.cursize < 0xFFFF);
    pkt->WriteShort(sb.cursize);
    pkt->WriteBuffer(sb.data, sb.cursize);
}

uint32 HuffmanPacket::GetLength()
{
    return sb.cursize;
}

uint32 HuffmanPacket::GetSize() { 
    return sb.cursize; 
}

void HuffmanPacket::WriteBuffer(byte *buf, int len)
{
    MSG_WriteData(&sb, buf, len);
}

void HuffmanPacket::ReadBuffer(byte *buf, int len)
{
    MSG_ReadData(&sb, buf, len);
}

void HuffmanPacket::WriteByte(byte b)
{
    MSG_WriteByte(&sb, b);
}

byte HuffmanPacket::ReadByte()
{
    if (ispeek) {
	ispeek = false;
	return peek;
    }
    return MSG_ReadByte(&sb);
}

byte HuffmanPacket::PeekByte()
{
    if (ispeek)
	return peek;
    peek = MSG_ReadByte(&sb);
    ispeek = true;
    return peek;
}

void HuffmanPacket::WriteBool (bool b)
{
    MSG_WriteBits(&sb, (int)b, 1);
}

bool HuffmanPacket::ReadBool ()
{
    return (bool)MSG_ReadBits(&sb, 1);
}

void HuffmanPacket::WriteShort(uint16 s)
{
    MSG_WriteShort (&sb, (int)s);
}

uint16 HuffmanPacket::ReadShort()
{
    return (uint16)MSG_ReadShort (&sb);
}

void HuffmanPacket::WriteInt(uint32 u)
{
    union {
	int i;
	uint32 u;
    } dat;

    dat.u = u;

    MSG_WriteLong(&sb, dat.i);
}

uint32 HuffmanPacket::ReadInt()
{
    union {
	int i;
	uint32 u;
    } dat;

    dat.i = MSG_ReadLong(&sb);
    return dat.u;
}

void HuffmanPacket::WriteIntNoSwap(uint32 i)
{
    WriteBuffer((byte *)&i, sizeof(uint32));
}

uint32 HuffmanPacket::ReadIntNoSwap()
{
    uint32 ret;
    ReadBuffer((byte *)&ret, sizeof(uint32));
    return ret;
}

void HuffmanPacket::WriteFloat(float f)
{
    MSG_WriteFloat(&sb, f);
}

float HuffmanPacket::ReadFloat()
{
    return MSG_ReadFloat(&sb);
}

void HuffmanPacket::WriteString(string& s)
{
    MSG_WriteLong(&sb, s.length());
    WriteBuffer((byte *) s.c_str(), s.length());
}

void HuffmanPacket::ReadString(string& s)
{
    int strLen = MSG_ReadLong(&sb);
    // xxx how do we make this more efficient???
    byte *buf = new byte[strLen];
    ReadBuffer(buf,  strLen);
    s = string((char *)buf, strLen);
    delete[] buf;
}
